/**
 * @file lcdprof.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "lcdprof.h"
#include "util.h"

#define CHAR_DEF    '='

static void print_value(LCDPROF *prof)
{
    printf("CHCNT_X=%d\n", prof->chcnt_x);
    printf("CHCNT_Y=%d\n", prof->chcnt_y);
    printf("CHOFS_X=%d\n", prof->chofs_x);
    printf("CHOFS_Y=%d\n", prof->chofs_y);
    printf("CHGAP_X=%d\n", prof->chgap_x);
    printf("CHGAP_Y=%d\n", prof->chgap_y);
    printf("PIXSIZ_X=%d\n", prof->pixsiz_x);
    printf("PIXSIZ_Y=%d\n", prof->pixsiz_y);
    printf("PIXGAP_X=%d\n", prof->pixgap_x);
    printf("PIXGAP_Y=%d\n", prof->pixgap_y);
    printf("FGCOL=%3d, %3d, %3d\n", prof->fgcol.r, prof->fgcol.g, prof->fgcol.b);
    printf("BGCOL=%3d, %3d, %3d\n", prof->bgcol.r, prof->bgcol.g, prof->bgcol.b);
    printf("CONTRAST=%d\n", prof->contrast);
}

static int pickup_value(LCDPROF *prof, const char *text)
{
    char key[BUFSIZ];
    char value[BUFSIZ];
    char tmp[BUFSIZ];
    int i;

    /*
     * Copy the master data to a temporary variable.
     */
    strcpy(tmp, text);

    /*
     * Separate the key and value.
     */
    for (i = 0; i < (int)strlen(tmp); i++) {
        if (tmp[i] == CHAR_DEF) {
            tmp[i] = ' ';
        }
    }

    /*
     * Pickup the value.
     */
    if (sscanf(tmp, "%s %s", key, value) == 2) {
        if (strcmp("CHCNT_X", key) == 0) { prof->chcnt_x = atoi(value); }
        if (strcmp("CHCNT_Y", key) == 0) { prof->chcnt_y = atoi(value); }
        if (strcmp("CHOFS_X", key) == 0) { prof->chofs_x = atoi(value); }
        if (strcmp("CHOFS_Y", key) == 0) { prof->chofs_y = atoi(value); }
        if (strcmp("CHGAP_X", key) == 0) { prof->chgap_x = atoi(value); }
        if (strcmp("CHGAP_Y", key) == 0) { prof->chgap_y = atoi(value); }
        if (strcmp("PIXSIZ_X", key) == 0) { prof->pixsiz_x = atoi(value); }
        if (strcmp("PIXSIZ_Y", key) == 0) { prof->pixsiz_y = atoi(value); }
        if (strcmp("PIXGAP_X", key) == 0) { prof->pixgap_x = atoi(value); }
        if (strcmp("PIXGAP_Y", key) == 0) { prof->pixgap_y = atoi(value); }
        if (strcmp("FGCOL.R", key) == 0) { prof->fgcol.r = atoi(value); }
        if (strcmp("FGCOL.G", key) == 0) { prof->fgcol.g = atoi(value); }
        if (strcmp("FGCOL.B", key) == 0) { prof->fgcol.b = atoi(value); }
        if (strcmp("BGCOL.R", key) == 0) { prof->bgcol.r = atoi(value); }
        if (strcmp("BGCOL.G", key) == 0) { prof->bgcol.g = atoi(value); }
        if (strcmp("BGCOL.B", key) == 0) { prof->bgcol.b = atoi(value); }
        if (strcmp("CONTRAST", key) == 0) { prof->contrast = atoi(value); }
        return 0;
    }

    return -1;
}

int lcdprof_read(const char *filename, LCDPROF *prof, int verbose)
{
    char buf[BUFSIZ];
    FILE *fp;

    /*
     * Check the file name.
     */
    if (filename == NULL) {
        return -1;
    }
    if (strlen(filename) == 0) {
        return -1;
    }

    /*
     * Open the target file.
     */
    fp = fopen(filename, "r");
    if (fp == NULL) {
        return -1;
    }

    /*
     * Read the configuration values.
     */
    while (fgets(buf, sizeof(buf), fp) != NULL) {
        util_remove_comment(buf);
        pickup_value(prof, buf);
    }
    fclose(fp);

    /*
     * Print the configuration values if user need it.
     */
    if (verbose) {
        print_value(prof);
    }

    return 0;
}

