/**
 * @file util.c
 * @author Copyright(C) 2012 Shinichiro Nakamura
 */

/*
 * ===============================================================
 *  LCD Image Tool
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include <string.h>
#include <ctype.h>
#include "util.h"

#define C_ESC   '\\'
#define C_END   '\0'
#define C_HEX   'x'

int util_remove_comment(char *text)
{
    int i;
    int n;
    n = strlen(text);
    for (i = 0; i < n; i++) {
        if (text[i] == '#') {
            text[i] = '\0';
        }
    }
    return 0;
}

static int get_hexval(char c)
{
    if (('0' <= c) && (c <= '9')) {
        return 0 + (c - '0');
    }
    if (('A' <= c) && (c <= 'F')) {
        return 10 + (c - 'A');
    }
    if (('a' <= c) && (c <= 'f')) {
        return 10 + (c - 'a');
    }
    return -1;
}

int util_format(const char *text, char *buf, size_t siz, int *bytes)
{
    int retval = 0;
    int count = 0;
    char *src = (char *)text;
    char *des = buf;
    *bytes = 0;
    while (*src) {
        if (*src == C_ESC) {
            src++;
            if (*src == C_END) {
                retval = 1;
                break;
            } else if (*src == C_ESC) {
                src++;
                *des++ = C_ESC;
                count++;
            } else if (*src == C_HEX) {
                char tmp = 0;
                char c = 0;
                src++;
                if (*src == C_END) {
                    retval = 2;
                    break;
                }
                if (!isxdigit(*src)) {
                    retval = 3;
                    break;
                }
                tmp = get_hexval(*src++);
                if (tmp < 0) {
                    retval = 4;
                    break;
                }
                c += tmp * 0x10;
                if (*src == C_END) {
                    retval = 5;
                    break;
                }
                if (!isxdigit(*src)) {
                    retval = 6;
                    break;
                }
                tmp = get_hexval(*src++);
                if (tmp < 0) {
                    retval = 7;
                    break;
                }
                c += tmp * 0x01;
                *des++ = c;
                count++;
            } else {
                retval = 8;
                break;
            }
        } else {
            *des++ = *src++;
            count++;
        }
    }
    *des = '\0';
    *bytes = count;
    return retval;
}

