package aipo.webservice.util;

/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

import java.util.Calendar;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;

import com.aimluck.eip.cayenne.om.portlet.EipTScheduleMap;
import com.aimluck.eip.schedule.util.ScheduleUtils;

/**
 * <HR>
 * リマインダー用スケジュール情報ユーティリティクラス
 * <p>
 * 
 * Webサービスのスケジュール情報ユーティリティ
 * <P>
 * <HR>
 * <P>
 * 
 */
public class WsScheduleUtils {

  /** スケジュール繰り返しパターン：通常 */
  private static final String EIP_T_SCHEDULE_REPEAT_PATTERN_NORMAL = "N";

  /** スケジュール繰り返しパターン：期間 */
  private static final String EIP_T_SCHEDULE_REPEAT_PATTERN_SPAN = "S";

  /** スケジュール繰り返しパターン：日またぎ */
  private static final String EIP_T_SCHEDULE_REPEAT_PATTERN_ACROSS = "Z";

  /** スケジュールマップの時刻昇順による順序付け */
  private static final Comparator<EipTScheduleMap> eipTScheduleMapComparatorByTimeAsc =
    new EipTScheduleMapComparatorByTimeAsc();

  /** スケジュールマップの時刻降順による順序付け */
  private static final Comparator<EipTScheduleMap> eipTScheduleMapComparatorByTimeDesc =
    new EipTScheduleMapComparatorByTimeDesc();

  /**
   * スケジュール範囲内判定
   * <p>
   * 繰り返し予定が、判定日時～判定日時+確認間隔の範囲内かチェックする。<br/>
   * 通常、日またぎ予定は一律対象とする。<br>
   * また、期間のみのスケジュールは対象外とする。<br>
   * 
   * @param checkDate
   *          チェック対象日時
   * @param scPtn
   *          スケジュールの繰り返しパターン
   * @param noticeTiming
   *          確認間隔（n分前）
   * @param scStartDate
   *          スケジュールの開始日時
   * @param scEndDate
   *          スケジュールの終了日時
   * @return boolean 表示対象の場合true、表示対象外の場合false
   */
  public static boolean isReminderRange(Date checkDate, String scPtn,
      int noticeTiming, Date scStartDate, Date scEndDate) throws Exception {

    boolean result = false;

    if (scPtn.startsWith(EIP_T_SCHEDULE_REPEAT_PATTERN_NORMAL)
      || scPtn.startsWith(EIP_T_SCHEDULE_REPEAT_PATTERN_ACROSS)) {
      return true;
    }

    if (scPtn.startsWith(EIP_T_SCHEDULE_REPEAT_PATTERN_SPAN)) {
      return false;
    }

    // 判定終了日時を作成
    Calendar checkEndDateCal = Calendar.getInstance();
    checkEndDateCal.setTime(checkDate);
    checkEndDateCal.add(Calendar.MINUTE, noticeTiming);

    // 判定日時範囲外チェック
    result =
      ScheduleUtils.isView(
        checkDate,
        checkEndDateCal.getTime(),
        scPtn,
        scStartDate,
        scEndDate);

    return result;
  }

  /**
   * スケジュールマップの時刻ソート
   * <p>
   * スケジュールマップを時刻でソートする。<br>
   * 昇順/降順をパラメータによって判定し、呼び出すクラスを決定する。<br>
   * 
   * @param list
   *          スケジュールマップ一覧
   * @param isAsc
   *          trueの場合：昇順ソート、falseの場合：降順ソート
   */
  public static void sortByTime(List<EipTScheduleMap> list, boolean isAsc) {
    Comparator<EipTScheduleMap> comparator = null;
    if (isAsc) {
      comparator = eipTScheduleMapComparatorByTimeAsc;
    } else {
      comparator = eipTScheduleMapComparatorByTimeDesc;
    }
    Collections.sort(list, comparator);
  }

}
