/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * http://aipostyle.com/
 *
 * Copyright(C) 2011 avanza Co.,Ltd. All rights reserved.
 * http://www.avnz.co.jp/
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.addressbook;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.eip.addressbookuser.util.AddressBookUserUtils;
import com.aimluck.eip.cayenne.om.portlet.EipMAddressGroup;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.ResultList;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.util.ALEipUtils;

/**
 * 携帯用の会社情報一覧取得用クラスです
 */
public class CellAddressBookGroupSelectData extends AddressBookGroupSelectData {
  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(CellAddressBookGroupSelectData.class
      .getName());

  /** ログインユーザーID */
  private int loginuserid;

  /**
   * 初期処理
   * 
   * @param action
   *          アクションクラス
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @throws ALPageNotFoundException
   * @throws ALDBErrorException
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {
    loginuserid = ALEipUtils.getUserId(rundata);
    super.init(action, rundata, context);
  }

  /**
   * 会社情報グループの一覧取得処理
   * 
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @return 会社情報リスト
   */
  @Override
  protected ResultList<EipMAddressGroup> selectList(RunData rundata,
      Context context) {
    try {
      SelectQuery<EipMAddressGroup> query = getSelectQuery(rundata, context);
      buildSelectQueryForListView(query);
      buildSelectQueryForListViewSort(query, rundata, context);
      return query.getResultList();
    } catch (Exception ex) {
      String msg =
        "会社情報の一覧取得に失敗しました。[" + ALEipUtils.getUserLoginName(loginuserid) + "]";
      logger.error(msg, ex);
      return null;
    }
  }

  /**
   * ResultData に値を格納して返します。（一覧データ） <BR>
   * 
   * @param obj
   *          DBから取得し個人アドレスグループ情報
   * @return 画面表示用データ
   * @see com.aimluck.eip.common.ALAbstractSelectData#getListData(java.lang.Object)
   */
  @Override
  protected Object getResultData(EipMAddressGroup record) {
    try {
      AddressBookGroupResultData rd = new AddressBookGroupResultData();
      rd.initField();
      if (record.getOwnerId().intValue() == 1) {
        EipMAddressGroup defaultGroup =
          AddressBookUserUtils.getDefaultEipMAddressGroup();
        rd.setGroupId(defaultGroup.getGroupId().intValue());
        rd.setGroupName(AddressBookUserUtils.EMPTY_COMPANY_NAME);
      } else {
        rd.setGroupId(record.getGroupId().intValue());
        rd.setGroupName(record.getGroupName());
      }
      return rd;
    } catch (Exception ex) {
      String msg =
        "会社情報の一覧取得に失敗しました。[" + ALEipUtils.getUserLoginName(loginuserid) + "]";
      logger.error(msg, ex);
      return null;
    }
  }

  /**
   * 検索条件を設定した SelectQuery を返します。 <BR>
   * 
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @return SQLオブジェクト
   */
  protected SelectQuery<EipMAddressGroup> getSelectQuery(RunData rundata,
      Context context) {
    SelectQuery<EipMAddressGroup> query =
      Database.query(EipMAddressGroup.class);
    Expression exp1 =
      ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, Integer
        .valueOf(ALEipUtils.getUserId(rundata)));
    Expression exp2 =
      ExpressionFactory.matchExp(EipMAddressGroup.OWNER_ID_PROPERTY, 1);
    query.setQualifier(exp1.orExp(exp2));
    query.orderAscending(EipMAddressGroup.GROUP_NAME_PROPERTY);
    return query;
  }
}
