/*
 * Aipo is a groupware program developed by Aimluck,Inc.
 * Copyright (C) 2004-2011 Aimluck,Inc.
 * http://www.aipo.com
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package com.aimluck.eip.webmail;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import org.apache.cayenne.exp.Expression;
import org.apache.cayenne.exp.ExpressionFactory;
import org.apache.jetspeed.services.logging.JetspeedLogFactoryService;
import org.apache.jetspeed.services.logging.JetspeedLogger;
import org.apache.turbine.util.RunData;
import org.apache.velocity.context.Context;

import com.aimluck.commons.field.ALNumberField;
import com.aimluck.commons.field.ALStringField;
import com.aimluck.eip.cayenne.om.portlet.AvzTMailSend;
import com.aimluck.eip.cayenne.om.portlet.EipMMailAccount;
import com.aimluck.eip.cayenne.om.portlet.EipTMail;
import com.aimluck.eip.cayenne.om.portlet.EipTMailFilter;
import com.aimluck.eip.cayenne.om.portlet.EipTMailFolder;
import com.aimluck.eip.common.ALAbstractFormData;
import com.aimluck.eip.common.ALDBErrorException;
import com.aimluck.eip.common.ALEipConstants;
import com.aimluck.eip.common.ALEipUser;
import com.aimluck.eip.common.ALPageNotFoundException;
import com.aimluck.eip.mail.ALFolder;
import com.aimluck.eip.mail.ALMailFactoryService;
import com.aimluck.eip.mail.ALMailHandler;
import com.aimluck.eip.mail.util.ALMailUtils;
import com.aimluck.eip.modules.actions.common.ALAction;
import com.aimluck.eip.orm.Database;
import com.aimluck.eip.orm.query.SelectQuery;
import com.aimluck.eip.services.eventlog.ALEventlogConstants;
import com.aimluck.eip.services.eventlog.ALEventlogFactoryService;
import com.aimluck.eip.services.storage.ALStorageService;
import com.aimluck.eip.util.ALEipUtils;
import com.aimluck.eip.webmail.util.WebMailUtils;

/**
 * 共有フォルダのフォルダフォームデータを管理するクラス <BR>
 * 
 */
public class WebMailFolderFormData extends ALAbstractFormData {

  /** logger */
  private static final JetspeedLogger logger =
    JetspeedLogFactoryService.getLogger(WebMailFolderFormData.class.getName());

  /** フォルダ名 */
  private ALStringField folder_name;

  /** アクセス権限フラグ */
  private ALNumberField access_flag;

  /** ログインユーザー */
  private ALEipUser login_user;

  /** メールアカウント */
  private EipMMailAccount mailAccount;

  private String folderId = null;

  // add by motegi start
  /** アカウントに紐付くフォルダリスト */
  // change start 受入テスト障害 190
  // private List<WebMailFolderResultData> folderList;
  private List<ExtFolderInfo> folderList;

  // change end 受入テスト障害 190
  /** 親フォルダ ID */
  private ALNumberField parent_folder_id;

  /** ログインユーザーID */
  private int user_id;

  //
  /**
   * 初期処理
   * 
   * @param action
   *          アクションクラス
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @see com.aimluck.eip.common.ALAbstractFormData#init(com.aimluck.eip.modules.actions.common.ALAction,
   *      org.apache.turbine.util.RunData, org.apache.velocity.context.Context)
   * 
   */
  @Override
  public void init(ALAction action, RunData rundata, Context context)
      throws ALPageNotFoundException, ALDBErrorException {

    super.init(action, rundata, context);

    int mailAccountId = 0;

    // 自ポートレットからのリクエストであれば、パラメータを展開しセッションに保存する。
    if (ALEipUtils.isMatch(rundata, context)) {
      try {
        // セッションからアカウントIDを取得する。
        mailAccountId =
          Integer.parseInt(ALEipUtils.getTemp(
            rundata,
            context,
            WebMailUtils.ACCOUNT_ID));
      } catch (Exception e) {
        return;
      }

      if (rundata.getParameters().containsKey(ALEipConstants.ENTITY_ID)) {
        String entityId =
          rundata.getParameters().getString(ALEipConstants.ENTITY_ID);
        if (!"new".equals(entityId)) {
          ALEipUtils.setTemp(
            rundata,
            context,
            ALEipConstants.ENTITY_ID,
            entityId);
          folderId = entityId;
        }
      }
    }

    login_user = ALEipUtils.getALEipUser(rundata);
    // add start
    user_id = ALEipUtils.getUserId(rundata);
    // add end

    // メールアカウントを取得する
    mailAccount =
      ALMailUtils.getMailAccount(
        (int) login_user.getUserId().getValue(),
        mailAccountId);
    if (mailAccount == null) {
      return;
    }

    if (folderId != null) {
      // 指定されたフォルダがアカウントのものかどうかチェックする
      EipTMailFolder folder =
        WebMailUtils.getEipTMailFolder(mailAccount, folderId);
      if (folder == null) {
        logger.error("[WebMail Folder] mail folder was not found.");
        return;
      }
    }

  }

  /**
   * 画面表示項目を設定します <BR>
   * 
   * @param rundata
   *          JetSpeedランデータ
   * @param context
   *          JetSpeedコンテキスト
   * @param msgList
   *          エラーメッセージリスト
   * @return true:処理成功 false:処理失敗
   */
  @Override
  protected boolean setFormData(RunData rundata, Context context,
      List<String> msgList) throws ALPageNotFoundException, ALDBErrorException {

    boolean res = super.setFormData(rundata, context, msgList);

    if (res) {
      // if (ALEipConstants.MODE_UPDATE.equals(getMode())) {
      // folderId = ALEipUtils.getTemp(rundata, context,
      // WebMailUtils.FOLDER_ID);
      // }

      // add start
      // フォルダ一覧を取得して設定する
      try {
        if (ALEipConstants.MODE_NEW_FORM.equals(getMode())) {
          // loadMailFolderList(ALMailUtils.getEipTMailFolderAll(mailAccount));
          loadMailFolderList(rundata, context, null);
        }
      } catch (Exception e) {
        throw new ALDBErrorException();
      }
      // add end
    }
    return res;
  }

  /**
   * 画面フィールド初期化
   * 
   * @see com.aimluck.eip.common.ALData#initField()
   */
  @Override
  public void initField() {
    // フォルダ名
    folder_name = new ALStringField();
    folder_name.setFieldName("フォルダ名");
    folder_name.setTrim(true);

    // add start
    // 親フォルダ
    parent_folder_id = new ALNumberField();
    parent_folder_id.setFieldName("親フォルダ");
    // parent_folder_id.setValue(0);

    // add emd
  }

  /**
   * バリデータ初期化
   * 
   * @see com.aimluck.eip.common.ALAbstractFormData#setValidator()
   */
  @Override
  protected void setValidator() {
    // フォルダ名必須項目
    folder_name.setNotNull(true);
    // フォルダ名の文字数制限
    folder_name.limitMaxLength(128);

    // add start
    // 親フォルダ必須項目
    parent_folder_id.setNotNull(true);
    // add end
  }

  /**
   * バリデーション
   * 
   * @param msgList
   *          エラーメッセージリスト
   * @return TRUE 成功 FALSE 失敗
   * @see com.aimluck.eip.common.ALAbstractFormData#validate(java.util.ArrayList)
   */
  @Override
  protected boolean validate(List<String> msgList) {
    // フォルダ名
    folder_name.validate(msgList);

    // add start
    // 親フォルダ
    parent_folder_id.validate(msgList);
    // add end

    // 同じフォルダ名が無いかどうか確かめる
    if (existsFolderName(folder_name.getValue(), ALEipConstants.MODE_UPDATE
      .equals(getMode()))) {
      msgList.add("このフォルダ名と同じフォルダがすでに存在するため、登録できません。フォルダ名を変更してください。");
    }

    // add start 2012.1.21 受入障害対応No.228 フォルダ名禁則文字チェック
    if (folder_name.getValue().matches(".*[\t:\"|/*<>?\\\\]+.*")) {
      msgList.add("フォルダ名には次の文字は使えません。\t : \" | / * < > ? \\ ");
    }
    // add end

    return (msgList.size() == 0);
  }

  /**
   * 同じアカウントに同じ名前のフォルダがあるかどうか調べます。
   * 
   * @param fname
   *          フォルダ名
   * @param is_update
   *          更新時呼出の場合true
   * @return true:同じフォルダあり false:なし
   */
  private boolean existsFolderName(String fname, boolean is_update) {
    if (fname == null || "".equals(fname)) {
      return false;
    }

    try {
      SelectQuery<EipTMailFolder> query = Database.query(EipTMailFolder.class);
      Expression exp =
        ExpressionFactory.matchExp(EipTMailFolder.FOLDER_NAME_PROPERTY, fname);
      if (is_update) {
        exp =
          exp.andExp(ExpressionFactory.noMatchDbExp(
            EipTMailFolder.FOLDER_ID_PK_COLUMN,
            folderId));
      }
      Expression exp2 =
        ExpressionFactory.matchExp(
          EipTMailFolder.EIP_MMAIL_ACCOUNT_PROPERTY,
          mailAccount);

      List<EipTMailFolder> list =
        query.setQualifier(exp.andExp(exp2)).fetchList();
      if (list != null && list.size() > 0) {
        return true;
      }
    } catch (Exception e) {
      return true;
    }
    return false;
  }

  /**
   * フォルダをデータベースから読み出します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   * @throws ALDBErrorException
   */
  @Override
  protected boolean loadFormData(RunData rundata, Context context,
      List<String> msgList) throws ALDBErrorException {
    try {
      // オブジェクトモデルを取得
      EipTMailFolder folder =
        WebMailUtils.getEipTMailFolder(mailAccount, folderId);
      if (folder == null) {
        return false;
      }

      // フォルダ名
      folder_name.setValue(folder.getFolderName());

      // add start
      // 親フォルダ
      parent_folder_id.setValue(folder.getParentFolderId());

      // remove start 受入テスト障害 190
      // フォルダ種別と同じフォルダ一覧を取得
      // List<EipTMailFolder> folder_list =
      // ALMailUtils
      // .getEipTMailFolderByKind(mailAccount, folder.getFolderKind());
      // 自身のフォルダを一覧より除く
      // folder_list.remove(folder);
      // フォルダ一覧をロード
      // loadMailFolderList(folder_list);
      // remove end 受入テスト障害 190
      loadMailFolderList(rundata, context, folder);
      removeChildFolders(folder);
      // add end

    } catch (Exception ex) {
      // logger.error("Exception", ex);
      // add start
      String msg =
        "フォルダ作成（編集）　フォルダ情報取得に失敗しました。["
          + ALEipUtils.getBaseUser(user_id).getUserName()
          + "]";
      logger.error(msg, ex);
      // add end
      throw new ALDBErrorException();
    }
    return true;
  }

  /**
   * フォルダをデータベースとファイルシステムから削除します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean deleteFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // String folderId =
      // ALEipUtils.getTemp(rundata, context, WebMailUtils.FOLDER_ID);
      // int delete_id = Integer.parseInt(folderId);

      // デフォルトのフォルダは削除不可。
      // if (mailAccount.getDefaultFolderId() == Integer.parseInt(folderId)) {
      // return false;
      // }

      // 削除するフォルダオブジェクトモデルを取得する．
      EipTMailFolder folder =
        WebMailUtils.getEipTMailFolder(mailAccount, folderId);

      // add start

      // フォルダ種別（受信トレイor送信トレイ）
      String folder_kind = folder.getFolderKind();

      // デフォルトフォルダの設定
      EipTMailFolder defaultFolder = null;
      if (ALMailUtils.FOLDER_KIND_RECEIVE.equals(folder_kind)) {
        // フォルダ種別が受信トレイの場合 「受信トレイ」
        defaultFolder =
          WebMailUtils.getEipTMailFolder(mailAccount, mailAccount
            .getDefaultFolderId()
            .toString());
      } else {
        // フォルダ種別が送信トレイの場合 「送信トレイ」
        defaultFolder =
          WebMailUtils.getEipTMailFolder(mailAccount, mailAccount
            .getDefaultSendFolderId()
            .toString());
      }

      // ルートフォルダ、受信トレイ、送信トレイ、下書き、ごみ箱は削除不可
      if (defaultFolder.getFolderId().equals(folderId)
        || folder.getFolderType().equals(ALMailUtils.FOLDER_TYPE_DRAFT)
        || folder.getFolderType().equals(ALMailUtils.FOLDER_TYPE_TRASH)
        || folder.getParentFolderId() == (WebMailUtils.ROOT_PARENT_FODLER_ID)) {
        return false;
      }

      // 送受信したメールを保持するローカルフォルダクラス
      int type_mail =
        (ALMailUtils.FOLDER_KIND_RECEIVE.equals(folder_kind))
          ? ALFolder.TYPE_RECEIVE
          : ALFolder.TYPE_SEND;
      ALMailHandler handler =
        ALMailFactoryService.getInstance().getMailHandler();
      ALFolder local_folder =
        handler.getALFolder(type_mail, Database.getDomainName(), ALEipUtils
          .getUserId(rundata), Integer.valueOf(mailAccount.getAccountId()));

      // 削除対象フォルダの子フォルダ階層を取得
      List<ExtFolderInfo> folder_hierarchy_list =
        WebMailUtils.getFolderList(folder.getFolderId(), mailAccount
          .getAccountId(), folder.getFolderKind());

      String child_folder_id = null;
      EipTMailFolder child_folder = null;

      // 振り分け先として指定してあるフィルタは、振り分け先をデフォルトに変更
      SelectQuery<EipTMailFilter> query_filter =
        Database.query(EipTMailFilter.class);

      Expression exp_filter = null;
      List<EipTMailFilter> filters = null;

      // 子フォルダの削除
      // 子フォルダを階層を辿って再帰的に削除
      if (folder_hierarchy_list != null) {
        for (int i = 0; i < folder_hierarchy_list.size(); i++) {

          // 子フォルダ情報取得
          child_folder_id =
            Integer.toString(folder_hierarchy_list.get(i).getFolderId());
          child_folder =
            WebMailUtils.getEipTMailFolder(mailAccount, child_folder_id);

          // 子フォルダを振り分け先として指定してあるフィルタ情報を取得
          exp_filter =
            ExpressionFactory.matchDbExp(
              EipTMailFilter.EIP_TMAIL_FOLDER_PROPERTY,
              child_folder);
          filters = query_filter.setQualifier(exp_filter).fetchList();
          // 当フォルダ振り分け先として指定してあるフィルタは、振り分け先をデフォルトフォルダに変更
          if (filters != null && filters.size() != 0) {
            for (EipTMailFilter filter : filters) {
              filter.setEipTMailFolder(defaultFolder);
            }
          }

          // フォルダ種別によってフォルダ内のメールを削除
          if (ALMailUtils.FOLDER_KIND_RECEIVE.equals(folder_kind)) {
            // 受信トレイの場合

            // フォルダ内の受信メールをDBより取得
            List<EipTMail> folderMails = ALMailUtils.getEipTMails(child_folder);

            if (folderMails != null && folderMails.size() != 0) {
              // 受信メールファイル削除
              for (EipTMail record : folderMails) {
                ALStorageService.deleteFile(local_folder.getFullName()
                  + ALStorageService.separator()
                  + record.getFilePath());
              }
              // 受信メールデータ削除
              Database.deleteAll(folderMails);
            }

          } else {
            // 送信トレイの場合

            // フォルダ内の送信メールをDBより取得
            List<AvzTMailSend> folderMails =
              ALMailUtils.getEipTMailsSend(child_folder);

            if (folderMails != null && folderMails.size() != 0) {
              // 送信メールファイル削除
              for (AvzTMailSend record : folderMails) {
                ALStorageService.deleteFile(local_folder.getFullName()
                  + ALStorageService.separator()
                  + record.getFilePath());
              }
              // 送信メールデータ削除
              Database.deleteAll(folderMails);
            }
          }

          // 子フォルダを削除
          Database.delete(child_folder);
        }
      }

      // 対象フォルダの削除

      // 削除対象フォルダを振り分け先として指定してあるフィルタ情報を取得
      exp_filter =
        ExpressionFactory.matchDbExp(
          EipTMailFilter.EIP_TMAIL_FOLDER_PROPERTY,
          folder);
      filters = query_filter.setQualifier(exp_filter).fetchList();
      // 当フォルダを振り分け先として指定してあるフィルタは、振り分け先をデフォルトフォルダに変更
      if (filters != null && filters.size() != 0) {
        for (EipTMailFilter filter : filters) {
          filter.setEipTMailFolder(defaultFolder);
        }
      }

      // フォルダ種別によってフォルダ内のメールを削除
      if (ALMailUtils.FOLDER_KIND_RECEIVE.equals(folder_kind)) {
        // 受信トレイの場合

        // フォルダ内の受信メールをDBより取得
        List<EipTMail> folderMails = ALMailUtils.getEipTMails(folder);

        if (folderMails != null && folderMails.size() != 0) {
          // 受信メールファイル削除
          for (EipTMail record : folderMails) {
            ALStorageService.deleteFile(local_folder.getFullName()
              + ALStorageService.separator()
              + record.getFilePath());
          }
          // 受信メールデータ削除
          Database.deleteAll(folderMails);
        }

      } else {
        // 送信トレイの場合

        // フォルダ内の送信メールをDBより取得
        List<AvzTMailSend> folderMails = ALMailUtils.getEipTMailsSend(folder);

        if (folderMails != null && folderMails.size() != 0) {
          // 送信メールファイル削除
          for (AvzTMailSend record : folderMails) {
            ALStorageService.deleteFile(local_folder.getFullName()
              + ALStorageService.separator()
              + record.getFilePath());
          }
          // 送信メールデータ削除
          Database.deleteAll(folderMails);
        }
      }
      // add end

      // remove start
      // 一緒に削除するメール
      // List<EipTMail> folderMails = ALMailUtils.getEipTMails(folder);

      // 振り分け先として指定してあるフィルタは、振り分け先をデフォルトに変更
      // SelectQuery<EipTMailFilter> query =
      // Database.query(EipTMailFilter.class);

      // Expression exp =
      // ExpressionFactory.matchDbExp(
      // EipTMailFilter.EIP_TMAIL_FOLDER_PROPERTY,
      // folder);

      // List<EipTMailFilter> filters = query.setQualifier(exp).fetchList();
      // if (filters != null && filters.size() != 0) {
      // EipTMailFolder defaultFolder =
      // WebMailUtils.getEipTMailFolder(mailAccount, mailAccount
      // .getDefaultFolderId()
      // .toString());
      // for (EipTMailFilter filter : filters) {
      // filter.setEipTMailFolder(defaultFolder);
      // }
      // }

      // ローカルファイルに保存されているメールのパスのリスト
      // List<String> mailPaths = new ArrayList<String>();
      // if (folderMails != null && folderMails.size() > 0) {
      // for (EipTMail mail : folderMails) {
      // mailPaths.add(mail.getFilePath());
      // }
      // }
      // remove end

      // フォルダ情報を削除
      Database.delete(folder);
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        folder.getFolderId(),
        ALEventlogConstants.PORTLET_TYPE_WEBMAIL_FOLDER,
        folder.getFolderName());

      // remove start
      // ローカルファイルに保存されているファイルを削除する．
      // if (mailPaths.size() > 0) {
      // int size = mailPaths.size();
      // for (int k = 0; k < size; k++) {
      // ALStorageService.deleteFile(ALMailUtils.getLocalurl()
      // + mailPaths.get(k));
      // }
      // }
      // remove end

      return true;
    } catch (Throwable t) {
      Database.rollback();
      // logger.error(t);
      // add start
      String msg =
        "フォルダ削除に失敗しました。[" + ALEipUtils.getBaseUser(user_id).getUserName() + "]";
      logger.error(msg, t);
      msgList.add(msg);
      // add end
      return false;
    }
  }

  /**
   * フォルダをデータベースとファイルシステムに格納します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗
   */
  @Override
  protected boolean insertFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // 新規オブジェクトモデル
      EipTMailFolder folder = Database.create(EipTMailFolder.class);

      // フォルダ名
      folder.setFolderName(folder_name.getValue());
      // 作成日
      folder.setCreateDate(Calendar.getInstance().getTime());
      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());
      // アカウントID
      folder.setEipMMailAccount(mailAccount);
      // add start
      // 親フォルダ ID
      folder.setParentFolderId(Integer.valueOf((int) parent_folder_id
        .getValue()));
      // フォルダ容量
      folder.setFolderVolume(0L);
      // フォルダタイプ
      folder.setFolderType(ALMailUtils.FOLDER_TYPE_NORMAL);
      // フォルダ種別
      // 親フォルダのフォルダ種別を取得する
      EipTMailFolder parent_folder =
        WebMailUtils
          .getEipTMailFolder(mailAccount, parent_folder_id.toString());
      if (parent_folder == null) {
        return false;
      }
      folder.setFolderKind(parent_folder.getFolderKind());
      // add end

      // フォルダを登録
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        folder.getFolderId(),
        ALEventlogConstants.PORTLET_TYPE_WEBMAIL_FOLDER,
        folder_name.getValue());
      return true;
    } catch (Throwable t) {
      // logger.error(t);
      // add start
      String msg =
        "フォルダ作成に失敗しました。[" + ALEipUtils.getBaseUser(user_id).getUserName() + "]";
      logger.error(msg, t);
      msgList.add(msg);
      // add end
      return false;
    }
  }

  /**
   * データベースとファイルシステムに格納されているフォルダを更新します。 <BR>
   * 
   * @param rundata
   * @param context
   * @param msgList
   * @return TRUE 成功 FALSE 失敗O
   */
  @Override
  protected boolean updateFormData(RunData rundata, Context context,
      List<String> msgList) {
    try {
      // オブジェクトモデルを取得
      EipTMailFolder folder =
        WebMailUtils.getEipTMailFolder(mailAccount, folderId);
      if (folder == null) {
        return false;
      }

      // フォルダ名
      folder.setFolderName(folder_name.getValue());

      // add start
      // 親フォルダ ID
      folder.setParentFolderId(Integer.valueOf((int) parent_folder_id
        .getValue()));
      // add end

      // 更新日
      folder.setUpdateDate(Calendar.getInstance().getTime());

      // フォルダを更新
      Database.commit();

      // イベントログに保存
      ALEventlogFactoryService.getInstance().getEventlogHandler().log(
        folder.getFolderId(),
        ALEventlogConstants.PORTLET_TYPE_WEBMAIL_FOLDER,
        folder_name.getValue());
      return true;
    } catch (Throwable t) {
      Database.rollback();
      // logger.error(t);
      // add start
      String msg =
        "フォルダ更新に失敗しました。[" + ALEipUtils.getBaseUser(user_id).getUserName() + "]";
      logger.error(msg, t);
      msgList.add(msg);
      // add end
      return false;
    }
  }

  /**
   * 選択中のメールアカウントを取得する． <BR>
   * 
   * @return
   */
  public EipMMailAccount getMailAccount() {
    return mailAccount;
  }

  /**
   * フォルダ名を取得する． <BR>
   * 
   * @return
   */
  public ALStringField getFolderName() {
    return folder_name;
  }

  public ALNumberField getAccessFlag() {
    return access_flag;
  }

  /**
   * フォルダIDを取得する． <BR>
   * 
   * @return
   */
  public String getFolderId() {
    return folderId;
  }

  // add start
  /**
   * 親フォルダIDを取得する． <BR>
   * 
   * @return
   */
  public ALNumberField getParentFolderId() {
    return parent_folder_id;
  }

  /**
   * 
   * @param id
   * @return
   */
  public boolean isMatch(int id1, long id2) {
    return id1 == (int) id2;
  }

  /**
   * フォルダ一覧をロードします。<BR>
   * 
   * @param mailFolders
   *          フォルダ情報オブジェクトリスト
   */
  // change start 受入テスト障害 190
  // private void loadMailFolderList(List<EipTMailFolder> mailFolders) {
  // if (mailAccount == null) {
  // return;
  // }
  // try {
  // folderList = new ArrayList<WebMailFolderResultData>();
  //
  // for (EipTMailFolder folder : mailFolders) {
  // // ルートフォルダ、下書き、ごみ箱は除く
  // if (folder.getFolderType().equals(ALMailUtils.FOLDER_TYPE_NORMAL)) {
  // if (folder.getParentFolderId() != WebMailUtils.ROOT_PARENT_FODLER_ID) {
  // WebMailFolderResultData rd = new WebMailFolderResultData();
  // rd.initField();
  // rd.setFolderId(folder.getFolderId().intValue());
  // rd.setFolderName(folder.getFolderName());
  // folderList.add(rd);
  // }
  // }
  // }
  // } catch (Exception ex) {
  // // logger.error("Exception", ex);
  // // add start
  // String msg =
  // "プルダウン表示用のフォルダ情報の一覧取得に失敗しました。["
  // + ALEipUtils.getBaseUser(user_id).getUserName()
  // + "]";
  // logger.error(msg, ex);
  // // add end
  // }
  // }
  public void loadMailFolderList(RunData rundata, Context context,
      EipTMailFolder targetFolder) {
    if (mailAccount == null) {
      return;
    }
    try {
      List<ExtFolderInfo> tmpFolderList = new ArrayList<ExtFolderInfo>();

      List<ExtFolderInfo> tmp =
        WebMailUtils.getFolderList(
          WebMailUtils.ROOT_PARENT_FODLER_ID,
          mailAccount.getAccountId(),
          WebMailConsts.RECEIVE_FOLDER);

      tmpFolderList.addAll(tmp);

      tmp =
        WebMailUtils.getFolderList(
          WebMailUtils.ROOT_PARENT_FODLER_ID,
          mailAccount.getAccountId(),
          WebMailConsts.SEND_FOLDER);

      tmpFolderList.addAll(tmp);

      folderList = new ArrayList<ExtFolderInfo>();
      for (ExtFolderInfo folder : tmpFolderList) {
        if (targetFolder == null
          || targetFolder.getFolderKind().equals(folder.getFolderKind())) {
          if (folder.getFolderType().equals(ALMailUtils.FOLDER_TYPE_NORMAL)) {
            // if (folder.getParentFolderId() !=
            // WebMailUtils.ROOT_PARENT_FODLER_ID) {
            folderList.add(folder);
            // }
          }
        }
      }

    } catch (Exception ex) {
      String msg =
        "プルダウン表示用のフォルダ情報の一覧取得に失敗しました。["
          + ALEipUtils.getBaseUser(user_id).getUserName()
          + "]";
      logger.error(msg, ex);
    }
  }

  // change end 受入テスト障害 190

  /**
   * フォルダ一覧を取得する． <BR>
   * 
   * @return
   */
  // change start 受入テスト障害 190
  // public List<WebMailFolderResultData> getFolderList() {
  public List<ExtFolderInfo> getFolderList() {
    // change end 受入テスト障害 190
    return folderList;
  }

  /**
   * フォルダ編集画面の親フォルダ選択プルダウンに、更新対象のフォルダの下に連なるフォルダ（子、孫・・・・） を除外する。
   * 
   * @param targetFolder
   *          更新対象フォルダ
   */
  private void removeChildFolders(EipTMailFolder targetFolder) {

    List<ExtFolderInfo> exclude_list =
      WebMailUtils.getFolderList(targetFolder.getFolderId(), mailAccount
        .getAccountId(), targetFolder.getFolderKind());

    // 自分を除外条件に含める
    if (exclude_list == null) {
      exclude_list = new ArrayList<ExtFolderInfo>(0);
    }
    ExtFolderInfo my = new ExtFolderInfo();
    my.setFolderId(targetFolder.getFolderId());
    exclude_list.add(my);

    List<ExtFolderInfo> tmpList = new ArrayList<ExtFolderInfo>();

    if (exclude_list != null) {
      for (ExtFolderInfo d : folderList) {
        boolean found = false;
        for (ExtFolderInfo f : exclude_list) {
          if (f.getFolderId() == d.getFolderId()) {
            // 除外対象
            found = true;
            break;
          }
        }
        // 除外に含まれていなければ表示対象
        if (!found) {
          tmpList.add(d);
        }
      }

      folderList = tmpList;
    }
  }
  // add end

}
