
/* File: misc.c */

/*
 * Copyright (c) 1997 Ben Harrison, James E. Wilson, Robert A. Koeneke
 *
 * This software may be copied and distributed for educational, research,
 * and not for profit purposes provided that this copyright and statement
 * are included in all such copies.  Other copyrights may also apply.
 */

/* Purpose: misc code */

#include "angband.h"




/*
 * Converts stat num into a six-char (right justified) string
 */
void cnv_stat(int val, char *out_val)
{
	/* Above 18 */
	if (val > 18)
	{
		int bonus = (val - 18);

		if (bonus >= 420)
		{
			sprintf(out_val, "  ****");
		}
		else
		{
			sprintf(out_val, "  %02d.%01d", (18 + bonus / 10),  bonus % 10);
		}
	}

	/* From 3 to 18 */
	else
	{
		sprintf(out_val, "  %2d.0", val);
	}
}


/*
 * Modify a stat value by a "modifier", return new value
 *
 * Stats go up: 3,4,...,17,18,18/10,18/20,...,18/220
 * Or even: 18/13, 18/23, 18/33, ..., 18/220
 *
 * Stats go down: 18/220, 18/210,..., 18/10, 18, 17, ..., 3
 * Or even: 18/13, 18/03, 18, 17, ..., 3
 */
s16b modify_stat_value(int value, int amount)
{
	int    i;

	/* Reward */
	if (amount > 0)
	{
		/* Apply each point */
		for (i = 0; i < amount; i++)
		{
			/* One point at a time */
			if (value < 18) value++;

			/* Ten "points" at a time */
			else value += 10;
		}
	}

	/* Penalty */
	else if (amount < 0)
	{
		/* Apply each point */
		for (i = 0; i < (0 - amount); i++)
		{
			/* Ten points at a time */
			if (value >= 18+10) value -= 10;

			/* Hack -- prevent weirdness */
			else if (value > 18) value = 18;

			/* One point at a time */
			else if (value > 3) value--;
		}
	}

	/* Return new value */
	return (value);
}



/*
 * Print character info at given row, column in a 13 char field
 */
static void prt_field(cptr info, int row, int col)
{
	/* Dump 13 spaces to clear */
	c_put_str(TERM_WHITE, "             ", row, col);

	/* Dump the info itself */
	c_put_str(TERM_L_BLUE, info, row, col);
}


/*
 *  Whether daytime or not
 */
bool is_daytime(void)
{
	s32b len = TURNS_PER_TICK * TOWN_DAWN;
	if ((turn % len) < (len / 2))
		return TRUE;
	else
		return FALSE;
}

/*
 * Extract day, hour, min
 */
void extract_day_hour_min(int *day, int *hour, int *min)
{
	s32b turn_in_today = (turn + A_DAY / 4 + start_hour * 1440 + start_min * 60) % A_DAY;
	*day = (turn + A_DAY / 4) / A_DAY + 1;
	*hour = (24 * turn_in_today / A_DAY) % 24;
	*min = (1440 * turn_in_today / A_DAY) % 60;
}

/*
 * Print time
 */
void prt_time(creature_type *player_ptr)
{
	int day, hour, min;

	/* Dump 13 spaces to clear */
	c_put_str(TERM_WHITE, "             ", ROW_DAY, COL_DAY);

	extract_day_hour_min(&day, &hour, &min);

	/* Dump the info itself */
#ifdef JP
	if (day < 1000) c_put_str(TERM_WHITE, format("%2d", day), ROW_DAY, COL_DAY);
	else c_put_str(TERM_WHITE, "***", ROW_DAY, COL_DAY);
#else
	if (day < 1000) c_put_str(TERM_WHITE, format("Day%3d", day), ROW_DAY, COL_DAY);
	else c_put_str(TERM_WHITE, "Day***", ROW_DAY, COL_DAY);
#endif

	c_put_str(TERM_WHITE, format("%2d:%02d", hour, min), ROW_DAY, COL_DAY+7);
}


cptr map_name(floor_type *floor_ptr)
{
	if (inside_quest && is_fixed_quest_idx(inside_quest)
	    && (quest[inside_quest].flags & QUEST_FLAG_PRESET))
#ifdef JP
		return "NGXg";
#else
		return "Quest";
#endif
	else if (wild_mode)
#ifdef JP
		return "n";
#else
		return "Surface";
#endif
	else if (fight_arena_mode)
#ifdef JP
		return "A[i";
#else
		return "Arena";
#endif
	else if (gamble_arena_mode)
#ifdef JP
		return "Z";
#else
		return "Monster Arena";
#endif
	else if (!floor_ptr->floor_level && town_num)
		return town[town_num].name;
	else
		return d_name+dungeon_info[floor_ptr->dun_type].name;
}

/*
 * Print dungeon
 */
static void prt_dungeon(void)
{
	floor_type *floor_ptr = get_floor_ptr(player_ptr);
	cptr dungeon_name;
	int col;

	/* Dump 13 spaces to clear */
	c_put_str(TERM_WHITE, "             ", ROW_DUNGEON, COL_DUNGEON);

	dungeon_name = map_name(floor_ptr);

	col = COL_DUNGEON + 6 - strlen(dungeon_name)/2;
	if (col < 0) col = 0;

	/* Dump the info itself */
	c_put_str(TERM_L_UMBER, format("%s",dungeon_name),
		  ROW_DUNGEON, col);
}




/*
 * Print character stat in given row, column
 */
static void prt_stat(creature_type *cr_ptr, int stat)
{
	char tmp[32];

	/* Display "injured" stat */
	if (cr_ptr->stat_cur[stat] < cr_ptr->stat_max[stat])
	{
		put_str(stat_names_reduced[stat], ROW_STAT + stat, 0);
		cnv_stat(cr_ptr->stat_use[stat], tmp);
		c_put_str(TERM_YELLOW, tmp, ROW_STAT + stat, COL_STAT + 6);
	}

	/* Display "healthy" stat */
	else
	{
		put_str(stat_names[stat], ROW_STAT + stat, 0);
		cnv_stat(cr_ptr->stat_use[stat], tmp);
		c_put_str(TERM_L_GREEN, tmp, ROW_STAT + stat, COL_STAT + 6);
	}

	/* Indicate natural maximum */
	if (cr_ptr->stat_max[stat] == cr_ptr->stat_mod_max_max[stat])
	{
#ifdef JP
		/* {ɂԂȂ悤ɕ\ʒuύX */
		put_str("!", ROW_STAT + stat, 5);
#else
		put_str("!", ROW_STAT + stat, 3);
#endif

	}
}


/*
 *  Data structure for status bar
 */
#define BAR_TSUYOSHI 0
#define BAR_HALLUCINATION 1
#define BAR_BLINDNESS 2
#define BAR_PARALYZE 3
#define BAR_CONFUSE 4
#define BAR_POISONED 5
#define BAR_AFRAID 6
#define BAR_LEVITATE 7
#define BAR_REFLECTION 8
#define BAR_PASSWALL 9
#define BAR_WRAITH 10
#define BAR_PROTEVIL 11
#define BAR_KAWARIMI 12
#define BAR_MAGICDEFENSE 13
#define BAR_EXPAND 14
#define BAR_STONESKIN 15
#define BAR_MULTISHADOW 16
#define BAR_REGMAGIC 17
#define BAR_ULTIMATE 18
#define BAR_INVULN 19
#define BAR_IMMACID 20
#define BAR_RESACID 21
#define BAR_IMMELEC 22
#define BAR_RESELEC 23
#define BAR_IMMFIRE 24
#define BAR_RESFIRE 25
#define BAR_IMMCOLD 26
#define BAR_RESCOLD 27
#define BAR_RESPOIS 28
#define BAR_RESNETH 29
#define BAR_RESTIME 30
#define BAR_DUSTROBE 31
#define BAR_SHFIRE 32
#define BAR_TOUKI 33
#define BAR_SHHOLY 34
#define BAR_EYEEYE 35
#define BAR_BLESSED 36
#define BAR_HEROISM 37
#define BAR_BERSERK 38
#define BAR_ATTKFIRE 39
#define BAR_ATTKCOLD 40
#define BAR_ATTKELEC 41
#define BAR_ATTKACID 42
#define BAR_ATTKPOIS 43
#define BAR_ATTKCONF 44
#define BAR_SENSEUNSEEN 45
#define BAR_TELEPATHY 46
#define BAR_REGENERATION 47
#define BAR_INFRAVISION 48
#define BAR_STEALTH 49
#define BAR_SUPERSTEALTH 50
#define BAR_RECALL 51
#define BAR_ALTER 52
#define BAR_SHCOLD 53
#define BAR_SHELEC 54
#define BAR_SHSHADOW 55
#define BAR_MIGHT 56
#define BAR_BUILD 57
#define BAR_ANTIMULTI 58
#define BAR_ANTITELE 59
#define BAR_ANTIMAGIC 60
#define BAR_PATIENCE 61
#define BAR_REVENGE 62
#define BAR_RUNESWORD 63
#define BAR_VAMPILIC 64
#define BAR_CURE 65
#define BAR_ESP_EVIL 66

static struct {
	byte attr;
	cptr sstr;
	cptr lstr;
} bar[]
#ifdef JP
= {
	{TERM_YELLOW, "", "悵"},
	{TERM_VIOLET, "", "o"},
	{TERM_L_DARK, "", "Ӗ"},
	{TERM_RED, "", ""},
	{TERM_VIOLET, "", ""},
	{TERM_GREEN, "", ""},
	{TERM_BLUE, "", "|"},
	{TERM_L_BLUE, "", "V"},
	{TERM_SLATE, "", ""},
	{TERM_SLATE, "", "ǔ"},
	{TERM_L_DARK, "H", "H"},
	{TERM_SLATE, "", "h"},
	{TERM_VIOLET, "", "ςg"},
	{TERM_YELLOW, "", "@Z"},
	{TERM_L_UMBER, "L", "L"},
	{TERM_WHITE, "", "Δ"},
	{TERM_L_BLUE, "", "g"},
	{TERM_SLATE, "h", "@h"},
	{TERM_YELLOW, "", ""},
	{TERM_YELLOW, "", "G"},
	{TERM_L_GREEN, "_", "_Ɖu"},
	{TERM_GREEN, "_", "ώ_"},
	{TERM_L_BLUE, "d", "dƉu"},
	{TERM_BLUE, "d", "ϓd"},
	{TERM_L_RED, "", "ΖƉu"},
	{TERM_RED, "", "ω"},
	{TERM_WHITE, "", "Ɖu"},
	{TERM_SLATE, "", "ϗ"},
	{TERM_GREEN, "", "ϓ"},
	{TERM_L_DARK, "", "ϒn"},
	{TERM_L_BLUE, "", "ώ"},
	{TERM_L_DARK, "", "I["},
	{TERM_L_RED, "I", "΃I["},
	{TERM_WHITE, "", "C"},
	{TERM_WHITE, "", "I["},
	{TERM_VIOLET, "", "ڂɂ͖"},
	{TERM_WHITE, "j", "j"},
	{TERM_WHITE, "E", "E"},
	{TERM_RED, "", ""},
	{TERM_L_RED, "", ""},
	{TERM_WHITE, "", ""},
	{TERM_L_BLUE, "d", "d"},
	{TERM_SLATE, "_", "_"},
	{TERM_L_GREEN, "", ""},
	{TERM_RED, "", "Ō"},
	{TERM_L_BLUE, "", ""},
	{TERM_ORANGE, "e", "epV"},
	{TERM_L_BLUE, "", ""},
	{TERM_L_RED, "", "ԊO"},
	{TERM_UMBER, "B", "B"},
	{TERM_YELLOW, "B", "B"},
	{TERM_WHITE, "A", "A"},
	{TERM_WHITE, "", "ϗe"},
	/* Hex */
	{TERM_WHITE, "I", "XI["},
	{TERM_BLUE, "I", "dI["},
	{TERM_L_DARK, "I", "eI["},
	{TERM_YELLOW, "r", "r͋"},
	{TERM_RED, "", "̋"},
	{TERM_L_DARK, "B", "B"},
	{TERM_ORANGE, "e", "e|"},
	{TERM_RED, "", "@"},
	{TERM_SLATE, "", "䖝"},
	{TERM_SLATE, "", "鍐"},
	{TERM_L_DARK, "", ""},
	{TERM_RED, "z", "zŌ"},
	{TERM_WHITE, "", ""},
	{TERM_L_DARK, "", "׈m"},
	{0, NULL, NULL}
};
#else
= {
	{TERM_YELLOW, "Ts", "Tsuyoshi"},
	{TERM_VIOLET, "Ha", "Halluc"},
	{TERM_L_DARK, "Bl", "Blind"},
	{TERM_RED, "Pa", "Paralyzed"},
	{TERM_VIOLET, "Cf", "Confused"},
	{TERM_GREEN, "Po", "Poisoned"},
	{TERM_BLUE, "Af", "Afraid"},
	{TERM_L_BLUE, "Lv", "Levit"},
	{TERM_SLATE, "Rf", "Reflect"},
	{TERM_SLATE, "Pw", "PassWall"},
	{TERM_L_DARK, "Wr", "Wraith"},
	{TERM_SLATE, "Ev", "PrtEvl"},
	{TERM_VIOLET, "Kw", "Kawarimi"},
	{TERM_YELLOW, "Md", "MgcArm"},
	{TERM_L_UMBER, "Eh", "Expand"},
	{TERM_WHITE, "Ss", "StnSkn"},
	{TERM_L_BLUE, "Ms", "MltShdw"},
	{TERM_SLATE, "Rm", "ResMag"},
	{TERM_YELLOW, "Ul", "Ultima"},
	{TERM_YELLOW, "Iv", "Invuln"},
	{TERM_L_GREEN, "IAc", "ImmAcid"},
	{TERM_GREEN, "Ac", "Acid"},
	{TERM_L_BLUE, "IEl", "ImmElec"},
	{TERM_BLUE, "El", "Elec"},
	{TERM_L_RED, "IFi", "ImmFire"},
	{TERM_RED, "Fi", "Fire"},
	{TERM_WHITE, "ICo", "ImmCold"},
	{TERM_SLATE, "Co", "Cold"},
	{TERM_GREEN, "Po", "Pois"},
	{TERM_L_DARK, "Nt", "Nthr"},
	{TERM_L_BLUE, "Ti", "Time"},
	{TERM_L_DARK, "Mr", "Mirr"},
	{TERM_L_RED, "SFi", "SFire"},
	{TERM_WHITE, "Fo", "Force"},
	{TERM_WHITE, "Ho", "Holy"},
	{TERM_VIOLET, "Ee", "EyeEye"},
	{TERM_WHITE, "Bs", "Bless"},
	{TERM_WHITE, "He", "Hero"},
	{TERM_RED, "Br", "Berserk"},
	{TERM_L_RED, "BFi", "BFire"},
	{TERM_WHITE, "BCo", "BCold"},
	{TERM_L_BLUE, "BEl", "BElec"},
	{TERM_SLATE, "BAc", "BAcid"},
	{TERM_L_GREEN, "BPo", "BPois"},
	{TERM_RED, "TCf", "TchCnf"},
	{TERM_L_BLUE, "Se", "SInv"},
	{TERM_ORANGE, "Te", "Telepa"},
	{TERM_L_BLUE, "Rg", "Regen"},
	{TERM_L_RED, "If", "Infr"},
	{TERM_UMBER, "Sl", "Stealth"},
	{TERM_YELLOW, "Stlt", "Stealth"},
	{TERM_WHITE, "Rc", "Recall"},
	{TERM_WHITE, "Al", "Alter"},
	/* Hex */
	{TERM_WHITE, "SCo", "SCold"},
	{TERM_BLUE, "SEl", "SElec"},
	{TERM_L_DARK, "SSh", "SShadow"},
	{TERM_YELLOW, "EMi", "ExMight"},
	{TERM_RED, "Bu", "BuildUp"},
	{TERM_L_DARK, "AMl", "AntiMulti"},
	{TERM_ORANGE, "AT", "AntiTele"},
	{TERM_RED, "AM", "AntiMagic"},
	{TERM_SLATE, "Pa", "Patience"},
	{TERM_SLATE, "Rv", "Revenge"},
	{TERM_L_DARK, "Rs", "RuneSword"},
	{TERM_RED, "Vm", "Vampiric"},
	{TERM_WHITE, "Cu", "Cure"},
	{TERM_L_DARK, "ET", "EvilTele"},
	{0, NULL, NULL}
};
#endif

#define ADD_FLG(FLG) (bar_flags[FLG / 32] |= (1L << (FLG % 32)))
#define IS_FLG(FLG) (bar_flags[FLG / 32] & (1L << (FLG % 32)))


/*
 *  Show status bar
 */
static void prt_status(creature_type *cr_ptr)
{
	u32b bar_flags[3];
	int wid, hgt, row_statbar, max_col_statbar;
	int i, col = 0, num = 0;
	int space = 2;

	Term_get_size(&wid, &hgt);
	row_statbar = hgt + ROW_STATBAR;
	max_col_statbar = wid + MAX_COL_STATBAR;

	Term_erase(0, row_statbar, max_col_statbar);

	bar_flags[0] = bar_flags[1] = bar_flags[2] = 0L;

	/* Tsuyoshi  */
	if (cr_ptr->tsuyoshi) ADD_FLG(BAR_TSUYOSHI);

	/* Hallucinating */
	if (cr_ptr->image) ADD_FLG(BAR_HALLUCINATION);

	/* Blindness */
	if (cr_ptr->blind) ADD_FLG(BAR_BLINDNESS);

	/* Paralysis */
	if (cr_ptr->paralyzed) ADD_FLG(BAR_PARALYZE);

	/* Confusion */
	if (cr_ptr->confused) ADD_FLG(BAR_CONFUSE);

	/* Posioned */
	if (cr_ptr->poisoned) ADD_FLG(BAR_POISONED);

	/* Times see-invisible */
	if (cr_ptr->tim_invis) ADD_FLG(BAR_SENSEUNSEEN);

	/* Timed esp */
	if (IS_TIM_ESP(cr_ptr)) ADD_FLG(BAR_TELEPATHY);

	/* Timed regenerate */
	if (cr_ptr->tim_regen) ADD_FLG(BAR_REGENERATION);

	/* Timed infra-vision */
	if (cr_ptr->tim_infra) ADD_FLG(BAR_INFRAVISION);

	/* Protection from evil */
	if (cr_ptr->protevil) ADD_FLG(BAR_PROTEVIL);

	/* Invulnerability */
	if (IS_INVULN(cr_ptr)) ADD_FLG(BAR_INVULN);

	/* Wraith form */
	if (cr_ptr->wraith_form) ADD_FLG(BAR_WRAITH);

	/* Kabenuke */
	if (cr_ptr->kabenuke) ADD_FLG(BAR_PASSWALL);

	if (cr_ptr->tim_reflect) ADD_FLG(BAR_REFLECTION);

	/* Heroism */
	if (IS_HERO(cr_ptr)) ADD_FLG(BAR_HEROISM);

	/* Super Heroism / berserk */
	if (cr_ptr->shero) ADD_FLG(BAR_BERSERK);

	/* Blessed */
	if (IS_BLESSED(cr_ptr)) ADD_FLG(BAR_BLESSED);

	/* Shield */
	if (cr_ptr->magicdef) ADD_FLG(BAR_MAGICDEFENSE);

	if (cr_ptr->tsubureru) ADD_FLG(BAR_EXPAND);

	if (cr_ptr->shield) ADD_FLG(BAR_STONESKIN);
	
	if (cr_ptr->special_defense & NINJA_KAWARIMI) ADD_FLG(BAR_KAWARIMI);

	/* Oppose Acid */
	if (cr_ptr->special_defense & DEFENSE_ACID) ADD_FLG(BAR_IMMACID);
	if (IS_OPPOSE_ACID(cr_ptr)) ADD_FLG(BAR_RESACID);

	/* Oppose Lightning */
	if (cr_ptr->special_defense & DEFENSE_ELEC) ADD_FLG(BAR_IMMELEC);
	if (IS_OPPOSE_ELEC(cr_ptr)) ADD_FLG(BAR_RESELEC);

	/* Oppose Fire */
	if (cr_ptr->special_defense & DEFENSE_FIRE) ADD_FLG(BAR_IMMFIRE);
	if (IS_OPPOSE_FIRE(cr_ptr)) ADD_FLG(BAR_RESFIRE);

	/* Oppose Cold */
	if (cr_ptr->special_defense & DEFENSE_COLD) ADD_FLG(BAR_IMMCOLD);
	if (IS_OPPOSE_COLD(cr_ptr)) ADD_FLG(BAR_RESCOLD);

	/* Oppose Poison */
	if (IS_OPPOSE_POIS(cr_ptr)) ADD_FLG(BAR_RESPOIS);

	/* Word of Recall */
	if (cr_ptr->word_recall) ADD_FLG(BAR_RECALL);

	/* Alter realiry */
	if (cr_ptr->alter_reality) ADD_FLG(BAR_ALTER);

	/* Afraid */
	if (cr_ptr->afraid) ADD_FLG(BAR_AFRAID);

	/* Resist time */
	if (cr_ptr->tim_res_time) ADD_FLG(BAR_RESTIME);

	if (cr_ptr->multishadow) ADD_FLG(BAR_MULTISHADOW);

	/* Confusing Hands */
	if (cr_ptr->special_attack & ATTACK_CONFUSE) ADD_FLG(BAR_ATTKCONF);

	if (cr_ptr->resist_magic) ADD_FLG(BAR_REGMAGIC);

	/* Ultimate-resistance */
	if (cr_ptr->ult_res) ADD_FLG(BAR_ULTIMATE);

	/* tim levitation */
	if (cr_ptr->tim_levitation) ADD_FLG(BAR_LEVITATE);

	if (cr_ptr->tim_res_nether) ADD_FLG(BAR_RESNETH);

	if (cr_ptr->dustrobe) ADD_FLG(BAR_DUSTROBE);

	/* Mahouken */
	if (cr_ptr->special_attack & ATTACK_FIRE) ADD_FLG(BAR_ATTKFIRE);
	if (cr_ptr->special_attack & ATTACK_COLD) ADD_FLG(BAR_ATTKCOLD);
	if (cr_ptr->special_attack & ATTACK_ELEC) ADD_FLG(BAR_ATTKELEC);
	if (cr_ptr->special_attack & ATTACK_ACID) ADD_FLG(BAR_ATTKACID);
	if (cr_ptr->special_attack & ATTACK_POIS) ADD_FLG(BAR_ATTKPOIS);
	if (cr_ptr->special_defense & NINJA_S_STEALTH) ADD_FLG(BAR_SUPERSTEALTH);

	if (cr_ptr->tim_sh_fire) ADD_FLG(BAR_SHFIRE);

	/* tim stealth */
	if (IS_TIM_STEALTH(cr_ptr)) ADD_FLG(BAR_STEALTH);

	if (cr_ptr->tim_sh_touki) ADD_FLG(BAR_TOUKI);

	/* Holy aura */
	if (cr_ptr->tim_sh_holy) ADD_FLG(BAR_SHHOLY);

	/* An Eye for an Eye */
	if (cr_ptr->tim_eyeeye) ADD_FLG(BAR_EYEEYE);

	/* Hex spells */
	if (cr_ptr->realm1 == REALM_HEX)
	{
		if (hex_spelling(cr_ptr, HEX_BLESS)) ADD_FLG(BAR_BLESSED);
		if (hex_spelling(cr_ptr, HEX_DEMON_AURA)) { ADD_FLG(BAR_SHFIRE); ADD_FLG(BAR_REGENERATION); }
		if (hex_spelling(cr_ptr, HEX_XTRA_MIGHT)) ADD_FLG(BAR_MIGHT);
		if (hex_spelling(cr_ptr, HEX_DETECT_EVIL)) ADD_FLG(BAR_ESP_EVIL);
		if (hex_spelling(cr_ptr, HEX_ICE_ARMOR)) ADD_FLG(BAR_SHCOLD);
		if (hex_spelling(cr_ptr, HEX_RUNESWORD)) ADD_FLG(BAR_RUNESWORD);
		if (hex_spelling(cr_ptr, HEX_BUILDING)) ADD_FLG(BAR_BUILD);
		if (hex_spelling(cr_ptr, HEX_ANTI_TELE)) ADD_FLG(BAR_ANTITELE);
		if (hex_spelling(cr_ptr, HEX_SHOCK_CLOAK)) ADD_FLG(BAR_SHELEC);
		if (hex_spelling(cr_ptr, HEX_SHADOW_CLOAK)) ADD_FLG(BAR_SHSHADOW);
		if (hex_spelling(cr_ptr, HEX_CONFUSION)) ADD_FLG(BAR_ATTKCONF);
		if (hex_spelling(cr_ptr, HEX_EYE_FOR_EYE)) ADD_FLG(BAR_EYEEYE);
		if (hex_spelling(cr_ptr, HEX_ANTI_MULTI)) ADD_FLG(BAR_ANTIMULTI);
		if (hex_spelling(cr_ptr, HEX_VAMP_BLADE)) ADD_FLG(BAR_VAMPILIC);
		if (hex_spelling(cr_ptr, HEX_ANTI_MAGIC)) ADD_FLG(BAR_ANTIMAGIC);
		if (hex_spelling(cr_ptr, HEX_CURE_LIGHT) ||
			hex_spelling(cr_ptr, HEX_CURE_SERIOUS) ||
			hex_spelling(cr_ptr, HEX_CURE_CRITICAL)) ADD_FLG(BAR_CURE);

		if (cr_ptr->magic_num2[2])
		{
			if (cr_ptr->magic_num2[1] == 1) ADD_FLG(BAR_PATIENCE);
			if (cr_ptr->magic_num2[1] == 2) ADD_FLG(BAR_REVENGE);
		}
	}

	/* Calcurate length */
	for (i = 0; bar[i].sstr; i++)
	{
		if (IS_FLG(i))
		{
			col += strlen(bar[i].lstr) + 1;
			num++;
		}
	}

	/* If there are not excess spaces for long strings, use short one */
	if (col - 1 > max_col_statbar)
	{
		space = 0;
		col = 0;

		for (i = 0; bar[i].sstr; i++)
		{
			if (IS_FLG(i))
			{
				col += strlen(bar[i].sstr);
			}
		}

		/* If there are excess spaces for short string, use more */
		if (col - 1 <= max_col_statbar - (num-1))
		{
			space = 1;
			col += num - 1;
		}
	}


	/* Centering display column */
	col = (max_col_statbar - col) / 2;

	/* Display status bar */
	for (i = 0; bar[i].sstr; i++)
	{
		if (IS_FLG(i))
		{
			cptr str;
			if (space == 2) str = bar[i].lstr;
			else str = bar[i].sstr;

			c_put_str(bar[i].attr, str, row_statbar, col);
			col += strlen(str);
			if (space > 0) col++;
			if (col > max_col_statbar) break;
		}
	}
}


/*
 * Prints level
 */
static void prt_level(creature_type *cr_ptr)
{
	char tmp[32];

	sprintf(tmp, "%2d/%2d", cr_ptr->lev, cr_ptr->max_lev);

	if (cr_ptr->lev >= cr_ptr->max_plv)
	{
		put_str("LEV", ROW_LEVEL, 0);				
		c_put_str(TERM_L_GREEN, tmp, ROW_LEVEL, COL_LEVEL + 3);
	}
	else
	{
		put_str("lev", ROW_LEVEL, 0);
		c_put_str(TERM_YELLOW, tmp, ROW_LEVEL, COL_LEVEL + 3);
	}
}


/*
 * Display the experience
 */
static void prt_exp(creature_type *cr_ptr)
{
	char out_val[32];

	if ((!exp_need)||(has_cf_creature(cr_ptr, CF_ANDROID)))
	{
		(void)sprintf(out_val, "%9ld", (long)cr_ptr->exp);
	}
	else
	{
		if (cr_ptr->lev >= cr_ptr->max_lev)
		{
			(void)sprintf(out_val, "*********");
		}
		else
		{
			(void)sprintf(out_val, "%9ld", (long)(creature_exp[cr_ptr->lev - 1] * cr_ptr->expfact / 100L) - cr_ptr->exp);
		}
	}

	if (cr_ptr->exp >= cr_ptr->max_exp)
	{
		if (has_cf_creature(cr_ptr, CF_ANDROID)) put_str("Cst ", ROW_EXP, 0);
		else put_str("EXP", ROW_EXP, COL_EXP);
		c_put_str(TERM_L_GREEN, out_val, ROW_EXP, COL_EXP + 3);

	}
	else
	{
		put_str("Exp", ROW_EXP, COL_EXP);
		c_put_str(TERM_YELLOW, out_val, ROW_EXP, COL_EXP + 3);
	}
}

/*
 * Prints current gold
 */
static void prt_gold(creature_type *cr_ptr)
{
	char tmp[32];

#ifdef JP
	put_str(" ", ROW_GOLD, COL_GOLD);
#else
	put_str("AU ", ROW_GOLD, COL_GOLD);
#endif

	sprintf(tmp, "%9ld", (long)cr_ptr->au);
	c_put_str(TERM_L_GREEN, tmp, ROW_GOLD, COL_GOLD + 3);
}



/*
 * Prints current AC
 */
static void prt_ac(creature_type *cr_ptr)
{
	char tmp[32];

#ifdef JP
/* AC ̕\ύXĂ */
	put_str("  AC (     )", ROW_AC, COL_AC);
	sprintf(tmp, "%5d", cr_ptr->dis_ac + cr_ptr->dis_to_a);
	c_put_str(TERM_L_GREEN, tmp, ROW_AC, COL_AC + 6);
#else
	put_str("Cur AC ", ROW_AC, COL_AC);
	sprintf(tmp, "%5d", cr_ptr->dis_ac + cr_ptr->dis_to_a);
	c_put_str(TERM_L_GREEN, tmp, ROW_AC, COL_AC + 7);
#endif

}


/*
 * Prints Cur/Max hit points
 */
static void prt_hp(creature_type *cr_ptr)
{
/* qbg|Cg̕\@ύX */
	char tmp[32];
  
	byte color;
  
	/* ^Cg */
/*	put_str(" goElo", ROW_HPMP, COL_HPMP); */

	put_str("HP", ROW_CURHP, COL_CURHP);

	/* ݂̃qbg|Cg */
	sprintf(tmp, "%5ld", cr_ptr->chp);

	if (cr_ptr->chp >= cr_ptr->mhp)
	{
		color = TERM_L_GREEN;
	}
	else if (cr_ptr->chp > (cr_ptr->mhp * hitpoint_warn) / 10)
	{
		color = TERM_YELLOW;
	}
	else
	{
		color = TERM_RED;
	}

	c_put_str(color, tmp, ROW_CURHP, COL_CURHP + 2);

	/* ؂ */
	put_str("/", ROW_CURHP, COL_CURHP + 7);

	/* őqbg|Cg */
	sprintf(tmp, "%5ld", cr_ptr->mhp);
	color = TERM_L_GREEN;

	c_put_str(color, tmp, ROW_CURHP, COL_CURHP + 8);
}


/*
 * Prints players max/cur spell points
 */
static void prt_sp(creature_type *cr_ptr)
{
/* }WbN|Cg̕\@ύXĂ */
	char tmp[32];
	byte color;

	/* ^Cg */
/*	put_str(" lo / ő", ROW_MAXSP, COL_MAXSP); */

#ifdef JP
	put_str("MP", ROW_CURSP, COL_CURSP);
#else
	put_str("SP", ROW_CURSP, COL_CURSP);
#endif

	/* ݂̃}WbN|Cg */
	sprintf(tmp, "%5ld", cr_ptr->csp);

	if (cr_ptr->csp >= cr_ptr->msp)
	{
		color = TERM_L_GREEN;
	}
	else if (cr_ptr->csp > (cr_ptr->msp * mana_warn) / 10)
	{
		color = TERM_YELLOW;
	}
	else
	{
		color = TERM_RED;
	}

	c_put_str(color, tmp, ROW_CURSP, COL_CURSP + 2);

	/* ؂ */
	put_str( "/", ROW_CURSP, COL_CURSP + 7 );

	/* ő}WbN|Cg */
	sprintf(tmp, "%5d", cr_ptr->msp);
	color = TERM_L_GREEN;

	c_put_str(color, tmp, ROW_CURSP, COL_CURSP + 8);
}



// Prints depth in stat area
static void prt_depth(creature_type *creature_ptr)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);
	char depths[32];
	int wid, hgt, row_depth, col_depth;
	byte attr = TERM_WHITE;

	Term_get_size(&wid, &hgt);
	col_depth = wid + COL_DEPTH;
	row_depth = hgt + ROW_DEPTH;

	if (!floor_ptr->floor_level)
	{
#ifdef JP
		strcpy(depths, "n");
#else
		strcpy(depths, "Surf.");
#endif
	}
	else if (inside_quest && !floor_ptr->dun_type)
	{
#ifdef JP
		strcpy(depths, "n");
#else
		strcpy(depths, "Quest");
#endif
	}
	else
	{
#ifdef JP
		if (depth_in_feet) (void)sprintf(depths, "%d ft", floor_ptr->floor_level * 50);
		else (void)sprintf(depths, "%d K", floor_ptr->floor_level);
#else
		if (depth_in_feet) (void)sprintf(depths, "%d ft", floor_ptr->floor_level * 50);
		else (void)sprintf(depths, "Lev %d", floor_ptr->floor_level);
#endif


		/* Get color of level based on feeling  -JSV- */
		switch (creature_ptr->floor_feeling)
		{
		case  0: attr = TERM_SLATE;   break; /* Unknown */
		case  1: attr = TERM_L_BLUE;  break; /* Special */
		case  2: attr = TERM_VIOLET;  break; /* Horrible visions */
		case  3: attr = TERM_RED;     break; /* Very dangerous */
		case  4: attr = TERM_L_RED;   break; /* Very bad feeling */
		case  5: attr = TERM_ORANGE;  break; /* Bad feeling */
		case  6: attr = TERM_YELLOW;  break; /* Nervous */
		case  7: attr = TERM_L_UMBER; break; /* Luck is turning */
		case  8: attr = TERM_L_WHITE; break; /* Don't like */
		case  9: attr = TERM_WHITE;   break; /* Reasonably safe */
		case 10: attr = TERM_WHITE;   break; /* Boring place */
		}
	}

	/* Right-Adjust the "depth", and clear old values */
	if(wizard)
	{
		c_prt(attr, format("%7s F:%3d", depths, player_ptr->floor_id), row_depth, col_depth - 6);
	}
	else
	{
		c_prt(attr, format("%7s", depths), row_depth, col_depth);
	}
}


/*
 * Prints status of hunger
 */
static void prt_hunger(creature_type *cr_ptr)
{
	byte color;
	char *expression;
	char tmp[30];

	if (cr_ptr->food < PY_FOOD_FAINT)
	{
		color = TERM_RED;
#ifdef JP
		expression = "  ";
		c_put_str(TERM_RED, "  ", ROW_HUNGRY, COL_HUNGRY);
#else
		expression = "Weak  ";
		c_put_str(TERM_RED, "Weak  ", ROW_HUNGRY, COL_HUNGRY);
#endif

	}
	else if (cr_ptr->food < PY_FOOD_WEAK)
	{
		color = TERM_ORANGE;
#ifdef JP
		expression = "  ";
		c_put_str(TERM_ORANGE, "  ", ROW_HUNGRY, COL_HUNGRY);
#else
		expression = "  ";
		c_put_str(TERM_ORANGE, "Weak  ", ROW_HUNGRY, COL_HUNGRY);
#endif

	}
	else if (cr_ptr->food < PY_FOOD_ALERT)
	{
		color = TERM_YELLOW;
#ifdef JP
		expression = "  ";
		c_put_str(TERM_YELLOW, "  ", ROW_HUNGRY, COL_HUNGRY);
#else
		expression = "Hungry";
		c_put_str(TERM_YELLOW, "Hungry", ROW_HUNGRY, COL_HUNGRY);
#endif
	}
	else if (cr_ptr->food < PY_FOOD_FULL)
	{
		color = TERM_L_GREEN;
		expression = "      ";
		c_put_str(TERM_L_GREEN, "      ", ROW_HUNGRY, COL_HUNGRY);
	}
	else if (cr_ptr->food < PY_FOOD_MAX)
	{
		color = TERM_L_GREEN;
#ifdef JP
		expression = "  ";
#else
		expression = "Full  ";
#endif

	}
	else
	{
		color = TERM_GREEN;
#ifdef JP
		expression = "H߂";
#else
		expression = "Gorged";
#endif

	}

	if(wizard) sprintf(tmp, "%6s(%d)", expression, cr_ptr->food);
	else sprintf(tmp, "%6s", expression);

	c_put_str(color, tmp, ROW_HUNGRY, COL_HUNGRY);
}


/*
 * Prints Searching, Resting, Paralysis, or 'count' status
 * Display is always exactly 10 characters wide (see below)
 *
 * This function was a major bottleneck when resting, so a lot of
 * the text formatting code was optimized in place below.
 */
static void prt_state(creature_type *cr_ptr)
{
	byte attr = TERM_WHITE;

	char text[5];

	/* Repeating */
	if (command_rep)
	{
		if (command_rep > 999)
		{
#ifdef JP
sprintf(text, "%2d00", command_rep / 100);
#else
			(void)sprintf(text, "%2d00", command_rep / 100);
#endif

		}
		else
		{
#ifdef JP
sprintf(text, "  %2d", command_rep);
#else
			(void)sprintf(text, "  %2d", command_rep);
#endif

		}
	}

	/* Action */
	else
	{
		switch(cr_ptr->action)
		{
			case ACTION_SEARCH:
			{
#ifdef JP
				strcpy(text, "T");
#else
				strcpy(text, "Sear");
#endif
				break;
			}
			case ACTION_REST:
			{
				int i;

				/* Start with "Rest" */
#ifdef JP
				strcpy(text, "    ");
#else
				strcpy(text, "    ");
#endif


				/* Extensive (timed) rest */
				if (resting >= 1000)
				{
					i = resting / 100;
					text[3] = '0';
					text[2] = '0';
					text[1] = '0' + (i % 10);
					text[0] = '0' + (i / 10);
				}

				/* Long (timed) rest */
				else if (resting >= 100)
				{
					i = resting;
					text[3] = '0' + (i % 10);
					i = i / 10;
					text[2] = '0' + (i % 10);
					text[1] = '0' + (i / 10);
				}

				/* Medium (timed) rest */
				else if (resting >= 10)
				{
					i = resting;
					text[3] = '0' + (i % 10);
					text[2] = '0' + (i / 10);
				}

				/* Short (timed) rest */
				else if (resting > 0)
				{
					i = resting;
					text[3] = '0' + (i);
				}

				/* Rest until healed */
				else if (resting == -1)
				{
					text[0] = text[1] = text[2] = text[3] = '*';
				}

				/* Rest until done */
				else if (resting == -2)
				{
					text[0] = text[1] = text[2] = text[3] = '&';
				}
				break;
			}
			case ACTION_LEARN:
			{
#ifdef JP
				strcpy(text, "wK");
#else
				strcpy(text, "lear");
#endif
				if (new_mane) attr = TERM_L_RED;
				break;
			}
			case ACTION_FISH:
			{
#ifdef JP
				strcpy(text, "ނ");
#else
				strcpy(text, "fish");
#endif
				break;
			}
			case ACTION_KAMAE:
			{
				int i;
				for (i = 0; i < MAX_KAMAE; i++)
					if (cr_ptr->special_defense & (KAMAE_GENBU << i)) break;
				switch (i)
				{
					case 0: attr = TERM_GREEN;break;
					case 1: attr = TERM_WHITE;break;
					case 2: attr = TERM_L_BLUE;break;
					case 3: attr = TERM_L_RED;break;
				}
				strcpy(text, kamae_shurui[i].desc);
				break;
			}
			case ACTION_KATA:
			{
				int i;
				for (i = 0; i < MAX_KATA; i++)
					if (cr_ptr->special_defense & (KATA_IAI << i)) break;
				strcpy(text, kata_shurui[i].desc);
				break;
			}
			case ACTION_SING:
			{
#ifdef JP
				strcpy(text, "  ");
#else
				strcpy(text, "Sing");
#endif
				break;
			}
			case ACTION_HAYAGAKE:
			{
#ifdef JP
				strcpy(text, "");
#else
				strcpy(text, "Fast");
#endif
				break;
			}
			case ACTION_SPELL:
			{
#ifdef JP
				strcpy(text, "r");
#else
				strcpy(text, "Spel");
#endif
				break;
			}
			default:
			{
				strcpy(text, "    ");
				break;
			}
		}
	}

	/* Display the info (or blanks) */
	c_put_str(attr, format("%5.5s",text), ROW_STATE, COL_STATE);
}


/*
 * Prints the speed of a character.			-CJS-
 */
static void prt_speed(creature_type *cr_ptr)
{
	int i = cr_ptr->speed;
	bool is_fast = IS_FAST(cr_ptr);

	byte attr = TERM_WHITE;
	char buf[32] = "";
	int wid, hgt, row_speed, col_speed;

	Term_get_size(&wid, &hgt);
	col_speed = wid + COL_SPEED;
	row_speed = hgt + ROW_SPEED;

	/* Hack -- Visually "undo" the Search Mode Slowdown */
	if (cr_ptr->action == ACTION_SEARCH && !cr_ptr->lightspeed) i += 10;

	/* Fast */
	if (i > 110)
	{
		if (cr_ptr->riding)
		{
			creature_type *m_ptr = &creature_list[cr_ptr->riding];
			if (m_ptr->fast && !m_ptr->slow) attr = TERM_L_BLUE;
			else if (m_ptr->slow && !m_ptr->fast) attr = TERM_VIOLET;
			else attr = TERM_GREEN;
		}
		else if ((is_fast && !cr_ptr->slow) || cr_ptr->lightspeed) attr = TERM_YELLOW;
		else if (cr_ptr->slow && !is_fast) attr = TERM_VIOLET;
		else attr = TERM_L_GREEN;
#ifdef JP
		sprintf(buf, "%s(+%d)", (cr_ptr->riding ? "n" : ""), (i - 110));
#else
		sprintf(buf, "Fast(+%d)", (i - 110));
#endif

	}

	/* Slow */
	else if (i < 110)
	{
		if (cr_ptr->riding)
		{
			creature_type *m_ptr = &creature_list[cr_ptr->riding];
			if (m_ptr->fast && !m_ptr->slow) attr = TERM_L_BLUE;
			else if (m_ptr->slow && !m_ptr->fast) attr = TERM_VIOLET;
			else attr = TERM_RED;
		}
		else if (is_fast && !cr_ptr->slow) attr = TERM_YELLOW;
		else if (cr_ptr->slow && !is_fast) attr = TERM_VIOLET;
		else attr = TERM_L_UMBER;
#ifdef JP
		sprintf(buf, "%s(-%d)", (cr_ptr->riding ? "n" : ""), (110 - i));
#else
		sprintf(buf, "Slow(-%d)", (110 - i));
#endif
	}
	else if (cr_ptr->riding)
	{
		attr = TERM_GREEN;
#ifdef JP
		strcpy(buf, "n");
#else
		strcpy(buf, "Riding");
#endif
	}

	/* Display the speed */
	c_put_str(attr, format("%-9s", buf), row_speed, col_speed);
}


static void prt_study(creature_type *cr_ptr)
{
	int wid, hgt, row_study, col_study;

	Term_get_size(&wid, &hgt);
	col_study = wid + COL_STUDY;
	row_study = hgt + ROW_STUDY;

	if (cr_ptr->new_spells)
	{
#ifdef JP
		put_str("wK", row_study, col_study);
#else
		put_str("Stud", row_study, col_study);
#endif

	}
	else
	{
		put_str("    ", row_study, col_study);
	}
}


static void prt_imitation(creature_type *cr_ptr)
{
	int wid, hgt, row_study, col_study;

	Term_get_size(&wid, &hgt);
	col_study = wid + COL_STUDY;
	row_study = hgt + ROW_STUDY;

	if (cr_ptr->cls_idx == CLASS_IMITATOR)
	{
		if (cr_ptr->mane_num)
		{
			byte attr;
			if (new_mane) attr = TERM_L_RED;
			else attr = TERM_WHITE;
#ifdef JP
			c_put_str(attr, "܂", row_study, col_study);
#else
			c_put_str(attr, "Imit", row_study, col_study);
#endif
		}
		else
		{
			put_str("    ", row_study, col_study);
		}
	}
}


static void prt_cut(creature_type *cr_ptr)
{
	int c = cr_ptr->cut;

	if (c > 1000)
	{
#ifdef JP
		c_put_str(TERM_L_RED, "v      ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_L_RED, "Mortal wound", ROW_CUT, COL_CUT);
#endif

	}
	else if (c > 200)
	{
#ifdef JP
		c_put_str(TERM_RED, "Ђǂ[  ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_RED, "Deep gash   ", ROW_CUT, COL_CUT);
#endif

	}
	else if (c > 100)
	{
#ifdef JP
		c_put_str(TERM_RED, "d        ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_RED, "Severe cut  ", ROW_CUT, COL_CUT);
#endif

	}
	else if (c > 50)
	{
#ifdef JP
		c_put_str(TERM_ORANGE, "ςȏ    ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_ORANGE, "Nasty cut   ", ROW_CUT, COL_CUT);
#endif

	}
	else if (c > 25)
	{
#ifdef JP
		c_put_str(TERM_ORANGE, "Ђǂ    ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_ORANGE, "Bad cut     ", ROW_CUT, COL_CUT);
#endif

	}
	else if (c > 10)
	{
#ifdef JP
		c_put_str(TERM_YELLOW, "y        ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_YELLOW, "Light cut   ", ROW_CUT, COL_CUT);
#endif

	}
	else if (c)
	{
#ifdef JP
		c_put_str(TERM_YELLOW, "菝    ", ROW_CUT, COL_CUT);
#else
		c_put_str(TERM_YELLOW, "Graze       ", ROW_CUT, COL_CUT);
#endif

	}
	else
	{
		put_str("            ", ROW_CUT, COL_CUT);
	}
}



static void prt_stun(creature_type *cr_ptr)
{
	int s = cr_ptr->stun;

	if (s > 100)
	{
#ifdef JP
		c_put_str(TERM_RED, "ӎs  ", ROW_STUN, COL_STUN);
#else
		c_put_str(TERM_RED, "Knocked out ", ROW_STUN, COL_STUN);
#endif

	}
	else if (s > 50)
	{
#ifdef JP
		c_put_str(TERM_ORANGE, "ЂǂNO  ", ROW_STUN, COL_STUN);
#else
		c_put_str(TERM_ORANGE, "Heavy stun  ", ROW_STUN, COL_STUN);
#endif

	}
	else if (s)
	{
#ifdef JP
		c_put_str(TERM_ORANGE, "NO        ", ROW_STUN, COL_STUN);
#else
		c_put_str(TERM_ORANGE, "Stun        ", ROW_STUN, COL_STUN);
#endif

	}
	else
	{
		put_str("            ", ROW_STUN, COL_STUN);
	}
}



/*
 * Redraw the "monster health bar"	-DRS-
 * Rather extensive modifications by	-BEN-
 *
 * The "monster health bar" provides visual feedback on the "health"
 * of the monster currently being "tracked".  There are several ways
 * to "track" a monster, including targetting it, attacking it, and
 * affecting it (and nobody else) with a ranged attack.
 *
 * Display the monster health bar (affectionately known as the
 * "health-o-meter").  Clear health bar if nothing is being tracked.
 * Auto-track current target monster when bored.  Note that the
 * health-bar stops tracking any monster that "disappears".
 */
static void health_redraw(creature_type *cr_ptr, bool riding)
{
	int row, col;
	creature_type *m_ptr;
	species_type *species_ptr;
	char k[2];

	if (riding)
	{
		health_who = cr_ptr->riding;
		row = ROW_RIDING_INFO;

		col = COL_RIDING_INFO;
	}
	else
	{
		row = ROW_INFO;
		col = COL_INFO;
	}

	m_ptr = &creature_list[health_who];
	species_ptr = &species_info[m_ptr->species_idx];

	/* Not tracking */

	if (!health_who)
	{
		// Erase the health bar
		Term_erase(col, row, 20);
	}


	/* Tracking an unseen monster */
	if (!m_ptr->ml)
	{
		/* Indicate that the monster health is "unknown" */
		Term_putstr(col, row, 16, TERM_WHITE, "  HP[----------]");
	}

	/* Tracking a hallucinatory monster */
	else if (cr_ptr->image)
	{
		/* Indicate that the monster health is "unknown" */
		Term_putstr(col, row, 16, TERM_WHITE, "  HP[----------]");
	}

	/* Tracking a dead monster (???) */
	else if (m_ptr->chp < 0)
	{
		/* Indicate that the monster health is "unknown" */
		Term_putstr(col, row, 16, TERM_WHITE, "  HP[----------]");
	}

	/* Tracking a visible monster */
	else
	{
		/* Extract the "percent" of health */
		int pct = 100L * m_ptr->chp / m_ptr->mhp;
		int pct2 = 100L * m_ptr->chp / m_ptr->mmhp;

		/* Convert percent into "health" */
		int len = (pct2 < 10) ? 1 : (pct2 < 90) ? (pct2 / 10 + 1) : 10;

		/* Default to almost dead */
		byte attr = TERM_RED;

		/* Invulnerable */
		if (m_ptr->invuln) attr = TERM_WHITE;

		/* Asleep */
		else if (m_ptr->paralyzed) attr = TERM_BLUE;

		/* Afraid */
		else if (m_ptr->afraid) attr = TERM_VIOLET;

		/* Healthy */
		else if (pct >= 100) attr = TERM_L_GREEN;

		/* Somewhat Wounded */
		else if (pct >= 60) attr = TERM_YELLOW;

		/* Wounded */
		else if (pct >= 25) attr = TERM_ORANGE;

		/* Badly wounded */
		else if (pct >= 10) attr = TERM_L_RED;

		/* Default to "unknown" */
		Term_putstr(col, row, 16, TERM_WHITE, "  HP[----------]");

		/* Dump the current "health" (use '*' symbols) */
		Term_putstr(col + 5, row, len, attr, "**********");

		if(health_who)
		{
			k[0] = species_ptr->x_char;
			k[1] = 0;
			Term_putstr(col, row, 1, species_ptr->x_attr, k);
		}

	}


}



/*
 * Display basic info (mostly left of map)
 */
static void prt_frame_basic(creature_type *cr_ptr)
{
	int i;

	/* Level/Experience */
	prt_level(cr_ptr);
	prt_exp(cr_ptr);

	/* All Stats */
	for (i = 0; i < 6; i++) prt_stat(cr_ptr, i);

	/* Armor */
	prt_ac(cr_ptr);

	/* Hitpoints */
	prt_hp(cr_ptr);

	/* Spellpoints */
	prt_sp(cr_ptr);

	/* Gold */
	prt_gold(cr_ptr);

	/* Current depth */
	prt_depth(cr_ptr);

	/* Special */
//TODO	//health_redraw(cr_ptr, FALSE);
}


/*
 * Display extra info (mostly below map)
 */
static void prt_frame_extra(creature_type *cr_ptr)
{
	/* Cut/Stun */
	prt_cut(cr_ptr);
	prt_stun(cr_ptr);

	/* Food */
	prt_hunger(cr_ptr);

	/* State */
	prt_state(cr_ptr);

	/* Speed */
	prt_speed(cr_ptr);

	/* Study spells */
	prt_study(cr_ptr);

	prt_imitation(cr_ptr);

	prt_status(cr_ptr);
}


/*
 * Hack -- display inventory in sub-windows
 */
static void fix_inven(creature_type *cr_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_INVEN))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Display inventory */
		display_inven(cr_ptr);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}



/*
 * Hack -- display equipment in sub-windows
 */
static void fix_equip(creature_type *player_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_EQUIP))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Display equipment */
		display_equip(player_ptr);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display equipment in sub-windows
 */
static void fix_spell(creature_type *creature_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_SPELL))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Display spell list */
		display_spell_list(creature_ptr);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display character in sub-windows
 */
static void fix_player(creature_type *player_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_PLAYER))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		update_playtime();

		/* Display player */
		display_creature_status(0, player_ptr);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}



/*
 * Hack -- display recent messages in sub-windows
 *
 * XXX XXX XXX Adjust for width and split messages
 */
static void fix_message(void)
{
	int j, i;
	int w, h;
	int x, y;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_MESSAGE))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Get size */
		Term_get_size(&w, &h);

		/* Dump messages */
		for (i = 0; i < h; i++)
		{
			/* Dump the message on the appropriate line */
			Term_putstr(0, (h - 1) - i, -1, (byte)((i < now_message) ? TERM_WHITE : TERM_SLATE), message_str((s16b)i));

			/* Cursor */
			Term_locate(&x, &y);

			/* Clear to end of line */
			Term_erase(x, y, 255);
		}

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display overhead view in sub-windows
 *
 * Note that the "player" symbol does NOT appear on the map.
 */
static void fix_overhead(creature_type *watcher_ptr)
{
	int j;

	int cy, cx;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;
		int wid, hgt;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_OVERHEAD))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Full map in too small window is useless  */
		Term_get_size(&wid, &hgt);
		if (wid > COL_MAP + 2 && hgt > ROW_MAP + 2)
		{
			/* Redraw map */
			display_map(watcher_ptr, &cy, &cx);

			/* Fresh */
			Term_fresh();
		}

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display dungeon view in sub-windows
 */
static void fix_dungeon(creature_type *creature_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_DUNGEON))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Redraw dungeon view */
		display_dungeon(creature_ptr);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display monster recall in sub-windows
 */
static void fix_monster(creature_type *cr_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_MONSTER))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Display monster race info */
		if (species_window_idx) display_roff(species_window_idx);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Hack -- display object recall in sub-windows
 */
static void fix_object(creature_type *creature_ptr)
{
	int j;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		term *old = Term;

		/* No window */
		if (!angband_term[j]) continue;

		/* No relevant flags */
		if (!(window_flag[j] & (PW_OBJECT))) continue;

		/* Activate */
		Term_activate(angband_term[j]);

		/* Display monster race info */
		if (play_window_object_kind_idx) display_koff(creature_ptr, play_window_object_kind_idx);

		/* Fresh */
		Term_fresh();

		/* Restore */
		Term_activate(old);
	}
}


/*
 * Calculate number of spells player should have, and forget,
 * or remember, spells until that number is properly reflected.
 *
 * Note that this function induces various "status" messages,
 * which must be bypasses until the character is created.
 */
static void calc_spells(creature_type *cr_ptr, bool message)
{
	int			i, j, k, levels;
	int			num_allowed;
	int                     num_boukyaku = 0;

	magic_type		*s_ptr;
	int which;
	int bonus = 0;


	cptr p;

	/* Hack -- must be literate */
	if (!m_info[cr_ptr->cls_idx].spell_book) return;

	/* Hack -- wait for creation */
	if (!character_generated) return;

	if ((cr_ptr->cls_idx == CLASS_SORCERER) || (cr_ptr->cls_idx == CLASS_RED_MAGE))
	{
		cr_ptr->new_spells = 0;
		return;
	}

	p = spell_category_name(m_info[cr_ptr->cls_idx].spell_book);

	/* Determine the number of spells allowed */
	levels = cr_ptr->lev - m_info[cr_ptr->cls_idx].spell_first + 1;

	/* Hack -- no negative spells */
	if (levels < 0) levels = 0;

	/* Extract total allowed spells */
	num_allowed = (adj_mag_study[cr_ptr->stat_ind[m_info[cr_ptr->cls_idx].spell_stat]] * levels / 2);

	if ((cr_ptr->cls_idx != CLASS_SAMURAI) && (m_info[cr_ptr->cls_idx].spell_book != TV_LIFE_BOOK))
	{
		bonus = 4;
	}
	if (cr_ptr->cls_idx == CLASS_SAMURAI)
	{
		num_allowed = 32;
	}
	else if (cr_ptr->realm2 == REALM_NONE)
	{
		num_allowed = (num_allowed+1)/2;
		if (num_allowed>(32+bonus)) num_allowed = 32+bonus;
	}
	else if ((cr_ptr->cls_idx == CLASS_MAGE) || (cr_ptr->cls_idx == CLASS_PRIEST))
	{
		if (num_allowed>(96+bonus)) num_allowed = 96+bonus;
	}
	else
	{
		if (num_allowed>(80+bonus)) num_allowed = 80+bonus;
	}

	/* Count the number of spells we know */
	for (j = 0; j < 64; j++)
	{
		/* Count known spells */
		if ((j < 32) ?
		    (cr_ptr->spell_forgotten1 & (1L << j)) :
		    (cr_ptr->spell_forgotten2 & (1L << (j - 32))))
		{
			num_boukyaku++;
		}
	}

	/* See how many spells we must forget or may learn */
	cr_ptr->new_spells = num_allowed + cr_ptr->add_spells + num_boukyaku - cr_ptr->learned_spells;

	/* Forget spells which are too hard */

	for (i = 63; i >= 0; i--)
	{
		/* Efficiency -- all done */
		if (!cr_ptr->spell_learned1 && !cr_ptr->spell_learned2) break;

		/* Access the spell */
		j = cr_ptr->spell_order[i];

	/* Skip non-spells */
		if (j >= 99) continue;


		/* Get the spell */
		if (!is_magic((j < 32) ? cr_ptr->realm1 : cr_ptr->realm2))
		{
			if (j < 32)
				s_ptr = &technic_info[cr_ptr->realm1 - MIN_TECHNIC][j];
			else
				s_ptr = &technic_info[cr_ptr->realm2 - MIN_TECHNIC][j%32];
		}
		else if (j < 32)
			s_ptr = &m_info[cr_ptr->cls_idx].info[cr_ptr->realm1-1][j];
		else
			s_ptr = &m_info[cr_ptr->cls_idx].info[cr_ptr->realm2-1][j%32];

		/* Skip spells we are allowed to know */
		if (s_ptr->slevel <= cr_ptr->lev) continue;

		/* Is it known? */
		if ((j < 32) ?
		    (cr_ptr->spell_learned1 & (1L << j)) :
		    (cr_ptr->spell_learned2 & (1L << (j - 32))))
		{
			/* Mark as forgotten */
			if (j < 32)
			{
				cr_ptr->spell_forgotten1 |= (1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_forgotten2 |= (1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* No longer known */
			if (j < 32)
			{
				cr_ptr->spell_learned1 &= ~(1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_learned2 &= ~(1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* Message */
#ifdef JP
			if(message) msg_format("%s%sYĂ܂B",
				   do_spell(cr_ptr, which, j%32, SPELL_NAME), p );
#else
			if(message) msg_format("You have forgotten the %s of %s.", p,
					do_spell(cr_ptr, which, j%32, SPELL_NAME));
#endif


			/* One more can be learned */
			cr_ptr->new_spells++;
		}
	}


	/* Forget spells if we know too many spells */
	for (i = 63; i >= 0; i--)
	{
		/* Stop when possible */
		if (cr_ptr->new_spells >= 0) break;

		/* Efficiency -- all done */
		if (!cr_ptr->spell_learned1 && !cr_ptr->spell_learned2) break;

		/* Get the (i+1)th spell learned */
		j = cr_ptr->spell_order[i];

		/* Skip unknown spells */
		if (j >= 99) continue;

		/* Forget it (if learned) */
		if ((j < 32) ?
		    (cr_ptr->spell_learned1 & (1L << j)) :
		    (cr_ptr->spell_learned2 & (1L << (j - 32))))
		{
			/* Mark as forgotten */
			if (j < 32)
			{
				cr_ptr->spell_forgotten1 |= (1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_forgotten2 |= (1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* No longer known */
			if (j < 32)
			{
				cr_ptr->spell_learned1 &= ~(1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_learned2 &= ~(1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* Message */
#ifdef JP
			if(message) msg_format("%s%sYĂ܂B",
				   do_spell(cr_ptr, which, j%32, SPELL_NAME), p );
#else
			if(message) msg_format("You have forgotten the %s of %s.", p,
				   do_spell(cr_ptr, which, j%32, SPELL_NAME));
#endif


			/* One more can be learned */
			cr_ptr->new_spells++;
		}
	}


	/* Check for spells to remember */
	for (i = 0; i < 64; i++)
	{
		/* None left to remember */
		if (cr_ptr->new_spells <= 0) break;

		/* Efficiency -- all done */
		if (!cr_ptr->spell_forgotten1 && !cr_ptr->spell_forgotten2) break;

		/* Get the next spell we learned */
		j = cr_ptr->spell_order[i];

		/* Skip unknown spells */
		if (j >= 99) break;

		/* Access the spell */
		if (!is_magic((j < 32) ? cr_ptr->realm1 : cr_ptr->realm2))
		{
			if (j < 32)
				s_ptr = &technic_info[cr_ptr->realm1 - MIN_TECHNIC][j];
			else
				s_ptr = &technic_info[cr_ptr->realm2 - MIN_TECHNIC][j%32];
		}
		else if (j<32)
			s_ptr = &m_info[cr_ptr->cls_idx].info[cr_ptr->realm1-1][j];
		else
			s_ptr = &m_info[cr_ptr->cls_idx].info[cr_ptr->realm2-1][j%32];

		/* Skip spells we cannot remember */
		if (s_ptr->slevel > cr_ptr->lev) continue;

		/* First set of spells */
		if ((j < 32) ?
		    (cr_ptr->spell_forgotten1 & (1L << j)) :
		    (cr_ptr->spell_forgotten2 & (1L << (j - 32))))
		{
			/* No longer forgotten */
			if (j < 32)
			{
				cr_ptr->spell_forgotten1 &= ~(1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_forgotten2 &= ~(1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* Known once more */
			if (j < 32)
			{
				cr_ptr->spell_learned1 |= (1L << j);
				which = cr_ptr->realm1;
			}
			else
			{
				cr_ptr->spell_learned2 |= (1L << (j - 32));
				which = cr_ptr->realm2;
			}

			/* Message */
#ifdef JP
			if(message) msg_format("%s%svoB",
				   do_spell(cr_ptr, which, j%32, SPELL_NAME), p );
#else
			if(message) msg_format("You have remembered the %s of %s.",
				   p, do_spell(cr_ptr, which, j%32, SPELL_NAME));
#endif


			/* One less can be learned */
			cr_ptr->new_spells--;
		}
	}

	k = 0;

	if (cr_ptr->realm2 == REALM_NONE)
	{
		/* Count spells that can be learned */
		for (j = 0; j < 32; j++)
		{
			if (!is_magic(cr_ptr->realm1)) s_ptr = &technic_info[cr_ptr->realm1-MIN_TECHNIC][j];
			else s_ptr = &m_info[cr_ptr->cls_idx].info[cr_ptr->realm1-1][j];

			/* Skip spells we cannot remember */
			if (s_ptr->slevel > cr_ptr->lev) continue;

			/* Skip spells we already know */
			if (cr_ptr->spell_learned1 & (1L << j))
			{
				continue;
			}

			/* Count it */
			k++;
		}
		if (k>32) k = 32;
		if ((cr_ptr->new_spells > k) && ((m_info[cr_ptr->cls_idx].spell_book == TV_LIFE_BOOK) || (m_info[cr_ptr->cls_idx].spell_book == TV_HISSATSU_BOOK))) cr_ptr->new_spells = k;
	}

	if (cr_ptr->new_spells < 0) cr_ptr->new_spells = 0;

	/* Spell count changed */
	if (cr_ptr->old_spells != cr_ptr->new_spells)
	{
		/* Message if needed */
		if (cr_ptr->new_spells)
		{
			/* Message */
#ifdef JP
			if( cr_ptr->new_spells < 10 ){
				if(message) msg_format(" %d %swׂB", cr_ptr->new_spells, p);
			}else{
				if(message) msg_format(" %d %swׂB", cr_ptr->new_spells, p);
			}
#else
			if(message) msg_format("You can learn %d more %s%s.",
				   cr_ptr->new_spells, p,
				   (cr_ptr->new_spells != 1) ? "s" : "");
#endif

		}

		/* Save the new_spells value */
		cr_ptr->old_spells = cr_ptr->new_spells;

		/* Redraw Study Status */
		play_redraw |= (PR_STUDY);

		/* Redraw object recall */
		play_window |= (PW_OBJECT);
	}
}


/*
 * Calculate maximum mana.  You do not need to know any spells.
 * Note that mana is lowered by heavy (or inappropriate) armor.
 *
 * This function induces status messages.
 */
static void calc_mana(creature_type *cr_ptr, bool message)
{
	int		msp, levels, cur_wgt, max_wgt;

	object_type	*o_ptr;

	levels = cr_ptr->lev;

	switch(cr_ptr->cls_idx)
	{
		case CLASS_SAMURAI:
			msp = (adj_mag_mana[cr_ptr->stat_ind[m_info[cr_ptr->cls_idx].spell_stat]] + 10) * 2;
			if (msp) msp += (msp * race_info[cr_ptr->race_idx1].r_adj[m_info[cr_ptr->cls_idx].spell_stat] / 20);
			break;

		default:
			/* Extract total mana */
			msp = adj_mag_mana[cr_ptr->stat_ind[STAT_INT]] * (levels+3) / 4;

			/* Hack -- usually add one mana */
			msp++;

			if (msp) msp += (msp * race_info[cr_ptr->race_idx1].r_adj[m_info[cr_ptr->cls_idx].spell_stat] / 20);

			if (msp && (cr_ptr->chara_idx == CHARA_MUNCHKIN)) msp += msp / 2;

			/* Hack: High mages have a 25% mana bonus */
			if (msp && (cr_ptr->cls_idx == CLASS_MAGE)) msp += msp / 6;
			if (msp && (cr_ptr->cls_idx == CLASS_HIGH_MAGE)) msp += msp / 4;
	}

	/* Only mages are affected */
	if (m_info[cr_ptr->cls_idx].spell_xtra & MAGIC_GLOVE_REDUCE_MANA)
	{
		u32b flgs[TR_FLAG_SIZE];

		/* Assume player is not encumbered by gloves */
		cr_ptr->cumber_glove = FALSE;

		/* Get the gloves */
		o_ptr = get_equipped_slot_ptr(cr_ptr, INVEN_SLOT_ARMS, 1);

		/* Examine the gloves */
		object_flags(o_ptr, flgs);

		/* Normal gloves hurt mage-type spells */
		if (o_ptr->k_idx &&
		    !(have_flag(flgs, TR_FREE_ACT)) &&
		    !(have_flag(flgs, TR_MAGIC_MASTERY)) &&
		    !((have_flag(flgs, TR_DEX)) && (o_ptr->pval > 0)))
		{
			/* Encumbered */
			cr_ptr->cumber_glove = TRUE;

			/* Reduce mana */
			msp = (3 * msp) / 4;
		}
	}


	/* Assume player not encumbered by armor */
	cr_ptr->cumber_armor = FALSE;

	/* Weigh the armor */
	cur_wgt = 0;


	/*TODO:ADJUST
	if(cr_ptr->inventory[].tval> TV_SWORD) cur_wgt += cr_ptr->inventory[].weight;
	if(cr_ptr->inventory[].tval> TV_SWORD) cur_wgt += cr_ptr->inventory[].weight;
	cur_wgt += cr_ptr->inventory[].weight;
	cur_wgt += cr_ptr->inventory[].weight;
	//TODO cur_wgt += cr_ptr->inventory[].weight;
	cur_wgt += cr_ptr->inventory[].weight;
	cur_wgt += cr_ptr->inventory[].weight;

	// Subtract a percentage of maximum mana.
	switch (cr_ptr->cls_idx)
	{
		// For these classes, mana is halved if armour 
		// is 30 pounds over their weight limit.
		case CLASS_MAGE:
		case CLASS_HIGH_MAGE:
		case CLASS_BLUE_MAGE:
		case CLASS_MONK:
		case CLASS_FORCETRAINER:
		case CLASS_SORCERER:
		{
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight;
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight;
			break;
		}

		// Mana halved if armour is 40 pounds over weight limit.
		case CLASS_PRIEST:
		case CLASS_BARD:
		case CLASS_TOURIST:
		{
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight*2/3;
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight*2/3;
			break;
		}

		case CLASS_MINDCRAFTER:
		case CLASS_BEASTMASTER:
		case CLASS_MIRROR_MASTER:
		{
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/2;
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/2;
			break;
		}

		// Mana halved if armour is 50 pounds over weight limit.
		case CLASS_ROGUE:
		case CLASS_RANGER:
		case CLASS_RED_MAGE:
		case CLASS_WARRIOR_MAGE:
		{
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/3;
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/3;
			break;
		}

		// Mana halved if armour is 60 pounds over weight limit.
		case CLASS_PALADIN:
		case CLASS_CHAOS_WARRIOR:
		{
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/5;
			if (cr_ptr->inventory[].tval <= TV_SWORD) cur_wgt += cr_ptr->inventory[].weight/5;
			break;
		}

		// For new classes created, but not yet added to this formula.
		default:
		{
			break;
		}
	}
	*/

	/* Determine the weight allowance */
	max_wgt = m_info[cr_ptr->cls_idx].spell_weight;

	/* Heavy armor penalizes mana by a percentage.  -LM- */
	if ((cur_wgt - max_wgt) > 0)
	{
		/* Encumbered */
		cr_ptr->cumber_armor = TRUE;

		/* Subtract a percentage of maximum mana. */
		switch (cr_ptr->cls_idx)
		{
			/* For these classes, mana is halved if armour 
			 * is 30 pounds over their weight limit. */
			case CLASS_MAGE:
			case CLASS_HIGH_MAGE:
			case CLASS_BLUE_MAGE:
			{
				msp -= msp * (cur_wgt - max_wgt) / 600;
				break;
			}

			/* Mana halved if armour is 40 pounds over weight limit. */
			case CLASS_PRIEST:
			case CLASS_MINDCRAFTER:
			case CLASS_BEASTMASTER:
			case CLASS_BARD:
			case CLASS_FORCETRAINER:
			case CLASS_TOURIST:
			case CLASS_MIRROR_MASTER:
			{
				msp -= msp * (cur_wgt - max_wgt) / 800;
				break;
			}

			case CLASS_SORCERER:
			{
				msp -= msp * (cur_wgt - max_wgt) / 900;
				break;
			}

			/* Mana halved if armour is 50 pounds over weight limit. */
			case CLASS_ROGUE:
			case CLASS_RANGER:
			case CLASS_MONK:
			case CLASS_RED_MAGE:
			{
				msp -= msp * (cur_wgt - max_wgt) / 1000;
				break;
			}

			/* Mana halved if armour is 60 pounds over weight limit. */
			case CLASS_PALADIN:
			case CLASS_CHAOS_WARRIOR:
			case CLASS_WARRIOR_MAGE:
			{
				msp -= msp * (cur_wgt - max_wgt) / 1200;
				break;
			}

			case CLASS_SAMURAI:
			{
				cr_ptr->cumber_armor = FALSE;
				break;
			}

			/* For new classes created, but not yet added to this formula. */
			default:
			{
				msp -= msp * (cur_wgt - max_wgt) / 800;
				break;
			}
		}
	}

	/* Mana can never be negative */
	if (msp < 0) msp = 0;


	/* Maximum mana has changed */
	if (cr_ptr->msp != msp)
	{
		/* Enforce maximum */
		if ((cr_ptr->csp >= msp) && (cr_ptr->cls_idx != CLASS_SAMURAI))
		{
			cr_ptr->csp = msp;
			cr_ptr->csp_frac = 0;
		}

#ifdef JP
		/* xAbv̎͏㏸ʂ\ */
		if ((level_up == 1) && (msp > cr_ptr->msp))
		{
			if(message) msg_format("ő}WbNE|Cg %d I",
				   (msp - cr_ptr->msp));
		}
#endif
		/* Save new mana */
		cr_ptr->msp = msp;

		/* Display mana later */
		play_redraw |= (PR_MANA);

		/* Window stuff */
		play_window |= (PW_PLAYER);
		play_window |= (PW_SPELL);
	}

	/* Take note when "glove state" changes */
	if (cr_ptr->old_cumber_glove != cr_ptr->cumber_glove)
	{
		/* Message */
		if (cr_ptr->cumber_glove)
		{
#ifdef JP
			if(message) msg_print("肪ĎɂB");
#else
			if(message) msg_print("Your covered hands feel unsuitable for spellcasting.");
#endif

		}
		else
		{
#ifdef JP
			if(message) msg_print("̎̏ԂȂAƎ₷B");
#else
			if(message) msg_print("Your hands feel more suitable for spellcasting.");
#endif

		}

		/* Save it */
		cr_ptr->old_cumber_glove = cr_ptr->cumber_glove;
	}


	/* Take note when "armor state" changes */
	if (cr_ptr->old_cumber_armor != cr_ptr->cumber_armor)
	{
		/* Message */
		if (cr_ptr->cumber_armor)
		{
#ifdef JP
			if(message) msg_print("̏dœ݂ȂĂ܂ĂB");
#else
			if(message) msg_print("The weight of your equipment encumbers your movement.");
#endif

		}
		else
		{
#ifdef JP
			if(message) msg_print("Ɗyɑ̂𓮂悤ɂȂB");
#else
			if(message) msg_print("You feel able to move more freely.");
#endif

		}

		/* Save it */
		cr_ptr->old_cumber_armor = cr_ptr->cumber_armor;
	}
}



/*
 * Calculate the players (maximal) hit points
 * Adjust current hitpoints if necessary
 */
static void calc_hitpoints(creature_type *cr_ptr, bool message)
{
	int bonus, mhp;
	byte tmp_hitdice;

	/* Un-inflate "half-hitpoint bonus per level" value */
	bonus = ((int)(adj_con_mhp[cr_ptr->stat_ind[STAT_CON]]) - 128) * cr_ptr->lev / 4;

	/* Divine Bonuses */
	if(cr_ptr->dr >= 0){
		bonus += adj_dr_mhp[cr_ptr->dr] * cr_ptr->hitdice;
	}

	/* Calculate hitpoints */
	mhp = cr_ptr->base_hp[cr_ptr->lev - 1];

	if (cr_ptr->mimic_form)
	{
		if (cr_ptr->cls_idx == CLASS_SORCERER) tmp_hitdice = mimic_info[cr_ptr->mimic_form].r_mhp/2;
		else tmp_hitdice = (byte)mimic_info[cr_ptr->mimic_form].r_mhp;

		if (cr_ptr->cls_idx != INDEX_NONE) tmp_hitdice += classkill_info[cr_ptr->cls_idx].c_mhp;
		if (cr_ptr->chara_idx != INDEX_NONE) tmp_hitdice += chara_info[cr_ptr->chara_idx].a_mhp;

		mhp = mhp * tmp_hitdice / cr_ptr->hitdice;
	}

	if (cr_ptr->cls_idx == CLASS_SORCERER)
	{
		if (cr_ptr->lev < 30)
			mhp = (mhp * (45+cr_ptr->lev) / 100);
		else
			mhp = (mhp * 75 / 100);
		bonus = (bonus * 65 / 100);
	}

	mhp += bonus;

	if (cr_ptr->cls_idx == CLASS_BERSERKER)
	{
		mhp = mhp * (110+(((cr_ptr->lev + 40) * (cr_ptr->lev + 40) - 1550) / 110))/100;
	}

	/* Always have at least one hitpoint per level */
	if (mhp < cr_ptr->lev + 1) mhp = cr_ptr->lev + 1;

	/* Factor in the hero / superhero settings */
	if (IS_HERO(cr_ptr)) mhp += 10;
	if (cr_ptr->shero && (cr_ptr->cls_idx != CLASS_BERSERKER)) mhp += 30;
	if (cr_ptr->tsuyoshi) mhp += 50;

	/* Factor in the hex spell settings */
	if (hex_spelling(cr_ptr, HEX_XTRA_MIGHT)) mhp += 15;
	if (hex_spelling(cr_ptr, HEX_BUILDING)) mhp += 60;

	/* New maximum hitpoints */
	if (cr_ptr->mhp != mhp)
	{
		/* Enforce maximum */
		if (cr_ptr->chp >= mhp)
		{
			cr_ptr->chp = mhp;
			cr_ptr->chp_frac = 0;
		}

#ifdef JP
		/* xAbv̎͏㏸ʂ\ */
		if ((level_up == 1) && (mhp > cr_ptr->mhp))
		{
			if(message) msg_format("őqbgE|Cg %d I", (mhp - cr_ptr->mhp) );
		}
#endif
		/* Save the new max-hitpoints */
		cr_ptr->mhp = mhp;
		cr_ptr->mmhp = mhp;

		/* Display hitpoints (later) */
		play_redraw |= (PR_HP);

		/* Window stuff */
		play_window |= (PW_PLAYER);
	}
}



/*
 * Extract and set the current "lite radius"
 *
 * SWD: Experimental modification: multiple light sources have additive effect.
 *
 */
static void calc_lite(creature_type *creature_ptr)
{
	int i;
	object_type *o_ptr;
	u32b flgs[TR_FLAG_SIZE];
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	// Assume no light
	creature_ptr->cur_lite = 0;

	// Loop through all wielded items
	for (i = 0; i < INVEN_TOTAL; i++)
	{
		o_ptr = &creature_ptr->inventory[i];

		// Examine actual lites
		if (GET_INVEN_SLOT_TYPE(creature_ptr, i) == INVEN_SLOT_LITE && (o_ptr->k_idx) && (o_ptr->tval == TV_LITE))
		{
			if (o_ptr->name2 == EGO_LITE_DARKNESS)
			{
				if (o_ptr->sval == SV_LITE_TORCH)
				{
					creature_ptr->cur_lite -= 2;
				}

				// Lanterns (with fuel) provide more lite
				else if (o_ptr->sval == SV_LITE_LANTERN)
				{
					creature_ptr->cur_lite -= 2;
				}

				else if (o_ptr->sval == SV_LITE_FEANOR)
				{
					creature_ptr->cur_lite -= 3;
				}
			}
			/* Torches (with fuel) provide some lite */
			else if ((o_ptr->sval == SV_LITE_TORCH) && (o_ptr->xtra4 > 0))
			{
				creature_ptr->cur_lite += 2;
			}

			/* Lanterns (with fuel) provide more lite */
			else if ((o_ptr->sval == SV_LITE_LANTERN) && (o_ptr->xtra4 > 0))
			{
				creature_ptr->cur_lite += 2;
			}

			else if (o_ptr->sval == SV_LITE_FEANOR || o_ptr->sval == SV_LITE_UDUN)
			{
				creature_ptr->cur_lite += 3;
			}

			/* Artifact Lites provide permanent, bright, lite */
			else if (object_is_fixed_artifact(o_ptr))
			{
				creature_ptr->cur_lite += 4;
			}

			if (o_ptr->name2 == EGO_LITE_SHINE) creature_ptr->cur_lite++;
		}
		else
		{
			/* Skip empty slots */
			if (!o_ptr->k_idx) continue;

			/* Extract the flags */
			object_flags(o_ptr, flgs);

			/* does this item glow? */
			if (have_flag(flgs, TR_LITE))
			{
				if ((o_ptr->name2 == EGO_DARK) || (o_ptr->name1 == ART_NIGHT)) creature_ptr->cur_lite--;
				else creature_ptr->cur_lite++;
			}
		}

	}

	/* max radius is 14 (was 5) without rewriting other code -- */
	/* see cave.c:update_lite() and defines.h:LITE_MAX */
	if (dungeon_info[floor_ptr->dun_type].flags1 & DF1_DARKNESS && creature_ptr->cur_lite > 1)
		creature_ptr->cur_lite = 1;

	/*
	 * check if the player doesn't have light radius, 
	 * but does weakly glow as an intrinsic.
	 */
	if (creature_ptr->cur_lite <= 0 && creature_ptr->lite) creature_ptr->cur_lite++;

	if (creature_ptr->cur_lite > 14) creature_ptr->cur_lite = 14;
	if (creature_ptr->cur_lite < 0) creature_ptr->cur_lite = 0;

	/* end experimental mods */

	/* Notice changes in the "lite radius" */
	if (creature_ptr->old_lite != creature_ptr->cur_lite)
	{
		/* Update stuff */
		/* Hack -- PU_MON_LITE for monsters' darkness */
		update |= (PU_LITE | PU_MON_LITE | PU_MONSTERS);

		/* Remember the old lite */
		creature_ptr->old_lite = creature_ptr->cur_lite;

		if ((creature_ptr->cur_lite > 0) && (creature_ptr->special_defense & NINJA_S_STEALTH))
			set_superstealth(creature_ptr, FALSE);
	}
}



/*
 * Computes current weight limit.
 */
u32b weight_limit(creature_type *cr_ptr)
{
	u32b i;

	// Weight limit based only on strength
	// Constant was 100
	i = (u32b)adj_str_wgt[cr_ptr->stat_ind[STAT_STR]] * 25;
	i *= cr_ptr->size / 10 * cr_ptr->size / 10 * cr_ptr->size / 10;
	i += (u32b)adj_str_wgt[cr_ptr->stat_ind[STAT_STR]] * 25;
	if (cr_ptr->cls_idx == CLASS_BERSERKER) i = i * 3 / 2;

	// Return the result
	return i;
}

/*
 * Calculate the players current "state", taking into account
 * not only race/class intrinsics, but also objects being worn
 * and temporary spell effects.
 *
 * See also calc_mana() and calc_hitpoints().
 *
 * Take note of the new "speed code", in particular, a very strong
 * player will start slowing down as soon as he reaches 150 pounds,
 * but not until he reaches 450 pounds will he be half as fast as
 * a normal kobold.  This both hurts and helps the player, hurts
 * because in the old days a player could just avoid 300 pounds,
 * and helps because now carrying 300 pounds is not very painful.
 *
 * The "weapon" and "bow" do *not* add to the bonuses to hit or to
 * damage, since that would affect non-combat things.  These values
 * are actually added in later, at the appropriate place.
 *
 * This function induces various "status" messages.
 */
void calc_bonuses(creature_type *creature_ptr, bool message)
{
	int             i, j, k, hold;
	int             new_speed;
	int             body_size;
	int             default_hand = 0;
	int             empty_hands_status = empty_hands(creature_ptr, TRUE);
	int             extra_blows[2];
	int             extra_shots;

	floor_type      *floor_ptr = get_floor_ptr(creature_ptr);
	object_type     *o_ptr;
	u32b flgs[TR_FLAG_SIZE];
	bool            omoi = FALSE;
	bool            yoiyami = FALSE;
	bool            down_saving = FALSE;
#if 0
	bool            have_dd_s = FALSE, have_dd_t = FALSE;
#endif
	bool            have_sw = FALSE, have_kabe = FALSE;
	bool            easy_2weapon = FALSE;
	bool            riding_levitation = FALSE;
	s16b this_object_idx, next_object_idx = 0;

	race_type *tmp_rcreature_ptr;
	race_type *tmp_rcreature_ptr2;

	/* Save the old vision stuff */
	bool old_telepathy = creature_ptr->telepathy;
	bool old_esp_animal = creature_ptr->esp_animal;
	bool old_esp_undead = creature_ptr->esp_undead;
	bool old_esp_demon = creature_ptr->esp_demon;
	bool old_esp_orc = creature_ptr->esp_orc;
	bool old_esp_troll = creature_ptr->esp_troll;
	bool old_esp_giant = creature_ptr->esp_giant;
	bool old_esp_dragon = creature_ptr->esp_dragon;
	bool old_esp_human = creature_ptr->esp_human;
	bool old_esp_evil = creature_ptr->esp_evil;
	bool old_esp_good = creature_ptr->esp_good;
	bool old_esp_nonliving = creature_ptr->esp_nonliving;
	bool old_esp_unique = creature_ptr->esp_unique;
	bool old_see_inv = creature_ptr->see_inv;
	bool old_mighty_throw = creature_ptr->mighty_throw;

	/* Save the old armor class */
	bool old_dis_ac = (bool)creature_ptr->dis_ac;
	bool old_dis_to_a = (bool)creature_ptr->dis_to_a;

	species_type *species_ptr = &species_info[creature_ptr->species_idx];

	/* Clear extra blows/shots */
	extra_blows[0] = extra_blows[1] = extra_shots = 0;

	/* Clear the stat modifiers */
	for (i = 0; i < 6; i++) creature_ptr->stat_add[i] = 0;

	/* Clear the Displayed/Real armor class */
	creature_ptr->dis_ac = creature_ptr->ac = 0;

	/* Clear the Displayed/Real Bonuses */
	creature_ptr->dis_to_h[0] = creature_ptr->to_h[0] = 0;
	creature_ptr->dis_to_h[1] = creature_ptr->to_h[1] = 0;
	creature_ptr->dis_to_d[0] = creature_ptr->to_d[0] = 0;
	creature_ptr->dis_to_d[1] = creature_ptr->to_d[1] = 0;
	creature_ptr->dis_to_h_b = creature_ptr->to_h_b = 0;
	creature_ptr->dis_to_a = creature_ptr->to_a = 0;
	creature_ptr->to_h_m = 0;
	creature_ptr->to_d_m = 0;

	creature_ptr->to_m_chance = 0;

	/* Clear the Extra Dice Bonuses */
	creature_ptr->to_dd[0] = creature_ptr->to_ds[0] = 0;
	creature_ptr->to_dd[1] = creature_ptr->to_ds[1] = 0;

	/* Clear the Activate Rate */
	creature_ptr->to_ar[0] = creature_ptr->to_ar[0] = 100; 

	/* Start with "normal" speed */
	new_speed = 110;
	if(creature_ptr->dr >= 0) new_speed += adj_dr_speed[creature_ptr->dr];

	/* Plus AC on Divine Rank */
	if(creature_ptr->dr >= 0){
		creature_ptr->dis_to_a += adj_dr_ac[creature_ptr->dr];
		creature_ptr->to_a += adj_dr_ac[creature_ptr->dr];
	}

	if(has_cf_creature(creature_ptr, CF_HUMANOID))
	{
		creature_ptr->num_hand = 2;
		creature_ptr->num_head = 1;
		creature_ptr->num_tail = 0;
		creature_ptr->num_foot = 2;
	}
	else
	{
		creature_ptr->num_hand = 0;
		creature_ptr->num_head = 0;
		creature_ptr->num_tail = 0;
		creature_ptr->num_foot = 0;
	}

	/* Start with a single blow per turn */
	creature_ptr->num_blow[0] = 1;
	creature_ptr->num_blow[1] = 1;

	/* Start with a single shot per turn */
	creature_ptr->num_fire = 100;

	/* Reset the "xtra" tval */
	creature_ptr->tval_xtra = 0;

	/* Reset the "ammo" tval */
	creature_ptr->tval_ammo = 0;

	/* Clear all the flags */
	creature_ptr->cursed = 0L;
	creature_ptr->bless_blade = FALSE;
	creature_ptr->xtra_might = FALSE;
	creature_ptr->impact[0] = FALSE;
	creature_ptr->impact[1] = FALSE;
	creature_ptr->pass_wall = FALSE;
	creature_ptr->kill_wall = FALSE;
	creature_ptr->dec_mana = FALSE;
	creature_ptr->easy_spell = FALSE;
	creature_ptr->heavy_spell = FALSE;
	creature_ptr->see_inv = FALSE;
	creature_ptr->free_act = FALSE;
	creature_ptr->slow_digest = FALSE;
	creature_ptr->regenerate = FALSE;
	creature_ptr->can_swim = FALSE;
	creature_ptr->levitation = FALSE;
	creature_ptr->hold_life = FALSE;
	creature_ptr->telepathy = FALSE;
	creature_ptr->esp_animal = FALSE;
	creature_ptr->esp_undead = FALSE;
	creature_ptr->esp_demon = FALSE;
	creature_ptr->esp_orc = FALSE;
	creature_ptr->esp_troll = FALSE;
	creature_ptr->esp_giant = FALSE;
	creature_ptr->esp_dragon = FALSE;
	creature_ptr->esp_human = FALSE;
	creature_ptr->esp_evil = FALSE;
	creature_ptr->esp_good = FALSE;
	creature_ptr->esp_nonliving = FALSE;
	creature_ptr->esp_unique = FALSE;
	creature_ptr->lite = FALSE;
	creature_ptr->sustain_str = FALSE;
	creature_ptr->sustain_int = FALSE;
	creature_ptr->sustain_wis = FALSE;
	creature_ptr->sustain_con = FALSE;
	creature_ptr->sustain_dex = FALSE;
	creature_ptr->sustain_chr = FALSE;
	creature_ptr->resist_acid = FALSE;
	creature_ptr->resist_elec = FALSE;
	creature_ptr->resist_fire = FALSE;
	creature_ptr->resist_cold = FALSE;
	creature_ptr->resist_pois = FALSE;
	creature_ptr->resist_conf = FALSE;
	creature_ptr->resist_sound = FALSE;
	creature_ptr->resist_lite = FALSE;
	creature_ptr->resist_dark = FALSE;
	creature_ptr->resist_chaos = FALSE;
	creature_ptr->resist_disen = FALSE;
	creature_ptr->resist_shard = FALSE;
	creature_ptr->resist_nexus = FALSE;
	creature_ptr->resist_blind = FALSE;
	creature_ptr->resist_neth = FALSE;
	creature_ptr->resist_time = FALSE;
	creature_ptr->resist_fear = FALSE;
	creature_ptr->reflect = FALSE;
	creature_ptr->sh_fire = FALSE;
	creature_ptr->sh_elec = FALSE;
	creature_ptr->sh_cold = FALSE;
	creature_ptr->anti_magic = FALSE;
	creature_ptr->anti_tele = FALSE;
	creature_ptr->warning = FALSE;
	creature_ptr->mighty_throw = FALSE;
	creature_ptr->see_nocto = FALSE;

	creature_ptr->immune_acid = FALSE;
	creature_ptr->immune_elec = FALSE;
	creature_ptr->immune_fire = FALSE;
	creature_ptr->immune_cold = FALSE;

	for(i = 0; i < INVEN_TOTAL; i++)
		creature_ptr->two_handed[i] = -1;

	creature_ptr->can_melee[0] = FALSE;
	creature_ptr->can_melee[1] = FALSE;
	creature_ptr->no_flowed = FALSE;

	for(i = 0; i < STAT_MAX; i++)
		creature_ptr->stat_mod_max_max[i] = creature_ptr->stat_max_max[i];

	if (creature_ptr->mimic_form) tmp_rcreature_ptr = &mimic_info[creature_ptr->mimic_form];
	else tmp_rcreature_ptr = &race_info[creature_ptr->race_idx1];
	tmp_rcreature_ptr2 = &race_info[creature_ptr->race_idx2];

	creature_ptr->size = body_size = calc_bodysize(creature_ptr->ht, creature_ptr->wt);

	/* Base infravision (purely racial) */
	creature_ptr->see_infra = tmp_rcreature_ptr->infra;
	j = 0;
	k = 1;


	/* Base skills */
	creature_ptr->skill_dis = 5;
	creature_ptr->skill_dev = 5;
	creature_ptr->skill_rob = 5 + (body_size - 10) / 2 * 3;
	creature_ptr->skill_eva = 5 - (body_size - 10) / 2 * 3;
	creature_ptr->skill_vol = 5;
	creature_ptr->skill_stl = 3 - body_size / 3;
	creature_ptr->skill_srh = 10;
	creature_ptr->skill_fos = 10;
	creature_ptr->skill_thn = body_size;
	creature_ptr->skill_thb = 10;
	creature_ptr->skill_tht = 10;
	creature_ptr->skill_dig = (body_size - 10) * 2;


	// species adjust
	creature_ptr->ac += species_ptr->ac;
	creature_ptr->dis_ac += species_ptr->ac;
	new_speed += species_ptr->speed;



	if(IS_PURE(creature_ptr))
	{
		creature_ptr->skill_dis += (-5 + race_info[creature_ptr->race_idx1].r_dis);
		creature_ptr->skill_dev += (-5 + race_info[creature_ptr->race_idx1].r_dev);
		creature_ptr->skill_rob += (-5 + race_info[creature_ptr->race_idx1].r_rob);
		creature_ptr->skill_eva += (-5 + race_info[creature_ptr->race_idx1].r_eva);
		creature_ptr->skill_vol += (-5 + race_info[creature_ptr->race_idx1].r_vol);
		creature_ptr->skill_stl += race_info[creature_ptr->race_idx1].r_stl;
		creature_ptr->skill_srh += (-10 + race_info[creature_ptr->race_idx1].r_srh);
		creature_ptr->skill_fos += (-10 + race_info[creature_ptr->race_idx1].r_fos);
		creature_ptr->skill_thn += (-10 + race_info[creature_ptr->race_idx1].r_thn);
		creature_ptr->skill_thb += (-10 + race_info[creature_ptr->race_idx1].r_thb);
		creature_ptr->skill_tht += (-10 + race_info[creature_ptr->race_idx1].r_thb);
	}
	else
	{
		creature_ptr->skill_dis += (-5 + race_info[creature_ptr->race_idx1].r_s_dis);
		creature_ptr->skill_dev += (-5 + race_info[creature_ptr->race_idx1].r_s_dev);
		creature_ptr->skill_rob += (-5 + race_info[creature_ptr->race_idx1].r_s_rob);
		creature_ptr->skill_eva += (-5 + race_info[creature_ptr->race_idx1].r_s_eva);
		creature_ptr->skill_vol += (-5 + race_info[creature_ptr->race_idx1].r_s_vol);
		creature_ptr->skill_stl += race_info[creature_ptr->race_idx1].r_s_stl;
		creature_ptr->skill_srh += (-10 + race_info[creature_ptr->race_idx1].r_s_srh);
		creature_ptr->skill_fos += (-10 + race_info[creature_ptr->race_idx1].r_s_fos);
		creature_ptr->skill_thn += (-10 + race_info[creature_ptr->race_idx1].r_s_thn);
		creature_ptr->skill_thb += (-10 + race_info[creature_ptr->race_idx1].r_s_thb);
		creature_ptr->skill_tht += (-10 + race_info[creature_ptr->race_idx1].r_s_thb);

		creature_ptr->skill_dis += race_info[creature_ptr->race_idx2].r_s_dis;
		creature_ptr->skill_dev += race_info[creature_ptr->race_idx2].r_s_dev;
		creature_ptr->skill_rob += race_info[creature_ptr->race_idx2].r_s_rob;
		creature_ptr->skill_eva += race_info[creature_ptr->race_idx2].r_s_eva;
		creature_ptr->skill_vol += race_info[creature_ptr->race_idx2].r_s_vol;
		creature_ptr->skill_stl += race_info[creature_ptr->race_idx2].r_s_stl;
		creature_ptr->skill_srh += race_info[creature_ptr->race_idx2].r_s_srh;
		creature_ptr->skill_fos += race_info[creature_ptr->race_idx2].r_s_fos;
		creature_ptr->skill_thn += race_info[creature_ptr->race_idx2].r_s_thn;
		creature_ptr->skill_thb += race_info[creature_ptr->race_idx2].r_s_thb;
		creature_ptr->skill_tht += race_info[creature_ptr->race_idx2].r_s_thb;

	}

	if(creature_ptr->cls_idx != INDEX_NONE)
	{
		creature_ptr->skill_dis += classkill_info[creature_ptr->cls_idx].c_dis;
		creature_ptr->skill_dev += classkill_info[creature_ptr->cls_idx].c_dev;
		creature_ptr->skill_rob += classkill_info[creature_ptr->cls_idx].c_sav;
		creature_ptr->skill_eva += classkill_info[creature_ptr->cls_idx].c_sav;
		creature_ptr->skill_vol += classkill_info[creature_ptr->cls_idx].c_sav;
		creature_ptr->skill_stl += classkill_info[creature_ptr->cls_idx].c_stl;
		creature_ptr->skill_srh += classkill_info[creature_ptr->cls_idx].c_srh;
		creature_ptr->skill_fos += classkill_info[creature_ptr->cls_idx].c_fos;
		creature_ptr->skill_thn += classkill_info[creature_ptr->cls_idx].c_thn;
		creature_ptr->skill_thb += classkill_info[creature_ptr->cls_idx].c_thb;
		creature_ptr->skill_tht += classkill_info[creature_ptr->cls_idx].c_thb;
	}
	else
	{
		creature_ptr->skill_dis += classkill_info[CLASS_TOURIST].c_dis;
		creature_ptr->skill_dev += classkill_info[CLASS_TOURIST].c_dev;
		creature_ptr->skill_rob += classkill_info[CLASS_TOURIST].c_sav;
		creature_ptr->skill_eva += classkill_info[CLASS_TOURIST].c_sav;
		creature_ptr->skill_vol += classkill_info[CLASS_TOURIST].c_sav;
		creature_ptr->skill_stl += classkill_info[CLASS_TOURIST].c_stl;
		creature_ptr->skill_srh += classkill_info[CLASS_TOURIST].c_srh;
		creature_ptr->skill_fos += classkill_info[CLASS_TOURIST].c_fos;
		creature_ptr->skill_thn += classkill_info[CLASS_TOURIST].c_thn;
		creature_ptr->skill_thb += classkill_info[CLASS_TOURIST].c_thb;
		creature_ptr->skill_tht += classkill_info[CLASS_TOURIST].c_thb;
	}

	if(creature_ptr->chara_idx != INDEX_NONE)
	{
		creature_ptr->skill_dis += chara_info[creature_ptr->chara_idx].a_dis;
		creature_ptr->skill_dev += chara_info[creature_ptr->chara_idx].a_dev;
		creature_ptr->skill_rob += chara_info[creature_ptr->chara_idx].a_sav;
		creature_ptr->skill_eva += chara_info[creature_ptr->chara_idx].a_sav;
		creature_ptr->skill_vol += chara_info[creature_ptr->chara_idx].a_sav;
		creature_ptr->skill_stl += chara_info[creature_ptr->chara_idx].a_stl;
		creature_ptr->skill_srh += chara_info[creature_ptr->chara_idx].a_srh;
		creature_ptr->skill_fos += chara_info[creature_ptr->chara_idx].a_fos;
		creature_ptr->skill_thn += chara_info[creature_ptr->chara_idx].a_thn;
		creature_ptr->skill_thb += chara_info[creature_ptr->chara_idx].a_thb;
		creature_ptr->skill_tht += chara_info[creature_ptr->chara_idx].a_thb;
	}

	for(i = 0; i < max_authorities_idx; i++)
	{
		if(HAS_AUTHORITY(creature_ptr, i))
		{
			creature_ptr->skill_dis += authority_info[i].a_dis;
			creature_ptr->skill_dev += authority_info[i].a_dev;
			creature_ptr->skill_rob += authority_info[i].a_sav;
			creature_ptr->skill_eva += authority_info[i].a_sav;
			creature_ptr->skill_vol += authority_info[i].a_sav;
			creature_ptr->skill_stl += authority_info[i].a_stl;
			creature_ptr->skill_srh += authority_info[i].a_srh;
			creature_ptr->skill_fos += authority_info[i].a_fos;
			creature_ptr->skill_thn += authority_info[i].a_thn;
			creature_ptr->skill_thb += authority_info[i].a_thb;
			creature_ptr->skill_tht += authority_info[i].a_thb;
		}
	}

	if(creature_ptr->dr < 0 && creature_ptr->patron_idx != INDEX_NONE && creature_ptr->patron_idx != creature_ptr->species_idx)
	{
		creature_type *patron_ptr = find_unique_instance(creature_ptr->patron_idx);
		if(patron_ptr)
		{
			for(i = 0; i < max_authorities_idx; i++)
			{
				if(HAS_AUTHORITY(patron_ptr, i))
				{
					creature_ptr->skill_dis += authority_info[i].w_dis;
					creature_ptr->skill_dev += authority_info[i].w_dev;
					creature_ptr->skill_rob += authority_info[i].w_sav;
					creature_ptr->skill_eva += authority_info[i].w_sav;
					creature_ptr->skill_vol += authority_info[i].w_sav;
					creature_ptr->skill_stl += authority_info[i].w_stl;
					creature_ptr->skill_srh += authority_info[i].w_srh;
					creature_ptr->skill_fos += authority_info[i].w_fos;
					creature_ptr->skill_thn += authority_info[i].w_thn;
					creature_ptr->skill_thb += authority_info[i].w_thb;
					creature_ptr->skill_tht += authority_info[i].w_thb;
				}
			}
		}
	}

	if(creature_ptr->dr >= 0)
	{
		creature_ptr->skill_rob += adj_dr_saving[creature_ptr->dr];
		creature_ptr->skill_eva += adj_dr_saving[creature_ptr->dr];
		creature_ptr->skill_vol += adj_dr_saving[creature_ptr->dr];

		for(i = 0; i < STAT_MAX; i++)
			creature_ptr->stat_mod_max_max[i] += creature_ptr->dr / 4 * 10;
	}

	creature_ptr->see_infra = (creature_ptr->see_infra + j) / k;

	if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 0) creature_ptr->can_melee[0] = TRUE;
	if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 1)
	{
		creature_ptr->can_melee[1] = TRUE;
		if (!creature_ptr->can_melee[0]) default_hand = 1;
	}

	if (CAN_TWO_HANDS_WIELDING(creature_ptr))
	{
		if (creature_ptr->can_melee[0] && (empty_hands(creature_ptr, FALSE) == EMPTY_HAND_LARM) &&
			object_allow_two_hands_wielding(creature_ptr, get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1)))
		{
			//TODO creature_ptr->two_handed = TRUE;
		}
		else if (creature_ptr->can_melee[1] && (empty_hands(creature_ptr, FALSE) == EMPTY_HAND_RARM) &&
			object_allow_two_hands_wielding(creature_ptr, get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 2)))
		{
			//TODO creature_ptr->two_handed = TRUE;
		}
		else
		{
			switch (creature_ptr->cls_idx)
			{
			case CLASS_MONK:
			case CLASS_FORCETRAINER:
			case CLASS_BERSERKER:
				if (empty_hands(creature_ptr, FALSE) == (EMPTY_HAND_RARM | EMPTY_HAND_LARM))
				{
					creature_ptr->can_melee[0] = TRUE;
					// TODO creature_ptr->two_handed = TRUE;
				}
				break;
			}
		}
	}

	if (!creature_ptr->can_melee[0] && !creature_ptr->can_melee[1])
	{
		if (empty_hands_status & EMPTY_HAND_RARM) creature_ptr->can_melee[0] = TRUE;
		else if (empty_hands_status == EMPTY_HAND_LARM)
		{
			creature_ptr->can_melee[1] = TRUE;
			default_hand = 1;
		}
	}

	if (creature_ptr->special_defense & KAMAE_MASK)
	{
		if (!(empty_hands_status & EMPTY_HAND_RARM))
		{
			set_action(creature_ptr, ACTION_NONE);
		}
	}

	switch (creature_ptr->cls_idx)
	{
		case CLASS_WARRIOR:
			if (creature_ptr->lev > 29) creature_ptr->resist_fear = TRUE;
			if (creature_ptr->lev > 44) creature_ptr->regenerate = TRUE;
			break;
		case CLASS_PALADIN:
			if (creature_ptr->lev > 39) creature_ptr->resist_fear = TRUE;
			break;
		case CLASS_CHAOS_WARRIOR:
			if (creature_ptr->lev > 29) creature_ptr->resist_chaos = TRUE;
			if (creature_ptr->lev > 39) creature_ptr->resist_fear = TRUE;
			break;
		case CLASS_MINDCRAFTER:
			if (creature_ptr->lev >  9) creature_ptr->resist_fear = TRUE;
			if (creature_ptr->lev > 19) creature_ptr->sustain_wis = TRUE;
			if (creature_ptr->lev > 29) creature_ptr->resist_conf = TRUE;
			if (creature_ptr->lev > 39) creature_ptr->telepathy = TRUE;
			break;
		case CLASS_MONK:
		case CLASS_FORCETRAINER:
			/* Unencumbered Monks become faster every 10 levels */
			if (!(heavy_armor(creature_ptr)))
			{
				if (!(race_is_(creature_ptr, RACE_KLACKON) ||
				      race_is_(creature_ptr, RACE_SPRITE) ||
				      (creature_ptr->chara_idx == CHARA_MUNCHKIN)))
					new_speed += (creature_ptr->lev) / 10;

				/* Free action if unencumbered at level 25 */
				if  (creature_ptr->lev > 24)
					creature_ptr->free_act = TRUE;
			}
			break;
		case CLASS_SORCERER:
			creature_ptr->to_a -= 50;
			creature_ptr->dis_to_a -= 50;
			break;
		case CLASS_BARD:
			creature_ptr->resist_sound = TRUE;
			break;
		case CLASS_SAMURAI:
			if (creature_ptr->lev > 29) creature_ptr->resist_fear = TRUE;
			break;
		case CLASS_BERSERKER:
			creature_ptr->shero = 1;
			creature_ptr->sustain_str = TRUE;
			creature_ptr->sustain_dex = TRUE;
			creature_ptr->sustain_con = TRUE;
			creature_ptr->regenerate = TRUE;
			creature_ptr->free_act = TRUE;
			new_speed += 2;
			if (creature_ptr->lev > 29) new_speed++;
			if (creature_ptr->lev > 39) new_speed++;
			if (creature_ptr->lev > 44) new_speed++;
			if (creature_ptr->lev > 49) new_speed++;
			creature_ptr->to_a += 10+creature_ptr->lev/2;
			creature_ptr->dis_to_a += 10+creature_ptr->lev/2;
			creature_ptr->skill_dig += (100+creature_ptr->lev*8);
			if (creature_ptr->lev > 39) creature_ptr->reflect = TRUE;
			play_redraw |= PR_STATUS;
			break;
		case CLASS_MIRROR_MASTER:
			if (creature_ptr->lev > 39) creature_ptr->reflect = TRUE;
			break;
		case CLASS_NINJA:
			/* Unencumbered Ninjas become faster every 10 levels */
			if (heavy_armor(creature_ptr))
			{
				new_speed -= (creature_ptr->lev) / 10;
				creature_ptr->skill_stl -= (creature_ptr->lev)/10;
			}
			else if ((!get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1)->k_idx || creature_ptr->can_melee[0]) &&
			         (!get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 2)->k_idx || creature_ptr->can_melee[1]))
			{
				new_speed += 3;
				if (!(race_is_(creature_ptr, RACE_KLACKON) ||
				      race_is_(creature_ptr, RACE_SPRITE) ||
				      (creature_ptr->chara_idx == CHARA_MUNCHKIN)))
					new_speed += (creature_ptr->lev) / 10;
				creature_ptr->skill_stl += (creature_ptr->lev)/10;

				/* Free action if unencumbered at level 25 */
				if  (creature_ptr->lev > 24)
					creature_ptr->free_act = TRUE;
			}
			if ((!get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1)->k_idx || creature_ptr->can_melee[0]) &&
			    (!get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 2)->k_idx || creature_ptr->can_melee[1]))
			{
				creature_ptr->to_a += creature_ptr->lev/2+5;
				creature_ptr->dis_to_a += creature_ptr->lev/2+5;
			}
			creature_ptr->slow_digest = TRUE;
			creature_ptr->resist_fear = TRUE;
			if (creature_ptr->lev > 19) creature_ptr->resist_pois = TRUE;
			if (creature_ptr->lev > 24) creature_ptr->sustain_dex = TRUE;
			if (creature_ptr->lev > 29) creature_ptr->see_inv = TRUE;
			if (creature_ptr->lev > 44)
			{
				creature_ptr->oppose_pois = 1;
				play_redraw |= PR_STATUS;
			}
			creature_ptr->see_nocto = TRUE;
			break;
	}

	/***** Races ****/

	// TODO: Mimic Race control to creature.c
	if (!creature_ptr->mimic_form)
	{
		set_resistance(creature_ptr);

		if(IS_RACE(creature_ptr, RACE_SPRITE))
		{
			creature_ptr->levitation = TRUE;
			new_speed += (creature_ptr->lev) / 10;
		}
		if(IS_RACE(creature_ptr, RACE_ENT))
		{
			/* Ents dig like maniacs, but only with their hands. */
			if (!get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1)->k_idx) 
				creature_ptr->skill_dig += creature_ptr->lev * 10;
			/* Ents get tougher and stronger as they age, but lose dexterity. */
			if (creature_ptr->lev > 25) creature_ptr->stat_add[STAT_STR]++;
			if (creature_ptr->lev > 40) creature_ptr->stat_add[STAT_STR]++;
			if (creature_ptr->lev > 45) creature_ptr->stat_add[STAT_STR]++;

			if (creature_ptr->lev > 25) creature_ptr->stat_add[STAT_DEX]--;
			if (creature_ptr->lev > 40) creature_ptr->stat_add[STAT_DEX]--;
			if (creature_ptr->lev > 45) creature_ptr->stat_add[STAT_DEX]--;

			if (creature_ptr->lev > 25) creature_ptr->stat_add[STAT_CON]++;
			if (creature_ptr->lev > 40) creature_ptr->stat_add[STAT_CON]++;
			if (creature_ptr->lev > 45) creature_ptr->stat_add[STAT_CON]++;
		}
		if(IS_RACE(creature_ptr, RACE_KLACKON))
		{
			new_speed += (creature_ptr->lev) / 10;
		}
		//TODO
		/*
			case LICH:
			creature_ptr->pass_wall = TRUE;
			break;
		*/
	}

	set_unreached_race_level_penalty(creature_ptr);

	if (creature_ptr->ult_res || (creature_ptr->special_defense & KATA_MUSOU))
	{
		creature_ptr->see_inv = TRUE;
		creature_ptr->free_act = TRUE;
		creature_ptr->slow_digest = TRUE;
		creature_ptr->regenerate = TRUE;
		creature_ptr->levitation = TRUE;
		creature_ptr->hold_life = TRUE;
		creature_ptr->telepathy = TRUE;
		creature_ptr->lite = TRUE;
		creature_ptr->sustain_str = TRUE;
		creature_ptr->sustain_int = TRUE;
		creature_ptr->sustain_wis = TRUE;
		creature_ptr->sustain_con = TRUE;
		creature_ptr->sustain_dex = TRUE;
		creature_ptr->sustain_chr = TRUE;
		creature_ptr->resist_acid = TRUE;
		creature_ptr->resist_elec = TRUE;
		creature_ptr->resist_fire = TRUE;
		creature_ptr->resist_cold = TRUE;
		creature_ptr->resist_pois = TRUE;
		creature_ptr->resist_conf = TRUE;
		creature_ptr->resist_sound = TRUE;
		creature_ptr->resist_lite = TRUE;
		creature_ptr->resist_dark = TRUE;
		creature_ptr->resist_chaos = TRUE;
		creature_ptr->resist_disen = TRUE;
		creature_ptr->resist_shard = TRUE;
		creature_ptr->resist_nexus = TRUE;
		creature_ptr->resist_blind = TRUE;
		creature_ptr->resist_neth = TRUE;
		creature_ptr->resist_fear = TRUE;
		creature_ptr->reflect = TRUE;
		creature_ptr->sh_fire = TRUE;
		creature_ptr->sh_elec = TRUE;
		creature_ptr->sh_cold = TRUE;
		creature_ptr->to_a += 100;
		creature_ptr->dis_to_a += 100;
	}
	/* Temporary shield */
	else if (creature_ptr->tsubureru || creature_ptr->shield || creature_ptr->magicdef)
	{
		creature_ptr->to_a += 50;
		creature_ptr->dis_to_a += 50;
	}

	if (creature_ptr->tim_res_nether)
	{
		creature_ptr->resist_neth = TRUE;
	}
	if (creature_ptr->tim_sh_fire)
	{
		creature_ptr->sh_fire = TRUE;
	}
	if (creature_ptr->tim_res_time)
	{
		creature_ptr->resist_time = TRUE;
	}


	/* Sexy Gal */
	if (creature_ptr->chara_idx == CHARA_SEXY) creature_ptr->cursed |= (TRC_AGGRAVATE);
	if (creature_ptr->chara_idx == CHARA_NAMAKE) creature_ptr->to_m_chance += 10;
	if (creature_ptr->chara_idx == CHARA_KIREMONO) creature_ptr->to_m_chance -= 3;
	if (creature_ptr->chara_idx == CHARA_CHARGEMAN) creature_ptr->resist_conf = TRUE;
	if ((creature_ptr->chara_idx == CHARA_GAMAN) || (creature_ptr->chara_idx == CHARA_CHIKARA)) creature_ptr->to_m_chance++;

	/* Lucky man */
	//TODO if (creature_ptr->chara_idx == CHARA_LUCKY) creature_ptr->flags14 |= RF14_GOOD_LUCK;

	if (creature_ptr->chara_idx == CHARA_MUNCHKIN)
	{
		creature_ptr->resist_blind = TRUE;
		creature_ptr->resist_conf  = TRUE;
		creature_ptr->hold_life = TRUE;
		if (creature_ptr->cls_idx != CLASS_NINJA) creature_ptr->lite = TRUE;

		if ((creature_ptr->race_idx1 != RACE_KLACKON) && (creature_ptr->race_idx1 != RACE_SPRITE))
			/* Munchkin become faster */
			new_speed += (creature_ptr->lev) / 10 + 5;
	}

	if (music_singing(creature_ptr, MUSIC_WALL))
	{
		creature_ptr->kill_wall = TRUE;
	}

	/* Hack -- apply racial/class stat maxes */
	/* Apply the racial modifiers */
	for (i = 0; i < 6; i++)
	{
		if(IS_PURE(creature_ptr))
			creature_ptr->stat_add[i] += tmp_rcreature_ptr->r_adj[i];
		else
		{
			creature_ptr->stat_add[i] += tmp_rcreature_ptr->r_s_adj[i];
			creature_ptr->stat_add[i] += tmp_rcreature_ptr2->r_s_adj[i];
		}

		if(creature_ptr->cls_idx != INDEX_NONE)
		{
			creature_ptr->stat_add[i] += classkill_info[creature_ptr->cls_idx].c_adj[i];
			if(creature_ptr->cls_bonus) creature_ptr->stat_add[i] += classkill_info[creature_ptr->cls_idx].c_adj_b[i];
		}

		if(creature_ptr->chara_idx != INDEX_NONE)
			creature_ptr->stat_add[i] += chara_info[creature_ptr->chara_idx].a_adj[i];

		for(j = 0; j < max_authorities_idx; j++)
			if(HAS_AUTHORITY(creature_ptr, j))
				creature_ptr->stat_add[i] += authority_info[j].a_adj[i];

		if(creature_ptr->dr < 0 && creature_ptr->patron_idx != INDEX_NONE && creature_ptr->patron_idx != creature_ptr->species_idx)
		{
			creature_type *patron_ptr = find_unique_instance(creature_ptr->patron_idx);
			if(patron_ptr)
			{
				for(j = 0; j < max_authorities_idx; j++)
					if(HAS_AUTHORITY(patron_ptr, j))
						creature_ptr->stat_add[i] += authority_info[j].a_adj[i];
			}
		}

	}




	for(i = 0; i < max_creature_flag_idx; i++)
	{
		if(has_cf_creature(creature_ptr, i))
		{
			creature_ptr->stat_add[STAT_STR] += creature_flag_info[i].adj[STAT_STR];
			creature_ptr->stat_add[STAT_INT] += creature_flag_info[i].adj[STAT_INT];
			creature_ptr->stat_add[STAT_WIS] += creature_flag_info[i].adj[STAT_WIS];
			creature_ptr->stat_add[STAT_DEX] += creature_flag_info[i].adj[STAT_DEX];
			creature_ptr->stat_add[STAT_CON] += creature_flag_info[i].adj[STAT_CON];
			creature_ptr->stat_add[STAT_CHA] += creature_flag_info[i].adj[STAT_CHA];

			creature_ptr->skill_dis += creature_flag_info[i].dis;
			creature_ptr->skill_dev += creature_flag_info[i].dev;
			creature_ptr->skill_stl += creature_flag_info[i].stl;
			creature_ptr->skill_srh += creature_flag_info[i].srh;
			creature_ptr->skill_dig += creature_flag_info[i].dig;
			creature_ptr->skill_thb += creature_flag_info[i].thb;
			creature_ptr->skill_thn += creature_flag_info[i].thn;
			creature_ptr->skill_tht += creature_flag_info[i].tht;

			creature_ptr->skill_rob += creature_flag_info[i].rob;
			creature_ptr->skill_eva += creature_flag_info[i].eva;
			creature_ptr->skill_vol += creature_flag_info[i].vol;

			creature_ptr->see_infra += creature_flag_info[i].infra;

		}
	}

	{
		if (has_cf_creature(creature_ptr, CF_FLESH_ROT))
		{
			creature_ptr->regenerate = FALSE;
			/* Cancel innate regeneration */
		}

		if (has_cf_creature(creature_ptr, CF_MAGIC_RES))
		{
			creature_ptr->skill_rob += creature_ptr->lev / 5;
			creature_ptr->skill_eva += creature_ptr->lev / 5;
			creature_ptr->skill_vol += creature_ptr->lev / 5;
		}

		if (has_cf_creature(creature_ptr, CF_ELEC_TOUC))
		{
			creature_ptr->sh_elec = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_FIRE_BODY))
		{
			creature_ptr->sh_fire = TRUE;
			creature_ptr->lite = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_WART_SKIN))
		{
			creature_ptr->stat_add[STAT_CHA] -= 2;
			creature_ptr->to_a += 5;
			creature_ptr->dis_to_a += 5;
		}

		if (has_cf_creature(creature_ptr, CF_SCALES))
		{
			creature_ptr->stat_add[STAT_CHA] -= 1;
			creature_ptr->to_a += 10;
			creature_ptr->dis_to_a += 10;
		}

		if (has_cf_creature(creature_ptr, CF_IRON_SKIN))
		{
			creature_ptr->stat_add[STAT_DEX] -= 1;
			creature_ptr->to_a += 25;
			creature_ptr->dis_to_a += 25;
		}

		if (has_cf_creature(creature_ptr, CF_WINGS))
		{
			creature_ptr->levitation = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_FEARLESS))
		{
			creature_ptr->resist_fear = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_REGEN))
		{
			creature_ptr->regenerate = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_ESP))
		{
			creature_ptr->telepathy = TRUE;
		}

		if (has_cf_creature(creature_ptr, CF_LIMBER))
		{
			creature_ptr->stat_add[STAT_DEX] += 3;
		}

		if (has_cf_creature(creature_ptr, CF_ARTHRITIS))
		{
			creature_ptr->stat_add[STAT_DEX] -= 3;
		}

		if (has_cf_creature(creature_ptr, CF_MOTION))
		{
			creature_ptr->free_act = TRUE;
			creature_ptr->skill_stl += 1;
		}

		if (has_cf_creature(creature_ptr, CF_ILL_NORM))
		{
			creature_ptr->stat_add[STAT_CHA] = 0;
		}
	}

	if (creature_ptr->tsuyoshi)
	{
		creature_ptr->stat_add[STAT_STR] += 4;
		creature_ptr->stat_add[STAT_CON] += 4;
	}

	/* Scan the usable creature_ptr->inventory */
	for (i = 0; i < INVEN_TOTAL; i++)
	{
		int bonus_to_h, bonus_to_d, slot;

		// Skip no equip
		if (!creature_ptr->equip_now[i]) continue;

		slot = GET_INVEN_SLOT_TYPE(creature_ptr, i);

		o_ptr = &creature_ptr->inventory[i];
		/* Skip non-objects */
		if (!o_ptr->k_idx) continue;

		/* Extract the item flags */
		object_flags(o_ptr, flgs);

		creature_ptr->cursed |= (o_ptr->curse_flags & (0xFFFFFFF0L));
		if (o_ptr->name1 == ART_CHAINSWORD) creature_ptr->cursed |= TRC_CHAINSWORD;

		/* Affect stats */
		if (have_flag(flgs, TR_STR)) creature_ptr->stat_add[STAT_STR] += o_ptr->pval;
		if (have_flag(flgs, TR_INT)) creature_ptr->stat_add[STAT_INT] += o_ptr->pval;
		if (have_flag(flgs, TR_WIS)) creature_ptr->stat_add[STAT_WIS] += o_ptr->pval;
		if (have_flag(flgs, TR_DEX)) creature_ptr->stat_add[STAT_DEX] += o_ptr->pval;
		if (have_flag(flgs, TR_CON)) creature_ptr->stat_add[STAT_CON] += o_ptr->pval;
		if (have_flag(flgs, TR_CHR)) creature_ptr->stat_add[STAT_CHA] += o_ptr->pval;

		if (have_flag(flgs, TR_MAGIC_MASTERY))    creature_ptr->skill_dev += 8*o_ptr->pval;

		/* Affect stealth */
		if (have_flag(flgs, TR_STEALTH)) creature_ptr->skill_stl += o_ptr->pval;

		/* Affect searching ability (factor of five) */
		if (have_flag(flgs, TR_SEARCH)) creature_ptr->skill_srh += (o_ptr->pval * 5);

		/* Affect searching frequency (factor of five) */
		if (have_flag(flgs, TR_SEARCH)) creature_ptr->skill_fos += (o_ptr->pval * 5);

		/* Affect infravision */
		if (have_flag(flgs, TR_INFRA)) creature_ptr->see_infra += o_ptr->pval;

		/* Affect digging (factor of 20) */
		if (have_flag(flgs, TR_TUNNEL)) creature_ptr->skill_dig += (o_ptr->pval * 20);

		if (have_flag(flgs, TR_SPEED)) new_speed += o_ptr->pval;

		/* Affect blows */
		if (have_flag(flgs, TR_BLOWS))
		{
			//TODO adjust
			extra_blows[0] += o_ptr->pval;
			extra_blows[1] += o_ptr->pval;
		}

		/* Hack -- cause earthquakes */
		if (have_flag(flgs, TR_IMPACT)) creature_ptr->impact[(i == get_equipped_slot_idx(creature_ptr, INVEN_SLOT_HAND, 1)) ? 0 : 1] = TRUE;

		/* Boost shots */
		if (have_flag(flgs, TR_XTRA_SHOTS)) extra_shots++;

		/* Various flags */
		if (have_flag(flgs, TR_AGGRAVATE))   creature_ptr->cursed |= TRC_AGGRAVATE;
		if (have_flag(flgs, TR_DRAIN_EXP))   creature_ptr->cursed |= TRC_DRAIN_EXP;
		if (have_flag(flgs, TR_TY_CURSE))    creature_ptr->cursed |= TRC_TY_CURSE;
		if (have_flag(flgs, TR_DEC_MANA))    creature_ptr->dec_mana = TRUE;
		if (have_flag(flgs, TR_BLESSED))     creature_ptr->bless_blade = TRUE;
		if (have_flag(flgs, TR_XTRA_MIGHT))  creature_ptr->xtra_might = TRUE;
		if (have_flag(flgs, TR_SLOW_DIGEST)) creature_ptr->slow_digest = TRUE;
		if (have_flag(flgs, TR_REGEN))       creature_ptr->regenerate = TRUE;

		if (have_flag(flgs, TR_TELEPATHY))   creature_ptr->telepathy = TRUE;
		if (have_flag(flgs, TR_ESP_ANIMAL))  creature_ptr->esp_animal = TRUE;
		if (have_flag(flgs, TR_ESP_UNDEAD))  creature_ptr->esp_undead = TRUE;
		if (have_flag(flgs, TR_ESP_DEMON))   creature_ptr->esp_demon = TRUE;
		if (have_flag(flgs, TR_ESP_ORC))     creature_ptr->esp_orc = TRUE;
		if (have_flag(flgs, TR_ESP_TROLL))   creature_ptr->esp_troll = TRUE;
		if (have_flag(flgs, TR_ESP_GIANT))   creature_ptr->esp_giant = TRUE;
		if (have_flag(flgs, TR_ESP_DRAGON))  creature_ptr->esp_dragon = TRUE;
		if (have_flag(flgs, TR_ESP_HUMAN))   creature_ptr->esp_human = TRUE;
		if (have_flag(flgs, TR_ESP_EVIL))    creature_ptr->esp_evil = TRUE;
		if (have_flag(flgs, TR_ESP_GOOD))    creature_ptr->esp_good = TRUE;
		if (have_flag(flgs, TR_ESP_NONLIVING)) creature_ptr->esp_nonliving = TRUE;
		if (have_flag(flgs, TR_ESP_UNIQUE))  creature_ptr->esp_unique = TRUE;

		if (have_flag(flgs, TR_SEE_INVIS))   creature_ptr->see_inv = TRUE;
		if (have_flag(flgs, TR_LEVITATION))  creature_ptr->levitation = TRUE;
		if (have_flag(flgs, TR_FREE_ACT))    creature_ptr->free_act = TRUE;
		if (have_flag(flgs, TR_HOLD_LIFE))   creature_ptr->hold_life = TRUE;

		if (have_flag(flgs, TR_WARNING)){
			if (!o_ptr->inscription || !(my_strchr(quark_str(o_ptr->inscription),'$')))
			  creature_ptr->warning = TRUE;
		}

		if (have_flag(flgs, TR_TELEPORT))
		{
			if (object_is_cursed(o_ptr)) creature_ptr->cursed |= TRC_TELEPORT;
			else
			{
				cptr insc = quark_str(o_ptr->inscription);

				if (o_ptr->inscription && my_strchr(insc, '.'))
				{
					/*
					 * {.} will stop random teleportation.
					 */
				}
				else
				{
					/* Controlled random teleportation */
					creature_ptr->cursed |= TRC_TELEPORT_SELF;
				}
			}
		}

		/* Immunity flags */
		if (have_flag(flgs, TR_IM_FIRE)) creature_ptr->immune_fire = TRUE;
		if (have_flag(flgs, TR_IM_ACID)) creature_ptr->immune_acid = TRUE;
		if (have_flag(flgs, TR_IM_COLD)) creature_ptr->immune_cold = TRUE;
		if (have_flag(flgs, TR_IM_ELEC)) creature_ptr->immune_elec = TRUE;

		/* Resistance flags */
		if (have_flag(flgs, TR_RES_ACID))   creature_ptr->resist_acid += 1;
		if (have_flag(flgs, TR_RES_ELEC))   creature_ptr->resist_elec += 1;
		if (have_flag(flgs, TR_RES_FIRE))   creature_ptr->resist_fire += 1;
		if (have_flag(flgs, TR_RES_COLD))   creature_ptr->resist_cold += 1;
		if (have_flag(flgs, TR_RES_POIS))   creature_ptr->resist_pois += 1;
		if (have_flag(flgs, TR_RES_SOUND))  creature_ptr->resist_sound += 1;
		if (have_flag(flgs, TR_RES_LITE))   creature_ptr->resist_lite += 1;
		if (have_flag(flgs, TR_RES_DARK))   creature_ptr->resist_dark += 1;
		if (have_flag(flgs, TR_RES_CHAOS))  creature_ptr->resist_chaos += 1;
		if (have_flag(flgs, TR_RES_DISEN))  creature_ptr->resist_disen += 1;
		if (have_flag(flgs, TR_RES_SHARDS)) creature_ptr->resist_shard += 1;
		if (have_flag(flgs, TR_RES_NEXUS))  creature_ptr->resist_nexus += 1;
		if (have_flag(flgs, TR_RES_NETHER)) creature_ptr->resist_neth += 1;
		if (o_ptr->name2 == EGO_RING_RES_TIME) creature_ptr->resist_time += 1;

		if (have_flag(flgs, TR_RES_FEAR))   creature_ptr->resist_fear = TRUE;
		if (have_flag(flgs, TR_RES_CONF))   creature_ptr->resist_conf = TRUE;
		if (have_flag(flgs, TR_RES_BLIND))  creature_ptr->resist_blind = TRUE;

		if (have_flag(flgs, TR_REFLECT))  creature_ptr->reflect = TRUE;
		if (have_flag(flgs, TR_SH_FIRE))  creature_ptr->sh_fire = TRUE;
		if (have_flag(flgs, TR_SH_ELEC))  creature_ptr->sh_elec = TRUE;
		if (have_flag(flgs, TR_SH_COLD))  creature_ptr->sh_cold = TRUE;
		if (have_flag(flgs, TR_NO_MAGIC)) creature_ptr->anti_magic = TRUE;
		if (have_flag(flgs, TR_NO_TELE))  creature_ptr->anti_tele = TRUE;

		/* Sustain flags */
		if (have_flag(flgs, TR_SUST_STR)) creature_ptr->sustain_str = TRUE;
		if (have_flag(flgs, TR_SUST_INT)) creature_ptr->sustain_int = TRUE;
		if (have_flag(flgs, TR_SUST_WIS)) creature_ptr->sustain_wis = TRUE;
		if (have_flag(flgs, TR_SUST_DEX)) creature_ptr->sustain_dex = TRUE;
		if (have_flag(flgs, TR_SUST_CON)) creature_ptr->sustain_con = TRUE;
		if (have_flag(flgs, TR_SUST_CHR)) creature_ptr->sustain_chr = TRUE;

		if (o_ptr->name2 == EGO_YOIYAMI) yoiyami = TRUE;
		if (o_ptr->name2 == EGO_TWO_WEAPON) easy_2weapon = TRUE;


		if (o_ptr->name2 == EGO_RING_THROW) creature_ptr->mighty_throw = TRUE;
		if (have_flag(flgs, TR_EASY_SPELL)) creature_ptr->easy_spell = TRUE;
		if (o_ptr->name2 == EGO_AMU_FOOL) creature_ptr->heavy_spell = TRUE;
		if (o_ptr->name2 == EGO_AMU_NAIVETY) down_saving = TRUE;

		if (o_ptr->curse_flags & TRC_LOW_MAGIC)
		{
			if (o_ptr->curse_flags & TRC_HEAVY_CURSE)
			{
				creature_ptr->to_m_chance += 10;
			}
			else
			{
				creature_ptr->to_m_chance += 3;
			}
		}

		if (o_ptr->tval == TV_CAPTURE) continue;

		/* Modify the base armor class */
		/* The base armor class is always known */
		/* Apply the bonuses to armor class */
		/* Apply the mental bonuses to armor class, if known */

		creature_ptr->ac += o_ptr->ac;
		creature_ptr->dis_ac += o_ptr->ac;
		creature_ptr->to_a += o_ptr->to_a;
		if (object_is_known(o_ptr)) creature_ptr->dis_to_a += o_ptr->to_a;

		if (o_ptr->curse_flags & TRC_LOW_MELEE)
		{
			int slot = i - 0; //TODO 
			if (slot < 2)
			{
				if (o_ptr->curse_flags & TRC_HEAVY_CURSE)
				{
					creature_ptr->to_h[slot] -= 15;
					if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_h[slot] -= 15;
				}
				else
				{
					creature_ptr->to_h[slot] -= 5;
					if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_h[slot] -= 5;
				}
			}
			else
			{
				if (o_ptr->curse_flags & TRC_HEAVY_CURSE)
				{
					creature_ptr->to_h_b -= 15;
					if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_h_b -= 15;
				}
				else
				{
					creature_ptr->to_h_b -= 5;
					if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_h_b -= 5;
				}
			}
		}

		if (o_ptr->curse_flags & TRC_LOW_AC)
		{
			if (o_ptr->curse_flags & TRC_HEAVY_CURSE)
			{
				creature_ptr->to_a -= 30;
				if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_a -= 30;
			}
			else
			{
				creature_ptr->to_a -= 10;
				if (o_ptr->ident & IDENT_MENTAL) creature_ptr->dis_to_a -= 10;
			}
		}

		/* Hack -- do not apply "weapon" bonuses */
		if (slot == INVEN_SLOT_HAND && get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > i) continue;

		/* Hack -- do not apply "bow" bonuses */
		if (slot == INVEN_SLOT_BOW) continue;

		bonus_to_h = o_ptr->to_h;
		bonus_to_d = o_ptr->to_d;

		if (creature_ptr->cls_idx == CLASS_NINJA)
		{
			if (o_ptr->to_h > 0) bonus_to_h = (o_ptr->to_h+1)/2;
			if (o_ptr->to_d > 0) bonus_to_d = (o_ptr->to_d+1)/2;
		}

		/* To Bow and Natural attack */

		/* Apply the bonuses to hit/damage */
		creature_ptr->to_h_b += bonus_to_h;
		creature_ptr->to_h_m += bonus_to_h;
		creature_ptr->to_d_m += bonus_to_d;

		/* Apply the mental bonuses tp hit/damage, if known */
		if (object_is_known(o_ptr)) creature_ptr->dis_to_h_b += bonus_to_h;

		/* To Melee */
		if (GET_INVEN_SLOT_TYPE(creature_ptr, i) == INVEN_SLOT_RING && !creature_ptr->two_handed)
		{
			/* Apply the bonuses to hit/damage */
			creature_ptr->to_h[get_equipped_slot_idx(creature_ptr, INVEN_SLOT_RING, 1 + i)] += bonus_to_h;
			creature_ptr->to_d[get_equipped_slot_idx(creature_ptr, INVEN_SLOT_RING, 1 + i)] += bonus_to_d;

			/* Apply the mental bonuses tp hit/damage, if known */
			if (object_is_known(o_ptr))
			{
				creature_ptr->dis_to_h[get_equipped_slot_idx(creature_ptr, INVEN_SLOT_RING, 1 + i)] += bonus_to_h;
				creature_ptr->dis_to_d[get_equipped_slot_idx(creature_ptr, INVEN_SLOT_RING, 1 + i)] += bonus_to_d;
			}
		}
		else if (creature_ptr->can_melee[0] && creature_ptr->can_melee[1])
		{
			/* Apply the bonuses to hit/damage */
			creature_ptr->to_h[0] += (bonus_to_h > 0) ? (bonus_to_h+1)/2 : bonus_to_h;
			creature_ptr->to_h[1] += (bonus_to_h > 0) ? bonus_to_h/2 : bonus_to_h;
			creature_ptr->to_d[0] += (bonus_to_d > 0) ? (bonus_to_d+1)/2 : bonus_to_d;
			creature_ptr->to_d[1] += (bonus_to_d > 0) ? bonus_to_d/2 : bonus_to_d;

			/* Apply the mental bonuses tp hit/damage, if known */
			if (object_is_known(o_ptr))
			{
				creature_ptr->dis_to_h[0] += (bonus_to_h > 0) ? (bonus_to_h+1)/2 : bonus_to_h;
				creature_ptr->dis_to_h[1] += (bonus_to_h > 0) ? bonus_to_h/2 : bonus_to_h;
				creature_ptr->dis_to_d[0] += (bonus_to_d > 0) ? (bonus_to_d+1)/2 : bonus_to_d;
				creature_ptr->dis_to_d[1] += (bonus_to_d > 0) ? bonus_to_d/2 : bonus_to_d;
			}
		}
		else
		{
			/* Apply the bonuses to hit/damage */
			creature_ptr->to_h[default_hand] += bonus_to_h;
			creature_ptr->to_d[default_hand] += bonus_to_d;

			/* Apply the mental bonuses to hit/damage, if known */
			if (object_is_known(o_ptr))
			{
				creature_ptr->dis_to_h[default_hand] += bonus_to_h;
				creature_ptr->dis_to_d[default_hand] += bonus_to_d;
			}
		}
	}

	if (old_mighty_throw != creature_ptr->mighty_throw)
	{
		/* Redraw average damege display of Shuriken */
		play_window |= PW_INVEN;
	}

	if (creature_ptr->cursed & TRC_TELEPORT) creature_ptr->cursed &= ~(TRC_TELEPORT_SELF);

	if (((creature_ptr->cls_idx == CLASS_MONK) || (creature_ptr->cls_idx == CLASS_FORCETRAINER)) && !heavy_armor(creature_ptr))
	{
		/* TODO Monks get extra ac for armour _not worn_
		if (!(creature_ptr->inventory[INVEN_SLOT_BODY].k_idx))
		{
			creature_ptr->to_a += (creature_ptr->lev * 3) / 2;
			creature_ptr->dis_to_a += (creature_ptr->lev * 3) / 2;
		}
		if (!(creature_ptr->inventory[INVEN_SLOT_OUTER].k_idx) && (creature_ptr->lev > 15))
		{
			creature_ptr->to_a += ((creature_ptr->lev - 13) / 3);
			creature_ptr->dis_to_a += ((creature_ptr->lev - 13) / 3);
		}
		if (!(creature_ptr->inventory[INVEN_SLOT_2NDARM].k_idx) && (creature_ptr->lev > 10))
		{
			creature_ptr->to_a += ((creature_ptr->lev - 8) / 3);
			creature_ptr->dis_to_a += ((creature_ptr->lev - 8) / 3);
		}
		if (!(creature_ptr->inventory[INVEN_SLOT_1STHEAD].k_idx) && (creature_ptr->lev > 4))
		{
			creature_ptr->to_a += (creature_ptr->lev - 2) / 3;
			creature_ptr->dis_to_a += (creature_ptr->lev -2) / 3;
		}
		if (!(creature_ptr->inventory[INVEN_SLOT_1STHANDS].k_idx))
		{
			creature_ptr->to_a += (creature_ptr->lev / 2);
			creature_ptr->dis_to_a += (creature_ptr->lev / 2);
		}
		if (!(creature_ptr->inventory[INVEN_SLOT_FEET].k_idx))
		{
			creature_ptr->to_a += (creature_ptr->lev / 3);
			creature_ptr->dis_to_a += (creature_ptr->lev / 3);
		}
		*/
		if (creature_ptr->special_defense & KAMAE_BYAKKO)
		{
			creature_ptr->stat_add[STAT_STR] += 2;
			creature_ptr->stat_add[STAT_DEX] += 2;
			creature_ptr->stat_add[STAT_CON] -= 3;
		}
		else if (creature_ptr->special_defense & KAMAE_SEIRYU)
		{
		}
		else if (creature_ptr->special_defense & KAMAE_GENBU)
		{
			creature_ptr->stat_add[STAT_INT] -= 1;
			creature_ptr->stat_add[STAT_WIS] -= 1;
			creature_ptr->stat_add[STAT_DEX] -= 2;
			creature_ptr->stat_add[STAT_CON] += 3;
		}
		else if (creature_ptr->special_defense & KAMAE_SUZAKU)
		{
			creature_ptr->stat_add[STAT_STR] -= 2;
			creature_ptr->stat_add[STAT_INT] += 1;
			creature_ptr->stat_add[STAT_WIS] += 1;
			creature_ptr->stat_add[STAT_DEX] += 2;
			creature_ptr->stat_add[STAT_CON] -= 2;
		}
	}

	if (creature_ptr->special_defense & KATA_KOUKIJIN)
	{
		for (i = 0; i < 6; i++)
			creature_ptr->stat_add[i] += 5;
		creature_ptr->to_a -= 50;
		creature_ptr->dis_to_a -= 50;
	}

	/* Set Species Blow. */
	creature_ptr->blow[0] = species_ptr->blow[0];
	creature_ptr->blow[1] = species_ptr->blow[1];
	creature_ptr->blow[2] = species_ptr->blow[2];
	creature_ptr->blow[3] = species_ptr->blow[3];


	/* Hack -- aura of fire also provides light */
	if (creature_ptr->sh_fire) creature_ptr->lite = TRUE;

	/* Racial AC bonus */
	if(IS_PURE(creature_ptr))
	{
		creature_ptr->to_a += race_info[creature_ptr->race_idx1].ac_base + (race_info[creature_ptr->race_idx1].ac_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
		creature_ptr->dis_to_a += race_info[creature_ptr->race_idx1].ac_base + (race_info[creature_ptr->race_idx1].ac_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
	}
	else
	{
		creature_ptr->to_a += race_info[creature_ptr->race_idx1].ac_s_base + (race_info[creature_ptr->race_idx1].ac_s_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
		creature_ptr->dis_to_a += race_info[creature_ptr->race_idx1].ac_s_base + (race_info[creature_ptr->race_idx1].ac_s_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
		creature_ptr->to_a += race_info[creature_ptr->race_idx2].ac_s_base + (race_info[creature_ptr->race_idx2].ac_s_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
		creature_ptr->dis_to_a += race_info[creature_ptr->race_idx2].ac_s_base + (race_info[creature_ptr->race_idx2].ac_s_plus * (creature_ptr->lev < 30 ? creature_ptr->lev : 30 ) / 30);
	}

	/* Hex bonuses */
	if (creature_ptr->realm1 == REALM_HEX)
	{
		if (hex_spelling_any(creature_ptr)) creature_ptr->skill_stl -= (1 + creature_ptr->magic_num2[0]);
		if (hex_spelling(creature_ptr, HEX_DETECT_EVIL)) creature_ptr->esp_evil = TRUE;
		if (hex_spelling(creature_ptr, HEX_XTRA_MIGHT)) creature_ptr->stat_add[STAT_STR] += 4;
		if (hex_spelling(creature_ptr, HEX_BUILDING))
		{
			creature_ptr->stat_add[STAT_STR] += 4;
			creature_ptr->stat_add[STAT_DEX] += 4;
			creature_ptr->stat_add[STAT_CON] += 4;
		}
		if (hex_spelling(creature_ptr, HEX_DEMON_AURA))
		{
			creature_ptr->sh_fire = TRUE;
			creature_ptr->regenerate = TRUE;
		}
		if (hex_spelling(creature_ptr, HEX_ICE_ARMOR))
		{
			creature_ptr->sh_cold = TRUE; 
			creature_ptr->to_a += 30;
			creature_ptr->dis_to_a += 30;
		}
		if (hex_spelling(creature_ptr, HEX_SHOCK_CLOAK))
		{
			creature_ptr->sh_elec = TRUE;
			new_speed += 3;
		}
		for (i = 0; i <= INVEN_TOTAL; i++)
		{
			int ac = 0;
			if(!creature_ptr->equip_now[i]) continue; 
			o_ptr = &creature_ptr->inventory[i];
			if (!o_ptr->k_idx) continue;
			if (!object_is_armour(o_ptr)) continue;
			if (!object_is_cursed(o_ptr)) continue;
			ac += 5;
			if (o_ptr->curse_flags & TRC_HEAVY_CURSE) ac += 7;
			if (o_ptr->curse_flags & TRC_DIVINE_CURSE) ac += 13;
			creature_ptr->to_a += ac;
			creature_ptr->dis_to_a += ac;
		}
	}

	/* Calculate stats */
	for (i = 0; i < 6; i++)
	{
		int top, use, ind;

		/* Extract the new "stat_use" value for the stat */
		top = modify_stat_value(creature_ptr->stat_max[i], creature_ptr->stat_add[i]);

		/* Notice changes */
		if (creature_ptr->stat_top[i] != top)
		{
			/* Save the new value */
			creature_ptr->stat_top[i] = top;

			/* Redisplay the stats later */
			play_redraw |= (PR_STATS);

			/* Window stuff */
			play_window |= (PW_PLAYER);
		}


		/* Extract the new "stat_use" value for the stat */
		use = modify_stat_value(creature_ptr->stat_cur[i], creature_ptr->stat_add[i]);

		if ((i == STAT_CHA) && (has_cf_creature(creature_ptr, CF_ILL_NORM)))
		{
			/* 10 to 18/90 charisma, guaranteed, based on level */
			if (use < 8 + 2 * creature_ptr->lev)
			{
				use = 8 + 2 * creature_ptr->lev;
			}
		}

		/* Notice changes */
		if (creature_ptr->stat_use[i] != use)
		{
			/* Save the new value */
			creature_ptr->stat_use[i] = use;

			/* Redisplay the stats later */
			play_redraw |= (PR_STATS);

			/* Window stuff */
			play_window |= (PW_PLAYER);
		}


		/* Values: 3, 4, ..., 17 */
		if (use <= 18) ind = (use - 3);

		/* Ranges: 18/00-18/09, ..., 18/210-18/219 */
		else if (use <= 18+419) ind = (15 + (use - 18) / 10);

		/* Range: 18/220+ */
		else ind = (57);

		/* Notice changes */
		if (creature_ptr->stat_ind[i] != ind)
		{
			/* Save the new index */
			creature_ptr->stat_ind[i] = ind;

			/* Change in CON affects Hitpoints */
			if (i == STAT_CON)
			{
				creature_ptr->creature_update |= (CRU_HP);
			}

			/* Change in INT may affect Mana/Spells */
			else if (i == STAT_INT)
			{
				if (m_info[creature_ptr->cls_idx].spell_stat == STAT_INT)
				{
					creature_ptr->creature_update |= (CRU_MANA | CRU_SPELLS);
				}
			}

			/* Change in WIS may affect Mana/Spells */
			else if (i == STAT_WIS)
			{
				if (m_info[creature_ptr->cls_idx].spell_stat == STAT_WIS)
				{
					creature_ptr->creature_update |= (CRU_MANA | CRU_SPELLS);
				}
			}

			/* Change in WIS may affect Mana/Spells */
			else if (i == STAT_CHA)
			{
				if (m_info[creature_ptr->cls_idx].spell_stat == STAT_CHA)
				{
					creature_ptr->creature_update |= (CRU_MANA | CRU_SPELLS);
				}
			}

			/* Window stuff */
			play_window |= (PW_PLAYER);
		}
	}


	/* Apply temporary "stun" */
	if (creature_ptr->stun > 50)
	{
		creature_ptr->to_h[0] -= 20;
		creature_ptr->to_h[1] -= 20;
		creature_ptr->to_h_b  -= 20;
		creature_ptr->to_h_m  -= 20;
		creature_ptr->dis_to_h[0] -= 20;
		creature_ptr->dis_to_h[1] -= 20;
		creature_ptr->dis_to_h_b  -= 20;
		creature_ptr->to_d[0] -= 20;
		creature_ptr->to_d[1] -= 20;
		creature_ptr->to_d_m -= 20;
		creature_ptr->dis_to_d[0] -= 20;
		creature_ptr->dis_to_d[1] -= 20;
	}
	else if (creature_ptr->stun)
	{
		creature_ptr->to_h[0] -= 5;
		creature_ptr->to_h[1] -= 5;
		creature_ptr->to_h_b -= 5;
		creature_ptr->to_h_m -= 5;
		creature_ptr->dis_to_h[0] -= 5;
		creature_ptr->dis_to_h[1] -= 5;
		creature_ptr->dis_to_h_b -= 5;
		creature_ptr->to_d[0] -= 5;
		creature_ptr->to_d[1] -= 5;
		creature_ptr->to_d_m -= 5;
		creature_ptr->dis_to_d[0] -= 5;
		creature_ptr->dis_to_d[1] -= 5;
	}

	/* Wraith form */
	if (creature_ptr->wraith_form)
	{
		creature_ptr->reflect = TRUE;
		creature_ptr->pass_wall = TRUE;
	}

	if (creature_ptr->kabenuke)
	{
		creature_ptr->pass_wall = TRUE;
	}

	/* Temporary blessing */
	if (IS_BLESSED(creature_ptr))
	{
		creature_ptr->to_a += 5;
		creature_ptr->dis_to_a += 5;
		creature_ptr->to_h[0] += 10;
		creature_ptr->to_h[1] += 10;
		creature_ptr->to_h_b  += 10;
		creature_ptr->to_h_m  += 10;
		creature_ptr->dis_to_h[0] += 10;
		creature_ptr->dis_to_h[1] += 10;
		creature_ptr->dis_to_h_b += 10;
	}

	if (creature_ptr->magicdef)
	{
		creature_ptr->resist_blind = TRUE;
		creature_ptr->resist_conf = TRUE;
		creature_ptr->reflect = TRUE;
		creature_ptr->free_act = TRUE;
		creature_ptr->levitation = TRUE;
	}

	/* Temporary "Hero" */
	if (IS_HERO(creature_ptr))
	{
		creature_ptr->to_h[0] += 12;
		creature_ptr->to_h[1] += 12;
		creature_ptr->to_h_b  += 12;
		creature_ptr->to_h_m  += 12;
		creature_ptr->dis_to_h[0] += 12;
		creature_ptr->dis_to_h[1] += 12;
		creature_ptr->dis_to_h_b  += 12;
	}

	/* Temporary "Beserk" */
	if (creature_ptr->shero)
	{
		creature_ptr->to_h[0] += 12;
		creature_ptr->to_h[1] += 12;
		creature_ptr->to_h_b  -= 12;
		creature_ptr->to_h_m  += 12;
		creature_ptr->to_d[0] += 3+(creature_ptr->lev/5);
		creature_ptr->to_d[1] += 3+(creature_ptr->lev/5);
		creature_ptr->to_d_m  += 3+(creature_ptr->lev/5);
		creature_ptr->dis_to_h[0] += 12;
		creature_ptr->dis_to_h[1] += 12;
		creature_ptr->dis_to_h_b  -= 12;
		creature_ptr->dis_to_d[0] += 3+(creature_ptr->lev/5);
		creature_ptr->dis_to_d[1] += 3+(creature_ptr->lev/5);
		creature_ptr->to_a -= 10;
		creature_ptr->dis_to_a -= 10;
		creature_ptr->skill_stl -= 7;
		creature_ptr->skill_dev -= 20;
		creature_ptr->skill_rob += 20;
		creature_ptr->skill_eva += 20;
		creature_ptr->skill_vol -= 30;
		creature_ptr->skill_srh -= 15;
		creature_ptr->skill_fos -= 15;
		creature_ptr->skill_tht -= 20;
		creature_ptr->skill_dig += 30;
	}

	/* Temporary "fast" */
	if (IS_FAST(creature_ptr))
	{
		new_speed += 10;
	}

	/* Temporary "slow" */
	if (creature_ptr->slow)
	{
		new_speed -= 10;
	}

	/* Temporary "telepathy" */
	if (IS_TIM_ESP(creature_ptr))
	{
		creature_ptr->telepathy = TRUE;
	}

	if (creature_ptr->ele_immune)
	{
		if (creature_ptr->special_defense & DEFENSE_ACID)
			creature_ptr->immune_acid = TRUE;
		else if (creature_ptr->special_defense & DEFENSE_ELEC)
			creature_ptr->immune_elec = TRUE;
		else if (creature_ptr->special_defense & DEFENSE_FIRE)
			creature_ptr->immune_fire = TRUE;
		else if (creature_ptr->special_defense & DEFENSE_COLD)
			creature_ptr->immune_cold = TRUE;
	}

	/* Temporary see invisible */
	if (creature_ptr->tim_invis)
	{
		creature_ptr->see_inv = TRUE;
	}

	/* Temporary infravision boost */
	if (creature_ptr->tim_infra)
	{
		creature_ptr->see_infra+=3;
	}

	/* Temporary regeneration boost */
	if (creature_ptr->tim_regen)
	{
		creature_ptr->regenerate = TRUE;
	}

	/* Temporary levitation */
	if (creature_ptr->tim_levitation)
	{
		creature_ptr->levitation = TRUE;
	}

	/* Temporary reflection */
	if (creature_ptr->tim_reflect)
	{
		creature_ptr->reflect = TRUE;
	}

	/* Hack -- Hero/Shero -> Res fear */
	if (IS_HERO(creature_ptr) || creature_ptr->shero)
	{
		creature_ptr->resist_fear = TRUE;
	}


	/* Hack -- Telepathy Change */
	if (creature_ptr->telepathy != old_telepathy)
	{
		update |= (PU_MONSTERS);
	}

	if ((creature_ptr->esp_animal != old_esp_animal) ||
	    (creature_ptr->esp_undead != old_esp_undead) ||
	    (creature_ptr->esp_demon != old_esp_demon) ||
	    (creature_ptr->esp_orc != old_esp_orc) ||
	    (creature_ptr->esp_troll != old_esp_troll) ||
	    (creature_ptr->esp_giant != old_esp_giant) ||
	    (creature_ptr->esp_dragon != old_esp_dragon) ||
	    (creature_ptr->esp_human != old_esp_human) ||
	    (creature_ptr->esp_evil != old_esp_evil) ||
	    (creature_ptr->esp_good != old_esp_good) ||
	    (creature_ptr->esp_nonliving != old_esp_nonliving) ||
	    (creature_ptr->esp_unique != old_esp_unique))
	{
		update |= (PU_MONSTERS);
	}

	/* Hack -- See Invis Change */
	if (creature_ptr->see_inv != old_see_inv)
	{
		update |= (PU_MONSTERS);
	}

	/* Bloating slows the player down (a little) */
	if (creature_ptr->food >= PY_FOOD_MAX) new_speed -= 10;

	if (creature_ptr->special_defense & KAMAE_SUZAKU) new_speed += 10;

	if ((creature_ptr->can_melee[0] && (empty_hands_status & EMPTY_HAND_RARM)) ||
	    (creature_ptr->can_melee[1] && (empty_hands_status & EMPTY_HAND_LARM)))
	{
		creature_ptr->to_h[default_hand] += (creature_ptr->skill_exp[GINOU_SUDE]) / 200;
		creature_ptr->dis_to_h[default_hand] += (creature_ptr->skill_exp[GINOU_SUDE]) / 200;
		creature_ptr->to_d[default_hand] += creature_ptr->size * (10 + (creature_ptr->skill_exp[GINOU_SUDE]) / 200) / 100;
		creature_ptr->dis_to_d[default_hand] += creature_ptr->size * (10 + (creature_ptr->skill_exp[GINOU_SUDE]) / 200) / 100;
	}

	//TODO: adjust
	/*
	if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 1)
	{
		int penalty1, penalty2;
		penalty1 = ((100 - creature_ptr->skill_exp[GINOU_NITOURYU] / 160) - (130 - creature_ptr->inventory[].weight) / 8);
		penalty2 = ((100 - creature_ptr->skill_exp[GINOU_NITOURYU] / 160) - (130 - creature_ptr->inventory[].weight) / 8);
		if ((creature_ptr->inventory[].name1 == ART_QUICKTHORN) && (creature_ptr->inventory[].name1 == ART_TINYTHORN))
		{
			penalty1 = penalty1 / 2 - 5;
			penalty2 = penalty2 / 2 - 5;
			new_speed += 7;
			creature_ptr->to_a += 10;
			creature_ptr->dis_to_a += 10;
		}
		if (easy_2weapon)
		{
			if (penalty1 > 0) penalty1 /= 2;
			if (penalty2 > 0) penalty2 /= 2;
		}
		else if ((creature_ptr->inventory[].tval == TV_SWORD) && ((creature_ptr->inventory[].sval == SV_MAIN_GAUCHE) || (creature_ptr->inventory[].sval == SV_WAKIZASHI)))
		{
			penalty1 = MAX(0, penalty1 - 10);
			penalty2 = MAX(0, penalty2 - 10);
		}
		if ((creature_ptr->inventory[].name1 == ART_MUSASI_KATANA) && (creature_ptr->inventory[].name1 == ART_MUSASI_WAKIZASI))
		{
			penalty1 = MIN(0, penalty1);
			penalty2 = MIN(0, penalty2);
			creature_ptr->to_a += 10;
			creature_ptr->dis_to_a += 10;
		}
		else
		{
			if ((creature_ptr->inventory[].name1 == ART_MUSASI_KATANA) && (penalty1 > 0))
				penalty1 /= 2;
			if ((creature_ptr->inventory[].name1 == ART_MUSASI_WAKIZASI) && (penalty2 > 0))
				penalty2 /= 2;
		}
		if (creature_ptr->inventory[].tval == TV_POLEARM) penalty1 += 10;
		if (creature_ptr->inventory[].tval == TV_POLEARM) penalty2 += 10;
		creature_ptr->to_h[0] -= penalty1;
		creature_ptr->to_h[1] -= penalty2;
		creature_ptr->dis_to_h[0] -= penalty1;
		creature_ptr->dis_to_h[1] -= penalty2;
	}
	*/

	/* Extract the current weight (in tenth pounds) */
	j = creature_ptr->total_weight;

	if (!creature_ptr->riding)
	{
		/* Extract the "weight limit" (in tenth pounds) */
		i = (int)weight_limit(creature_ptr);
	}
	else
	{
		creature_type *riding_m_ptr = &creature_list[creature_ptr->riding];
		species_type *riding_r_ptr = &species_info[riding_m_ptr->species_idx];
		int speed = riding_m_ptr->speed;

		if (riding_m_ptr->speed > 110)
		{
			new_speed = 110 + (s16b)((speed - 110) * (creature_ptr->skill_exp[GINOU_RIDING] * 3 + creature_ptr->lev * 160L - 10000L) / (22000L));
			if (new_speed < 110) new_speed = 110;
		}
		else
		{
			new_speed = speed;
		}
		new_speed += (creature_ptr->skill_exp[GINOU_RIDING] + creature_ptr->lev *160L)/3200;
		if (riding_m_ptr->fast) new_speed += 10;
		if (riding_m_ptr->slow) new_speed -= 10;
		riding_levitation = can_fly_species(riding_r_ptr) ? TRUE : FALSE;
		if (can_swim_species(riding_r_ptr) || is_aquatic_species(riding_r_ptr)) creature_ptr->can_swim = TRUE;

		if (!is_pass_wall_species(riding_r_ptr)) creature_ptr->pass_wall = FALSE;
		if (is_kill_wall_species(riding_r_ptr)) creature_ptr->kill_wall = TRUE;

		if (creature_ptr->skill_exp[GINOU_RIDING] < RIDING_EXP_SKILLED) j += (creature_ptr->wt * 3 * (RIDING_EXP_SKILLED - creature_ptr->skill_exp[GINOU_RIDING])) / RIDING_EXP_SKILLED;

		/* Extract the "weight limit" */
		i = 1500 + riding_r_ptr->level * 25;
	}

	/* XXX XXX XXX Apply "encumbrance" from weight */
	//TODO CHECK
	if (j > i) new_speed -= ((j - i) / (i / 5 + 1));

	/* Searching slows the player down */
	if (creature_ptr->action == ACTION_SEARCH) new_speed -= 10;

	/* Actual Modifier Bonuses (Un-inflate stat bonuses) */
	creature_ptr->to_a += ((int)(adj_dex_ta[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->to_d[0] += ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_d[1] += ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_d_m  += ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_h[0] += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->to_h[1] += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->to_h_b  += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->to_h_m  += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->to_h[0] += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_h[1] += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_h_b  += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->to_h_m  += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);

	/* Displayed Modifier Bonuses (Un-inflate stat bonuses) */
	creature_ptr->dis_to_a += ((int)(adj_dex_ta[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->dis_to_d[0] += ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->dis_to_d[1] += ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->dis_to_h[0] += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->dis_to_h[1] += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->dis_to_h_b  += ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);
	creature_ptr->dis_to_h[0] += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->dis_to_h[1] += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);
	creature_ptr->dis_to_h_b  += ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128);


	/* Obtain the "hold" value */
	hold = adj_str_hold[creature_ptr->stat_ind[STAT_STR]] * creature_ptr->size / 10 * creature_ptr->size / 10;


	// Examine the "current bow"
	o_ptr = get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_BOW, 1);

	if(o_ptr->k_idx)
	{
		/* Assume not heavy */
		creature_ptr->heavy_shoot = FALSE;

		/* It is hard to carholdry a heavy bow */
		if (hold < o_ptr->weight / 10)
		{
			/* Hard to wield a heavy bow */
			creature_ptr->to_h_b  += 2 * (hold - o_ptr->weight / 10);
			creature_ptr->dis_to_h_b  += 2 * (hold - o_ptr->weight / 10);

			/* Heavy Bow */
			creature_ptr->heavy_shoot = TRUE;
		}

		/* Compute "extra shots" if needed */
		if (o_ptr->k_idx)
		{
			/* Analyze the launcher */
			switch (o_ptr->sval)
			{
				case SV_SLING:
				{
					creature_ptr->tval_ammo = TV_SHOT;
					break;
				}

				case SV_SHORT_BOW:
				case SV_LONG_BOW:
				case SV_NAMAKE_BOW:
				{
					creature_ptr->tval_ammo = TV_ARROW;
					break;
				}

				case SV_LIGHT_XBOW:
				case SV_HEAVY_XBOW:
				{
					creature_ptr->tval_ammo = TV_BOLT;
					break;
				}
				case SV_CRIMSON:
				{
					creature_ptr->tval_ammo = TV_NO_AMMO;
					break;
				}
			}

			/* Apply special flags */
			if (o_ptr->k_idx && !creature_ptr->heavy_shoot)
			{
				/* Extra shots */
				creature_ptr->num_fire += (extra_shots * 100);

				/* Hack -- Rangers love Bows */
				if ((creature_ptr->cls_idx == CLASS_RANGER) &&
				    (creature_ptr->tval_ammo == TV_ARROW))
				{
					creature_ptr->num_fire += (creature_ptr->lev * 4);
				}

				if ((creature_ptr->cls_idx == CLASS_CAVALRY) &&
				    (creature_ptr->tval_ammo == TV_ARROW))
				{
					creature_ptr->num_fire += (creature_ptr->lev * 3);
				}

				if (creature_ptr->cls_idx == CLASS_ARCHER)
				{
					if (creature_ptr->tval_ammo == TV_ARROW)
						creature_ptr->num_fire += ((creature_ptr->lev * 5)+50);
					else if ((creature_ptr->tval_ammo == TV_BOLT) || (creature_ptr->tval_ammo == TV_SHOT))
						creature_ptr->num_fire += (creature_ptr->lev * 4);
				}

				/*
				 * Addendum -- also "Reward" high level warriors,
				 * with _any_ missile weapon -- TY
				 */
				if (creature_ptr->cls_idx == CLASS_WARRIOR &&
				   (creature_ptr->tval_ammo <= TV_BOLT) &&
				   (creature_ptr->tval_ammo >= TV_SHOT))
				{
					creature_ptr->num_fire += (creature_ptr->lev * 2);
				}
				if ((creature_ptr->cls_idx == CLASS_ROGUE) &&
				    (creature_ptr->tval_ammo == TV_SHOT))
				{
					creature_ptr->num_fire += (creature_ptr->lev * 4);
				}

				/* Snipers love Cross bows */
				if ((creature_ptr->cls_idx == CLASS_SNIPER) &&
					(creature_ptr->tval_ammo == TV_BOLT))
				{
					creature_ptr->to_h_b += (10 + (creature_ptr->lev / 5));
					creature_ptr->dis_to_h_b += (10 + (creature_ptr->lev / 5));
				}
			}
		}
	}

	if (creature_ptr->two_handed) hold *= 2;

	for(i = 0 ; i < MAX_HANDS ; i++)
	{
		/* Examine the "main weapon" */
		o_ptr = get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 1 + i);

		object_flags(o_ptr, flgs);

		/* Assume not heavy */
		creature_ptr->heavy_wield[i] = FALSE;
		creature_ptr->icky_wield[i] = FALSE;
		creature_ptr->riding_wield[i] = FALSE;

		if (!(get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > i)) {creature_ptr->num_blow[i]=1;continue;}
		/* It is hard to hold a heavy weapon */
		if (hold < o_ptr->weight / 10)
		{
			/* Hard to wield a heavy weapon */
			creature_ptr->to_h[i] += 2 * (hold - o_ptr->weight / 10);
			creature_ptr->dis_to_h[i] += 2 * (hold - o_ptr->weight / 10);

			/* Heavy weapon */
			creature_ptr->heavy_wield[i] = TRUE;
		}
		else if (creature_ptr->two_handed && (hold < o_ptr->weight/5)) omoi = TRUE;

		if ((i == 1) && (o_ptr->tval == TV_SWORD) && ((o_ptr->sval == SV_MAIN_GAUCHE) || (o_ptr->sval == SV_WAKIZASHI)))
		{
			creature_ptr->to_a += 5;
			creature_ptr->dis_to_a += 5;
		}

		/* Normal weapons */
		if (o_ptr->k_idx && !creature_ptr->heavy_wield[i])
		{
			int str_index, dex_index;

			int num = 0, wgt = 0, mul = 0, div = 0;

			/* Analyze the class */
			switch (creature_ptr->cls_idx)
			{
				/* Warrior */
				case CLASS_WARRIOR:
					num = 6; wgt = 70; mul = 5; break;

				/* Berserker */
				case CLASS_BERSERKER:
					num = 6; wgt = 70; mul = 7; break;

				/* Mage */
				case CLASS_MAGE:
				case CLASS_HIGH_MAGE:
				case CLASS_BLUE_MAGE:
					num = 3; wgt = 100; mul = 2; break;

				/* Priest, Mindcrafter, Magic-Eater */
				case CLASS_PRIEST:
				case CLASS_MAGIC_EATER:
				case CLASS_MINDCRAFTER:
					num = 5; wgt = 100; mul = 3; break;

				/* Rogue */
				case CLASS_ROGUE:
					num = 5; wgt = 40; mul = 3; break;

				/* Ranger */
				case CLASS_RANGER:
					num = 5; wgt = 70; mul = 4; break;

				/* Paladin */
				case CLASS_PALADIN:
				case CLASS_SAMURAI:
					num = 5; wgt = 70; mul = 4; break;

				/* Weaponsmith */
				case CLASS_SMITH:
					num = 5; wgt = 150; mul = 5; break;

				/* Warrior-Mage */
				case CLASS_WARRIOR_MAGE:
				case CLASS_RED_MAGE:
					num = 5; wgt = 70; mul = 3; break;

				/* Chaos Warrior */
				case CLASS_CHAOS_WARRIOR:
					num = 5; wgt = 70; mul = 4; break;

				/* Monk */
				case CLASS_MONK:
					num = 5; wgt = 60; mul = 3; break;

				/* Tourist */
				case CLASS_TOURIST:
					num = 4; wgt = 100; mul = 3; break;

				/* Imitator */
				case CLASS_IMITATOR:
					num = 5; wgt = 70; mul = 4; break;

				/* Beastmaster */
				case CLASS_BEASTMASTER:
					num = 5; wgt = 70; mul = 3; break;

				/* Cavalry */
				case CLASS_CAVALRY:
					if ((creature_ptr->riding) && (have_flag(flgs, TR_RIDING))) {num = 5; wgt = 70; mul = 4;}
					else {num = 5; wgt = 100; mul = 3;}
					break;

				/* Sorcerer */
				case CLASS_SORCERER:
					num = 1; wgt = 1; mul = 1; break;

				/* Archer, Bard */
				case CLASS_ARCHER:
				case CLASS_BARD:
					num = 4; wgt = 70; mul = 2; break;

				/* ForceTrainer */
				case CLASS_FORCETRAINER:
					num = 4; wgt = 60; mul = 2; break;

				/* Mirror Master, Sniper */
				case CLASS_MIRROR_MASTER:
				case CLASS_SNIPER:
					num = 3; wgt = 100; mul = 3; break;

				/* Ninja */
				case CLASS_NINJA:
					num = 4; wgt = 20; mul = 1; break;

				/* None */
				default:
					num = 4; wgt = 100; mul = 3; break;

			}

			/* Hex - extra mights gives +1 bonus to max blows */
			if (hex_spelling(creature_ptr, HEX_XTRA_MIGHT) || hex_spelling(creature_ptr, HEX_BUILDING)) { num++; wgt /= 2; mul += 2; }

			/* Enforce a minimum "weight" (tenth pounds) */
			div = ((o_ptr->weight < wgt) ? wgt : o_ptr->weight);

			/* Access the strength vs weight */
			str_index = (adj_str_blow[creature_ptr->stat_ind[STAT_STR]] * mul / div);

			if (creature_ptr->two_handed && !omoi) str_index++;
			if (creature_ptr->cls_idx == CLASS_NINJA) str_index = MAX(0, str_index-1);

			/* Maximal value */
			if (str_index > 11) str_index = 11;

			/* Index by dexterity */
			dex_index = (adj_dex_blow[creature_ptr->stat_ind[STAT_DEX]]);

			/* Maximal value */
			if (dex_index > 11) dex_index = 11;

			/* Use the blows table */
			creature_ptr->num_blow[i] = blows_table[str_index][dex_index];

			/* Maximal value */
			if (creature_ptr->num_blow[i] > num) creature_ptr->num_blow[i] = num;

			/* Add in the "bonus blows" */
			creature_ptr->num_blow[i] += extra_blows[i];


			if (creature_ptr->cls_idx == CLASS_WARRIOR) creature_ptr->num_blow[i] += (creature_ptr->lev / 40);
			else if (creature_ptr->cls_idx == CLASS_BERSERKER)
			{
				creature_ptr->num_blow[i] += (creature_ptr->lev / 23);
			}
			else if ((creature_ptr->cls_idx == CLASS_ROGUE) && (o_ptr->weight < 50) && (creature_ptr->stat_ind[STAT_DEX] >= 30)) creature_ptr->num_blow[i] ++;

			if (creature_ptr->special_defense & KATA_FUUJIN) creature_ptr->num_blow[i] -= 1;

			if ((o_ptr->tval == TV_SWORD) && (o_ptr->sval == SV_DOKUBARI)) creature_ptr->num_blow[i] = 1;


			/* Require at least one blow */
			if (creature_ptr->num_blow[i] < 1) creature_ptr->num_blow[i] = 1;

			/* Boost digging skill by weapon weight */
			creature_ptr->skill_dig += (o_ptr->weight / 10);
		}

		/* Assume okay */
		/* Priest weapon penalty for non-blessed edged weapons */
		if ((creature_ptr->cls_idx == CLASS_PRIEST) && (!(have_flag(flgs, TR_BLESSED))) &&
		    ((o_ptr->tval == TV_SWORD) || (o_ptr->tval == TV_POLEARM)))
		{
			/* Reduce the real bonuses */
			creature_ptr->to_h[i] -= 2;
			creature_ptr->to_d[i] -= 2;

			/* Reduce the mental bonuses */
			creature_ptr->dis_to_h[i] -= 2;
			creature_ptr->dis_to_d[i] -= 2;

			/* Icky weapon */
			creature_ptr->icky_wield[i] = TRUE;
		}
		else if (creature_ptr->cls_idx == CLASS_BERSERKER)
		{
			creature_ptr->to_h[i] += creature_ptr->lev/5;
			creature_ptr->to_d[i] += creature_ptr->lev/6;
			creature_ptr->dis_to_h[i] += creature_ptr->lev/5;
			creature_ptr->dis_to_d[i] += creature_ptr->lev/6;
			if (((i == 0) && !creature_ptr->can_melee[1]) || creature_ptr->two_handed)
			{
				creature_ptr->to_h[i] += creature_ptr->lev/5;
				creature_ptr->to_d[i] += creature_ptr->lev/6;
				creature_ptr->dis_to_h[i] += creature_ptr->lev/5;
				creature_ptr->dis_to_d[i] += creature_ptr->lev/6;
			}
		}
		else if (creature_ptr->cls_idx == CLASS_SORCERER)
		{
			if (!((o_ptr->tval == TV_HAFTED) && ((o_ptr->sval == SV_WIZSTAFF) || (o_ptr->sval == SV_NAMAKE_HAMMER))))
			{
				/* Reduce the real bonuses */
				creature_ptr->to_h[i] -= 200;
				creature_ptr->to_d[i] -= 200;

				/* Reduce the mental bonuses */
				creature_ptr->dis_to_h[i] -= 200;
				creature_ptr->dis_to_d[i] -= 200;

				/* Icky weapon */
				creature_ptr->icky_wield[i] = TRUE;
			}
			else
			{
				/* Reduce the real bonuses */
				creature_ptr->to_h[i] -= 30;
				creature_ptr->to_d[i] -= 10;

				/* Reduce the mental bonuses */
				creature_ptr->dis_to_h[i] -= 30;
				creature_ptr->dis_to_d[i] -= 10;
			}
		}
		/* Hex bonuses */
		if (creature_ptr->realm1 == REALM_HEX)
		{
			if (object_is_cursed(o_ptr))
			{
				if (o_ptr->curse_flags & (TRC_CURSED)) { creature_ptr->to_h[i] += 5; creature_ptr->dis_to_h[i] += 5; }
				if (o_ptr->curse_flags & (TRC_HEAVY_CURSE)) { creature_ptr->to_h[i] += 7; creature_ptr->dis_to_h[i] += 7; }
				if (o_ptr->curse_flags & (TRC_DIVINE_CURSE)) { creature_ptr->to_h[i] += 13; creature_ptr->dis_to_h[i] += 13; }
				if (o_ptr->curse_flags & (TRC_TY_CURSE)) { creature_ptr->to_h[i] += 5; creature_ptr->dis_to_h[i] += 5; }
				if (hex_spelling(creature_ptr, HEX_RUNESWORD))
				{
					if (o_ptr->curse_flags & (TRC_CURSED)) { creature_ptr->to_d[i] += 5; creature_ptr->dis_to_d[i] += 5; }
					if (o_ptr->curse_flags & (TRC_HEAVY_CURSE)) { creature_ptr->to_d[i] += 7; creature_ptr->dis_to_d[i] += 7; }
					if (o_ptr->curse_flags & (TRC_DIVINE_CURSE)) { creature_ptr->to_d[i] += 13; creature_ptr->dis_to_d[i] += 13; }
				}
			}
		}
		if (creature_ptr->riding)
		{
			if ((o_ptr->tval == TV_POLEARM) && ((o_ptr->sval == SV_LANCE) || (o_ptr->sval == SV_HEAVY_LANCE)))
			{
				creature_ptr->to_h[i] +=15;
				creature_ptr->dis_to_h[i] +=15;
				creature_ptr->to_dd[i] += 2;
			}
			else if (!(have_flag(flgs, TR_RIDING)))
			{
				int penalty;
				if ((creature_ptr->cls_idx == CLASS_BEASTMASTER) || (creature_ptr->cls_idx == CLASS_CAVALRY))
				{
					penalty = 5;
				}
				else
				{
					penalty = species_info[creature_list[creature_ptr->riding].species_idx].level - creature_ptr->skill_exp[GINOU_RIDING] / 80;
					penalty += 30;
					if (penalty < 30) penalty = 30;
				}
				creature_ptr->to_h[i] -= penalty;
				creature_ptr->dis_to_h[i] -= penalty;

				/* Riding weapon */
				creature_ptr->riding_wield[i] = TRUE;
			}
		}
	}

	if (creature_ptr->riding)
	{
		int penalty = 0;

		creature_ptr->riding_two_handed = FALSE;

		if (creature_ptr->two_handed || (empty_hands(creature_ptr, FALSE) == EMPTY_HAND_NONE)) creature_ptr->riding_two_handed = TRUE;
		else if (creature_ptr->pet_extra_flags & PF_RYOUTE)
		{
			switch (creature_ptr->cls_idx)
			{
			case CLASS_MONK:
			case CLASS_FORCETRAINER:
			case CLASS_BERSERKER:
				if ((empty_hands(creature_ptr, FALSE) != EMPTY_HAND_NONE) && !get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND))
					creature_ptr->riding_two_handed = TRUE;
				break;
			}
		}

		if ((creature_ptr->cls_idx == CLASS_BEASTMASTER) || (creature_ptr->cls_idx == CLASS_CAVALRY))
		{
			if (creature_ptr->tval_ammo != TV_ARROW) penalty = 5;
		}
		else
		{
			penalty = species_info[creature_list[creature_ptr->riding].species_idx].level - creature_ptr->skill_exp[GINOU_RIDING] / 80;
			penalty += 30;
			if (penalty < 30) penalty = 30;
		}
		if (creature_ptr->tval_ammo == TV_BOLT) penalty *= 2;
		creature_ptr->to_h_b -= penalty;
		creature_ptr->dis_to_h_b -= penalty;
	}

	/* Different calculation for monks with empty hands */
	if (((creature_ptr->cls_idx == CLASS_MONK) || (creature_ptr->cls_idx == CLASS_FORCETRAINER) || (creature_ptr->cls_idx == CLASS_BERSERKER)) &&
		(empty_hands_status & EMPTY_HAND_RARM) && !creature_ptr->can_melee[1])
	{
		int blow_base = creature_ptr->lev + adj_dex_blow[creature_ptr->stat_ind[STAT_DEX]];
		creature_ptr->num_blow[0] = 0;

		if (creature_ptr->cls_idx == CLASS_FORCETRAINER)
		{
			if (blow_base > 18) creature_ptr->num_blow[0]++;
			if (blow_base > 31) creature_ptr->num_blow[0]++;
			if (blow_base > 44) creature_ptr->num_blow[0]++;
			if (blow_base > 58) creature_ptr->num_blow[0]++;
			if (creature_ptr->magic_num1[0])
			{
				creature_ptr->to_d[0] += (s16b)(creature_ptr->magic_num1[0]/5);
				creature_ptr->dis_to_d[0] += (s16b)(creature_ptr->magic_num1[0]/5);
			}
		}
		else
		{
			if (blow_base > 12) creature_ptr->num_blow[0]++;
			if (blow_base > 22) creature_ptr->num_blow[0]++;
			if (blow_base > 31) creature_ptr->num_blow[0]++;
			if (blow_base > 39) creature_ptr->num_blow[0]++;
			if (blow_base > 46) creature_ptr->num_blow[0]++;
			if (blow_base > 53) creature_ptr->num_blow[0]++;
			if (blow_base > 59) creature_ptr->num_blow[0]++;
		}

		if (heavy_armor(creature_ptr) && (creature_ptr->cls_idx != CLASS_BERSERKER))
			creature_ptr->num_blow[0] /= 2;
		else
		{
			creature_ptr->to_h[0] += (creature_ptr->lev / 3);
			creature_ptr->dis_to_h[0] += (creature_ptr->lev / 3);

			creature_ptr->to_d[0] += (creature_ptr->lev / 6);
			creature_ptr->dis_to_d[0] += (creature_ptr->lev / 6);
		}

		if (creature_ptr->special_defense & KAMAE_BYAKKO)
		{
			creature_ptr->to_a -= 40;
			creature_ptr->dis_to_a -= 40;
			
		}
		else if (creature_ptr->special_defense & KAMAE_SEIRYU)
		{
			creature_ptr->to_a -= 50;
			creature_ptr->dis_to_a -= 50;
			creature_ptr->resist_acid = TRUE;
			creature_ptr->resist_fire = TRUE;
			creature_ptr->resist_elec = TRUE;
			creature_ptr->resist_cold = TRUE;
			creature_ptr->resist_pois = TRUE;
			creature_ptr->sh_fire = TRUE;
			creature_ptr->sh_elec = TRUE;
			creature_ptr->sh_cold = TRUE;
			creature_ptr->levitation = TRUE;
		}
		else if (creature_ptr->special_defense & KAMAE_GENBU)
		{
			creature_ptr->to_a += (creature_ptr->lev*creature_ptr->lev)/50;
			creature_ptr->dis_to_a += (creature_ptr->lev*creature_ptr->lev)/50;
			creature_ptr->reflect = TRUE;
			creature_ptr->num_blow[0] -= 2;
			if ((creature_ptr->cls_idx == CLASS_MONK) && (creature_ptr->lev > 42)) creature_ptr->num_blow[0]--;
			if (creature_ptr->num_blow[0] < 0) creature_ptr->num_blow[0] = 0;
		}
		else if (creature_ptr->special_defense & KAMAE_SUZAKU)
		{
			creature_ptr->to_h[0] -= (creature_ptr->lev / 3);
			creature_ptr->to_d[0] -= (creature_ptr->lev / 6);

			creature_ptr->dis_to_h[0] -= (creature_ptr->lev / 3);
			creature_ptr->dis_to_d[0] -= (creature_ptr->lev / 6);
			creature_ptr->num_blow[0] /= 2;
			creature_ptr->levitation = TRUE;
		}

		creature_ptr->num_blow[0] += 1+extra_blows[0];
	}

	if (creature_ptr->riding) creature_ptr->levitation = riding_levitation;

	monk_armour_aux = FALSE;

	if (heavy_armor(creature_ptr))
	{
		monk_armour_aux = TRUE;
	}

	for (i = 0; i < 2; i++)
	{
		if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 0)
		{
			int tval = get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, i+1)->tval - TV_WEAPON_BEGIN;
			int sval = get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, i+1)->sval;
			int boost = (creature_ptr->weapon_exp[tval][sval] - WEAPON_EXP_BEGINNER) / 200;

			creature_ptr->to_h[i] += boost > 0 ? boost : 0;
			creature_ptr->dis_to_h[i] += boost > 0 ? boost : 0;

			if ((creature_ptr->cls_idx == CLASS_MONK) || (creature_ptr->cls_idx == CLASS_FORCETRAINER))
			{
				if (!skill_info[creature_ptr->cls_idx].w_max[tval][sval])
				{
					creature_ptr->to_h[i] -= 40;
					creature_ptr->dis_to_h[i] -= 40;
					creature_ptr->icky_wield[i] = TRUE;
				}
			}
			else if (creature_ptr->cls_idx == CLASS_NINJA)
			{
				if ((skill_info[CLASS_NINJA].w_max[tval][sval] <= WEAPON_EXP_BEGINNER) ||
					(get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, 2)->tval == TV_SHIELD))
				{
					creature_ptr->to_h[i] -= 40;
					creature_ptr->dis_to_h[i] -= 40;
					creature_ptr->icky_wield[i] = TRUE;
					creature_ptr->num_blow[i] /= 2;
					if (creature_ptr->num_blow[i] < 1) creature_ptr->num_blow[i] = 1;
				}
			}

			if (get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_HAND, i+1)->name1 == ART_IRON_BALL) creature_ptr->good -= 1000;
		}
	}

	/* Maximum speed is (+99). (internally it's 110 + 99) */
	/* Temporary lightspeed forces to be maximum speed */
	if ((creature_ptr->lightspeed && !creature_ptr->riding) || (new_speed > 209))
	{
		new_speed = 209;
	}

	/* Minimum speed is (-99). (internally it's 110 - 99) */
	if (new_speed < 11) new_speed = 11;

	/* Display the speed (if needed) */
	if (creature_ptr->speed != (byte)new_speed)
	{
		creature_ptr->speed = (byte)new_speed;
		play_redraw |= (PR_SPEED);
	}

	if (yoiyami)
	{
		if (creature_ptr->to_a > (0 - creature_ptr->ac))
			creature_ptr->to_a = 0 - creature_ptr->ac;
		if (creature_ptr->dis_to_a > (0 - creature_ptr->dis_ac))
			creature_ptr->dis_to_a = 0 - creature_ptr->dis_ac;
	}

	/* Redraw armor (if needed) */
	if ((creature_ptr->dis_ac != old_dis_ac) || (creature_ptr->dis_to_a != old_dis_to_a))
	{
		/* Redraw */
		play_redraw |= (PR_ARMOR);

		/* Window stuff */
		play_window |= (PW_PLAYER);
	}


	if (creature_ptr->two_handed && !omoi)
	{
		int bonus_to_h=0, bonus_to_d=0;
		bonus_to_d = ((int)(adj_str_td[creature_ptr->stat_ind[STAT_STR]]) - 128)/2;
		bonus_to_h = ((int)(adj_str_th[creature_ptr->stat_ind[STAT_STR]]) - 128) + ((int)(adj_dex_th[creature_ptr->stat_ind[STAT_DEX]]) - 128);

		creature_ptr->to_h[default_hand] += MAX(bonus_to_h,1);
		creature_ptr->dis_to_h[default_hand] += MAX(bonus_to_h,1);
		creature_ptr->to_d[default_hand] += MAX(bonus_to_d,1);
		creature_ptr->dis_to_d[default_hand] += MAX(bonus_to_d,1);
	}

	//TODO if (((creature_ptr->cls_idx == CLASS_MONK) || (creature_ptr->cls_idx == CLASS_FORCETRAINER) || (creature_ptr->cls_idx == CLASS_BERSERKER)) && (empty_hands(creature_ptr, FALSE) == (EMPTY_HAND_RARM | EMPTY_HAND_LARM))) creature_ptr->two_handed = FALSE;

	/* Affect Skill -- stealth (bonus one) */
	creature_ptr->skill_stl += 1;

	if (IS_TIM_STEALTH(creature_ptr)) creature_ptr->skill_stl += 99;

	/* Affect Skill -- disarming (DEX and INT) */
	creature_ptr->skill_dis += adj_dex_dis[creature_ptr->stat_ind[STAT_DEX]];
	creature_ptr->skill_dis += adj_int_dis[creature_ptr->stat_ind[STAT_INT]];

	/* Affect Skill -- magic devices (INT) */
	creature_ptr->skill_dev += adj_int_dev[creature_ptr->stat_ind[STAT_INT]];

	/* Affect Skill -- saving throw (WIS) */
	creature_ptr->skill_rob += (adj_sav[creature_ptr->stat_ind[STAT_STR]] + adj_sav[creature_ptr->stat_ind[STAT_CON]]);
	creature_ptr->skill_eva += (adj_sav[creature_ptr->stat_ind[STAT_INT]] + adj_sav[creature_ptr->stat_ind[STAT_DEX]]);
	creature_ptr->skill_vol += (adj_sav[creature_ptr->stat_ind[STAT_WIS]] + adj_sav[creature_ptr->stat_ind[STAT_CHA]]);

	/* Affect Skill -- digging (STR) */
	creature_ptr->skill_dig += adj_str_dig[creature_ptr->stat_ind[STAT_STR]];

	/* Affect Skill -- disarming (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_dis += (classkill_info[creature_ptr->cls_idx].x_dis * creature_ptr->lev / 10);
	else creature_ptr->skill_dis += (classkill_info[CLASS_TOURIST].x_dis * creature_ptr->lev / 10);

	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_dis += (chara_info[creature_ptr->chara_idx].a_dis * creature_ptr->lev / 50);

	/* Affect Skill -- magic devices (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_dev += (classkill_info[creature_ptr->cls_idx].x_dev * creature_ptr->lev / 10);
	else creature_ptr->skill_dev += (classkill_info[CLASS_TOURIST].x_dev * creature_ptr->lev / 10);

	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_dev += (chara_info[creature_ptr->chara_idx].a_dev * creature_ptr->lev / 50);

	/* Affect Skill -- saving throw (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_rob += (classkill_info[creature_ptr->cls_idx].x_sav * creature_ptr->lev / 10);
	else creature_ptr->skill_rob += (classkill_info[CLASS_TOURIST].x_sav * creature_ptr->lev / 10);
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_eva += (classkill_info[creature_ptr->cls_idx].x_sav * creature_ptr->lev / 10);
	else creature_ptr->skill_rob += (classkill_info[CLASS_TOURIST].x_sav * creature_ptr->lev / 10);
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_vol += (classkill_info[creature_ptr->cls_idx].x_sav * creature_ptr->lev / 10);
	else creature_ptr->skill_rob += (classkill_info[CLASS_TOURIST].x_sav * creature_ptr->lev / 10);

	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_rob += (chara_info[creature_ptr->chara_idx].a_sav * creature_ptr->lev / 50);
	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_eva += (chara_info[creature_ptr->chara_idx].a_sav * creature_ptr->lev / 50);
	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_vol += (chara_info[creature_ptr->chara_idx].a_sav * creature_ptr->lev / 50);

	/* Affect Skill -- stealth (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_stl += (classkill_info[creature_ptr->cls_idx].x_stl * creature_ptr->lev / 10);
	else creature_ptr->skill_stl += (classkill_info[CLASS_TOURIST].x_stl * creature_ptr->lev / 10);

	/* Affect Skill -- search ability (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_srh += (classkill_info[creature_ptr->cls_idx].x_srh * creature_ptr->lev / 10);
	else creature_ptr->skill_srh += (classkill_info[CLASS_TOURIST].x_srh * creature_ptr->lev / 10);

	/* Affect Skill -- search frequency (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_fos += (classkill_info[creature_ptr->cls_idx].x_fos * creature_ptr->lev / 10);
	else  creature_ptr->skill_fos += (classkill_info[CLASS_TOURIST].x_fos * creature_ptr->lev / 10);

	/* Affect Skill -- combat (normal) (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_thn += (classkill_info[creature_ptr->cls_idx].x_thn * creature_ptr->lev / 10);
	else creature_ptr->skill_thn += (classkill_info[CLASS_TOURIST].x_thn * creature_ptr->lev / 10);
	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_thn += (chara_info[creature_ptr->chara_idx].a_thn * creature_ptr->lev / 50);

	/* Affect Skill -- combat (shooting) (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_thb += (classkill_info[creature_ptr->cls_idx].x_thb * creature_ptr->lev / 10);
	else creature_ptr->skill_thb += (classkill_info[CLASS_TOURIST].x_thb * creature_ptr->lev / 10);
	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_thb += (chara_info[creature_ptr->chara_idx].a_thb * creature_ptr->lev / 50);

	/* Affect Skill -- combat (throwing) (Level, by Class) */
	if(creature_ptr->cls_idx != INDEX_NONE) creature_ptr->skill_tht += (classkill_info[creature_ptr->cls_idx].x_thb * creature_ptr->lev / 10);
	else creature_ptr->skill_tht += (classkill_info[CLASS_TOURIST].x_thb * creature_ptr->lev / 10);
	if(creature_ptr->chara_idx != INDEX_NONE) creature_ptr->skill_tht += (chara_info[creature_ptr->chara_idx].a_thb * creature_ptr->lev / 50);


	if ((race_is_(creature_ptr, RACE_S_FAIRY)) && (creature_ptr->chara_idx != CHARA_SEXY) && (creature_ptr->cursed & TRC_AGGRAVATE))
	{
		creature_ptr->cursed &= ~(TRC_AGGRAVATE);
		creature_ptr->skill_stl = MIN(creature_ptr->skill_stl - 3, (creature_ptr->skill_stl + 2) / 2);
	}

	/* Limit Skill -- stealth from 0 to 30 */
	if (creature_ptr->skill_stl > 30) creature_ptr->skill_stl = 30;
	if (creature_ptr->skill_stl < 0) creature_ptr->skill_stl = 0;

	/* Limit Skill -- digging from 1 up */
	if (creature_ptr->skill_dig < 1) creature_ptr->skill_dig = 1;

	if (creature_ptr->anti_magic && (creature_ptr->skill_rob < (90 + creature_ptr->lev))) creature_ptr->skill_rob = 90 + creature_ptr->lev;
	if (creature_ptr->anti_magic && (creature_ptr->skill_eva < (90 + creature_ptr->lev))) creature_ptr->skill_eva = 90 + creature_ptr->lev;
	if (creature_ptr->anti_magic && (creature_ptr->skill_vol < (90 + creature_ptr->lev))) creature_ptr->skill_vol = 90 + creature_ptr->lev;

	if (creature_ptr->tsubureru) creature_ptr->skill_rob = 10;
	if (creature_ptr->tsubureru) creature_ptr->skill_eva = 10;
	if (creature_ptr->tsubureru) creature_ptr->skill_vol = 10;

	if ((creature_ptr->ult_res || creature_ptr->resist_magic || creature_ptr->magicdef) && (creature_ptr->skill_rob < (95 + creature_ptr->lev))) creature_ptr->skill_rob = 95 + creature_ptr->lev;
	if ((creature_ptr->ult_res || creature_ptr->resist_magic || creature_ptr->magicdef) && (creature_ptr->skill_eva < (95 + creature_ptr->lev))) creature_ptr->skill_eva = 95 + creature_ptr->lev;
	if ((creature_ptr->ult_res || creature_ptr->resist_magic || creature_ptr->magicdef) && (creature_ptr->skill_vol < (95 + creature_ptr->lev))) creature_ptr->skill_vol = 95 + creature_ptr->lev;

	if (down_saving) creature_ptr->skill_rob /= 2;
	if (down_saving) creature_ptr->skill_eva /= 2;
	if (down_saving) creature_ptr->skill_vol /= 2;

	/* Hack -- Each elemental immunity includes resistance */
	if (creature_ptr->immune_acid) creature_ptr->resist_acid = TRUE;
	if (creature_ptr->immune_elec) creature_ptr->resist_elec = TRUE;
	if (creature_ptr->immune_fire) creature_ptr->resist_fire = TRUE;
	if (creature_ptr->immune_cold) creature_ptr->resist_cold = TRUE;

	/* Set Possible Equipment*/
	if(creature_ptr->race_idx1 != INDEX_NONE)
		creature_ptr->possible_equipment = race_info[creature_ptr->race_idx1].possible_equipment;
	else
		creature_ptr->possible_equipment = 0;



	/* calc karmas and bonuses */
	for(i = 0; i < MAX_KARMA; i++)
		creature_ptr->karmas_cur[i] = creature_ptr->karmas_cur[i];

	for(i = 0; i < MAX_KARMA; i++)
		creature_ptr->karmas_rank[i] = calc_rank(creature_ptr->karmas_cur[i]);


	/* calc alignments and bonuses */
	if(creature_ptr->race_idx1 != INDEX_NONE)
	{
		creature_ptr->good    = tmp_rcreature_ptr->good;
		creature_ptr->evil    = tmp_rcreature_ptr->evil;
		creature_ptr->order   = tmp_rcreature_ptr->order;
		creature_ptr->chaos   = tmp_rcreature_ptr->chaos;
		creature_ptr->balance = tmp_rcreature_ptr->balance;
	}
	else
	{
		creature_ptr->good    = 0;
		creature_ptr->evil    = 0;
		creature_ptr->order   = 0;
		creature_ptr->chaos   = 0;
		creature_ptr->balance = 0;
	}

	for(i = 0; i < MAX_KARMA; i++)
	{
		creature_ptr->good += creature_ptr->karmas[i] * karma[i].good_adj;
		creature_ptr->evil += creature_ptr->karmas[i] * karma[i].evil_adj;
		creature_ptr->order += creature_ptr->karmas[i] * karma[i].order_adj;
		creature_ptr->chaos += creature_ptr->karmas[i] * karma[i].chaos_adj;
	}

	creature_ptr->good_rank = calc_rank(creature_ptr->good);
	creature_ptr->evil_rank = calc_rank(creature_ptr->evil);
	creature_ptr->order_rank = calc_rank(creature_ptr->order);
	creature_ptr->chaos_rank = calc_rank(creature_ptr->chaos);

	creature_ptr->balance -= ((creature_ptr->good_rank - creature_ptr->evil_rank) * (creature_ptr->good_rank - creature_ptr->evil_rank) - 100) * 10;
	creature_ptr->balance -= ((creature_ptr->order_rank - creature_ptr->chaos_rank) * (creature_ptr->order_rank - creature_ptr->chaos_rank) - 100) * 10;

	creature_ptr->balance_rank = calc_rank(creature_ptr->balance);

	/* Take note when "heavy bow" changes */
	if (creature_ptr->old_heavy_shoot != creature_ptr->heavy_shoot)
	{
		/* Message */
		if (creature_ptr->heavy_shoot)
		{
#ifdef JP
			if(message) msg_print("ȏd|𑕔Ă̂͑ςB");
#else
			if(message) msg_print("You have trouble wielding such a heavy bow.");
#endif

		}
		else if (get_equipped_slot_ptr(creature_ptr, INVEN_SLOT_BOW, 1)->k_idx)
		{
#ifdef JP
			if(message) msg_print("̋|Ȃ瑕ĂĂhȂB");
#else
			if(message) msg_print("You have no trouble wielding your bow.");
#endif

		}
		else
		{
#ifdef JP
			if(message) msg_print("d|𑕔͂đ̂yɂȂB");
#else
			if(message) msg_print("You feel relieved to put down your heavy bow.");
#endif

		}

		/* Save it */
		creature_ptr->old_heavy_shoot = creature_ptr->heavy_shoot;
	}

	for (i = 0 ; i < 2 ; i++)
	{
		/* Take note when "heavy weapon" changes */
		if (creature_ptr->old_heavy_wield[i] != creature_ptr->heavy_wield[i])
		{
			/* Message */
			if (creature_ptr->heavy_wield[i])
			{
#ifdef JP
				if(message) msg_print("ȏd𑕔Ă̂͑ςB");
#else
				if(message) msg_print("You have trouble wielding such a heavy weapon.");
#endif

			}
			else if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 0)
			{
#ifdef JP
				if(message) msg_print("Ȃ瑕ĂĂhȂB");
#else
				if(message) msg_print("You have no trouble wielding your weapon.");
#endif

			}
			else if (creature_ptr->heavy_wield[1-i])
			{
#ifdef JP
				if(message) msg_print("܂킪dB");
#else
				if(message) msg_print("You have still trouble wielding a heavy weapon.");
#endif

			}
			else
			{
#ifdef JP
				if(message) msg_print("d𑕔͂đ̂yɂȂB");
#else
				if(message) msg_print("You feel relieved to put down your heavy weapon.");
#endif

			}

			/* Save it */
			creature_ptr->old_heavy_wield[i] = creature_ptr->heavy_wield[i];
		}

		/* Take note when "heavy weapon" changes */
		if (creature_ptr->old_riding_wield[i] != creature_ptr->riding_wield[i])
		{
			/* Message */
			if (creature_ptr->riding_wield[i])
			{
#ifdef JP
				if(message) msg_print("̕͏nɎgɂ͂ނȂ悤B");
#else
				if(message) msg_print("This weapon is not suitable for use while riding.");
#endif

			}
			else if (!creature_ptr->riding)
			{
#ifdef JP
				if(message) msg_print("͓̕kŎg₷B");
#else
				if(message) msg_print("This weapon was not suitable for use while riding.");
#endif

			}
			else if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 0)
			{
#ifdef JP
				if(message) msg_print("Ȃnɂ҂肾B");
#else
				if(message) msg_print("This weapon is suitable for use while riding.");
#endif

			}
			/* Save it */
			creature_ptr->old_riding_wield[i] = creature_ptr->riding_wield[i];
		}

		/* Take note when "illegal weapon" changes */
		if (creature_ptr->old_icky_wield[i] != creature_ptr->icky_wield[i])
		{
			/* Message */
			if (creature_ptr->icky_wield[i])
			{
#ifdef JP
				if(message) msg_print("̑͂ǂɂӂ킵ȂCB");
#else
				if(message) msg_print("You do not feel comfortable with your weapon.");
#endif
			}
			else if (get_equipped_slot_num(creature_ptr, INVEN_SLOT_HAND) > 0)
			{
#ifdef JP
				if(message) msg_print("͎̑ɂӂ킵CB");
#else
				if(message) msg_print("You feel comfortable with your weapon.");
#endif

			}
			else
			{
#ifdef JP
				if(message) msg_print("͂琏ƋCyɂȂB");
#else
				if(message) msg_print("You feel more comfortable after removing your weapon.");
#endif

			}

			/* Save it */
			creature_ptr->old_icky_wield[i] = creature_ptr->icky_wield[i];
		}
	}

	if (creature_ptr->riding && (creature_ptr->old_riding_two_handed != creature_ptr->riding_two_handed))
	{
		/* Message */
		if (creature_ptr->riding_two_handed)
		{
#ifdef JP
		if(message) msg_format("%sn𑀂ȂB", (empty_hands(creature_ptr, FALSE) == EMPTY_HAND_NONE) ? "肪ӂĂ" : "");
#else
			if(message) msg_print("You are using both hand for fighting, and you can't control a riding pet.");
#endif
		}
		else
		{
#ifdef JP
			if(message) msg_format("%sn𑀂悤ɂȂB", (empty_hands(creature_ptr, FALSE) == EMPTY_HAND_NONE) ? "肪󂢂" : "");
#else
			if(message) msg_print("You began to control riding pet with one hand.");
#endif
		}

		creature_ptr->old_riding_two_handed = creature_ptr->riding_two_handed;
	}

	if (((creature_ptr->cls_idx == CLASS_MONK) || (creature_ptr->cls_idx == CLASS_FORCETRAINER) || (creature_ptr->cls_idx == CLASS_NINJA)) && (monk_armour_aux != monk_notify_aux))
	{
		if (heavy_armor(creature_ptr))
		{
#ifdef JP
            if(message) msg_print("dăoXȂB");
#else
			if(message) msg_print("The weight of your armor disrupts your balance.");
#endif
		}
		else
#ifdef JP
            if(message) msg_print("oXƂ悤ɂȂB");
#else
			if(message) msg_print("You regain your balance.");
#endif

		monk_notify_aux = monk_armour_aux;
	}

	for (i = 0; i < INVEN_TOTAL; i++)
	{
		if ((creature_ptr->inventory[i].tval == TV_NATURE_BOOK) && (creature_ptr->inventory[i].sval == 2)) have_sw = TRUE;
		if ((creature_ptr->inventory[i].tval == TV_CRAFT_BOOK) && (creature_ptr->inventory[i].sval == 2)) have_kabe = TRUE;
	}

	if(is_in_this_floor(creature_ptr))
	{
		for (this_object_idx = floor_ptr->cave[creature_ptr->fy][creature_ptr->fx].object_idx; this_object_idx; this_object_idx = next_object_idx)
		{
			object_type *o_ptr;

			/* Acquire object */
			o_ptr = &object_list[this_object_idx];

			/* Acquire next object */
			next_object_idx = o_ptr->next_object_idx;

			if ((o_ptr->tval == TV_NATURE_BOOK) && (o_ptr->sval == 2)) have_sw = TRUE;
			if ((o_ptr->tval == TV_CRAFT_BOOK) && (o_ptr->sval == 2)) have_kabe = TRUE;
		}
	}

	if (creature_ptr->pass_wall && !creature_ptr->kill_wall) creature_ptr->no_flowed = TRUE;
	if (have_sw && ((creature_ptr->realm1 == REALM_NATURE) || (creature_ptr->realm2 == REALM_NATURE) || (creature_ptr->cls_idx == CLASS_SORCERER)))
	{
		magic_type *s_ptr = &m_info[creature_ptr->cls_idx].info[REALM_NATURE-1][SPELL_SW];
		if (creature_ptr->lev >= s_ptr->slevel) creature_ptr->no_flowed = TRUE;
	}

	if (have_kabe && ((creature_ptr->realm1 == REALM_CRAFT) || (creature_ptr->realm2 == REALM_CRAFT) || (creature_ptr->cls_idx == CLASS_SORCERER)))
	{
		magic_type *s_ptr = &m_info[creature_ptr->cls_idx].info[REALM_CRAFT-1][SPELL_KABE];
		if (creature_ptr->lev >= s_ptr->slevel) creature_ptr->no_flowed = TRUE;
	}

	// TODO: Evasion was adjusted by speed and size.
	// creature_ptr->skill_eva += (creature_ptr->speed * creature_ptr->size / 5);


}



/*
 * Handle "cr_ptr->creature_update"
 */
void notice_stuff(creature_type *cr_ptr)
{
	/* Notice stuff */
	if (!cr_ptr->creature_update) return;

	/* Actually do auto-destroy */
	if (cr_ptr->creature_update & (CRN_AUTODESTROY))
	{
		cr_ptr->creature_update &= ~(CRN_AUTODESTROY);
		autopick_delayed_alter(cr_ptr);
	}

	/* Combine the pack */
	if (cr_ptr->creature_update & (CRU_COMBINE))
	{
		cr_ptr->creature_update &= ~(CRU_COMBINE);
		combine_pack(cr_ptr);
	}

	/* Reorder the pack */
	if (cr_ptr->creature_update & (CRU_REORDER))
	{
		cr_ptr->creature_update &= ~(CRU_REORDER);
		reorder_pack(cr_ptr);
	}
}

// Handle "update"
void update_creature(creature_type *creature_ptr, bool message)
{
	floor_type *floor_ptr = get_floor_ptr(creature_ptr);

	// Update stuff
	if (!creature_ptr->creature_update) return;

	calculate_upkeep_servant(creature_ptr);
	set_creature_equip_slot_num(creature_ptr);

	if (creature_ptr->creature_update & (CRU_BONUS))
	{
		creature_ptr->creature_update &= ~(CRU_BONUS);
		calc_bonuses(creature_ptr, message);
	}

	if (creature_ptr->creature_update & (CRU_TORCH))
	{
		update &= ~(CRU_TORCH);
		calc_lite(creature_ptr);
	}

	if (creature_ptr->creature_update & (CRU_HP))
	{
		creature_ptr->creature_update &= ~(CRU_HP);
		calc_hitpoints(creature_ptr, message);
	}

	if (creature_ptr->creature_update & (CRU_MANA))
	{
		creature_ptr->creature_update &= ~(CRU_MANA);
		calc_mana(creature_ptr, message);
	}

	if (creature_ptr->creature_update & (CRU_SPELLS))
	{
		creature_ptr->creature_update &= ~(CRU_SPELLS);
		calc_spells(creature_ptr, message);
	}


	/* Character is not ready yet, no screen updates */
	if (!character_generated) return;

	if (update & (PU_UN_LITE))
	{
		update &= ~(PU_UN_LITE);
		forget_lite(floor_ptr);
	}

	if (update & (PU_UN_VIEW))
	{
		update &= ~(PU_UN_VIEW);
		forget_view(floor_ptr);
	}

	if (update & (PU_VIEW))
	{
		update &= ~(PU_VIEW);
		update_view(creature_ptr);
	}

	if (update & (PU_LITE))
	{
		update &= ~(PU_LITE);
		update_lite(creature_ptr);
	}


	if (update & (PU_FLOW))
	{
		update &= ~(PU_FLOW);
		update_flow(creature_ptr);
	}

	if (update & (PU_DISTANCE))
	{
		update &= ~(PU_DISTANCE);

		/* Still need to call update_creatures(FALSE) after update_creature_lite() */ 
		/* update &= ~(PU_MONSTERS); */

		update_creatures(TRUE);
	}

	if (update & (PU_MON_LITE))
	{
		update &= ~(PU_MON_LITE);
		//update_creature_lite(floor_ptr);
	}

	/*
	 * Mega-Hack -- Delayed visual update
	 * Only used if update_view(), update_lite() or update_creature_lite() was called
	 */
	if (update & (PU_DELAY_VIS))
	{
		update &= ~(PU_DELAY_VIS);
		delayed_visual_update(floor_ptr);
	}

	if (update & (PU_MONSTERS))
	{
		update &= ~(PU_MONSTERS);
		update_creatures(FALSE);
	}
}


/*
 * Handle "play_redraw"
 */
void redraw_stuff()
{
	/* Redraw stuff */
	if (!play_redraw) return;

	/* Character is not ready yet, no screen updates */
	if (!character_generated) return;

	/* Hack -- clear the screen */
	if (play_redraw & (PR_WIPE))
	{
		play_redraw &= ~(PR_WIPE);
		msg_print(NULL);
		Term_clear();
	}

	if (play_redraw & (PR_MAP))
	{
		play_redraw &= ~(PR_MAP);
		prt_map(player_ptr);
	}

	prt_frame_basic(player_ptr);

	if (play_redraw & (PR_BASIC))
	{
		play_redraw &= ~(PR_BASIC);
		play_redraw &= ~(PR_MISC | PR_TITLE | PR_STATS);
		play_redraw &= ~(PR_LEV | PR_EXP | PR_GOLD);
		play_redraw &= ~(PR_ARMOR | PR_HP | PR_MANA);
		play_redraw &= ~(PR_DEPTH | PR_HEALTH | PR_UHEALTH);
		prt_time(player_ptr);
		prt_dungeon();
	}

	if (play_redraw & (PR_MISC))
	{
		play_redraw &= ~(PR_MISC);
	}

	if (play_redraw & (PR_LEV))
	{
		play_redraw &= ~(PR_LEV);
		prt_level(player_ptr);
	}

	if (play_redraw & (PR_EXP))
	{
		play_redraw &= ~(PR_EXP);
		prt_exp(player_ptr);
	}

	if (play_redraw & (PR_STATS))
	{
		play_redraw &= ~(PR_STATS);
		prt_stat(player_ptr, STAT_STR);
		prt_stat(player_ptr, STAT_INT);
		prt_stat(player_ptr, STAT_WIS);
		prt_stat(player_ptr, STAT_DEX);
		prt_stat(player_ptr, STAT_CON);
		prt_stat(player_ptr, STAT_CHA);
	}

	if (play_redraw & (PR_STATUS))
	{
		play_redraw &= ~(PR_STATUS);
		prt_status(player_ptr);
	}

	if (play_redraw & (PR_ARMOR))
	{
		play_redraw &= ~(PR_ARMOR);
		prt_ac(player_ptr);
	}

	if (play_redraw & (PR_HP))
	{
		play_redraw &= ~(PR_HP);
		prt_hp(player_ptr);
	}

	if (play_redraw & (PR_MANA))
	{
		play_redraw &= ~(PR_MANA);
		prt_sp(player_ptr);
	}

	if (play_redraw & (PR_GOLD))
	{
		play_redraw &= ~(PR_GOLD);
		prt_gold(player_ptr);
	}

	if (play_redraw & (PR_DEPTH))
	{
		play_redraw &= ~(PR_DEPTH);
		prt_depth(player_ptr);
	}

	if (play_redraw & (PR_HEALTH))
	{
		play_redraw &= ~(PR_HEALTH);
		health_redraw(player_ptr, FALSE);
	}

	if (play_redraw & (PR_UHEALTH))
	{
		play_redraw &= ~(PR_UHEALTH);
		health_redraw(player_ptr, TRUE);
	}


	if (play_redraw & (PR_EXTRA))
	{
		play_redraw &= ~(PR_EXTRA);
		play_redraw &= ~(PR_CUT | PR_STUN);
		play_redraw &= ~(PR_HUNGER);
		play_redraw &= ~(PR_STATE | PR_SPEED | PR_STUDY | PR_IMITATION | PR_STATUS);
		prt_frame_extra(player_ptr);
	}

	if (play_redraw & (PR_CUT))
	{
		play_redraw &= ~(PR_CUT);
		prt_cut(player_ptr);
	}

	if (play_redraw & (PR_STUN))
	{
		play_redraw &= ~(PR_STUN);
		prt_stun(player_ptr);
	}

	if (play_redraw & (PR_HUNGER))
	{
		play_redraw &= ~(PR_HUNGER);
		prt_hunger(player_ptr);
	}

	if (play_redraw & (PR_STATE))
	{
		play_redraw &= ~(PR_STATE);
		prt_state(player_ptr);
	}

	if (play_redraw & (PR_SPEED))
	{
		play_redraw &= ~(PR_SPEED);
		prt_speed(player_ptr);
	}

	if (player_ptr->cls_idx == CLASS_IMITATOR)
	{
		if (play_redraw & (PR_IMITATION))
		{
			play_redraw &= ~(PR_IMITATION);
			prt_imitation(player_ptr);
		}
	}
	else if (play_redraw & (PR_STUDY))
	{
		play_redraw &= ~(PR_STUDY);
		prt_study(player_ptr);
	}
}


/*
 * Handle "play_window"
 */
void window_stuff(void)
{
	int j;

	u32b mask = 0L;


	/* Nothing to do */
	if (!play_window) return;

	/* Scan windows */
	for (j = 0; j < 8; j++)
	{
		/* Save usable flags */
		if (angband_term[j]) mask |= window_flag[j];
	}

	/* Apply usable flags */
	play_window &= mask;

	/* Nothing to do */
	if (!play_window) return;


	/* Display inventory */
	if (play_window & (PW_INVEN))
	{
		play_window &= ~(PW_INVEN);
		fix_inven(player_ptr);
	}

	/* Display equipment */
	if (play_window & (PW_EQUIP))
	{
		play_window &= ~(PW_EQUIP);
		fix_equip(player_ptr);
	}

	/* Display spell list */
	if (play_window & (PW_SPELL))
	{
		play_window &= ~(PW_SPELL);
		fix_spell(player_ptr);
	}

	/* Display player */
	if (play_window & (PW_PLAYER))
	{
		play_window &= ~(PW_PLAYER);
		fix_player(player_ptr);
	}

	/* Display overhead view */
	if (play_window & (PW_MESSAGE))
	{
		play_window &= ~(PW_MESSAGE);
		fix_message();
	}

	/* Display overhead view */
	if (play_window & (PW_OVERHEAD))
	{
		play_window &= ~(PW_OVERHEAD);
		fix_overhead(player_ptr);
	}

	/* Display overhead view */
	if (play_window & (PW_DUNGEON))
	{
		play_window &= ~(PW_DUNGEON);
		fix_dungeon(player_ptr);
	}

	/* Display monster recall */
	if (play_window & (PW_MONSTER))
	{
		play_window &= ~(PW_MONSTER);
		fix_monster(player_ptr);
	}

	/* Display object recall */
	if (play_window & (PW_OBJECT))
	{
		play_window &= ~(PW_OBJECT);
		fix_object(player_ptr);
	}
}


/*
 * Handle "update" and "play_redraw" and "play_window"
 */
void handle_stuff(void)
{
	/* Update stuff */
	if (update) update_creature(player_ptr, TRUE);

	/* Redraw stuff */
	if (play_redraw) redraw_stuff();

	/* Window stuff */
	if (play_window) window_stuff();
}


s16b empty_hands(creature_type *cr_ptr, bool riding_control)
{
	s16b status = 0x0000, least_zero_bit;
	int i;

	for(i = 0; i < MAX_HANDS; i++)
		if (!get_equipped_slot_ptr(cr_ptr, INVEN_SLOT_HAND, i)->k_idx) status |= 0x0001 << i;

	if (riding_control && status && cr_ptr->riding && !(cr_ptr->pet_extra_flags & PF_RYOUTE))
	{
		least_zero_bit = ~status & -(~status);
		status &= ~least_zero_bit;
	}

	return status;
}


bool heavy_armor(creature_type *cr_ptr)
{
	u16b monk_arm_wgt = 0;

	if ((cr_ptr->cls_idx != CLASS_MONK) && (cr_ptr->cls_idx != CLASS_FORCETRAINER) && (cr_ptr->cls_idx != CLASS_NINJA)) return FALSE;

	/* Weight the armor */
	//TODO
	//if(cr_ptr->inventory[].tval > TV_SWORD) monk_arm_wgt += cr_ptr->inventory[].weight;
	//if(cr_ptr->inventory[].tval > TV_SWORD) monk_arm_wgt += cr_ptr->inventory[].weight;
	//monk_arm_wgt += cr_ptr->inventory[].weight;
	//monk_arm_wgt += cr_ptr->inventory[].weight;
	//monk_arm_wgt += cr_ptr->inventory[].weight;
	//monk_arm_wgt += cr_ptr->inventory[].weight;
	//monk_arm_wgt += cr_ptr->inventory[].weight;

	return (monk_arm_wgt > (100 + (cr_ptr->lev * 4)) * cr_ptr->size / 10 * cr_ptr->size / 10);
}


/* Calculate alignment */
s16b calc_rank(s32b align){
	s32b t = 0, n; 
	if(align < 0) n = -align;
	else          n = align;
	while(t * t < n / 10) t++;
	if(align < 0) return (s16b)-t;
	else          return (s16b)t;
}

/* get subrace */
bool get_subrace(creature_type *cr_ptr, int n)
{
	if(cr_ptr->sub_race[n / 32] & (0x01 << (n % 32))) return TRUE;
	return FALSE;
}

/* set subrace */
void set_subrace(creature_type *cr_ptr, int n, bool b)
{
	unsigned int a = n;
	if(b) cr_ptr->sub_race[a / 32] = cr_ptr->sub_race[a / 32] | (0x01 << (a % 32));
	else cr_ptr->sub_race[a / 32] = cr_ptr->sub_race[a / 32] & !(0x01 << (a % 32));
}



/* Return Race Name */
cptr desc_race_name(creature_type *cr_ptr){
	char name[80];
	char subname[80];
	race_type *rcr_ptr = &race_info[cr_ptr->race_idx1];
	name[0] = '\0';
	subname[0] = '\0';

	if(cr_ptr->race_idx1 == INDEX_NONE) return format("");
	else if(cr_ptr->race_idx2 == INDEX_NONE) return format("%s", race_info[cr_ptr->race_idx1].title);

	if(has_cf_creature(cr_ptr, CF_GOLEM))
	{
#if JP
		strcat(subname, "^S[");
#else
		strcat(subname, "-type Golem");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_ANDROID))
	{
#if JP
		strcat(subname, "^AhCh");
#else
		strcat(subname, "-type Android");
#endif
	}


	if(has_cf_creature(cr_ptr, CF_VAMPIRE))
	{
#if JP
		strcat(subname, "̋zS");
#else
		strcat(subname, " Vampire");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_ZOMBIE))
	{
#if JP
		strcat(subname, "̃]r");
#else
		strcat(subname, " Zombie");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_SKELETON))
	{
#if JP
		strcat(subname, "̃XPg");
#else
		strcat(subname, " Skeleton");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_LICH))
	{
#if JP
		strcat(subname, "̃b`");
#else
		strcat(subname, " Lich");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_NAZGUL))
	{
#if JP
		strcat(subname, "̃iYO");
#else
		strcat(subname, " Nazgul");
#endif
	}

	if(has_cf_creature(cr_ptr, CF_PUELLA_MAGI))
	{
#if JP
		strcat(subname, "̖@");
#else
		strcat(subname, " Puella Magi");
#endif
	}


	if(subname[0])
	{
#if JP
		strcat(name, "");
#else
		strcat(name, "Ex-");
#endif
	}

	if(IS_PURE(cr_ptr))
	{
		if(!name[0])
		{
#if JP
			strcat(name, "");
#else
			strcat(name, "Pure-blooded ");
			strcat(name, sex_info[cr_ptr->sex].title);
			strcat(name, " ");
#endif
		}
		strcat(name, race_info[cr_ptr->race_idx1].title);
	}
	else
	{
		int i;
		for(i = 0; i < MAX_HALF_RACE_DESCRIPTION; i++)
		{
			if((cr_ptr->race_idx1 == half_race_desc[i].race1 && cr_ptr->race_idx2 == half_race_desc[i].race2) ||
			   (cr_ptr->race_idx2 == half_race_desc[i].race1 && cr_ptr->race_idx1 == half_race_desc[i].race2))
			{
				strcat(name, half_race_desc[i].title);
				break;
			}
		}

		if(!name[0])
		{
#if !JP
			strcat(name, sex_info[cr_ptr->sex].title);
			strcat(name, "Mixed ");
#endif
			strcat(name, race_info[cr_ptr->race_idx1].title);
#if JP
			strcat(name, "");
#else
			strcat(name, "and ");
#endif
			strcat(name, race_info[cr_ptr->race_idx2].title);
#if JP
			strcat(name, "̍");
#endif
		}
	}


	strcat(name, subname);

#if JP
	if(cr_ptr->sex != SEX_UNDEFINED)
	{
		strcat(name, "(");
		strcat(name, sex_info[cr_ptr->sex].title);
		strcat(name, ")");
	}
#endif

	return format("%s", name);
}


cptr get_class_desc(creature_type *cr_ptr){
	char name[80];
	name[0] = '\0';

	strcat(name, classkill_info[cr_ptr->cls_idx].title);

	if(has_cf_creature(cr_ptr, CF_PUELLA_MAGI))
	{
#if JP
		strcat(name, "/@");
#else
		strcat(name, "/Puella-Magi");
#endif
	}

	if(cr_ptr->realm1 != REALM_NONE && cr_ptr->realm1 < MAX_REALM)
	{
		strcat(name, " (");
		strcat(name, realm_names[cr_ptr->realm1]);
		if(cr_ptr->realm2 != REALM_NONE && cr_ptr->realm2 < MAX_REALM)
		{
			strcat(name, ", ");
			strcat(name, realm_names[cr_ptr->realm2]);
		}
		strcat(name, ")");
	}


	return format("%s", name);

}

