#ifndef ECHOES_DATA_H
#define ECHOES_DATA_H

#include <QList>

namespace stand
{
namespace model
{
class ControlItem;
}
namespace echoes
{

/*!
 *  @brief ECHOES で使用するデータを格納するクラスです．
 */
class Data
{
public:
    //! @brief コンストラクタ．メンバ変数は 0 初期化されます．
    Data();
    //! @brief コピーコンストラクタ．メンバ変数はディープコピーされます．
    Data(const Data &b);
    //! @brief デストラクタ．確保したメモリを解放します．
    virtual ~Data();

    /*!
     *  @brief ファイルからの読み込み．
     *  @param[in] fp 読み込みたいファイルを開いたファイルポインタ．
     *  @return 読み込み成否．
     */
    bool read(FILE *fp);
    /*!
     *  @brief ファイルへの書き込み．
     *  @param[in] fp 書き込みたいファイルを開いたファイルポインタ．
     */
    void write(FILE *fp) const;

    /*!
     *  @brief 格納する F0, Volume, Index を与えられた長さで初期化します．
     *  @attention このメソッドは操作前に持っていた情報を削除します．
     *  @param[in] length データ長．
     */
    void setLength(int length);

    inline bool isEmpty() const
    {
        return (!_f0 || !_volume || _length <= 0);
    }

    inline int *ticks() const
    {
        return _ticks;
    }
    inline int tickAt(int index) const
    {
        return _ticks[index];
    }

    inline double *f0() const
    {
        return _f0;
    }
    inline double f0At(int index) const
    {
        return _f0[index];
    }

    inline double *volume() const
    {
        return _volume;
    }
    inline double volumeAt(int index) const
    {
        return _volume[index];
    }

    inline int length() const
    {
        return _length;
    }

    inline void setFramePeriod(double framePeriod)
    {
        _framePeriod = framePeriod;
    }
    inline double framePeriod() const
    {
        return _framePeriod;
    }

    inline void setChangeIndex(int index)
    {
        _changeIndex = index;
    }
    inline int changeIndex() const
    {
        return _changeIndex;
    }

    bool operator ==(const Data &b) const;
    bool operator !=(const Data &b) const;
    bool operator < (const Data &b) const;
    bool operator <=(const Data &b) const;
    bool operator > (const Data &b) const;
    bool operator >=(const Data &b) const;

    Data &operator =(const Data &b);

private:
    void _destroy();
    void _allValueInit();

    double *_f0;            //! @brief 基本周波数列
    double *_volume;        //! @brief 振幅包絡
    int    *_ticks;         //! @brief インデクッスにおける tick 時刻
    int _length;            //! @brief 配列長
    double _framePeriod;    //! @brief フレーム長
    int _changeIndex;       //! @brief 音符が変化する位置
};

}
}

#endif // ECHOES_DATA_H
