#include "WaveAnalysisTask.h"

#include "WaveBuffer.h"

#include "EnvelopeComputingTask.h"
#include "TimeComputingTask.h"

#include "envelope/F0Computer.h"
#include "envelope/VolumeComputer.h"

#include "TrackItem.h"
#include "ControlItem.h"
#include "SequenceModel.h"

#include <QFileInfo>
#include <float.h>

using namespace stand::model;
using namespace stand::gui::helper;
using namespace stand::utility::envelope;

WaveAnalysisTask::WaveAnalysisTask(SequenceModel *model, TrackItem *target, const QString &path, double framePeriod, QObject *parent) :
    Task(parent)
{
    _framePeriod = std::max(2.0, framePeriod);
    _model = model;
    _track = target;
    _path = path;
}

WaveAnalysisTask::~WaveAnalysisTask()
{
}

bool WaveAnalysisTask::exec()
{
    if(!_track)
    {
        qDebug("WaveAnalysisTask::exec(); // Target track is NULL.");
        return false;
    }

    stand::io::WaveBuffer wave;
    if(!wave.read(_path.toLocal8Bit().data()))
    {
        qDebug("WaveAnalysisTask::exec(%s); // failed opening file.", _path.toUtf8().data());
        return false;
    }

    stand::utility::envelope::Envelope f0(0, 0, _framePeriod);
    stand::utility::envelope::Envelope volume(0, 0, _framePeriod);

    qDebug("WaveAnalysisTask::exec();");
    qDebug(" Target Wave file; %s", _path.toUtf8().data());

    EnvelopeComputingTask f0ComputingTask(&f0, &wave, new F0Computer);
    EnvelopeComputingTask volumeComputingTask(&volume, &wave, new VolumeComputer);

    qDebug(" Begin f0 computing.");
    f0ComputingTask.exec();

    qDebug(" Begin volume computing.");
    volumeComputingTask.exec();

    qDebug(" Begin time computing.");
    TimeComputingTask timeComputingTask(_model->tempo()->contour(), _framePeriod, f0.length());
    timeComputingTask.exec();

    TreeItem *pitchItem = _track->createControl("Pitch", &f0, timeComputingTask.data(), ControlItem::TypePitch);
    TreeItem *volumeItem = _track->createControl("Volume", &volume, timeComputingTask.data(), ControlItem::TypeNormal);

    double maxVal = DBL_MIN, minVal = DBL_MAX;
    for(int i = 0; i < volume.tLen(); i++)
    {
        double at = volume.at(i);
        if(maxVal < at)
        {
            maxVal = at;
        }
        if(minVal > at)
        {
            minVal = at;
        }
    }

    volumeItem->setData(ControlItem::MaximumValue, maxVal, Qt::EditRole);
    volumeItem->setData(ControlItem::MinimumValue, maxVal - 24.0, Qt::EditRole); // 表示範囲は最大値から -24dB まで…？

    qDebug(" WaveAnalysisTask finished.");

    return true;
}
