#include "ControlItem.h"

#include "envelope/Envelope.h"
#include "ControlCurve.h"

#include <QVector>

using namespace stand::model;
using stand::utility::ControlPoint;

ControlItem::ControlItem(const QString &name, double defaultValue, TreeItem *parent) :
    TreeItem(name, parent)
{
    _length = 0;
    setData(INT_MIN, defaultValue, Qt::EditRole);
}

ControlItem::ControlItem(const QString &name, const stand::utility::envelope::Envelope *e, const QVector<int> &ticks, TreeItem *parent) :
    TreeItem(name, parent)
{
    setEnvelope(e, ticks);
}

ControlItem::~ControlItem()
{
}

void ControlItem::replace(const QList<ControlPoint> &points, int first, int last)
{
    if(points.empty())
    {
        return;
    }
    removeColumn(std::min(points.first().tick, first), std::max(points.last().tick, last));

    // iterator を使って insert した場合，使用した iterator は無効になるんだってさ！！
    int i;
    for(i = 0; i < _contour.size(); i++)
    {
        if(points.first() < _contour[i])
        {
            break;
        }
    }
    for(int j = points.size() - 1; j >= 0; j--)
    {
        _contour.insert(i, points[j]);
    }
}

void ControlItem::setControl(const stand::sequence::ControlCurve *control)
{
    if(!control || control->data().empty())
    {
        return;
    }
    _contour[0].value = control->at(0);
    bool isDefaultSet = false;
    double defaultValue;

    for(int i = 0; i < control->data().size(); i++)
    {
        ControlPoint p = {control->data()[i].tickBegin, control->data()[i].val};
        setData(p.tick, p.value, 0);
        if(p.tick == 0)
        {
            isDefaultSet = true;
            defaultValue = p.value;
        }
    }

    if(isDefaultSet)
    {
        setData(INT_MIN, defaultValue, 0);
    }
    qSort(_contour);
}

void ControlItem::setEnvelope(const stand::utility::envelope::Envelope *e, const QVector<int> &ticks)
{
    if(!e || e->tLen() == 0)
    {
        return;
    }
    _contour.clear();
    for(int i = 0; i < e->tLen(); i++)
    {
        ControlPoint p = {i, e->at(i)};
        _contour.append(p);
    }
    for(int i = 0; i < _contour.size() && i < ticks.size(); i++)
    {
        _contour[i].tick = ticks[i];
    }
    _length = e->tLen();
}

int ControlItem::columnCount() const
{
    return _length;
}

QVariant ControlItem::data(int column, int role) const
{
    if(role != Qt::DisplayRole && role != Qt::EditRole)
    {
        return QVariant();
    }
    if(_contour.empty())
    {
        return QVariant();
    }

    ControlPoint p = {column, 0};
    QList<ControlPoint>::const_iterator it = qLowerBound(_contour.begin(), _contour.end(), p);
    if(it == _contour.end())
    {
        return _contour.last().value;
    }
    if((*it).tick == column || it == _contour.begin())
    {
        return (*it).value;
    }
    it--;
    return (*it).value;
}

void ControlItem::setData(int column, const QVariant &d, int role)
{
    if(role != Qt::DisplayRole && role != Qt::EditRole)
    {
        return;
    }
    ControlPoint p = {column, d.toDouble()};
    QList<ControlPoint>::iterator it = qLowerBound(_contour.begin(), _contour.end(), p);

    // 前と同じなら何もする必要がない．
    if(it != _contour.begin())
    {
        QList<ControlPoint>::iterator prev = it;
        prev--;
        if((*prev).value == p.value)
        {
            return;
        }
    }

    if(it == _contour.end())
    {
        _contour.insert(it, p);
        return;
    }

    // 後ろと同じなら後ろを前に持ってくる
    if((*it).tick == column)
    {
        QList<ControlPoint>::iterator next = it;
        next++;
        if(next != _contour.end() && (*next).value == p.value)
        {
            (*next).tick = p.tick;
            return;
        }
        (*it).value = p.value;
        return;
    }

    // そうもうまくないので挿入
    _contour.insert(it, p);
}

QList<ControlPoint>::iterator ControlItem::removeColumn(int first, int last)
{
    ControlPoint f = {first, 0}, l = {last, 0};
    QList<ControlPoint>::iterator begin = qLowerBound(_contour.begin(), _contour.end(), f);
    QList<ControlPoint>::iterator end = qUpperBound(_contour.begin(), _contour.end(), l);
    _contour.erase(begin, end);
    return begin;
}

QList<ControlPoint> ControlItem::contour(int first, int last) const
{
    QList<ControlPoint> ret;

    ControlPoint f = {first, 0}, l = {last, 0};
    QList<ControlPoint>::const_iterator begin = qLowerBound(_contour.begin(), _contour.end(), f);
    QList<ControlPoint>::const_iterator end = qUpperBound(_contour.begin(), _contour.end(), l);
    QList<ControlPoint>::const_iterator it;

    for(it = begin; it != end; it++)
    {
        ret.push_back((*it));
    }

    return ret;
}

void ControlItem::setUpDomElement(QDomDocument &doc, QDomElement &element)
{
    element.setTagName("control");

    if(!_contour.empty())
    {
        element.setAttribute("default_value", _contour[0].value);
    }

    for(int i = 1; i < _contour.size(); i++)
    {
        QDomElement e = doc.createElement("point");
        e.setAttribute("tick", _contour[i].tick);
        e.setAttribute("value", _contour[i].value);
        element.appendChild(e);
    }
}

