#ifndef CONTROLITEM_H
#define CONTROLITEM_H

#include "TreeItem.h"
#include "ControlPoint.h"

namespace stand
{
namespace utility
{
namespace envelope
{
class Envelope;
}
}
namespace sequence
{
class ControlCurve;
}
namespace model
{

class ControlItem : public TreeItem
{
    Q_OBJECT
public:

    explicit ControlItem(const QString &name, double defaultValue, TreeItem *parent = 0);
    explicit ControlItem(const QString &name, const stand::utility::envelope::Envelope *e, const QVector<int> &ticks, TreeItem *parent = 0);
    virtual ~ControlItem();

    virtual int columnCount() const;
    virtual QVariant data(int column, int role) const;
    virtual void setData(int column, const QVariant &d, int role);

    /*!
     *  @brief エンベロープと tick 時刻のインデックスから ControlItem のデータを作成します．
     *         このメソッドは以前持っていたデータを破棄します．
     *  @param[in] e 格納すべきエンベロープ．
     *  @param[in] ticks tick 時刻のインデックス．
     */
    void setEnvelope(const utility::envelope::Envelope *e, const QVector<int> &ticks);

    /*!
     *  @brief stand::sequence からの変換用メソッド．
     *  @attention 削除予定
     */
    void setControl(const stand::sequence::ControlCurve *control);

    /*!
     *  @brief 指定した範囲のデータを削除します．[first, last)の範囲で削除されます．
     *  @param[in] first 削除開始 tick
     *  @param[inl last 削除終了 tick
     *  @return 削除が始まった位置を示すイテレータ．
     */
    QList<stand::utility::ControlPoint>::iterator removeColumn(int first, int last);

    /*!
     *  @brief 指定した範囲のデータを points で指定されたデータに置き換えます．
     *         points はソートされている必要があります．
     *  @param[in] points 置き換えるデータ．
     *  @param[in] first 置き換え開始 tick
     *  @param[in] last 置き換え終了 tick
     */
    void replace(const QList<stand::utility::ControlPoint> &points, int first, int last);

    /*!
     *  @brief 指定された範囲のデータを取り出します．このメソッドはディープコピーを行います．
     *  @param[in] first 取り出し開始 tick
     *  @param[in] last 取り出し終了 tick
     *  @return 指定位置のパラメータのコピー．
     */
    QList<stand::utility::ControlPoint> contour(int first, int last) const;

    /*!
     *  @brief 全範囲のデータを取り出します．このメソッドはシャローコピーを行います．
     *  @return インスタンスの持つコントロールパラメータ．
     */
    inline const QList<stand::utility::ControlPoint> &contour() const
    {
        return _contour;
    }

    inline int length() const
    {
        if(_contour.empty())
        {
            return 0;
        }
        return _contour.last().tick;
    }

    inline void appendPoint(const stand::utility::ControlPoint &p)
    {
        _contour.push_back(p);
    }

signals:

protected:
    virtual void setUpDomElement(QDomDocument &doc, QDomElement &element);

private:
    QList<stand::utility::ControlPoint> _contour;

    int _length;

public:
    enum Property
    {
        ViewType,
        TimeType,
        FramePeriod,
        MinimumValue,
        MaximumValue
    };

    enum Time
    {
        TypeTick = 0,
        TypeFrame
    };

    enum View
    {
        TypeNormal = 0,
        TypePitch
    };
};

}
}

#endif // CONTROLITEM_H
