#include "AbstractPianoroll.h"

#include <QScrollBar>
#include <QScrollArea>
#include <QEvent>
#include <QPaintEvent>
#include <QPainter>
#include "../configure.h"

#include "ViewSettings.h"
#include "Utility.h"

using namespace stand::view;
using namespace stand::utility;

AbstractPianoroll::AbstractPianoroll(QWidget *parent) :
    QAbstractItemView(parent)
{
    verticalScrollBar()->setSizePolicy(QSizePolicy::Maximum, QSizePolicy::Preferred);
    horizontalScrollBar()->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Maximum);
    verticalScrollBar()->setAutoFillBackground(true);
    horizontalScrollBar()->setAutoFillBackground(true);

    _h = new QSlider(Qt::Horizontal, this);
    _v = new QSlider(Qt::Vertical, this);
    _h->setMaximumWidth(64);
    _v->setMaximumHeight(64);
    _h->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
    _v->setSizePolicy(QSizePolicy::Preferred, QSizePolicy::Preferred);
    _h->setAutoFillBackground(true);
    _v->setAutoFillBackground(true);
    _h->setRange(32, 256);
    _h->setValue(128);
    _v->setRange(8, 24);
    _v->setValue(16);
    verticalScrollBar()->setSingleStep(20);
    horizontalScrollBar()->setSingleStep(20);

    addScrollBarWidget(_h, Qt::AlignRight);
    addScrollBarWidget(_v, Qt::AlignBottom);

    setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOn);
    setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOn);

    _length = 0;
    _settings = NULL;
    _widget = NULL;
    _settingsIsDeletable = false;

    setSettings(new ViewSettings(1, 16, 64, this));
    _settingsIsDeletable = true;

    _setWidget(new QWidget(viewport()));
}

AbstractPianoroll::~AbstractPianoroll()
{
}

void AbstractPianoroll::releaseSettings()
{
    disconnect(this, SLOT(releaseSettings()));
    disconnect(this, SLOT(setMeasure));
    _settings = NULL;
    _settings = new ViewSettings(1, 16, 64, this);
    _settingsIsDeletable = true;
}

void AbstractPianoroll::setSettings(ViewSettings *s)
{
    if(_settings)
    {
        disconnect(_settings, SLOT(setBeatWidth(int)));
        disconnect(_settings, SLOT(setNoteHeight(int)));
        disconnect(_settings, SLOT(widgetResized(int,int)));
        if(_settingsIsDeletable)
        {
            delete _settings;
        }
    }

    _settingsIsDeletable = false;
    _settings = s;
    if(!s)
    {
        _settings = new ViewSettings(1, 16, 64, this);
        _settingsIsDeletable = true;
    }
    else
    {
        connect(s, SIGNAL(AboutToDelete()), this, SLOT(releaseSettings()));
    }
    connect(_h, SIGNAL(valueChanged(int)), _settings, SLOT(setBeatWidth(int)));
    connect(_v, SIGNAL(valueChanged(int)), _settings, SLOT(setNoteHeight(int)));
    connect(_settings, SIGNAL(StateChanged()), this, SLOT(setMeasure()));
    connect(this, SIGNAL(widgetResized(int,int)), _settings, SLOT(widgetResized(int,int)));
    setMeasure();
}

void AbstractPianoroll::_setWidget(QWidget *w)
{
    if(!w)
    {
        return;
    }
    delete _widget;
    horizontalScrollBar()->setValue(0);
    verticalScrollBar()->setValue(0);
    if(w->parentWidget() != viewport())
    {
        w->setParent(viewport());
    }
    _widget = w;
    _widget->setAutoFillBackground(true);
    _widget->installEventFilter(this);
    _widget->setParent(viewport());
    _updateScrollBars();
    _widget->show();
}

void AbstractPianoroll::scrollContentsBy(int dx, int dy)
{
    viewport()->scroll(dx, dy);
}

void AbstractPianoroll::_updateWidgetPosition()
{
    Qt::LayoutDirection dir = Qt::LeftToRight;
    QRect scrolled = QStyle::visualRect(dir, viewport()->rect(), QRect(QPoint(-horizontalScrollBar()->value(), -verticalScrollBar()->value()), _widget->size()));
    QRect aligned = QStyle::alignedRect(dir, Qt::AlignCenter, _widget->size(), viewport()->rect());
    _widget->move(_widget->width() < viewport()->width() ? aligned.x() : scrolled.x(),
                 _widget->height() < viewport()->height() ? aligned.y() : scrolled.y());
}

void AbstractPianoroll::_updateScrollBars()
{
    if(!_widget)
    {
        return;
    }
    QSize p = viewport()->size();

    QSize min = QSize(std::max(length(), viewport()->width() + _settings->beatWidth() * 12), _settings->noteHeight() * 128);
    QSize max = QSize(INT_MAX, _settings->noteHeight() * 128);

    QSize rs = p.expandedTo(min).boundedTo(max);
    _widget->resize(rs);
    emit widgetResized(min.width(), min.height());

    QScrollBar *hbar = horizontalScrollBar();
    QScrollBar *vbar = verticalScrollBar();

    hbar->setRange(0, length() - p.width());
    hbar->setPageStep(p.width());
    vbar->setRange(0, _settings->noteHeight() * 128 - p.height());
    vbar->setPageStep(p.height());
    _updateWidgetPosition();
}

bool AbstractPianoroll::eventFilter(QObject *o, QEvent *e)
{
    if(o == _widget && e->type() == QEvent::Resize)
    {
        _updateScrollBars();
        _widget->move(-horizontalScrollBar()->value(), -verticalScrollBar()->value());
    }

    if(o == _widget && e->type() == QEvent::Paint)
    {
        QPainter p(_widget);
        _drawStripe(&p, (QPaintEvent *)e);
        _drawBeat(&p, (QPaintEvent *)e);

        // PaintEvent なら握りつぶす
        return true;
    }

    return QAbstractItemView::eventFilter(o, e);
}

void AbstractPianoroll::resizeEvent(QResizeEvent *event)
{
    QAbstractItemView::resizeEvent(event);
    _updateScrollBars();
}


void AbstractPianoroll::setMeasure()
{
    if(!_widget)
    {
        return;
    }
    _updateScrollBars();
    _widget->update();
}

void AbstractPianoroll::_drawStripe(QPainter *p, QPaintEvent *e)
{
    int noteHeight = _settings->noteHeight();
    int beginNote = e->rect().y() / noteHeight;
    int endNote = e->rect().bottom() / noteHeight + 1;

    for(int i = beginNote; i < endNote; i++)
    {
        QColor c;
        int ca = 255;
        if(abs(i - 56) > 28)
        {
            ca = ca * 5 / 6;
        }
        if(abs(i - 52) > 45)
        {
            ca = ca * 3 / 4;
        }
        if(IsBlackKey(127 - i))
        {
            ca = ca * 4 / 5;
        }
        c = QColor(ca, ca, ca);
        int y = i * noteHeight;
        p->fillRect(e->rect().x(), y, e->rect().width(), noteHeight, c);
        p->setPen(QColor(128, 128, 128));
        if((127 - i) % 12 == 11)
        {
            p->setPen(QColor(64, 0 ,0));
        }
        p->drawLine(e->rect().x(), y, e->rect().x() + e->rect().width(), y);
    }

    if(_settings->tickAt(e->rect().x()) < 0)
    {
        int w = _settings->xAt(0) - e->rect().x();
        for(int i = beginNote; i < endNote; i++)
        {
            QColor c(Qt::gray);
            if(IsBlackKey(127 - i))
            {
                c = QColor(Qt::darkGray);
            }
            int y = i * noteHeight;
            p->fillRect(e->rect().x(), y, w, noteHeight, c);
            p->setPen(QColor(128, 128, 144));
            if((127 - i) % 12 == 11)
            {
                p->setPen(Qt::black);
            }
            p->drawLine(e->rect().x(), y, e->rect().x() + e->rect().width(), y);
        }
    }
}

int AbstractPianoroll::length()
{
    return _settings->beatWidth() * 12;
}

void AbstractPianoroll::_drawBeat(QPainter *p, QPaintEvent *e)
{
    int begin = _settings->tickAt(e->rect().x());
    int end = _settings->tickAt(e->rect().right());
    int beginMeasure = _settings->measureAt(begin) - 1;
    int endMeasure = _settings->measureAt(end) + 1;

    int prev = _settings->tickByMeasure(beginMeasure);
    for(int i = beginMeasure; i < endMeasure; i++)
    {
        int pres = _settings->tickByMeasure(i + 1);
        p->setPen(QColor(64, 64, 64));
        int x = _settings->xAt(pres);
        p->drawLine(x, e->rect().top(), x, e->rect().bottom());
        p->setPen(Qt::gray);
        for(int j = prev + stand::sequence::DefaultTicksPerBeat; j < pres; j += stand::sequence::DefaultTicksPerBeat)
        {
            x = _settings->xAt(j);
            p->drawLine(x, e->rect().top(), x, e->rect().bottom());
        }
        prev = pres;
    }
    p->setPen(Qt::red);
    p->drawLine(_settings->currentBarX(), e->rect().top(), _settings->currentBarX(), e->rect().bottom());
}

void AbstractPianoroll::mousePressEvent(QMouseEvent *e)
{
    if(e->button() != Qt::LeftButton)
    {
        QAbstractItemView::mousePressEvent(e);
        return;
    }
    _barMoved = true;
    _clickedPos = QPoint(e->pos().x() - _widget->x(), e->pos().y() - _widget->y());
    QAbstractItemView::mousePressEvent(e);
}

void AbstractPianoroll::mouseMoveEvent(QMouseEvent *event)
{
    _barMoved = false;
    QAbstractItemView::mouseMoveEvent(event);
}

void AbstractPianoroll::mouseReleaseEvent(QMouseEvent *event)
{
    if(_barMoved)
    {
        int x = _clickedPos.x();
        if(_settings->isSnap())
        {
            x = (int)(0.5 + (double)x / _settings->snapWidth()) * _settings->snapWidth();
        }
        QRect r[2];
        QRegion region;
        r[0] = QRect(_settings->currentBarX(), 0, 1, _widget->height());
        _settings->setCurrentBarTick(_settings->tickAt(x));
        r[1] = QRect(_settings->currentBarX(), 0, 1, _widget->height());
        region = region + r[0] + r[1];
        _widget->update(region);
    }
    if(event)
    {
        QAbstractItemView::mouseReleaseEvent(event);
    }
}
