#include "ControlArea.h"
#include "ControlView.h"
#include "ViewSettings.h"
#include "TrackView.h"

#include "TrackItem.h"

#include <QEvent>
#include <QPainter>
#include <QPaintEvent>
#include <QResizeEvent>
#include <QScrollBar>

using namespace stand::view;

ControlArea::ControlArea(TrackView *track, ViewSettings *s, stand::model::TrackItem *tracks, QWidget *parent) :
    ViewArea(parent)
{
    _settings = new ViewSettings(1, 16, 64, this);
    _settingsIsDeletable =true;

    delete _frame;
    FrameWidget *frame = new FrameWidget(s, this);
    _frame = frame;
    _frame->setAutoFillBackground(false);
    _frame->setAttribute(Qt::WA_TransparentForMouseEvents, true);
    QPalette p(palette());
    p.setColor(QPalette::Background, QColor(64, 64, 64));
    setAutoFillBackground(true);
    setPalette(p);
    setLineWidth(0);

    setSettings(s);
    setControls(tracks->controls());
    setTrackView(track);
}

ControlArea::~ControlArea()
{
    qDeleteAll(_views);
    _views.clear();
}

void ControlArea::addControl(stand::model::TreeItem *item)
{
    if(item->data(stand::model::ControlItem::ViewType, 0) != stand::model::ControlItem::TypeNormal)
    {
        return;
    }
    ControlView *v = new ControlView(item, _settings, this);
    addWidget(v);
    v->hide();
    v->setAttribute(Qt::WA_TransparentForMouseEvents, true);
    _views.push_back(v);
}

void ControlArea::setTrackView(TrackView *track)
{
    resize(track->width(), height());
    connect(track->horizontalScrollBar(), SIGNAL(valueChanged(int)), this, SLOT(horizontalScroll(int)));
    connect(track, SIGNAL(widgetResized(int,int)), this, SLOT(trackSizeChanged(int,int)));
    _hscrW = track->horizontalScrollBar()->width();
}

void ControlArea::trackSizeChanged(int w, int)
{
    for(int i = 0; i < _widgets.size(); i++)
    {
        _widgets[i]->setFixedWidth(w + _hscrW);
    }
    _frame->setFixedWidth(w + _hscrW);
}

void ControlArea::setSettings(ViewSettings *s)
{
    if(_settingsIsDeletable)
    {
        delete _settings;
    }
    _settings = s;
    if(!s)
    {
        _settings = new ViewSettings(1, 16, 64, this);
        _settingsIsDeletable =true;
    }
    else
    {
        connect(_settings, SIGNAL(AboutToDelete()), this, SLOT(releaseSetting()));
    }
    connect(_settings, SIGNAL(StateChanged()), this, SLOT(update()));
    connect(_settings, SIGNAL(resetAll()), this, SLOT(resetAll()));
    connect(_settings, SIGNAL(trackChanged()), this, SLOT(resetAll()));
    connect(_settings, SIGNAL(controlTrackChanged()), this, SLOT(controlChanged()));

    resetAll();
}

void ControlArea::controlChanged()
{
    for(int i = 0; i < _views.size(); i++)
    {
        _views[i]->hide();
        _views[i]->setAttribute(Qt::WA_TransparentForMouseEvents);
    }
    if(0 <= _settings->controlIdPrevious() && _settings->controlIdPrevious() < _views.size())
    {
        _views[_settings->controlIdPrevious()]->setIsTop(false);
        _views[_settings->controlIdPrevious()]->show();
    }
    _frame->raise();

    _views[_settings->controlId()]->raise();
    _views[_settings->controlId()]->setIsTop(true);
    _views[_settings->controlId()]->show();
    _views[_settings->controlId()]->setAttribute(Qt::WA_TransparentForMouseEvents, false);
}

void ControlArea::resetAll()
{
    qDeleteAll(_views);
    _views.clear();
    setControls(_settings->controls());
}

void ControlArea::releaseSetting()
{
    disconnect(this, SLOT(releaseSetting()));
    qDeleteAll(_views);
    clearWidgets();
    _views.clear();
    _settings = new ViewSettings(1, 16, 64, this);
    _settingsIsDeletable =true;
}

void ControlArea::setControls(const QList<stand::model::TreeItem *> &items)
{
    clearWidgets();
    qDeleteAll(_views);
    _views.clear();

    for(int i = 0; i < items.size(); i++)
    {
        addControl(items[i]);
    }
    if(0 <= _settings->controlId() && _settings->controlId() < _views.size())
    {
        addWidget(_views[_settings->controlId()]);
        _views[_settings->controlId()]->show();
        _views[_settings->controlId()]->setAttribute(Qt::WA_TransparentForMouseEvents, false);
        _currentIndex = 0;
    }
}

void ControlArea::resizeEvent(QResizeEvent *e)
{
    for(int i = 0; i < _views.size(); i++)
    {
        _views[i]->resize(e->size().width(), _views[i]->height());
    }
    ViewArea::resizeEvent(e);
}

void FrameWidget::paintEvent(QPaintEvent *e)
{
    QPainter p(this);
    QPen pen(QColor(0, 255, 255, 64));

    p.setPen(pen);
    p.drawLine(0, height() / 2, width(), height() / 2);
    p.setPen(pen);
    p.drawLine(0, height() / 4 * 3, width(), height() / 4 * 3);
    p.drawLine(0, height() / 4, width(), height() / 4);
    if(!_settings)
    {
        return;
    }

    pen = QPen(QColor(0, 255, 255, 128));
    p.setPen(pen);

    int begin = _settings->tickAt(e->rect().x()) / 120 * 120;
    int end = _settings->tickAt(e->rect().right()) / 120 * 120;

    for(int i = begin; i <= end; i += 120)
    {
        int div = 32;
        int x = _settings->xAt(i);
        if(i % 480 == 0)
        {
            div = 16;
        }
        if(_settings->isBar(i))
        {
            div = 8;
        }
        p.drawLine(x, 0, x, height() / div);
        p.drawLine(x, height(), x, height() - height() / div);
        p.drawLine(x, height() / 2 - height() / (div * 2), x, height() / 2 + height() / (div * 2));
    }
}

void FrameWidget::setSettings(ViewSettings *s)
{
    _settings = s;
}

