#include "MeasureView.h"

#include "ViewSettings.h"
#include "MeasureItem.h"
#include "ControlItem.h"
#include "../configure.h"

#include <QPainter>
#include <QPaintEvent>
#include <QLineEdit>
#include <QEvent>
#include <QKeyEvent>

#include <QStringList>

using namespace stand::view;

MeasureView::MeasureView(QWidget *parent) :
    QWidget(parent)
{
    _settingsIsDeletable = false;
    _settings = NULL;
    _editor = NULL;
    _current.type = 0;
    setSettings(_settings);
}

MeasureView::~MeasureView()
{
    if(_settingsIsDeletable)
    {
        delete _settings;
    }
    delete _editor;
}

void MeasureView::releaseSettings()
{
    disconnect(this, SLOT(releaseSettings()));
    _settings = new ViewSettings(1, 16, 64, this);
    _settingsIsDeletable = true;
}

void MeasureView::setSettings(ViewSettings *s)
{
    disconnect(this, SLOT(releaseSettings()));
    disconnect(this, SLOT(BarMoved(int,int)));
    disconnect(this, SLOT(widthChanged(int)));
    disconnect(this, SLOT(reset()));
    if(_settingsIsDeletable)
    {
        delete _settings;
    }
    _settings = s;
    if(s)
    {
        connect(_settings, SIGNAL(AboutToDelete()), this, SLOT(releaseSettings()));
        _settingsIsDeletable = false;
    }
    else
    {
        _settings = new ViewSettings(1, 16, 64, this);
        _settingsIsDeletable = true;
    }
    connect(_settings, SIGNAL(StateChanged()), this, SLOT(reset()));
    connect(_settings, SIGNAL(widthChanged(int)), this, SLOT(widthChanged(int)));
    connect(_settings, SIGNAL(BarMoved(int,int)), this, SLOT(BarMoved(int,int)));
    reset();
}

void MeasureView::reset()
{
    setFixedHeight(_settings->noteHeight() * 3 + 1);
    widthChanged(_settings->width());
    update();
}

void MeasureView::widthChanged(int w)
{
    setMinimumWidth(std::max(w, minimumWidth()));
}

void MeasureView::BarMoved(int oldX, int newX)
{
    QRect r1(oldX, 0, 1, height()), r2(newX, 0, 1, height());
    QRegion rgn;
    rgn = rgn + r1 + r2;
    update(rgn);
}

void MeasureView::paintEvent(QPaintEvent *e)
{
    int begin = _settings->tickAt(e->rect().x()) / 120 * 120;
    int end = _settings->tickAt(e->rect().right()) / 120 * 120;

    QPainter p(this);
    QFont font(p.font());
    font.setPointSize(_settings->noteHeight() - 4);
    p.setFont(font);

    _drawLines(begin, end, &p, e);

    QPen pen(QColor(255, 0, 0, 128));
    p.setPen(pen);
    p.drawLine(_settings->currentBarX(), 0, _settings->currentBarX(), height());

    _drawMeasure(begin, end, &p, e);
    _drawTempo(begin, end, &p, e);
}

void MeasureView::_drawLines(int begin, int end, QPainter *p, QPaintEvent *e)
{
    p->fillRect(e->rect(), QColor(96, 96, 96));
    QPen pen(QColor(0, 192, 192));
    p->setPen(pen);

    for(int i = 1; i <= 3; i++)
    {
        int y = i * _settings->noteHeight() - 1;
        p->drawLine(e->rect().x(), y, e->rect().right(), y);
    }

    for(int i = begin / 120 * 120; i <= end / 120 * 120; i += 120)
    {
        int div = 4;
        int x = _settings->xAt(i);
        if(i % 480 == 0)
        {
            div = 2;
        }
        if(_settings->isBar(i))
        {
            int measureCount = _settings->measureAt(i);
            if(measureCount >= 0)
            {
                measureCount++;
            }
            p->drawText(QRect(x + 4, 1, _settings->beatWidth() - 2, _settings->noteHeight() - 2), QString::number(measureCount), Qt::AlignLeft | Qt::AlignVCenter);
            div = 1;
            p->drawLine(x, 0, x, _settings->noteHeight() - 1);
        }
        for(int i = 2; i <= 3; i++)
        {
            int y = i * _settings->noteHeight() - 1;
            p->drawLine(x, y - _settings->noteHeight() / div, x, y);
        }
    }
}

void MeasureView::_drawTempo(int begin, int end, QPainter *p, QPaintEvent *e)
{
    if(!_settings->tempo())
    {
        return;
    }
    for(int i = 0; i < _settings->tempo()->contour().size(); i++)
    {
        if(_settings->tempo()->contour()[i].tick + 480 < begin)
        {
            continue;
        }
        if(_settings->tempo()->contour()[i].tick > end)
        {
            break;
        }
        int left = _settings->xAt(_settings->tempo()->contour()[i].tick);
        p->fillRect(left, _settings->noteHeight(), _settings->beatWidth(), _settings->noteHeight(), QColor(255, 255, 255, 64));
        QString text = QString::number(_settings->tempo()->contour()[i].value);
        p->drawText(QRect(left + 4, _settings->noteHeight() + 1, _settings->beatWidth() - 2, _settings->noteHeight() - 2), text, Qt::AlignLeft | Qt::AlignVCenter);
    }
    int left = _settings->xAt(0);
    double val = _settings->tempo()->data(0, 0).toDouble();
    p->fillRect(left, _settings->noteHeight(), _settings->beatWidth(), _settings->noteHeight(), QColor(192, 192, 255, 64));
    QString text = QString::number(val);
    p->drawText(QRect(left + 4, _settings->noteHeight() + 1, _settings->beatWidth() - 2, _settings->noteHeight() - 2), text, Qt::AlignLeft | Qt::AlignVCenter);
}

void MeasureView::_drawMeasure(int begin, int end, QPainter *p, QPaintEvent *e)
{
    if(!_settings->measure())
    {
        return;
    }
    QPen pen(QColor(0, 255, 255));
    p->setPen(pen);
    for(int i = 0; i < _settings->measure()->contour().size(); i++)
    {
        int tick = _settings->tickByMeasure(_settings->measure()->contour()[i].measure);
        if(tick + 480 < begin)
        {
            continue;
        }
        if(tick > end)
        {
            break;
        }
        int left = _settings->xAt(tick);
        p->fillRect(left, _settings->noteHeight() * 2, _settings->beatWidth(), _settings->noteHeight(), QColor(255, 255, 255, 64));
        QString text = QString::number(_settings->measure()->contour()[i].numerator) + tr("/") + QString::number(_settings->measure()->contour()[i].denominator);
        p->drawText(QRect(left + 4, _settings->noteHeight() * 2 + 1, _settings->beatWidth() - 2, _settings->noteHeight() - 2), text, Qt::AlignLeft | Qt::AlignVCenter);
    }
    int left = _settings->xAt(0);
    QPoint pos(_settings->measure()->data(0, 0).toPoint());
    p->fillRect(left, _settings->noteHeight() * 2, _settings->beatWidth(), _settings->noteHeight(), QColor(192, 192, 255, 64));
    QString text = QString::number(pos.x()) + "/" + QString::number(pos.y());
    p->drawText(QRect(left + 4, _settings->noteHeight() * 2 + 1, _settings->beatWidth() - 2, _settings->noteHeight() - 2), text, Qt::AlignLeft | Qt::AlignVCenter);
}

void MeasureView::mousePressEvent(QMouseEvent *e)
{
    if(e->button() & Qt::LeftButton)
    {
        delete _editor;
        _editor = NULL;
    }
}

void MeasureView::mouseDoubleClickEvent(QMouseEvent *e)
{
    delete _editor;
    disconnect(this, SLOT(getEditorData()));
    _current = _indexAt(e->pos());
    if(_current.type == _TypeMeasure && !_settings->isBar(_current.tick))
    {
        return;
    }
    int y = _current.type * _settings->noteHeight();
    QString text(_stringFromPoint(&_current));

    _editor = new QLineEdit(text, this);
    _editor->setGeometry(_settings->xAt(_current.tick), y, _settings->beatWidth(), _settings->noteHeight());
    connect(_editor, SIGNAL(returnPressed()), this, SLOT(getEditorData()));
    _editor->installEventFilter(this);
    _editor->show();
    _editor->setFocus();
}

QString MeasureView::_stringFromPoint(_Point *p)
{
    switch(p->type)
    {
    case _TypeTempo:
        return QString::number(p->tempo);
    case _TypeMeasure:
        return QString::number(p->up) + "/" + QString::number(p->down);
    }
    return QString();
}

bool MeasureView::eventFilter(QObject *o, QEvent *e)
{
    if(o == _editor)
    {
        QKeyEvent *key;
        switch(e->type())
        {
        case QEvent::KeyPress:
            key = (QKeyEvent *)(e);
            if(key->key() == Qt::Key_Escape)
            {
                _editor->hide();
            }
            break;
        case QEvent::FocusOut:
            _editor->hide();
            return true;
        }
    }
    return QWidget::eventFilter(o, e);
}

void MeasureView::getEditorData()
{
    bool f1, f2, isUpdated = false;
    QStringList sl;
    switch(_current.type)
    {
    case _TypeTempo:
        _current.tempo = _editor->text().toDouble(&f1);
        if(f1)
        {
            isUpdated = true;
        }
        break;
    case _TypeMeasure:
        sl = _editor->text().split("/", QString::KeepEmptyParts);
        if(sl.size() != 2)
        {
            break;
        }
        _current.up = sl.at(0).toInt(&f1);
        _current.down = sl.at(1).toInt(&f2);
        if(!f1 || !f2)
        {
            break;
        }
        if(_current.down & (_current.down - 1))
        {
            break;
        }
        if(0 < _current.up && _current.up < _current.down * 2)
        {
            isUpdated = true;
        }
        break;
    }
    if(isUpdated)
    {
        MeasureCommand *c = new MeasureCommand(_current, _settings);
        if(_settings->undoStack())
        {
            _settings->undoStack()->push(c);
        }
        else
        {
            c->redo();
            delete c;
        }
    }
    _editor->hide();
}

MeasureView::_Point MeasureView::_indexAt(QPoint pos)
{
    _Point p;
    bool isFound = false;
    int tick = _settings->tickAt(pos.x());
    if(_settings->noteHeight() <= pos.y() && pos.y() < _settings->noteHeight() * 2)
    {
        p.type = _TypeTempo;
        for(int i = 0; i < _settings->tempo()->contour().size(); i++)
        {
            if(tick <= _settings->tempo()->contour()[i].tick && _settings->tempo()->contour()[i].tick <= tick + stand::sequence::DefaultTicksPerBeat)
            {
                isFound = true;
                p.tick = _settings->tempo()->contour()[i].tick;
                p.tempo = _settings->tempo()->contour()[i].value;
            }
        }
    }
    else if(_settings->noteHeight() * 2 <= pos.y() && pos.y() < _settings->noteHeight() * 3)
    {
        p.type = _TypeMeasure;
        for(int i = 0; i < _settings->measure()->contour().size(); i++)
        {
            int measureTick = _settings->tickByMeasure(_settings->measure()->contour()[i].measure);
            if(tick - stand::sequence::DefaultTicksPerBeat / 2 <= measureTick && measureTick <= tick + stand::sequence::DefaultTicksPerBeat / 2)
            {
                isFound = true;
                p.tick = measureTick;
                p.up = _settings->measure()->contour()[i].numerator;
                p.down = _settings->measure()->contour()[i].denominator;
            }
        }
    }
    if(!isFound)
    {
        p.tick = _settings->tickAt(pos.x());
        if(_settings->isSnap())
        {
            int step = stand::sequence::DefaultTicksPerBeat / _settings->snapDiv();
            p.tick += step / 2;
            p.tick = p.tick / step * step;
        }
        p.tempo = _settings->tempo()->data(p.tick, 0).toDouble();
        QPoint point(_settings->measure()->data(p.tick, 0).toPoint());
        p.up = point.x();
        p.down = point.y();
    }
    return p;
}

MeasureView::MeasureCommand::MeasureCommand(_Point newPoint, ViewSettings *s)
{
    n = newPoint;
    settings = s;
    int measure;
    QPoint p;
    switch(newPoint.type)
    {
    case _TypeTempo:
        o.tick = n.tick;
        o.tempo = s->tempo()->data(o.tick, 0).toDouble();
        break;
    case _TypeMeasure:
        o.tick = n.tick = s->measureAt(n.tick);
        p = s->measure()->data(n.tick, 0).toPoint();
        o.up = p.x();
        o.down = p.y();
        break;
    }
}

void MeasureView::MeasureCommand::undo()
{
    switch(n.type)
    {
    case _TypeTempo:
        settings->tempo()->setData(o.tick, o.tempo, 0);
        break;
    case _TypeMeasure:
        settings->measure()->setData(o.tick, QPoint(o.up, o.down), 0);
        break;
    }
    settings->toggleReset();
}

void MeasureView::MeasureCommand::redo()
{
    switch(n.type)
    {
    case _TypeTempo:
        settings->tempo()->setData(n.tick, n.tempo, 0);
        break;
    case _TypeMeasure:
        settings->measure()->setData(n.tick, QPoint(n.up, n.down), 0);
        break;
    }
    settings->toggleReset();
}
