<?php
// vim: foldmethod=marker
/**
 *  Ethna_DB_SQLite.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: fe51d282cad3d1042ae28570da47e1dce6cb1e29 $
 */

// {{{ Ethna_DB_SQLite
/**
 *  SQLite用、データベースアクセスドライバ
 *
 *  DSN設定は以下の形式を用いること
 *  'sqlite:////path/to/dbfile?mode=0644'
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_SQLite extends Ethna_DB
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  string  $dsn DSN
     *  @param  bool    $persistent 持続接続設定
     */
    function Ethna_DB_SQLite($dsn, $persistent)
    {
        parent::Ethna_DB($dsn, $persistent);
    }

    /**
     *  DBに接続する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function connect()
    {
        $filename = $this->_dsninfo['database'];
        $mode = $this->_dsninfo['mode'];
        $error = '';
        if ($this->_persistent) {
            $this->_db = sqlite_popen($filename, $mode, $error);
        } else {
            $this->_db = sqlite_open($filename, $mode, $error);
        }

        if ($this->_db === false) {
             return Ethna::raiseError(
                       "could not connect database: $con_str detail: $error"
                   );
        }
        return 0;
    }

    /**
     *  DB接続を切断する
     *
     *  @access public
     */
    function disconnect()
    {
        //
        //   持続的接続は、SQLite の場合閉じることができる
        //   さらに、sqlite_close は値を返さない
        //

        if ($this->isValid()) {
            sqlite_close($this->_db);
        }
        $this->_db = NULL;
        return 0;
    }

    /**
     *  DBトランザクションを開始する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function begin()
    {
        return $this->__transact_query('BEGIN TRANSACTION');
    }

    /**
     *  DBトランザクションを中断する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function rollback()
    {
        return $this->__transact_query('ROLLBACK TRANSACTION');
    }

    /**
     *  DBトランザクションを終了する
     *
     *  @access public
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function commit()
    {
        return $this->__transact_query('COMMIT TRANSACTION');
    }

    /**
     *  SQL ステートメントを実行する準備を行い、
     *  Ethna_Statementオブジェクトを返します。
     *
     *  Ethna_Statement::execute() メソッドによ
     *  って実行される SQL ステートメントを準備
     *  します。 SQL ステートメントは、文が実行
     *  されるときに実際の値に置き換えられます。
     *
     *  (ステートメントにパラメータを指定する場合、
     *   ? で指定すること)
     *  @access public
     *  @param  $sql  実行を準備するSQL
     *  @return mixed 成功時にEthna_DB_Statement
     *                Ethna_Error:エラー
     */
    function prepare($sql)
    {
        $res = NULL;
        $stmt = new Ethna_DB_SQLite_Statement($this);
        $stmt->setLogger($this->_logger);

        //
        //   SQLite の手続き型APIでは、prepared stmt
        //   向けのAPIがないので、常にエミュレートされます
        //

        $stmt->setOption('__sql', trim($sql));
        $stmt->setOption('__emulation_mode', true);
        return $stmt; 
    }

    /**
     *  SQL ステートメントを実行し、作用した行数を返します。
     *  SELECT 文からは結果を返しません。SELECT 文を指定した
     *  場合は常に0が返ります。
     *
     *  @access public
     *  @param  $sql    実行するSQL
     *  @return mixed   0以上の数:作用した行数 Ethna_Error:エラー
     */
    function exec($sql)
    {
        $sql = ltrim($sql);
        if (preg_match('/^SELECT/i', $sql)) {
            return 0;
        }
        $this->_logger->log(LOG_DEBUG, $sql);
        $result = sqlite_exec($this->_db, $sql);
        if ($result === false) {
             return Ethna::raiseError(
                "could not exec your sql: "
              . $this->_dsn
              . " query: $sql ",
                E_DB_QUERY 
            );
        }
        return sqlite_changes($this->_db); 
    }

    /**
     *  直前のINSERTによるIDを取得する
     *
     *  @access public
     *  @param  string  $name (SQLiteでは不要)データベース依存のパラメータ
     *  @return mixed   int:直近のINSERTにより生成されたID null:エラー
     */
    function getInsertId($name = NULL)
    {
        return sqlite_last_insert_rowid($this->_db);
    }

    /**
     *  文字列をエスケープする 
     *
     *  @access public
     *  @param  string  $value  エスケープ対象の値 
     *  @return string  エスケープ済みの値
     */
    function escape($value)
    {
        return sqlite_escape_string($value);
    }

    /**
     *  テーブル定義情報を取得する
     *
     *  @access public
     *  @param  string  $table_name  取得対象のテーブル名
     *  @return mixed   array: PEAR::DBに準じたメタデータ
     *                  Ethna_Error::エラー
     */
    function getMetaData($table_name)
    {
        //   このメソッドはAppObject
        //   との連携に必要。 
        //
        //   以下3つのプラグマを組み合わせて取得する
        //
        //   pragma table_info('table_name');
        //   pragma index_list('table_name');
        //   pragma index_info('index_name');

        $table_info_result = $this->query("pragma table_info('$table_name')");
        if (Ethna::isError($table_info_result)) {
            return $table_info_result;
        }
        $mata = $table_info_result->fetchAll();
        $result = array();
        if (!empty($meta)) {
            foreach ($meta as $row) {
                $meta_row = array();

                $cid = $row['cid'];
                $colname = $row['name'];
                $type = $row['type'];
                $pk = $row['pk'];
                $notnull = $row['notnull'];
                $default = $row['dflt_value'];
                
                //  primary
                $meta_row['primary'] = ($pk) ? 1 : 0;
                //  seq
                $meta_row['seq'] = ($type == 'integer' && ($pk))
                            ? 1 : 0;
                //  type.
                //  @see http://sqlite.org/datatype3.html
                if (preg_match('/varchar/i', $type)) {
                    $meta_row['type'] = 'string';
                } else {
                    $type = strtolower($type);
                    switch ($type) {
                    case 'text':
                    case 'integer':
                    case 'float':
                    case 'decimal':
                    case 'date';
                    case 'boolean':
                        $meta_row['type'] = $type;
                        break;
                    case 'blob':
                    case 'binary':
                        $meta_row['type'] = 'binary';
                        break;
                    case 'datetime':
                    case 'timestamp':
                    case 'time':
                        $meta_row['type'] = 'datetime';
                        break;
                    }
                } 

                //  required
                $meta_row['required'] = ($notnull) ? 1 : 0;
                //  unique
                $index_list_result = $this->query("pragma index_list('$table_name')");
                if (Ethna::isError($index_list_result)) {
                    return $index_list_result;
                }
                $index_list = $index_list_result->fetchAll();
                if (!empty($index_list)) {
                    foreach ($index_list as $index) {
                        $index_name = $index['name'];
                        $unique = ($index['unique']) ? 1 : 0;
                        $index_info = $this->query("pragma index_info('$index_name')");
                        if (Ethna::isError($index_info)) {
                            return $index_info;
                        }
                        $info = $index_info->fetchAll();
                        if (!empty($info)) {
                            foreach ($info as $i) {
                                $index_cid = $i['cid'];
                                if ($index_cid == $cid) {
                                    $meta_row['unique'] = $unique;
                                }
                            }
                        }
                    }
                }
                $result[$colname] = $meta_row;
            }
        }
        return $result;
    }

    /**
     *  データベースのタイプを取得する
     *
     *  @access public
     *  @return string  データベースのタイプ
     */
    function getType()
    {
        return 'sqlite';
    }

    /**
     *  DBトランザクション関連のクエリを送信
     *  する内部関数です。
     *
     *  @access private 
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __transact_query($query)
    {
        if (!$this->isValid()) {
             return Ethna::raiseError(
                "Not Connected: "
              . $this->_dsn
            );
        }
        $this->_logger->log(LOG_DEBUG, $query);
        $result = sqlite_exec($this->_db, $query);
        if ($result === false) {
              return Ethna::raiseError(
                "$query transation failed: "
              . $this->_dsn,
                E_DB_QUERY 
            );
        }
        return 0; 
    }
}

// {{{ Ethna_DB_SQLite_Statement
/**
 *  SQLite ドライバの一部
 *
 *  実行前にはプリペアドステートメント。
 *  実行後には関連する結果セットを表す
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @access     public
 *  @package    Ethna
 */
class Ethna_DB_SQLite_Statement extends Ethna_DB_Statement
{
    /**
     *  コンストラクタ
     *
     *  @access public
     *  @param  Ethna_DB $db データベース接続 
     */
    function Ethna_DB_SQLite_Statement(&$db)
    {
        parent::Ethna_DB_Statement($db);
    } 

    /**
     *  プリペアドステートメント実行結果から
     *  次の行を取得します。 
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse 
     */
    function fetchRow()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            return sqlite_fetch_array($this->_result, SQLITE_NUM);
        case DB_FETCHMODE_ASSOC:
            return sqlite_fetch_array($this->_result, SQLITE_ASSOC);
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
    }
     
    /**
     *  プリペアドステートメント実行結果の
     *  結果セットを「全て」配列で返します
     *
     *  @access public
     *  @return mixed   結果がある場合は配列。ない場合はfalse
     */
    function fetchAll()
    {
        if (empty($this->_result)
         || is_resource($this->_result) == false) {
            return false;
        }
        switch ($this->_fetch_mode) {
        case DB_FETCHMODE_NUM:
            return sqlite_fetch_all($this->_result, SQLITE_NUM);
        case DB_FETCHMODE_ASSOC:
            return sqlite_fetch_all($this->_result, SQLITE_ASSOC);
        default:
            Ethna::raiseWarning(
               'Unknown fetch mode: ' . $this->fetch_mode,
               E_DB_GENERAL
            );
            return false;
        } 
    }

    /**
     *  直近の DELETE, INSERT, UPDATE 文によっ
     *  て作用した行数を返します。 
     *
     *  @access public
     *  @return int  作用した行数
     *               SELECT の場合は 0
     */
    function affectedRows()
    {
        return sqlite_changes($this->_db);
    }

    /**
     *  プリペアドステートメントのエミュレー
     *  ションロジックです。
     *
     *  @access protected
     *  @param  string $sql 実行するSQL (パラメータは ? として指定)
     *  @param  mixed  $param  パラメータの配列
     *  @return mixed   0:正常終了 Ethna_Error:エラー
     */
    function __emulatePstmt($sql, $param = array())
    {
        $fetchmode = ($this->_fetchmode == DB_FETCHMODE_NUM)
                   ? SQLITE_NUM
                   : SQLITE_ASSOC;
        $prepare_sql = $this->__getEmulatePstmtSQL($sql, $param);
        $result = sqlite_query($this->_db, $prepare_sql, $fetchmode);
        if ($result === false) {
            return Ethna::raiseError(
                "could not exec your sql: "
              . " query: $prepare_sql",
                E_DB_QUERY 
            );
        }
        $this->_result = $result;
        return 0;                
    }
}

?>
