<?php
/**
 *  Ethna_DB_PgSQL_Test.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @version    $Id: f1793c5165106d4dab9636cf294a1c3e928a1175 $
 */

require_once ETHNA_BASE . '/class/DB/Ethna_DB_PgSQL.php';

//  constant for testing
//  change here  your database server settings.
define('ETHNA_TEST_DB_PGSQL_VALIDDSN', 
       'pgsql://postgres:password@unix+localhost/test');
define('ETHNA_TEST_DB_PGSQL_DUMMYDSN', 
       'pgsql://dummy');


//{{{    Ethna_DB_PgSQL_Test
/**
 *  Test Case For Ethna_DB_PgSQL
 *
 *  @access public
 */
class Ethna_DB_PgSQL_Test extends Ethna_UnitTestBase
{
    var $pgdb;
    var $pg_dummy_db;

    function skip()
    {
        $this->pgdb = new Ethna_DB_PgSQL(ETHNA_TEST_DB_PGSQL_VALIDDSN, false);
        if (function_exists('pg_connect')) {
            $result = @$this->pgdb->connect();
            $this->skipIf(($result !== 0),
                          " because could not connect database -> "
                         . ETHNA_TEST_DB_PGSQL_VALIDDSN
            );
            @$this->pgdb->disconnect();
        } else {
            $this->skipIf(true,
                          " because your PHP does not support PostgreSQL"
            );
        }
    }

    function setUp()
    {
        $result = $this->pgdb->connect();  // valid connection
        $this->pgdb->setLogger($this->backend->getLogger()); 
        $this->pg_dummy_db = new Ethna_DB_PgSQL(ETHNA_TEST_DB_PGSQL_DUMMYDSN, false);  // dummy
    }

    function tearDown()
    {
        if ($this->pgdb->isValid()) {
            $this->pgdb->disconnect();
        }
    }

    function test_connect()
    {
        //  dummy connect
        $dummy_result = @$this->pg_dummy_db->connect();  // suppress error output.
        $this->assertTrue(($dummy_result !== 0));
        $this->assertTrue(Ethna::isError($dummy_result));

        $valid_result = $this->pgdb->connect();
        $this->assertTrue(($valid_result === 0));
        $this->assertFalse(Ethna::isError($valid_result));
    }

    function test_disconnect()
    {
        //  dummy disconnect
        $dummy_result = @$this->pg_dummy_db->disconnect();  // suppress error output.
        $this->assertTrue(($dummy_result !== 0));
        $this->assertTrue(Ethna::isError($dummy_result));

        //  valid disconnect
        $valid_result = $this->pgdb->disconnect();
        $this->assertTrue(($valid_result === 0));
        $this->assertFalse(Ethna::isError($valid_result));

        //  persistent connection(dummy)
        $dummy_persist_db = new Ethna_DB_PgSQL(ETHNA_TEST_DB_PGSQL_DUMMYDSN, true);  // dummy
        $valid_persist_result = $dummy_persist_db->disconnect();
        $this->assertTrue(($valid_persist_result === 0));
        $this->assertFalse(Ethna::isError($valid_persist_result));
    }

    function test_isValid()
    {
        $this->assertTrue($this->pgdb->isValid());
        $this->assertFalse($this->pg_dummy_db->isValid());
    }

    function test_getType()
    {
        $this->assertEqual('pgsql', $this->pgdb->getType());
        $this->assertEqual('pgsql', $this->pg_dummy_db->getType());
    }

    function test_prepare()
    {
        @$this->pgdb->exec("create table test (a integer)");
        $stmt = $this->pgdb->prepare("select * from test where a = ?");
        $this->assertFalse(Ethna::isError($stmt));
        $this->assertEqual($stmt->getOption('__sql'),
                           "select * from test where a = $1"
        ); 

        if (version_compare(PHP_VERSION, '5.1.0') === 1) {
            $this->assertFalse($stmt->getOption('__emulation_mode'));
        } else {
            $this->assertTrue($stmt->getOption('__emulation_mode'));
        }

        //  invalid sql(WHERE 句がない!)
        $stmt = @$this->pgdb->prepare("select * from test a = ?");
        $this->assertTrue(Ethna::isError($stmt));

        @$this->pgdb->exec("drop table test");
    }

    function test_exec()
    {
        @$this->pgdb->exec("create table test (a integer)");
        $result = $this->pgdb->exec("SELECT a from test");
        $this->assertEqual(0, $result);

        $result = $this->pgdb->exec("INSERT INTO test (a) values (1)");
        $this->assertEqual(1, $result);
        
        //  invalid sql(WHERE 句がない!)
        $result = @$this->pgdb->prepare("select * from test a = ?");
        $this->assertTrue(Ethna::isError($result));

        @$this->pgdb->exec("drop table test");
    } 

    function test_query()
    {
        @$this->pgdb->exec("create table test (a integer primary key, b text)");
        $result = $this->pgdb->exec("INSERT INTO test (a,b) values (1, 'a')");
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(1, $result);
        $result1 = $this->pgdb->exec("INSERT INTO test (a,b) values (2, 'b')");
        $this->assertFalse(Ethna::isError($result1));
        $this->assertEqual(1, $result1);
       
        //    fetchmode
        $stmt = $this->pgdb->query("SELECT * FROM test WHERE a = ?",
                           array(1));   //  FETCH_ASSOC
        $this->assertFalse(Ethna::isError($stmt));
        $row = $stmt->fetchRow();
        $this->assertEqual(1, $row['a']);
        $this->assertEqual('a', $row['b']);
        
        $stmt1 = $this->pgdb->query("SELECT * FROM test WHERE a = ?",
                           array(2),
                           DB_FETCHMODE_NUM);   //  FETCH_NUM
        $this->assertFalse(Ethna::isError($stmt1));
        $row1 = $stmt1->fetchRow();
        $this->assertEqual(2, $row1[0]);
        $this->assertEqual('b', $row1[1]);

        //    invalid sql
        $stmt2 = @$this->pgdb->query("SELECT * FROM test WHERE a = ",
                           array(2)); 
        $this->assertTrue(Ethna::isError($stmt2));

        @$this->pgdb->exec("drop table test");
    }

    function test_getInsertId()
    {
        @$this->pgdb->exec("create table test (a SERIAL primary key, b integer)");
        $result = $this->pgdb->exec("INSERT INTO test (b) values (1)");
        $this->assertFalse(Ethna::isError($result));

        $id = $this->pgdb->getInsertId('test_a_seq');
        $this->assertEqual(1, $id);

        $this->assertNULL($this->pgdb->getInsertId());
    }

    function test_getMetaData()
    {
        @$this->pgdb->exec("drop table test");
        $result = $this->pgdb->exec('CREATE TABLE test ( '
                        . ' a varchar(255) primary key, b date, '
                        . " c float, d integer DEFAULT 1, e numeric, f text DEFAULT 'hoge', "
                        . ' g time, h timestamp, i bytea UNIQUE, j boolean DEFAULT FALSE, '
                        . ' k serial NOT NULL)'
                         );
        $this->assertFalse(Ethna::isError($result));

        $meta = $this->pgdb->getMetaData('test');
        $this->assertFalse(Ethna::isError($meta));
        if (Ethna::isError($meta)) {
            $this->fail($meta->getMessage());
        }

        //   type
        $this->assertEqual('string', $meta['a']['type']);
        $this->assertEqual('date', $meta['b']['type']);
        $this->assertEqual('float', $meta['c']['type']);
        $this->assertEqual('integer', $meta['d']['type']);
        $this->assertEqual('decimal', $meta['e']['type']);
        $this->assertEqual('text', $meta['f']['type']);
        $this->assertEqual('time', $meta['g']['type']);
        $this->assertEqual('timestamp', $meta['h']['type']);
        $this->assertEqual('binary', $meta['i']['type']);
        $this->assertEqual('boolean', $meta['j']['type']);
        $this->assertEqual('integer', $meta['k']['type']);

        //    required
        $this->assertEqual(1, $meta['a']['required']);  // primary key
        $this->assertEqual(0, $meta['b']['required']);
        $this->assertEqual(0, $meta['c']['required']);
        $this->assertEqual(0, $meta['d']['required']);
        $this->assertEqual(0, $meta['e']['required']);
        $this->assertEqual(0, $meta['f']['required']);
        $this->assertEqual(0, $meta['g']['required']);
        $this->assertEqual(0, $meta['h']['required']);
        $this->assertEqual(0, $meta['i']['required']);
        $this->assertEqual(0, $meta['j']['required']);
        $this->assertEqual(1, $meta['k']['required']);  // NOT NULL

        //    unique 
        $this->assertEqual(0, $meta['a']['unique']);  // primary key
        $this->assertEqual(0, $meta['b']['unique']);
        $this->assertEqual(0, $meta['c']['unique']);
        $this->assertEqual(0, $meta['d']['unique']);
        $this->assertEqual(0, $meta['e']['unique']);
        $this->assertEqual(0, $meta['f']['unique']);
        $this->assertEqual(0, $meta['g']['unique']);
        $this->assertEqual(0, $meta['h']['unique']);
        $this->assertEqual(1, $meta['i']['unique']);  // UNIQUE
        $this->assertEqual(0, $meta['j']['unique']);
        $this->assertEqual(0, $meta['k']['unique']); 

        //    primary 
        $this->assertEqual(1, $meta['a']['primary']);  // primary key
        $this->assertEqual(0, $meta['b']['primary']);
        $this->assertEqual(0, $meta['c']['primary']);
        $this->assertEqual(0, $meta['d']['primary']);
        $this->assertEqual(0, $meta['e']['primary']);
        $this->assertEqual(0, $meta['f']['primary']);
        $this->assertEqual(0, $meta['g']['primary']);
        $this->assertEqual(0, $meta['h']['primary']);
        $this->assertEqual(0, $meta['i']['primary']);
        $this->assertEqual(0, $meta['j']['primary']);
        $this->assertEqual(0, $meta['k']['primary']);

        //    seq 
        $this->assertEqual(0, $meta['a']['seq']);
        $this->assertEqual(0, $meta['b']['seq']);
        $this->assertEqual(0, $meta['c']['seq']);
        $this->assertEqual(0, $meta['d']['seq']);
        $this->assertEqual(0, $meta['e']['seq']);
        $this->assertEqual(0, $meta['f']['seq']);
        $this->assertEqual(0, $meta['g']['seq']);
        $this->assertEqual(0, $meta['h']['seq']);
        $this->assertEqual(0, $meta['i']['seq']);
        $this->assertEqual(0, $meta['j']['seq']);
        $this->assertEqual(1, $meta['k']['seq']);

        @$this->pgdb->exec("drop table test");
    }

    function test_getMetaData_Multiple_Key()
    {
        @$this->pgdb->exec("drop table test");
        $result = $this->pgdb->exec('CREATE TABLE test ( '
                        . ' a varchar(255), b date, '
                        . " primary key (a,b) )"
                         );
        $this->assertFalse(Ethna::isError($result));

        $meta = $this->pgdb->getMetaData('test');
        $this->assertFalse(Ethna::isError($meta));
        if (Ethna::isError($meta)) {
            $this->fail($meta->getMessage());
        }

        $this->assertEqual(1, $meta['a']['primary']);  // primary key
        $this->assertEqual(1, $meta['b']['primary']);  // primary key

        $this->assertEqual(1, $meta['a']['required']);  // primary key
        $this->assertEqual(1, $meta['b']['required']);  // primary key

        @$this->pgdb->exec("drop table test");
    }
}

//{{{    Ethna_DB_PgSQL_Statement_Test
/**
 *  Test Case For Ethna_DB_PgSQL_Statement
 *
 *  @access public
 */
class Ethna_DB_PgSQL_Statement_Test extends Ethna_UnitTestBase
{
    var $pgdb;
    var $pgstmt;

    function skip()
    {
        $this->pgdb = new Ethna_DB_PgSQL(ETHNA_TEST_DB_PGSQL_VALIDDSN, false);
        if (function_exists('pg_connect')) {
            $result = @$this->pgdb->connect();
            $this->skipIf(($result !== 0),
                          " because could not connect database -> "
                         . ETHNA_TEST_DB_PGSQL_VALIDDSN
            );
            $this->pgdb->disconnect();
        } else {
            @$this->skipIf(true,
                          " because your PHP does not support PostgreSQL"
            );
        }
    }

    function setUp()
    {
        $result = $this->pgdb->connect();  // valid connection
        $this->pgdb->setLogger($this->backend->getLogger()); 
        $this->pgdb->exec("CREATE TABLE test (a integer PRIMARY KEY, b text)");
        $this->pgdb->exec("INSERT INTO test (a,b) VALUES(1, 'a')");
        $this->pgdb->exec("INSERT INTO test (a,b) VALUES(2, 'b')");
    }

    function tearDown()
    {
        $this->pgdb->exec("DROP TABLE test");
        if ($this->pgdb->isValid()) {
            $this->pgdb->disconnect();
        }
    }

    function test_stmt_exec()
    {
        //   normal result(INSERT)
        $sql = "INSERT INTO test (a,b) VALUES (?, ?)";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql, array(3,'c'));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   normal result(select)
        $sql = "SELECT a FROM test WHERE a = ? OR a = 2 OR a = ? OR 1 = 1";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql, array(1, 3));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   normal result(update)
        $sql = "UPDATE test set a = ?, b = ? WHERE a = 1 ";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql, array(1,'c'));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   duplicate key
        $result1 = @$this->pgstmt->exec($sql, array(3,'d'));
        $this->assertTrue(Ethna::isError($result1));
    }

    function test_stmt_fetchrow()
    {
        $sql = "SELECT * FROM test ORDER BY a";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql);
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);
        
        //   FETCHMODE_ASSOC
        $row1 = $this->pgstmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(1, $row1['a']);
        $this->assertEqual('a', $row1['b']);

        $row2 = $this->pgstmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(2, $row2['a']);
        $this->assertEqual('b', $row2['b']);
        $row3 = $this->pgstmt->fetchRow();
        $this->assertFalse($row3);

        $sql = "SELECT * FROM test WHERE a = ? AND 1 = 1";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $this->pgstmt->setFetchMode(DB_FETCHMODE_NUM);
        $result1 = $this->pgstmt->exec($sql, array(2));

        //   FETCHMODE_NUM
        $row1 = $this->pgstmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(2, $row1[0]);
        $this->assertEqual('b', $row1[1]);
        $row2 = $this->pgstmt->fetchRow();
        $this->assertFalse($row2);
    }

    function test_stmt_fetchall()
    {
        $sql = "SELECT * FROM test ORDER BY a";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql);
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   FETCHMODE_ASSOC
        $allrows = $this->pgstmt->fetchAll();
        $this->assertEqual(1, $allrows[0]['a']);
        $this->assertEqual('a', $allrows[0]['b']);
        $this->assertEqual(2, $allrows[1]['a']);
        $this->assertEqual('b', $allrows[1]['b']);
        $this->assertFalse(isset($allrows[2]['a']));
        $this->assertFalse(isset($allrows[2]['b']));

        //   FETCHMODE_NUM
        $this->pgstmt->setFetchMode(DB_FETCHMODE_NUM);
        $result1 = $this->pgstmt->exec($sql);

        $allrows = $this->pgstmt->fetchAll();
        $this->assertEqual(1, $allrows[0][0]);
        $this->assertEqual('a', $allrows[0][1]);
        $this->assertEqual(2, $allrows[1][0]);
        $this->assertEqual('b', $allrows[1][1]);
        $this->assertFalse(isset($allrows[2][0]));
        $this->assertFalse(isset($allrows[2][1]));
    }

    function test_stmt_affectedrows()
    {
        //    SELECT の結果の場合、結果は0となる
        $sql = "SELECT * FROM test ORDER BY a";
        $this->pgstmt = $this->pgdb->prepare($sql);
        $result = $this->pgstmt->exec($sql);

        $affected = $this->pgstmt->affectedRows();
        $this->assertEqual(0, $affected);

        //    INSERT 
        $insert_sql = "INSERT INTO test (a,b) VALUES (?,'c')";
        $this->pgstmt = $this->pgdb->prepare($insert_sql);
        $this->pgstmt->exec($insert_sql, array(3));
        $affected = $this->pgstmt->affectedRows();
        $this->assertEqual(1, $affected);
    
        //    UPDATE 
        $update_sql = "UPDATE test set b = ?";
        $this->pgstmt = $this->pgdb->prepare($update_sql);
        $this->pgstmt->exec($update_sql, array('d'));
        $affected = $this->pgstmt->affectedRows();
        $this->assertEqual(3, $affected);

        //    DELETE
        $delete_sql = "DELETE FROM test WHERE a = ? AND 1 = 1";
        $this->pgstmt = $this->pgdb->prepare($delete_sql);
        $this->pgstmt->exec($delete_sql, array(1));
        $affected = $this->pgstmt->affectedRows();
        $this->assertEqual(1, $affected);

        //    TRUNCATE は 結果が0 になる。
        //    なぜなら、pg_affected_rows は INSERT, UPDATE, DELETE
        //    のみに対応しているから
        $truncate_sql = "TRUNCATE TABLE test";
        $this->pgstmt = $this->pgdb->prepare($truncate_sql);
        $this->pgstmt->exec($truncate_sql);
        $affected = $this->pgstmt->affectedRows();
        $this->assertEqual(0, $affected);
    }
}

?>
