<?php
/**
 *  Ethna_DB_SQLite_Test.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @version    $Id: 64ea15a92675a1ebb450c178f120fca0c79f1904 $
 */

require_once ETHNA_BASE . '/class/DB/Ethna_DB_SQLite.php';
require_once ETHNA_BASE . '/class/DB/Ethna_DB_SQLite.php';

//  constant for testing
//  change here  your database server settings.
define('ETHNA_TEST_DB_SQLITE_VALIDDSN', 
       'sqlite:///ethnatest.db?mode=0644');

define('ETHNA_TEST_DB_SQLITE_DUMMYDSN', 
       'sqlite:////path/to/nonexistent');

//{{{  Ethna_DB_SQLite_Test
/**
 *  Test Case For Ethna_DB_SQLite
 *
 *  @access public
 */
class Ethna_DB_SQLite_Test extends Ethna_UnitTestBase
{
    var $sqlitedb;
    var $sqlite_dummy_db;

    function skip()
    {
        $this->sqlitedb = new Ethna_DB_SQLite(ETHNA_TEST_DB_SQLITE_VALIDDSN, false);
        if (function_exists('sqlite_open')) {
            $result = @$this->sqlitedb->connect();
            $this->skipIf(($result !== 0),
                          " because could not connect database -> "
                         . ETHNA_TEST_DB_SQLITE_VALIDDSN
            );
            $this->sqlitedb->disconnect();
        } else {
            $this->skipIf(true,
                          " because your PHP does not support SQLite"
            );
        }
    }

    function setUp()
    {
        $result = $this->sqlitedb->connect();  // valid connection
        $this->sqlitedb->setLogger($this->backend->getLogger()); 
        $this->sqlite_dummy_db = new Ethna_DB_SQLite(ETHNA_TEST_DB_SQLITE_DUMMYDSN, false);  // dummy, not connnected
    }

    function tearDown()
    {
        if ($this->sqlitedb->isValid()) {
            $this->sqlitedb->disconnect();
        }
    }

    function test_connect()
    {
        //  dummy connect
        $dummy_result = @$this->sqlite_dummy_db->connect();  // suppress error output.
        $this->assertTrue(($dummy_result !== 0));
        $this->assertTrue(Ethna::isError($dummy_result));

        $valid_result = $this->sqlitedb->connect();
        $this->assertTrue(($valid_result === 0));
        $this->assertFalse(Ethna::isError($valid_result));
    }

    function test_disconnect()
    {
        //
        //   どんな状態でも 0 が返るので、結果は常に正しくなる
        //

        //  dummy disconnect
        $dummy_result = @$this->sqlite_dummy_db->disconnect();  // suppress error output.
        $this->assertTrue(($dummy_result === 0));
        $this->assertFalse(Ethna::isError($dummy_result));

        //  valid disconnect
        $valid_result = $this->sqlitedb->disconnect();
        $this->assertTrue(($valid_result === 0));
        $this->assertFalse(Ethna::isError($valid_result));

        //  persistent connection(dummy)
        $dummy_persist_db = new Ethna_DB_SQLite(ETHNA_TEST_DB_SQLITE_DUMMYDSN, true);  // dummy
        $valid_persist_result = $dummy_persist_db->disconnect();
        $this->assertTrue(($valid_persist_result === 0));
        $this->assertFalse(Ethna::isError($valid_persist_result));
    }

    function test_isValid()
    {
        $this->assertTrue($this->sqlitedb->isValid());
        $this->assertFalse($this->sqlite_dummy_db->isValid());
    }

    function test_getType()
    {
        $this->assertEqual('sqlite', $this->sqlitedb->getType());
    }

    function test_prepare()
    {
        @$this->sqlitedb->exec("create table test (a integer)");
        $stmt = $this->sqlitedb->prepare("select * from test where a = ?");
        $this->assertFalse(Ethna::isError($stmt));
        $this->assertEqual($stmt->getOption('__sql'),
                           "select * from test where a = ?"
        ); 

        //  emulation mode は常に true
        $this->assertTrue($stmt->getOption('__emulation_mode'));

        //  invalid sql(WHERE 句がない!)
        //
        //  emulation mode の場合、SQL の間違いを検知できないの
        //  で、結果はエラーにならず、常に正しい値が帰ってくる
        //

        $stmt = @$this->sqlitedb->prepare("select * from test a = ?");
        $this->assertFalse(Ethna::isError($stmt));
        $this->assertPattern('/Statement/i',get_class($stmt));

        @$this->sqlitedb->exec("drop table test");
    }

    function test_exec()
    {
        @$this->sqlitedb->exec("create table test (a integer)");
        $result = $this->sqlitedb->exec("SELECT a from test");
        $this->assertEqual(0, $result);

        $result = $this->sqlitedb->exec("INSERT INTO test (a) values (1)");
        $this->assertEqual(1, $result);
        
        //  invalid sql(WHERE 句がない!)
        //
        //  emulation mode の場合、SQL の間違いを検知できないの
        //  で、結果はエラーにならず、常に正しい値が帰ってくる
        //
        $result = @$this->sqlitedb->prepare("select * from test a = ?");
        $this->assertFalse(Ethna::isError($result));
        $this->assertPattern('/Statement/i',get_class($result));

        @$this->sqlitedb->exec("drop table test");
    } 

    function test_query()
    {
        @$this->sqlitedb->exec("create table test (a integer primary key, b text)");
        $result = $this->sqlitedb->exec("INSERT INTO test (a,b) values (1, 'a')");
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(1, $result);
        $result1 = $this->sqlitedb->exec("INSERT INTO test (a,b) values (2, 'b')");
        $this->assertFalse(Ethna::isError($result1));
        $this->assertEqual(1, $result1);
       
        //    fetchmode
        $stmt = $this->sqlitedb->query("SELECT * FROM test WHERE a = ?",
                           array(1));   //  FETCH_ASSOC
        $this->assertFalse(Ethna::isError($stmt));
        $row = $stmt->fetchRow();
        $this->assertEqual(1, $row['a']);
        $this->assertEqual('a', $row['b']);
        
        $stmt1 = $this->sqlitedb->query("SELECT * FROM test WHERE a = ?",
                           array(2),
                           DB_FETCHMODE_NUM);   //  FETCH_NUM
        $this->assertFalse(Ethna::isError($stmt1));
        $row1 = $stmt1->fetchRow();
        $this->assertEqual(2, $row1[0]);
        $this->assertEqual('b', $row1[1]);

        //    invalid sql
        $stmt2 = @$this->sqlitedb->query("SELECT * FROM test WHERE a = ",
                           array(2)); 
        $this->assertTrue(Ethna::isError($stmt2));

        @$this->sqlitedb->exec("drop table test");
    }

    function test_getInsertId()
    {
        //
        //  SQLite では、sequence 型は存在せず、integer primary key が
        //  auto increment を意味する
        //

        @$this->sqlitedb->exec("create table test (a integer primary key, b integer)");
        $result = $this->sqlitedb->exec("INSERT INTO test (b) values (3)");
        $this->assertFalse(Ethna::isError($result));

        $id = $this->sqlitedb->getInsertId();
        $this->assertEqual(1, $id);

        $result = $this->sqlitedb->exec("INSERT INTO test (b) values (3)");
        $id = $this->sqlitedb->getInsertId();
        $this->assertEqual(2, $id);

        @$this->sqlitedb->exec("drop table test");
    }

    function test_getMetaData()
    {
        //  TODO: write this test.
    }
}

//{{{    Ethna_DB_SQLite_Statement_Test
/**
 *  Test Case For Ethna_DB_SQLite_Statement
 *
 *  @access public
 */
class Ethna_DB_SQLite_Statement_Test extends Ethna_UnitTestBase
{
    var $sqlitedb;
    var $sqlitestmt;

    function skip()
    {
        $this->sqlitedb = new Ethna_DB_SQLite(ETHNA_TEST_DB_SQLITE_VALIDDSN, false);
        if (function_exists('sqlite_open')) {
            $result = @$this->sqlitedb->connect();
            $this->skipIf(($result !== 0),
                          " because could not connect database -> "
                         . ETHNA_TEST_DB_SQLITE_VALIDDSN
            );
            $this->sqlitedb->disconnect();
        } else {
            @$this->skipIf(true,
                          " because your PHP does not support PostgreSQL"
            );
        }
    }

    function setUp()
    {
        $result = $this->sqlitedb->connect();  // valid connection
        $this->sqlitedb->setLogger($this->backend->getLogger()); 
        $this->sqlitedb->exec("CREATE TABLE test (a integer PRIMARY KEY, b text)");
        $this->sqlitedb->exec("INSERT INTO test (a,b) VALUES(1, 'a')");
        $this->sqlitedb->exec("INSERT INTO test (a,b) VALUES(2, 'b')");
    }

    function tearDown()
    {
        $this->sqlitedb->exec("DROP TABLE test");
        if ($this->sqlitedb->isValid()) {
            $this->sqlitedb->disconnect();
        }
    }

    function test_stmt_exec()
    {
        //   normal result(INSERT)
        $sql = "INSERT INTO test (a,b) VALUES (?, ?)";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql, array(3,'c'));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   normal result(select)
        $sql = "SELECT a FROM test WHERE a = ? OR a = 2 OR a = ? OR 1 = 1";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql, array(1, 3));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   normal result(update)
        $sql = "UPDATE test set a = ?, b = ? WHERE a = 1 ";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql, array(1,'c'));
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   duplicate key
        $result1 = @$this->sqlitestmt->exec($sql, array(3,'d'));
        $this->assertTrue(Ethna::isError($result1));
    }

    function test_stmt_fetchrow()
    {
        $sql = "SELECT * FROM test ORDER BY a";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql);
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);
        
        //   FETCHMODE_ASSOC
        $row1 = $this->sqlitestmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(1, $row1['a']);
        $this->assertEqual('a', $row1['b']);

        $row2 = $this->sqlitestmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(2, $row2['a']);
        $this->assertEqual('b', $row2['b']);
        $row3 = $this->sqlitestmt->fetchRow();
        $this->assertFalse($row3);

        $sql = "SELECT * FROM test WHERE a = ? AND 1 = 1";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $this->sqlitestmt->setFetchMode(DB_FETCHMODE_NUM);
        $result1 = $this->sqlitestmt->exec($sql, array(2));

        //   FETCHMODE_NUM
        $row1 = $this->sqlitestmt->fetchRow();
        $this->assertTrue(is_array($row1));
        $this->assertEqual(2, $row1[0]);
        $this->assertEqual('b', $row1[1]);
        $row2 = $this->sqlitestmt->fetchRow();
        $this->assertFalse($row2);
    }

    function test_stmt_fetchall()
    {
        $sql = "SELECT * FROM test ORDER BY a";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql);
        $this->assertFalse(Ethna::isError($result));
        $this->assertEqual(0, $result);

        //   FETCHMODE_ASSOC
        $allrows = $this->sqlitestmt->fetchAll();
        $this->assertEqual(1, $allrows[0]['a']);
        $this->assertEqual('a', $allrows[0]['b']);
        $this->assertEqual(2, $allrows[1]['a']);
        $this->assertEqual('b', $allrows[1]['b']);
        $this->assertFalse(isset($allrows[2]['a']));
        $this->assertFalse(isset($allrows[2]['b']));

        //   FETCHMODE_NUM
        $this->sqlitestmt->setFetchMode(DB_FETCHMODE_NUM);
        $result1 = $this->sqlitestmt->exec($sql);

        $allrows = $this->sqlitestmt->fetchAll();
        $this->assertEqual(1, $allrows[0][0]);
        $this->assertEqual('a', $allrows[0][1]);
        $this->assertEqual(2, $allrows[1][0]);
        $this->assertEqual('b', $allrows[1][1]);
        $this->assertFalse(isset($allrows[2][0]));
        $this->assertFalse(isset($allrows[2][1]));
    }

    function test_stmt_affectedrows()
    {
        //    SELECT の結果の場合、結果は0となる
        $sql = "SELECT * FROM test ORDER BY a";
        $this->sqlitestmt = $this->sqlitedb->prepare($sql);
        $result = $this->sqlitestmt->exec($sql);

        $affected = $this->sqlitestmt->affectedRows();
        $this->assertEqual(0, $affected);

        //    INSERT 
        $insert_sql = "INSERT INTO test (a,b) VALUES (?,'c')";
        $this->sqlitestmt = $this->sqlitedb->prepare($insert_sql);
        $this->sqlitestmt->exec($insert_sql, array(3));
        $affected = $this->sqlitestmt->affectedRows();
        $this->assertEqual(1, $affected);
    
        //    UPDATE 
        $update_sql = "UPDATE test set b = ?";
        $this->sqlitestmt = $this->sqlitedb->prepare($update_sql);
        $this->sqlitestmt->exec($update_sql, array('d'));
        $affected = $this->sqlitestmt->affectedRows();
        $this->assertEqual(3, $affected);

        //    DELETE
        $delete_sql = "DELETE FROM test WHERE a = ? AND 1 = 1";
        $this->sqlitestmt = $this->sqlitedb->prepare($delete_sql);
        $this->sqlitestmt->exec($delete_sql, array(1));
        $affected = $this->sqlitestmt->affectedRows();
        $this->assertEqual(1, $affected);

        //
        //    SQLite では TRUNCATE 文はサポートされていない
        //    @see http://sqlite.org/lang.html
        //
    }
}

?>
