<?php
/**
 *  Ethna_DB_Test.php
 *
 *  @author     Yoshinari Takaoka <takaoka@beatcraft.com>
 *  @version    $Id: da9b7fcc78d98598b1d3d5e4d9a9593d0c8eb72b $
 */

//{{{    Ethna_DB_Test
/**
 *  Test Case For Ethna_DB
 *
 *  @access public
 */
class Ethna_DB_Test extends Ethna_UnitTestBase
{
    var $_dummy_db;

    function setUp()
    {
        $dummy_dsn = "dummy";
        $this->_dummy_db = new Ethna_DB($dummy_dsn, false);
    }

    function tearDown()
    {
        // do nothing.
    }

    function test_getLogger()
    {
        $this->assertNULL($this->_dummy_db->getLogger());
    }

    function test_setLogger()
    {
        $expected = "dummy logger object";
        $this->_dummy_db->setLogger($expected);
        $this->assertNotNULL($this->_dummy_db->getLogger());
        $this->assertEqual($expected, $this->_dummy_db->getLogger());
    }

    function test_getDSN()
    {
        $this->assertEqual("dummy", $this->_dummy_db->getDSN());
    }

    function test_getDSNInfo()
    {
        $dsninfo = $this->_dummy_db->getDSNInfo();
        $this->assertNotNULL($dsninfo);
        $this->assertTrue(is_array($dsninfo));

        //    get from array
        $this->assertEqual('dummy', $dsninfo['phptype']);
        $this->assertEqual('dummy', $dsninfo['dbsyntax']);
        $this->assertFalse($dsninfo['username']);
        $this->assertFalse($dsninfo['password']);
        $this->assertFalse($dsninfo['protocol']);
        $this->assertFalse($dsninfo['hostspec']);
        $this->assertFalse($dsninfo['port']);
        $this->assertFalse($dsninfo['socket']);
        $this->assertFalse($dsninfo['database']);

        //    get with key 
        $this->assertEqual('dummy', $this->_dummy_db->getDSNInfo('phptype'));
        $this->assertEqual('dummy', $this->_dummy_db->getDSNInfo('dbsyntax'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('username'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('password'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('protocol'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('hostspec'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('port'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('socket'));
        $this->assertFalse($this->_dummy_db->getDSNInfo('database'));
        $this->assertNULL($this->_dummy_db->getDSNInfo('hoge'));
    }

    function test_parseDSN_dsn_array()
    {
        $dsn_array = array(
                        'phptype'  => true,
                        'dbsyntax' => true,
                        'username' => true,
                        'password' => true,
                        'protocol' => true,
                        'hostspec' => true,
                        'port'     => true,
                        'socket'   => true,
                        'database' => true,
                        'hoge' => true,
                        'fuga' => true,
                     );
        $array_db = new Ethna_DB($dsn_array, false);
        
        $this->assertTrue($array_db->getDSNInfo('phptype'));
        $this->assertTrue($array_db->getDSNInfo('dbsyntax'));
        $this->assertTrue($array_db->getDSNInfo('username'));
        $this->assertTrue($array_db->getDSNInfo('password'));
        $this->assertTrue($array_db->getDSNInfo('protocol'));
        $this->assertTrue($array_db->getDSNInfo('hostspec'));
        $this->assertTrue($array_db->getDSNInfo('port'));
        $this->assertTrue($array_db->getDSNInfo('socket'));
        $this->assertTrue($array_db->getDSNInfo('database'));
        $this->assertTrue($array_db->getDSNInfo('hoge'));
        $this->assertTrue($array_db->getDSNInfo('fuga'));
        $this->assertNULL($array_db->getDSNInfo('foo'));
    }

    function test_parseDSN_absolute()
    {
        $absolute_dsn = 'phptype(dbsyntax)://username:password@protocol'
                      . '+hostspec/database?option=8&another=true';
        $absolute_db = new Ethna_DB($absolute_dsn, false);
        
        $this->assertEqual('phptype', $absolute_db->getDSNInfo('phptype'));
        $this->assertEqual('dbsyntax', $absolute_db->getDSNInfo('dbsyntax'));
        $this->assertEqual('username', $absolute_db->getDSNInfo('username'));
        $this->assertEqual('password', $absolute_db->getDSNInfo('password'));
        $this->assertEqual('protocol', $absolute_db->getDSNInfo('protocol'));
        //   hostspec が解釈されるのは protocol が
        //   tcp のときだけ port も同様
        $this->assertFalse($absolute_db->getDSNInfo('hostspec'));
        $this->assertFalse($absolute_db->getDSNInfo('port'));
        $this->assertEqual('database', $absolute_db->getDSNInfo('database'));
        //   socket が解釈されるのは protocol が
        //   unix のときだけ
        $this->assertFalse($absolute_db->getDSNInfo('socket'));
        $this->assertEqual('8', $absolute_db->getDSNInfo('option'));
        $this->assertEqual('true', $absolute_db->getDSNInfo('another'));
    }

    function test_parseDSN_hostspec_tcp()
    {
        $tcp_dsn = 'phptype(dbsyntax)://username:password@tcp'
                      . '+hostspec:110/database?option=8&another=true';
        $tcp_db = new Ethna_DB($tcp_dsn, false);
        $this->assertEqual('tcp', $tcp_db->getDSNInfo('protocol'));
        $this->assertEqual('hostspec', $tcp_db->getDSNInfo('hostspec'));
        $this->assertEqual('110', $tcp_db->getDSNInfo('port'));
        //   socket が解釈されるのは protocol が
        //   unix のときだけ
        $this->assertFalse($tcp_db->getDSNInfo('socket'));
    }

    function test_parseDSN_hostspec_unix()
    {
        $unix_dsn = 'phptype://username:password@unix'
                      . '(hostspec:110)/database?option=8&another=true';
        $unix_db = new Ethna_DB($unix_dsn, false);
        $this->assertEqual('unix', $unix_db->getDSNInfo('protocol'));
        //   hostspec が解釈されるのは protocol が
        //   tcp のときだけ port も同様
        $this->assertFalse($unix_db->getDSNInfo('hostspec'));
        $this->assertFalse($unix_db->getDSNInfo('port'));
        //   socket が解釈されるのは protocol が
        //   unix のときだけ
        $this->assertEqual('hostspec:110', $unix_db->getDSNInfo('socket'));
    }

    function test_parseDSN_hostspec_simple()
    {
        $simple_dsn = 'phptype://';
        $simple_db = new Ethna_DB($simple_dsn, false);

        $this->assertEqual('phptype', $simple_db->getDSNInfo('phptype'));
        $this->assertEqual('phptype', $simple_db->getDSNInfo('dbsyntax'));
        $this->assertFalse($simple_db->getDSNInfo('username'));
        $this->assertFalse($simple_db->getDSNInfo('password'));
        //   '://' より前の部分しかない場合は protocol 以降は解釈されない
        $this->assertFalse($simple_db->getDSNInfo('protocol'));
        $this->assertFalse($simple_db->getDSNInfo('hostspec'));
        $this->assertFalse($simple_db->getDSNInfo('port'));
        $this->assertFalse($simple_db->getDSNInfo('database'));
        $this->assertFalse($simple_db->getDSNInfo('socket'));
        $this->assertFalse($simple_db->getDSNInfo('option'));
        $this->assertFalse($simple_db->getDSNInfo('another'));
    }

    function test_parseDSN_hostspec_simple1()
    {
        $simple_dsn = 'phptype(dbsyntax)';
        $simple_db = new Ethna_DB($simple_dsn, false);
        $this->assertEqual('phptype', $simple_db->getDSNInfo('phptype'));
        $this->assertEqual('dbsyntax', $simple_db->getDSNInfo('dbsyntax'));
        $this->assertFalse($simple_db->getDSNInfo('username'));
        $this->assertFalse($simple_db->getDSNInfo('password'));
        //   '://' より前の部分しかない場合は protocol 以降は解釈されない
        $this->assertFalse($simple_db->getDSNInfo('protocol'));
        $this->assertFalse($simple_db->getDSNInfo('hostspec'));
        $this->assertFalse($simple_db->getDSNInfo('port'));
        $this->assertFalse($simple_db->getDSNInfo('database'));
        $this->assertFalse($simple_db->getDSNInfo('socket'));
        $this->assertFalse($simple_db->getDSNInfo('option'));
        $this->assertFalse($simple_db->getDSNInfo('another'));
    }

    function test_parseDSN_hostspec_param_override()
    {
        $unix_dsn = 'phptype://username:password@unix'
                      . '(hostspec:110)/database?option=8&option=true';
        $unix_db = new Ethna_DB($unix_dsn, false);

        //  クエリストリングの引数で被るものがあった場合は早い者勝ち
        $this->assertEqual('8', $unix_db->getDSNInfo('option'));
    }

    function test_parseDSN_hostspec_plus_mode()
    {
        $sqlite_dsn = 'sqlite:////path/to/dbfile?mode=644';
        $sqlite_db = new Ethna_DB($sqlite_dsn, false);

        $this->assertEqual('/path/to/dbfile', $sqlite_db->getDSNInfo('database'));
        $this->assertEqual('644', $sqlite_db->getDSNInfo('mode'));
    }
}
 
?>
