/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.action;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringWriter;
import java.text.NumberFormat;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletResponse;

import jp.sf.fess.Constants;
import jp.sf.fess.form.IndexForm;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.LabelTypeHelper;
import jp.sf.fess.service.SearchService;

import org.apache.commons.lang.StringEscapeUtils;
import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.util.RequestUtil;
import org.seasar.struts.util.ResponseUtil;

public class IndexAction implements Serializable {

    private static final long serialVersionUID = 1L;

    public static final int DEFAULT_PAGE_SIZE = 20;

    public static final long DEFAULT_START_COUNT = 0;

    @ActionForm
    @Resource
    protected IndexForm indexForm;

    @Resource
    protected SearchService searchService;

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected LabelTypeHelper labelTypeHelper;

    public List<Map<String, Object>> documentItems;

    public String pageSize;

    public String currentPageNumber;

    public String allRecordCount;

    public String allPageCount;

    public boolean existNextPage;

    public boolean existPrevPage;

    public String currentStartRecordNumber;

    public String currentEndRecordNumber;

    public List<String> pageNumberList;

    public String execTime;

    public List<Map<String, String>> labelTypeItems;

    @Execute(validator = false)
    public String index() {
        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        // label
        labelTypeItems = labelTypeHelper.getLabelTypeItems();

        return "index.jsp";
    }

    protected String doSearch() {
        if (isMobile()) {
            return "/mobile/?redirect=true";
        }

        if (StringUtil.isBlank(indexForm.query)
                && StringUtil.isBlank(indexForm.labelTypeValue)) {
            // redirect to index page
            indexForm.query = null;
            return "index?redirect=true";
        }

        doSearchInternal();

        return "search.jsp";
    }

    private String doSearchInternal() {
        String query;
        if (StringUtil.isBlank(indexForm.query)) {
            if (StringUtil.isBlank(indexForm.labelTypeValue)) {
                query = "";
            } else {
                query = "label:\"" + indexForm.labelTypeValue + "\"";
            }
        } else {
            if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
                query = indexForm.query + " label:\""
                        + indexForm.labelTypeValue + "\"";
            } else {
                query = indexForm.query;
            }
        }

        // init pager
        if (StringUtil.isBlank(indexForm.start)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            try {
                Long.parseLong(indexForm.start);
            } catch (NumberFormatException e) {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                int num = Integer.parseInt(indexForm.num);
                if (num > 100) {
                    // max page size
                    indexForm.num = "100";
                }
            } catch (NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        long startTime = System.currentTimeMillis();
        documentItems = searchService.selectList(query,
                Integer.parseInt(indexForm.start),
                Integer.parseInt(indexForm.num));
        long execTime = System.currentTimeMillis() - startTime;

        NumberFormat nf = NumberFormat.getInstance(RequestUtil.getRequest()
                .getLocale());
        nf.setMaximumIntegerDigits(2);
        nf.setMaximumFractionDigits(2);
        try {
            this.execTime = nf.format(((double) execTime) / 1000);
        } catch (Exception e) {
        }

        Beans.copy(documentItems, this)
                .includes("pageSize", "currentPageNumber", "allRecordCount",
                        "allPageCount", "existNextPage", "existPrevPage",
                        "currentStartRecordNumber", "currentEndRecordNumber",
                        "pageNumberList").execute();

        // label
        labelTypeItems = labelTypeHelper.getLabelTypeItems();

        return query;
    }

    @Execute(validator = false)
    public String search() {
        return doSearch();
    }

    @Execute(validator = false)
    public String prev() {
        return doMove(-1);
    }

    @Execute(validator = false)
    public String next() {
        return doMove(1);
    }

    @Execute(validator = false)
    public String move() {
        return doMove(0);
    }

    protected String doMove(int move) {
        int pageSize = DEFAULT_PAGE_SIZE;
        if (StringUtil.isBlank(indexForm.num)) {
            indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
        } else {
            try {
                pageSize = Integer.parseInt(indexForm.num);
            } catch (NumberFormatException e) {
                indexForm.num = String.valueOf(DEFAULT_PAGE_SIZE);
            }
        }

        if (StringUtil.isBlank(indexForm.pn)) {
            indexForm.start = String.valueOf(DEFAULT_START_COUNT);
        } else {
            Integer pageNumber = Integer.parseInt(indexForm.pn);
            if (pageNumber != null && pageNumber > 0) {
                pageNumber = pageNumber + move;
                if (pageNumber < 1) {
                    pageNumber = 1;
                }
                indexForm.start = String.valueOf((pageNumber - 1) * pageSize);
            } else {
                indexForm.start = String.valueOf(DEFAULT_START_COUNT);
            }
        }

        return doSearch();
    }

    @Execute(validator = false)
    public String xml() throws IOException {
        switch (getFormatType()) {
        case SEARCH:
            return searchByXml();
        case LABEL:
            return labelByXml();
        default:
            ResponseUtil.getResponse().sendError(
                    HttpServletResponse.SC_NOT_FOUND);
            return null;
        }
    }

    private String searchByXml() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        StringBuilder buf = new StringBuilder(1000);
        String query = null;
        try {
            query = doSearchInternal();
            buf.append("<query>");
            buf.append(StringEscapeUtils.escapeXml(query));
            buf.append("</query>");
            buf.append("<exec-time>");
            buf.append(execTime);
            buf.append("</exec-time>");
            buf.append("<page-size>");
            buf.append(pageSize);
            buf.append("</page-size>");
            buf.append("<page-number>");
            buf.append(currentPageNumber);
            buf.append("</page-number>");
            buf.append("<record-count>");
            buf.append(allRecordCount);
            buf.append("</record-count>");
            buf.append("<page-count>");
            buf.append(allPageCount);
            buf.append("</page-count>");
            buf.append("<result>");
            for (Map<String, Object> document : documentItems) {
                buf.append("<doc>");
                for (Map.Entry<String, Object> entry : document.entrySet()) {
                    if (StringUtil.isNotBlank(entry.getKey())
                            && entry.getValue() != null) {
                        String tagName = StringUtil.decamelize(entry.getKey())
                                .replaceAll("_", "-").toLowerCase();
                        buf.append('<');
                        buf.append(tagName);
                        buf.append('>');
                        buf.append(StringEscapeUtils.escapeXml(entry.getValue()
                                .toString()));
                        buf.append("</");
                        buf.append(tagName);
                        buf.append('>');
                    }
                }
                buf.append("</doc>");
            }
            buf.append("</result>");
        } catch (Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeXmlResponse(status, buf.toString(), errMsg);
        return null;
    }

    private String labelByXml() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        StringBuilder buf = new StringBuilder(255);
        try {
            labelTypeItems = labelTypeHelper.getLabelTypeItems();
            buf.append("<record-count>");
            buf.append(labelTypeItems.size());
            buf.append("</record-count>");
            buf.append("<result>");
            for (Map<String, String> labelMap : labelTypeItems) {
                buf.append("<label>");
                buf.append("<name>");
                buf.append(StringEscapeUtils.escapeXml(labelMap
                        .get(Constants.ITEM_LABEL)));
                buf.append("</name>");
                buf.append("<value>");
                buf.append(StringEscapeUtils.escapeXml(labelMap
                        .get(Constants.ITEM_VALUE)));
                buf.append("</value>");
                buf.append("</label>");
            }
            buf.append("</result>");
        } catch (Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeXmlResponse(status, buf.toString(), errMsg);
        return null;
    }

    private void writeXmlResponse(int status, String body, String errMsg) {
        StringBuilder buf = new StringBuilder(1000);
        buf.append("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
        buf.append("<response>");
        buf.append("<version>");
        buf.append(Constants.WEB_API_VERSION);
        buf.append("</version>");
        buf.append("<status>");
        buf.append(status);
        buf.append("</status>");
        if (status == 0) {
            buf.append(body);
        } else {
            buf.append("<message>");
            buf.append(StringEscapeUtils.escapeXml(errMsg));
            buf.append("</message>");
        }
        buf.append("</response>");
        ResponseUtil.write(buf.toString(), "text/xml", Constants.UTF_8);

    }

    @Execute(validator = false)
    public String json() throws IOException {
        switch (getFormatType()) {
        case SEARCH:
            return searchByJson();
        case LABEL:
            return labelByJson();
        default:
            ResponseUtil.getResponse().sendError(
                    HttpServletResponse.SC_NOT_FOUND);
            return null;
        }
    }

    private String searchByJson() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        String query = null;
        StringBuilder buf = new StringBuilder(1000);
        try {
            query = doSearchInternal();
            buf.append("\"query\":\"");
            buf.append(StringEscapeUtils.escapeJavaScript(query));
            buf.append("\",");
            buf.append("\"execTime\":");
            buf.append(execTime);
            buf.append(',');
            buf.append("\"pageSize\":");
            buf.append(pageSize);
            buf.append(',');
            buf.append("\"pageNumber\":");
            buf.append(currentPageNumber);
            buf.append(',');
            buf.append("\"recordCount\":");
            buf.append(allRecordCount);
            buf.append(',');
            buf.append("\"pageCount\":");
            buf.append(allPageCount);
            if (!documentItems.isEmpty()) {
                buf.append(',');
                buf.append("\"result\":[");
                boolean first1 = true;
                for (Map<String, Object> document : documentItems) {
                    if (!first1) {
                        buf.append(',');
                    } else {
                        first1 = false;
                    }
                    buf.append('{');
                    boolean first2 = true;
                    for (Map.Entry<String, Object> entry : document.entrySet()) {
                        if (StringUtil.isNotBlank(entry.getKey())
                                && entry.getValue() != null) {
                            if (!first2) {
                                buf.append(',');
                            } else {
                                first2 = false;
                            }
                            buf.append('\"');
                            buf.append(escapeJsonString(entry.getKey()));
                            buf.append("\":\"");
                            buf.append(escapeJsonString(entry.getValue()
                                    .toString()));
                            buf.append('\"');
                        }
                    }
                    buf.append('}');
                }
                buf.append(']');
            }
        } catch (Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeJsonResponse(status, buf.toString(), errMsg);

        return null;
    }

    private String labelByJson() {
        int status = 0;
        String errMsg = Constants.EMPTY_STRING;
        StringBuilder buf = new StringBuilder(255);
        try {
            labelTypeItems = labelTypeHelper.getLabelTypeItems();
            buf.append("\"recordCount\":");
            buf.append(labelTypeItems.size());
            if (!labelTypeItems.isEmpty()) {
                buf.append(',');
                buf.append("\"result\":[");
                boolean first1 = true;
                for (Map<String, String> labelMap : labelTypeItems) {
                    if (!first1) {
                        buf.append(',');
                    } else {
                        first1 = false;
                    }
                    buf.append("{\"label\":\"");
                    buf.append(escapeJsonString(labelMap
                            .get(Constants.ITEM_LABEL)));
                    buf.append("\", \"value\":\"");
                    buf.append(escapeJsonString(labelMap
                            .get(Constants.ITEM_VALUE)));
                    buf.append("\"}");
                }
                buf.append(']');
            }
        } catch (Exception e) {
            status = 1;
            errMsg = e.getMessage();
        }

        writeJsonResponse(status, buf.toString(), errMsg);

        return null;
    }

    private void writeJsonResponse(int status, String body, String errMsg) {
        StringBuilder buf = new StringBuilder(1000);
        buf.append("{\"response\":");
        buf.append("{\"version\":");
        buf.append(Constants.WEB_API_VERSION);
        buf.append(',');
        buf.append("\"status\":");
        buf.append(status);
        buf.append(',');
        if (status == 0) {
            buf.append(body);
        } else {
            buf.append("\"message\":\"");
            buf.append(StringEscapeUtils.escapeXml(errMsg));
            buf.append('\"');
        }
        buf.append('}');
        buf.append('}');
        ResponseUtil.write(buf.toString(), "text/javascript+json",
                Constants.UTF_8);

    }

    private String escapeJsonString(String str) {
        if (str == null) {
            return "";
        }

        StringWriter out = new StringWriter(str.length() * 2);
        int sz;
        sz = str.length();
        for (int i = 0; i < sz; i++) {
            char ch = str.charAt(i);

            // handle unicode
            if (ch > 0xfff) {
                out.write("\\u" + hex(ch));
            } else if (ch > 0xff) {
                out.write("\\u0" + hex(ch));
            } else if (ch > 0x7f) {
                out.write("\\u00" + hex(ch));
            } else if (ch < 32) {
                switch (ch) {
                case '\b':
                    out.write('\\');
                    out.write('b');
                    break;
                case '\n':
                    out.write('\\');
                    out.write('n');
                    break;
                case '\t':
                    out.write('\\');
                    out.write('t');
                    break;
                case '\f':
                    out.write('\\');
                    out.write('f');
                    break;
                case '\r':
                    out.write('\\');
                    out.write('r');
                    break;
                default:
                    if (ch > 0xf) {
                        out.write("\\u00" + hex(ch));
                    } else {
                        out.write("\\u000" + hex(ch));
                    }
                    break;
                }
            } else {
                switch (ch) {
                case '"':
                    out.write("\\u0022");
                    break;
                case '\\':
                    out.write("\\u005C");
                    break;
                case '/':
                    out.write("\\u002F");
                    break;
                default:
                    out.write(ch);
                    break;
                }
            }
        }
        return out.toString();
    }

    private String hex(char ch) {
        return Integer.toHexString(ch).toUpperCase();
    }

    protected boolean isMobile() {
        if (browserTypeHelper == null) {
            return false;
        }

        return browserTypeHelper.isMobile();
    }

    public List<Map<String, String>> getLabelTypeItems() {
        return labelTypeItems;
    }

    public boolean isDisplayLabelTypeItems() {
        if (labelTypeItems != null) {
            return !labelTypeItems.isEmpty();
        } else {
            return false;
        }
    }

    public String getDisplayQuery() {
        StringBuilder buf = new StringBuilder();
        buf.append(indexForm.query);
        if (StringUtil.isNotBlank(indexForm.labelTypeValue)) {
            for (Map<String, String> map : labelTypeItems) {
                if (map.get(Constants.ITEM_VALUE).equals(
                        indexForm.labelTypeValue)) {
                    buf.append(' ');
                    buf.append(map.get(Constants.ITEM_LABEL));
                    break;
                }
            }
        }
        return buf.toString();
    }

    private FormatType getFormatType() {
        if (indexForm.type == null) {
            return FormatType.SEARCH;
        }
        String type = indexForm.type.toUpperCase();
        if (FormatType.SEARCH.name().equals(type)) {
            return FormatType.SEARCH;
        } else if (FormatType.LABEL.name().equals(type)) {
            return FormatType.LABEL;
        } else {
            // default
            return FormatType.SEARCH;
        }
    }

    private static enum FormatType {
        SEARCH, LABEL;
    }
}