/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.Document;
import jp.sf.fess.util.FessProperties;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.taglib.S2Functions;
import org.seasar.struts.util.RequestUtil;

public class ViewHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    protected static final String GOOGLE_MOBILE_TRANSCODER_LINK = "http://www.google.co.jp/gwt/n?u=";

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected FessProperties crawlerProperties;

    public int mobileDescriptionLength = 50;

    public int pcDescriptionLength = 200;

    public int mobileTitleLength = 50;

    public int pcTitleLength = 50;

    public String getContentTitle(Document document) {
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileTitleLength;
        } else {
            size = pcTitleLength;
        }

        String title;
        if (StringUtil.isNotBlank(document.title)) {
            title = document.title;
        } else {
            title = document.url;
        }
        return StringUtils.abbreviate(title, size);
    }

    public String getContentDescription(Document document) {
        String[] queries = (String[]) RequestUtil.getRequest().getAttribute(
                Constants.QUERIES);
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileDescriptionLength;
        } else {
            size = pcDescriptionLength;
        }

        if (StringUtil.isNotBlank(document.digest)) {
            return highlight(S2Functions.h(StringUtils.abbreviate(
                    document.digest, size)), queries);
        }

        if (document.cache != null) {
            if (queries != null && queries.length > 0) {
                int idx = document.cache.indexOf(queries[0]) - 10;
                return highlight(S2Functions.h(StringUtils.abbreviate(
                        document.cache, idx, size)), queries);
            } else {
                return S2Functions.h(StringUtils.abbreviate(document.cache,
                        size));
            }
        }
        return "";
    }

    protected String highlight(String content, String[] queries) {
        if (StringUtil.isBlank(content) || queries == null
                || queries.length == 0) {
            return content;
        }
        for (String query : queries) {
            content = content.replaceAll(query, "<em>" + query + "</em>");
        }
        return content;
    }

    public String getUrlLink(Document document) {
        if (browserTypeHelper.isMobile()) {
            String mobileTrasncoder = crawlerProperties.getProperty(
                    Constants.MOBILE_TRANSCODER_PROPERTY,
                    Constants.EMPTY_STRING);
            if (Constants.GOOGLE_MOBILE_TRANSCODER.equals(mobileTrasncoder)) {
                return getGoogleMobileTranscoderLink(document);
            }
        }
        return document.url;
    }

    protected String getGoogleMobileTranscoderLink(Document document) {
        StringBuilder buf = new StringBuilder(255);
        buf.append(GOOGLE_MOBILE_TRANSCODER_LINK);
        try {
            buf.append(URLEncoder.encode(document.url, Constants.UTF_8));
        } catch (UnsupportedEncodingException e) {
            return document.url;
        }
        return buf.toString();
    }
}
