/*
 * Copyright 2009-2012 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.crud.action.admin;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.crud.CommonConstants;
import jp.sf.fess.crud.CrudMessageException;
import jp.sf.fess.crud.util.SAStrutsUtil;
import jp.sf.fess.db.exentity.RequestHeader;
import jp.sf.fess.form.admin.RequestHeaderForm;
import jp.sf.fess.pager.RequestHeaderPager;
import jp.sf.fess.service.RequestHeaderService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.exception.ActionMessagesException;

public class BsRequestHeaderAction implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory
            .getLog(BsRequestHeaderAction.class);

    // for list

    public List<RequestHeader> requestHeaderItems;

    // for edit/confirm/delete

    @ActionForm
    @Resource
    protected RequestHeaderForm requestHeaderForm;

    @Resource
    protected RequestHeaderService requestHeaderService;

    @Resource
    protected RequestHeaderPager requestHeaderPager;

    protected String displayList(final boolean redirect) {
        // page navi
        requestHeaderItems = requestHeaderService
                .getRequestHeaderList(requestHeaderPager);

        // restore from pager
        Beans.copy(requestHeaderPager, requestHeaderForm.searchParams)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        if (redirect) {
            return "index?redirect=true";
        } else {
            return "index.jsp";
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String index() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "list/{pageNumber}")
    public String list() {
        // page navi
        if (StringUtil.isNotBlank(requestHeaderForm.pageNumber)) {
            try {
                requestHeaderPager.setCurrentPageNumber(Integer
                        .parseInt(requestHeaderForm.pageNumber));
            } catch (final NumberFormatException e) {
                if (log.isDebugEnabled()) {
                    log.debug("Invalid value: " + requestHeaderForm.pageNumber,
                            e);
                }
            }
        }

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String search() {
        Beans.copy(requestHeaderForm.searchParams, requestHeaderPager)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String reset() {
        requestHeaderPager.clear();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String back() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String editagain() {
        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "confirmpage/{crudMode}/{id}")
    public String confirmpage() {
        if (requestHeaderForm.crudMode != CommonConstants.CONFIRM_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.CONFIRM_MODE,
                            requestHeaderForm.crudMode });
        }

        loadRequestHeader();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String createpage() {
        // page navi
        requestHeaderForm.initialize();
        requestHeaderForm.crudMode = CommonConstants.CREATE_MODE;

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "editpage/{crudMode}/{id}")
    public String editpage() {
        if (requestHeaderForm.crudMode != CommonConstants.EDIT_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.EDIT_MODE,
                            requestHeaderForm.crudMode });
        }

        loadRequestHeader();

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String editfromconfirm() {
        requestHeaderForm.crudMode = CommonConstants.EDIT_MODE;

        loadRequestHeader();

        return "edit.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromcreate() {
        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromupdate() {
        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "deletepage/{crudMode}/{id}")
    public String deletepage() {
        if (requestHeaderForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            requestHeaderForm.crudMode });
        }

        loadRequestHeader();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String deletefromconfirm() {
        requestHeaderForm.crudMode = CommonConstants.DELETE_MODE;

        loadRequestHeader();

        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String create() {
        try {
            final RequestHeader requestHeader = createRequestHeader();
            requestHeaderService.store(requestHeader);
            SAStrutsUtil.addSessionMessage("success.crud_create_crud_table");

            return displayList(true);
        } catch (final ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (final Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_create_crud_table");
        }
    }

    @Execute(validator = true, input = "edit.jsp")
    public String update() {
        try {
            final RequestHeader requestHeader = createRequestHeader();
            requestHeaderService.store(requestHeader);
            SAStrutsUtil.addSessionMessage("success.crud_update_crud_table");

            return displayList(true);
        } catch (final ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (final Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_update_crud_table");
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String delete() {
        if (requestHeaderForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            requestHeaderForm.crudMode });
        }

        try {
            final RequestHeader requestHeader = requestHeaderService
                    .getRequestHeader(createKeyMap());
            if (requestHeader == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { requestHeaderForm.id });

            }

            requestHeaderService.delete(requestHeader);
            SAStrutsUtil.addSessionMessage("success.crud_delete_crud_table");

            return displayList(true);
        } catch (final ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (final CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (final Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_delete_crud_table");
        }
    }

    protected void loadRequestHeader() {

        final RequestHeader requestHeader = requestHeaderService
                .getRequestHeader(createKeyMap());
        if (requestHeader == null) {
            // throw an exception
            throw new ActionMessagesException(
                    "errors.crud_could_not_find_crud_table",

                    new Object[] { requestHeaderForm.id });

        }

        Beans.copy(requestHeader, requestHeaderForm)
                .excludes("searchParams", "mode").execute();
    }

    protected RequestHeader createRequestHeader() {
        RequestHeader requestHeader;
        if (requestHeaderForm.crudMode == CommonConstants.EDIT_MODE) {
            requestHeader = requestHeaderService
                    .getRequestHeader(createKeyMap());
            if (requestHeader == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { requestHeaderForm.id });

            }
        } else {
            requestHeader = new RequestHeader();
        }
        Beans.copy(requestHeaderForm, requestHeader)
                .excludes("searchParams", "mode").execute();

        return requestHeader;
    }

    protected Map<String, String> createKeyMap() {
        final Map<String, String> keys = new HashMap<String, String>();

        keys.put("id", requestHeaderForm.id);

        return keys;
    }
}
