/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.task;

import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.db.exentity.CrawlingSession;
import jp.sf.fess.helper.CrawlingSessionHelper;
import jp.sf.fess.helper.DatabaseHelper;
import jp.sf.fess.helper.FileSystemIndexHelper;
import jp.sf.fess.helper.PathMappingHelper;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.helper.WebIndexHelper;
import jp.sf.fess.service.CrawlingSessionService;
import jp.sf.fess.service.PathMappingService;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.FessProperties;

import org.apache.commons.lang.time.DateUtils;
import org.seasar.chronos.core.TaskTrigger;
import org.seasar.chronos.core.annotation.task.Task;
import org.seasar.chronos.core.trigger.CCronTrigger;
import org.seasar.framework.container.SingletonS2Container;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Task
public class CrawlTask implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(CrawlTask.class);

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    protected WebIndexHelper webIndexHelper;

    @Resource
    protected FileSystemIndexHelper fileSystemIndexHelper;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected CrawlingSessionService crawlingSessionService;

    @Resource
    protected PathMappingService pathMappingService;

    private CCronTrigger trigger;

    public TaskTrigger getTrigger() {
        if (trigger == null) {
            trigger = new CCronTrigger(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        } else {
            trigger.setExpression(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        }
        return trigger;
    }

    public void setExpression(String cronExpression) {
        if (trigger != null)
            trigger.setExpression(cronExpression);
    }

    public String getExpression() {
        if (trigger != null) {
            trigger.getCronExpression();
        }
        return null;
    }

    public void doExecute() {
        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        if (systemHelper.readyCrawlProcess()) {
            DatabaseHelper databaseHelper = SingletonS2Container
                .getComponent("databaseHelper");
            databaseHelper.optimize();

            PathMappingHelper pathMappingHelper = SingletonS2Container
                    .getComponent("pathMappingHelper");

            // create session id
            SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            String sessionId = sdf.format(new Date());
            systemHelper.setSessionId(sessionId);

            SimpleDateFormat dateFormat = new SimpleDateFormat(
                    "yyyy-MM-dd'T'HH:mm:ss.SSSZ");

            long totalTime = System.currentTimeMillis();
            Map<String, String> infoMap = new HashMap<String, String>();

            try {
                logger.info("Crawler Starting..");
                infoMap.put(Constants.CRAWLER_START_TIME, dateFormat
                        .format(new Date()));

                SolrServerGroup solrServerGroup = solrServerManager
                        .getUpdateSolrServerGroup();

                // store crawling session
                CrawlingSession crawlingSession = new CrawlingSession(sessionId);
                crawlingSessionService.store(crawlingSession);

                // setup path mapping
                pathMappingHelper.setPathMappingList(sessionId,
                        pathMappingService.getPathMappingList());

                // crawl web
                infoMap.put(Constants.WEB_CRAWLER_START_TIME, dateFormat
                        .format(new Date()));
                webIndexHelper.crawl(sessionId, solrServerGroup);
                infoMap.put(Constants.WEB_CRAWLER_END_TIME, dateFormat
                        .format(new Date()));

                // crawl file system
                infoMap.put(Constants.FS_CRAWLER_START_TIME, dateFormat
                        .format(new Date()));
                fileSystemIndexHelper.crawl(sessionId, solrServerGroup);
                infoMap.put(Constants.FS_CRAWLER_END_TIME, dateFormat
                        .format(new Date()));

                // clean up
                String dayForCleanupStr = crawlerProperties.getProperty(
                        Constants.DAY_FOR_CLEANUP_PROPERTY, "1");
                int dayForCleanup = -1;
                try {
                    dayForCleanup = Integer.parseInt(dayForCleanupStr);
                } catch (NumberFormatException e) {
                }
                if (dayForCleanup >= 0) {
                    Date date = DateUtils.addDays(new Date(), -1
                            * dayForCleanup);
                    String[] sessionIds = crawlingSessionService
                            .getSessionIdsBefore(date);
                    SolrServerGroup selectSolrServerGroup = solrServerManager
                            .getSelectSolrServerGroup();
                    for (String sid : sessionIds) {
                        // delete 
                        solrServerGroup.deleteByQuery("segment:" + sid);
                        if (selectSolrServerGroup != solrServerGroup) {
                            selectSolrServerGroup.deleteByQuery("segment:"
                                    + sid);
                        }
                    }
                }

                // optimize
                String optimizeStr = crawlerProperties.getProperty(
                        Constants.OPTIMIZE_PROPERTY, Constants.TRUE);
                String commitStr = crawlerProperties.getProperty(
                        Constants.COMMIT_PROPERTY, Constants.TRUE);
                if (Constants.TRUE.equalsIgnoreCase(optimizeStr)) {
                    infoMap.put(Constants.OPTIMIZE_START_TIME, dateFormat
                            .format(new Date()));
                    long startTime = System.currentTimeMillis();
                    solrServerGroup.optimize();
                    startTime = System.currentTimeMillis() - startTime;
                    infoMap.put(Constants.OPTIMIZE_END_TIME, dateFormat
                            .format(new Date()));
                    infoMap.put(Constants.OPTIMIZE_EXEC_TIME, Long
                            .toString(startTime));
                    if (logger.isInfoEnabled()) {
                        logger.info("[EXEC TIME] index optimize time: "
                                + startTime + "ms");
                    }
                } else if (Constants.TRUE.equalsIgnoreCase(commitStr)) {
                    infoMap.put(Constants.COMMIT_START_TIME, dateFormat
                            .format(new Date()));
                    long startTime = System.currentTimeMillis();
                    solrServerGroup.commit();
                    startTime = System.currentTimeMillis() - startTime;
                    infoMap.put(Constants.COMMIT_END_TIME, dateFormat
                            .format(new Date()));
                    infoMap.put(Constants.COMMIT_EXEC_TIME, Long
                            .toString(startTime));
                    if (logger.isInfoEnabled()) {
                        logger.info("[EXEC TIME] index commit time: "
                                + startTime + "ms");
                    }
                } else {
                    if (logger.isInfoEnabled()) {
                        logger.info("No index commit.");
                    }
                }

                String serverRotationStr = crawlerProperties.getProperty(
                        Constants.SERVER_ROTATION_PROPERTY, Constants.TRUE);
                if (Constants.TRUE.equalsIgnoreCase(serverRotationStr)) {
                    // apply
                    solrServerManager.applyNewServerGroup();
                }

                logger.info("Finished Crawler");
            } catch (Throwable t) {
                logger.warn("Interrupted a crawl task.", t);
            } finally {
                systemHelper.finishCrawlProcess();
                pathMappingHelper.removePathMappingList(sessionId);
                infoMap.put(Constants.CRAWLER_END_TIME, dateFormat
                        .format(new Date()));
                infoMap.put(Constants.CRAWLER_EXEC_TIME, Long.toString(System
                        .currentTimeMillis()
                        - totalTime));
                databaseHelper.optimize();
            }

            CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                    .getComponent("crawlingSessionHelper");
            crawlingSessionHelper.put(sessionId, infoMap);
        } else {
            logger.warn("Crawler is running now.");
        }
    }

    public void catchException(Exception e) {
        logger.error("Failed to execute crawl task.", e);
    }

}
