/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper.impl;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.helper.RoleQueryHelper;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.struts.util.RequestUtil;

public class QueryHelperImpl implements QueryHelper, Serializable {

    private static final long serialVersionUID = 1L;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Binding(bindingType = BindingType.MAY)
    @Resource
    protected RoleQueryHelper roleQueryHelper;

    public String[] responseFields = new String[] { "id", "score", "boost",
            "contentLength", "host", "site", "lastModified", "mimetype",
            "tstamp", "title", "digest", "url" };

    public String[] highlightingFields = new String[] { "digest", "cache" };

    public String[] supportedFields = new String[] { "url", "host", "site",
            "title", "content", "contentLength", "lastModified", "mimetype",
            "label" };

    public String sortPrefix = "sort:";

    public String[] supportedSortFields = new String[] { "tstamp",
            "contentLength", "lastModified" };

    public int highlightSnippetSize = 5;

    protected String shards;

    public boolean useBigram = true;

    /* (non-Javadoc)
     * @see jp.sf.fess.helper.QueryHelper#build(java.lang.String)
     */
    public SearchQuery build(String query) {

        SearchQuery searchQuery = buildQuery(query);
        if (!searchQuery.queryExists()) {
            return searchQuery.query("");
        }

        if (browserTypeHelper == null && roleQueryHelper == null) {
            return searchQuery;
        }

        StringBuilder queryBuf = new StringBuilder(255);
        queryBuf.append('(');
        queryBuf.append(searchQuery.getQuery());
        queryBuf.append(')');

        if (browserTypeHelper != null) {
            queryBuf.append(" AND type:\"");
            queryBuf.append(browserTypeHelper.getBrowserType());
            queryBuf.append('"');
        }

        if (roleQueryHelper != null) {
            List<String> roleList = roleQueryHelper.build();
            if (!roleList.isEmpty()) {
                queryBuf.append(" AND (");
                boolean isFirst = true;
                for (String role : roleList) {
                    if (isFirst) {
                        isFirst = false;
                    } else {
                        queryBuf.append(" OR ");

                    }
                    queryBuf.append("role:\"");
                    queryBuf.append(role);
                    queryBuf.append('"');
                }
                queryBuf.append(')');
            }
        }

        return searchQuery.query(queryBuf.toString());
    }

    protected SearchQuery buildQuery(String query) {
        String[] values = splitQuery(query);
        if (values.length == 0) {
            return new SearchQuery().query("");
        }

        // set queries to request
        RequestUtil.getRequest().setAttribute(Constants.QUERIES, values);

        SearchQuery searchQuery = new SearchQuery();

        StringBuilder queryBuf = new StringBuilder(255);
        StringBuilder titleBuf = new StringBuilder(255);
        StringBuilder contentBuf = new StringBuilder(255);
        for (String value : values) {
            boolean nonPrefix = false;
            // check prefix
            for (String field : supportedFields) {
                String prefix = field + ":";
                if (value.startsWith(prefix)
                        && value.length() != prefix.length()) {
                    if (queryBuf.length() > 0) {
                        queryBuf.append(" AND ");
                    }
                    queryBuf.append(prefix);
                    appendQueryValue(queryBuf, value.substring(prefix.length()));
                    nonPrefix = true;
                    break;
                }
            }

            // sort
            if (value.startsWith(sortPrefix)
                    && value.length() != sortPrefix.length()) {
                String[] sortFieldPairs = value.substring(sortPrefix.length())
                        .split(",");
                for (String sortFieldPairStr : sortFieldPairs) {
                    String[] sortFieldPair = sortFieldPairStr.split("\\.");
                    if (isSupportedSortField(sortFieldPair[0])) {
                        if (sortFieldPair.length == 1) {
                            searchQuery.addSortField(sortFieldPair[0],
                                    Constants.ASC);
                        } else {
                            searchQuery.addSortField(sortFieldPair[0],
                                    sortFieldPair[1]);
                        }
                    }
                }
                continue;
            }

            if (!nonPrefix) {
                // title
                if (titleBuf.length() > 0) {
                    titleBuf.append(" AND ");
                }
                titleBuf.append("title:");
                appendQueryValue(titleBuf, value);

                // content
                if (contentBuf.length() > 0) {
                    contentBuf.append(" AND ");
                }
                contentBuf.append("content:");
                appendQueryValue(contentBuf, value);
            }
        }

        if (titleBuf.length() > 0) {
            boolean append = false;
            if (queryBuf.length() > 0) {
                append = true;
            }
            if (append) {
                queryBuf.append(" AND (");
            }
            queryBuf.append('(').append(titleBuf.toString()).append(") OR (")
                    .append(contentBuf.toString()).append(')');
            if (append) {
                queryBuf.append(')');
            }
        }
        return searchQuery.query(queryBuf.toString());
    }

    private void appendQueryValue(StringBuilder buf, String value) {
        if (useBigram && value.length() == 1
                && !StringUtils.isAsciiPrintable(value)) {
            // if using bigram, add ?
            value = value + "?";
        }

        String tildeValue = null;
        String caretValue = null;
        int pos = value.indexOf('~');
        if (pos > 0) {
            tildeValue = value.substring(pos);
            value = value.substring(0, pos);
        }
        pos = value.indexOf('^');
        if (pos > 0) {
            caretValue = value.substring(pos);
            value = value.substring(0, pos);
        }
        boolean quoted = true;
        if (value.startsWith("[") || value.startsWith("{")) {
            quoted = false;
        }
        if (quoted && !value.contains(" ") && !value.contains("\u3000")) {
            quoted = false;
        }
        if (quoted) {
            buf.append('"');
        }
        buf.append(value);
        if (quoted) {
            buf.append('"');
        }
        if (tildeValue != null) {
            buf.append(tildeValue);
        } else if (caretValue != null) {
            buf.append(caretValue);
        }
    }

    private boolean isSupportedSortField(String field) {
        for (String f : supportedSortFields) {
            if (f.equals(field)) {
                return true;
            }
        }
        return false;
    }

    protected String[] splitQuery(String query) {
        List<String> valueList = new ArrayList<String>();
        StringBuilder buf = new StringBuilder();
        boolean quoted = false;
        boolean squareBracket = false;
        boolean curlyBracket = false;
        for (int i = 0; i < query.length(); i++) {
            char c = query.charAt(i);
            switch (c) {
            case '[':
                buf.append(c);
                if (!quoted && !curlyBracket) {
                    squareBracket = true;
                }
                break;
            case ']':
                buf.append(c);
                squareBracket = false;
                break;
            case '{':
                buf.append(c);
                if (!quoted && !squareBracket) {
                    curlyBracket = true;
                }
                break;
            case '}':
                buf.append(c);
                curlyBracket = false;
                break;
            case '"':
                if (!curlyBracket && !squareBracket) {
                    quoted = !quoted;
                } else {
                    buf.append(c);
                }
                break;
            case ' ':
            case '\u3000':
                if (quoted || curlyBracket || squareBracket) {
                    buf.append(c);
                } else {
                    if (buf.length() > 0) {
                        String str = buf.toString();
                        if (str.startsWith("[") || str.startsWith("{")) {
                            int pos = str.indexOf(" TO ");
                            if (pos < 0) {
                                str = str.replace('[', ' ');
                                str = str.replace(']', ' ');
                                str = str.replace('{', ' ');
                                str = str.replace('}', ' ');
                            }
                        }
                        valueList.add(str.trim());
                    }
                    buf = new StringBuilder();
                }
                break;
            default:
                buf.append(c);
                break;
            }
        }
        if (buf.length() > 0) {
            valueList.add(buf.toString());
        }
        return valueList.toArray(new String[valueList.size()]);
    }

    public String[] getResponseFields() {
        return responseFields;
    }

    public String[] getHighlightingFields() {
        return highlightingFields;
    }

    @Override
    public int getHighlightSnippetSize() {
        return highlightSnippetSize;
    }

    public String getShards() {
        return shards;
    }

    public void setShards(String shards) {
        this.shards = shards;
    }
}
