/*
 * Copyright 2009-2012 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.task;

import java.io.File;
import java.io.Serializable;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.helper.CrawlingSessionHelper;
import jp.sf.fess.helper.SystemHelper;
import jp.sf.fess.service.CrawlingSessionService;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.FessProperties;

import org.seasar.chronos.core.TaskTrigger;
import org.seasar.chronos.core.annotation.task.Task;
import org.seasar.chronos.core.trigger.CCronTrigger;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Task
public class CrawlTask implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(CrawlTask.class);

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected CrawlingSessionService crawlingSessionService;

    private CCronTrigger trigger;

    public TaskTrigger getTrigger() {
        if (trigger == null) {
            trigger = new CCronTrigger(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        } else {
            trigger.setExpression(crawlerProperties.getProperty(
                    Constants.CRON_EXPRESSION_PROPERTY,
                    Constants.DEFAULT_CRON_EXPRESSION));
        }
        return trigger;
    }

    public void setExpression(final String cronExpression) {
        if (trigger != null) {
            trigger.setExpression(cronExpression);
        }
    }

    public String getExpression() {
        if (trigger != null) {
            trigger.getCronExpression();
        }
        return null;
    }

    public void doExecute() {
        final SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        if (systemHelper.readyCrawlProcess()) {
            systemHelper.setForceStop(false);

            // create session id
            final SimpleDateFormat sdf = new SimpleDateFormat("yyyyMMddHHmmss");
            final String sessionId = sdf.format(new Date());
            systemHelper.setSessionId(sessionId);
            // store crawling session
            final CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                    .getComponent("crawlingSessionHelper");
            try {
                crawlingSessionHelper.store(sessionId);
                if (Constants.TRUE.equals(crawlerProperties.getProperty(
                        Constants.SNAPSHOT_REPLICATION_PROPERTY,
                        Constants.FALSE))) {
                    try {
                        // replication
                        doReplication(sessionId);
                    } finally {
                        try {
                            final String dayForCleanupStr = ((FessProperties) SingletonS2Container
                                    .getComponent("crawlerProperties"))
                                    .getProperty(
                                            Constants.DAY_FOR_CLEANUP_PROPERTY,
                                            "1");
                            int dayForCleanup = -1;
                            try {
                                dayForCleanup = Integer
                                        .parseInt(dayForCleanupStr);
                            } catch (final NumberFormatException e) {
                            }
                            crawlingSessionHelper.updateParams(sessionId, null,
                                    dayForCleanup);
                        } catch (final Exception e) {
                            logger.warn(
                                    "Failed to update crawling information.", e);
                        }
                        try {
                            crawlingSessionHelper.store(sessionId);
                        } catch (final Exception e) {
                            logger.warn(
                                    "Failed to store crawling information.", e);
                        }
                    }
                } else {
                    doCrawl(sessionId);
                }
            } finally {
                systemHelper.finishCrawlProcess();
            }
        } else {
            logger.warn("Crawler is running now.");
        }
    }

    public void doReplication(final String sessionId) {
        final String snapshotPath = crawlerProperties.getProperty(
                Constants.SNAPSHOT_PATH_PROPERTY, null);
        if (StringUtil.isBlank(snapshotPath)) {
            if (logger.isWarnEnabled()) {
                logger.warn("A snapshot path is a blank.");
            }
            return;
        }
        // TODO remote snapshot path
        final SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");
        final File snapshotDir = systemHelper.getSnapshotDir(snapshotPath);
        if (!snapshotDir.exists()) {
            if (logger.isWarnEnabled()) {
                logger.warn(snapshotPath + " does not exist.");
            }
            return;
        }

        if (logger.isInfoEnabled()) {
            logger.info("Starting Replication..");
        }

        final SimpleDateFormat dateFormat = new SimpleDateFormat(
                "yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        final long totalTime = System.currentTimeMillis();

        final CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                .getComponent("crawlingSessionHelper");

        crawlingSessionHelper.putToInfoMap(Constants.REPLICATION_START_TIME,
                dateFormat.format(new Date()));
        boolean completed = false;
        try {
            // TODO search lock?

            final SolrServerGroup solrServerGroup = solrServerManager
                    .getUpdateSolrServerGroup();

            // replication
            solrServerGroup.replicate(snapshotDir);

            // commit
            long startTime = System.currentTimeMillis();
            solrServerGroup.commit();
            startTime = System.currentTimeMillis() - startTime;
            if (logger.isInfoEnabled()) {
                logger.info("[EXEC TIME] replication commit time: " + startTime
                        + "ms");
            }

            final String serverRotationStr = crawlerProperties.getProperty(
                    Constants.SERVER_ROTATION_PROPERTY, Constants.TRUE);
            if (Constants.TRUE.equalsIgnoreCase(serverRotationStr)) {
                // apply
                solrServerManager.applyNewServerGroup();
            }

            completed = true;
        } catch (final Throwable t) { // NOPMD
            if (logger.isWarnEnabled()) {
                logger.warn("Interrupted a replication task.", t);
            }
        } finally {
            crawlingSessionHelper.putToInfoMap(Constants.REPLICATION_STATUS,
                    completed ? Constants.T : Constants.F);
            crawlingSessionHelper.putToInfoMap(Constants.REPLICATION_END_TIME,
                    dateFormat.format(new Date()));
            crawlingSessionHelper.putToInfoMap(Constants.REPLICATION_EXEC_TIME,
                    Long.toString(System.currentTimeMillis() - totalTime));
        }
        if (logger.isInfoEnabled()) {
            logger.info("Finished Replication");
        }
    }

    protected void doCrawl(final String sessionId) {
        SingletonS2Container.getComponent(SystemHelper.class).executeCrawler(
                sessionId);
    }

    public void catchException(final Exception e) {
        logger.error("Failed to execute crawl task.", e);
    }

}
