package jp.sf.fess.ds.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.db.exentity.DataCrawlingConfig;
import jp.sf.fess.ds.DataStore;
import jp.sf.fess.ds.IndexUpdateCallback;

import org.seasar.framework.util.OgnlUtil;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class AbstractDataStoreImpl implements DataStore {

    private static final Logger logger = LoggerFactory
            .getLogger(AbstractDataStoreImpl.class);

    public String mimeType = "application/datastore";

    protected boolean alive = true;

    protected Map<String, String> parseParameter(String value) {
        Map<String, String> paramMap = new HashMap<String, String>();
        if (value != null) {
            String[] lines = value.split("[\r\n]");
            for (String line : lines) {
                if (StringUtil.isNotBlank(line)) {
                    int pos = line.indexOf('=');
                    if (pos == 0) {
                        throw new FessSystemException(
                                "Invalid parameter. The key is null.");
                    } else if (pos > 0) {
                        if (pos < line.length()) {
                            paramMap.put(line.substring(0, pos).trim(), line
                                    .substring(pos + 1));
                        } else {
                            paramMap.put(line.substring(0, pos).trim(),
                                    Constants.EMPTY_STRING);
                        }
                    } else {
                        paramMap.put(line.trim(), Constants.EMPTY_STRING);
                    }
                }
            }
        }
        return paramMap;
    }

    protected Map<String, String> parseScript(String value) {
        Map<String, String> scriptMap = new HashMap<String, String>();
        if (value != null) {
            String[] lines = value.split("[\r\n]");
            for (String line : lines) {
                if (StringUtil.isNotBlank(line)) {
                    int pos = line.indexOf('=');
                    if (pos == 0) {
                        throw new FessSystemException(
                                "Invalid parameter. The key is null.");
                    } else if (pos > 0) {
                        if (pos < line.length()) {
                            scriptMap.put(line.substring(0, pos).trim(), line
                                    .substring(pos + 1).trim());
                        } else {
                            scriptMap.put(line.substring(0, pos).trim(),
                                    Constants.EMPTY_STRING);
                        }
                    } else {
                        scriptMap.put(line.trim(), Constants.EMPTY_STRING);
                    }
                }
            }
        }
        return scriptMap;
    }

    public void stop() {
        alive = false;
    }

    public void store(DataCrawlingConfig config, IndexUpdateCallback callback,
            Map<String, String> initParamMap) {
        Map<String, String> paramMap = parseParameter(config
                .getHandlerParameter());
        Map<String, String> scriptMap = parseScript(config.getHandlerScript());

        initParamMap.putAll(paramMap);
        paramMap = initParamMap;

        // default values
        Map<String, Object> defaultDataMap = new HashMap<String, Object>();
        // segment
        defaultDataMap.put("segment", initParamMap.get("sessionId"));
        // tstamp
        defaultDataMap.put("tstamp", Long.toString(new Date().getTime()));
        // boost
        defaultDataMap.put("boost", config.getBoost().toString());
        // type: browserType
        List<String> browserTypeList = new ArrayList<String>();
        for (String browserType : config.getBrowserTypeValues()) {
            browserTypeList.add(browserType);
        }
        defaultDataMap.put("type", browserTypeList);
        // label: labelType
        List<String> labelTypeList = new ArrayList<String>();
        for (String labelType : config.getLabelTypeValues()) {
            labelTypeList.add(labelType);
        }
        defaultDataMap.put("label", labelTypeList);
        // role: roleType
        List<String> roleTypeList = new ArrayList<String>();
        for (String roleType : config.getRoleTypeValues()) {
            roleTypeList.add(roleType);
        }
        defaultDataMap.put("role", roleTypeList);
        // mimetype
        defaultDataMap.put("mimetype", mimeType);
        // title
        // content
        // cache
        // digest
        // host
        // site
        // url
        // anchor
        // contentLength
        // lastModified
        // id

        storeData(callback, paramMap, scriptMap, defaultDataMap);

    }

    protected String convertValue(String template, Map<String, String> paramMap) {
        if (StringUtil.isEmpty(template)) {
            return Constants.EMPTY_STRING;
        }

        try {
            Object exp = OgnlUtil.parseExpression(template);
            Object value = OgnlUtil.getValue(exp, paramMap);
            if (value == null) {
                return Constants.EMPTY_STRING;
            }
            return value.toString();
        } catch (Exception e) {
            logger.warn("Invalid value format: " + template, e);
            return template;
        }
    }

    protected abstract void storeData(IndexUpdateCallback callback,
            Map<String, String> paramMap, Map<String, String> scriptMap,
            Map<String, Object> defaultDataMap);
}
