/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.solr;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.helper.IntervalControlHelper;
import jp.sf.fess.helper.SystemHelper;

import org.apache.solr.common.SolrInputDocument;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.container.annotation.tiger.Binding;
import org.seasar.framework.container.annotation.tiger.BindingType;
import org.seasar.robot.db.cbean.AccessResultCB;
import org.seasar.robot.db.exbhv.AccessResultBhv;
import org.seasar.robot.db.exbhv.AccessResultDataBhv;
import org.seasar.robot.db.exentity.AccessResult;
import org.seasar.robot.dbflute.cbean.PagingResultBean;
import org.seasar.robot.entity.AccessResultData;
import org.seasar.robot.service.DataService;
import org.seasar.robot.service.UrlFilterService;
import org.seasar.robot.service.UrlQueueService;
import org.seasar.robot.transformer.Transformer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexUpdater extends Thread {
    private static final Logger logger = LoggerFactory
            .getLogger(IndexUpdater.class);

    protected List<String> sessionIdList;

    protected SolrServerGroup solrServerGroup;

    @Resource
    protected DataService dataService;

    @Resource
    protected UrlQueueService urlQueueService;

    @Resource
    protected UrlFilterService urlFilterService;

    @Resource
    protected AccessResultBhv accessResultBhv;

    @Resource
    protected AccessResultDataBhv accessResultDataBhv;

    public int maxDocumentCacheSize = 10;

    protected boolean finishCrawling = false;

    public long updateInterval = 60000; // 1 min

    protected long executeTime;

    protected long documentSize;

    protected long commitPerCount = 0;

    protected int maxSolrErrorCount = 0;

    protected int unprocessedDocumentSize = 100;

    protected List<String> finishedSessionIdList = new ArrayList<String>();

    public long commitMarginTime = 10000; // 10ms

    public int maxEmptyListCount = 60; // 1hour

    public boolean threadDump = false;

    public IndexUpdater() {
    }

    public void addFinishedSessionId(String sessionId) {
        synchronized (finishedSessionIdList) {
            finishedSessionIdList.add(sessionId);
        }
    }

    private void deleteBySessionId(String sessionId) {
        try {
            urlFilterService.delete(sessionId);
        } catch (Exception e) {
            logger.warn("Failed to delete url filters: " + sessionId, e);
        }
        try {
            urlQueueService.delete(sessionId);
        } catch (Exception e) {
            logger.warn("Failed to delete url queues: " + sessionId, e);
        }
        try {
            dataService.delete(sessionId);
        } catch (Exception e) {
            logger.warn("Failed to delete data: " + sessionId, e);
        }
    }

    @Override
    public void run() {
        if (dataService == null) {
            throw new FessSystemException("DataService is null.");
        }

        if (logger.isDebugEnabled()) {
            logger.debug("Starting indexUpdater.");
        }

        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");

        executeTime = 0;
        documentSize = 0;

        AccessResultCB cb = new AccessResultCB();
        cb.setupSelect_AccessResultDataAsOne();
        cb.query().setSessionId_InScope(sessionIdList);
        cb.query().addOrderBy_CreateTime_Asc();
        cb.query().setStatus_Equal(org.seasar.robot.Constants.OK_STATUS);
        if (maxDocumentCacheSize <= 0) {
            maxDocumentCacheSize = 1;
        }
        cb.fetchFirst(maxDocumentCacheSize);
        cb.fetchPage(1);

        final List<SolrInputDocument> docList = new ArrayList<SolrInputDocument>();
        final List<org.seasar.robot.entity.AccessResult> accessResultList = new ArrayList<org.seasar.robot.entity.AccessResult>();
        final List<org.seasar.robot.db.exentity.AccessResultData> accessResultDataList = new ArrayList<org.seasar.robot.db.exentity.AccessResultData>();

        long updateTime = System.currentTimeMillis();
        int solrErrorCount = 0;
        int emptyListCount = 0;
        try {
            IntervalControlHelper intervalControlHelper = SingletonS2Container
                    .getComponent("intervalControlHelper");
            while (!finishCrawling || !accessResultList.isEmpty()) {
                try {
                    int sessionIdListSize = finishedSessionIdList.size();
                    intervalControlHelper.setCrawlerRunning(true);

                    updateTime = System.currentTimeMillis() - updateTime;

                    long interval = updateInterval - updateTime;
                    if (interval > 0) {
                        // sleep
                        try {
                            Thread.sleep(interval); // 1 min (default)
                        } catch (InterruptedException e) {
                            logger.warn("Interrupted index update.", e);
                        }
                    }

                    docList.clear();
                    accessResultList.clear();
                    accessResultDataList.clear();

                    if (logger.isDebugEnabled()) {
                        logger.debug("Processing documents in IndexUpdater queue.");
                    }

                    updateTime = System.currentTimeMillis();

                    PagingResultBean<AccessResult> arList = getAccessResultList(cb);
                    if (arList.isEmpty()) {
                        emptyListCount++;
                    } else {
                        emptyListCount = 0; // reset
                    }
                    while (!arList.isEmpty()) {
                        processAccessResults(docList, accessResultList,
                                accessResultDataList, arList);

                        cleanupAccessResults(accessResultList,
                                accessResultDataList);

                        if (logger.isDebugEnabled()) {
                            logger.debug("Getting documents in IndexUpdater queue.");
                        }
                        arList = getAccessResultList(cb);
                    }

                    if (!docList.isEmpty()) {
                        sendDocuments(docList);
                    }

                    synchronized (finishedSessionIdList) {
                        if (sessionIdListSize != 0
                                && sessionIdListSize == finishedSessionIdList
                                        .size()) {
                            cleanupFinishedSessionData();
                        }
                    }
                    executeTime += System.currentTimeMillis() - updateTime;

                    if (logger.isDebugEnabled()) {
                        logger.debug("Processed documents in IndexUpdater queue.");
                    }
                } catch (FessSolrException e) {
                    if (solrErrorCount > maxSolrErrorCount) {
                        throw e;
                    }
                    solrErrorCount++;
                    logger.warn(
                            "Failed to access a solr group. Retry to access.. "
                                    + solrErrorCount, e);
                } finally {
                    if (systemHelper.isForceStop()) {
                        finishCrawling = true;
                        if (logger.isDebugEnabled()) {
                            logger.debug("Stopped indexUpdater.");
                        }
                    }
                }

                if (emptyListCount >= maxEmptyListCount) {
                    if (logger.isInfoEnabled()) {
                        logger.info("Terminating indexUpdater. "
                                + "emptyListCount is over " + maxEmptyListCount
                                + ".");
                    }
                    // terminate crawling
                    finishCrawling = true;
                    systemHelper.setForceStop(true);
                    if (threadDump) {
                        printThreadDump();
                    }

                }
            }

            intervalControlHelper.setCrawlerRunning(true);

            if (logger.isDebugEnabled()) {
                logger.debug("Finished indexUpdater.");
            }
        } catch (Throwable t) {
            logger.error("IndexUpdater is terminated.", t);
            if (systemHelper.isCrawlProcessRunning()) {
                systemHelper.setForceStop(true);
            }
        }

        if (logger.isInfoEnabled()) {
            logger.info("[EXEC TIME] index update time: " + executeTime + "ms");
        }

    }

    private void printThreadDump() {
        for (Map.Entry<Thread, StackTraceElement[]> entry : Thread
                .getAllStackTraces().entrySet()) {
            logger.info("Thread: " + entry.getKey());
            StackTraceElement[] trace = entry.getValue();
            for (int i = 0; i < trace.length; i++)
                logger.info("\tat " + trace[i]);
        }
    }

    private void processAccessResults(
            final List<SolrInputDocument> docList,
            final List<org.seasar.robot.entity.AccessResult> accessResultList,
            final List<org.seasar.robot.db.exentity.AccessResultData> accessResultDataList,
            PagingResultBean<AccessResult> arList) {
        for (AccessResult accessResult : arList) {
            if (logger.isDebugEnabled()) {
                logger.debug("Indexing " + accessResult.getUrl());
            }
            accessResult.setStatus(Constants.DONE_STATUS);
            accessResultList.add(accessResult);

            if (accessResult.getHttpStatusCode() != 200) {
                // invalid page
                if (logger.isDebugEnabled()) {
                    logger.debug("Skipped. The response code is "
                            + accessResult.getHttpStatusCode() + ".");
                }
                continue;
            }

            AccessResultData accessResultData = accessResult
                    .getAccessResultData();
            if (accessResultData != null) {
                accessResult.setAccessResultData(null);
                accessResultDataList
                        .add((org.seasar.robot.db.exentity.AccessResultData) accessResultData);
                try {
                    Transformer transformer = SingletonS2Container
                            .getComponent(accessResultData.getTransformerName());
                    if (transformer == null) {
                        // no transformer
                        logger.warn("No transformer: "
                                + accessResultData.getTransformerName());
                        continue;
                    }
                    Map<String, Object> map = (Map<String, Object>) transformer
                            .getData(accessResultData);
                    if (map.isEmpty()) {
                        // no transformer
                        logger.warn("No data: " + accessResult.getUrl());
                        continue;
                    }

                    if (Constants.FALSE.equals(map
                            .get(Constants.INDEXING_TARGET))) {
                        if (logger.isDebugEnabled()) {
                            logger.debug("Skipped. "
                                    + "This document is not a index target. ");
                        }
                        continue;
                    } else {
                        map.remove(Constants.INDEXING_TARGET);
                    }

                    SolrInputDocument doc = new SolrInputDocument();
                    // add data
                    for (Map.Entry<String, Object> entry : map.entrySet()) {
                        if ("boost".equals(entry.getKey())) {
                            // boost
                            float documentBoost = Float.valueOf(entry
                                    .getValue().toString());
                            doc.setDocumentBoost(documentBoost);
                            if (logger.isDebugEnabled()) {
                                logger.debug("Set a document boost ("
                                        + documentBoost + ").");
                            }
                        }
                        doc.addField(entry.getKey(), entry.getValue());
                    }

                    docList.add(doc);
                    if (logger.isDebugEnabled()) {
                        logger.debug("Added the document. "
                                + "The number of a document cache is "
                                + docList.size() + ".");
                    }

                    if (docList.size() > maxDocumentCacheSize) {
                        sendDocuments(docList);
                    }
                    documentSize++;
                    // commit
                    if (commitPerCount > 0
                            && documentSize % commitPerCount == 0) {
                        if (!docList.isEmpty()) {
                            sendDocuments(docList);
                        }
                        commitDocuments();
                    }
                    if (logger.isDebugEnabled()) {
                        logger.debug("The number of an added document is "
                                + documentSize + ".");
                    }
                } catch (FessSolrException e) {
                    throw e;
                } catch (Exception e) {
                    logger.warn(
                            "Could not add a doc: " + accessResult.getUrl(), e);
                }
            } else {
                if (logger.isDebugEnabled()) {
                    logger.debug("Skipped. No content. ");
                }
            }

        }
    }

    private void cleanupAccessResults(
            final List<org.seasar.robot.entity.AccessResult> accessResultList,
            final List<org.seasar.robot.db.exentity.AccessResultData> accessResultDataList) {
        if (!accessResultList.isEmpty()) {
            long execTime = System.currentTimeMillis();
            int size = accessResultList.size();
            dataService.update(accessResultList);
            accessResultList.clear();
            if (logger.isDebugEnabled()) {
                logger.debug("Updated " + size
                        + " access results. The execution time is "
                        + (System.currentTimeMillis() - execTime) + "ms.");
            }
        }

        if (!accessResultDataList.isEmpty()) {
            long execTime = System.currentTimeMillis();
            int size = accessResultDataList.size();
            // clean up content
            accessResultDataBhv.batchDelete(accessResultDataList);
            accessResultDataList.clear();
            if (logger.isDebugEnabled()) {
                logger.debug("Deleted " + size
                        + " access result data. The execution time is "
                        + (System.currentTimeMillis() - execTime) + "ms.");
            }
        }
    }

    private PagingResultBean<AccessResult> getAccessResultList(AccessResultCB cb) {
        long execTime = System.currentTimeMillis();
        PagingResultBean<AccessResult> arList = accessResultBhv.selectPage(cb);
        if (!arList.isEmpty()) {
            for (AccessResult ar : arList.toArray(new AccessResult[arList
                    .size()])) {
                if (ar.getCreateTime().getTime() > execTime - commitMarginTime) {
                    arList.remove(ar);
                }
            }
        }
        if (logger.isInfoEnabled()) {
            logger.info("The number of a crawled document is "
                    + arList.getAllRecordCount() + ". The processing size is "
                    + arList.size() + ". The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
        if (arList.getAllRecordCount() > unprocessedDocumentSize) {
            if (logger.isInfoEnabled()) {
                logger.info("Stopped all crawler threads. " + " You have "
                        + arList.getAllRecordCount() + " (>"
                        + unprocessedDocumentSize + ") "
                        + " unprocessed documents.");
            }
            IntervalControlHelper intervalControlHelper = SingletonS2Container
                    .getComponent("intervalControlHelper");
            intervalControlHelper.setCrawlerRunning(false);
        }
        return arList;
    }

    private void cleanupFinishedSessionData() {
        long execTime = System.currentTimeMillis();
        // cleanup
        for (String sessionId : finishedSessionIdList) {
            long execTime2 = System.currentTimeMillis();
            if (logger.isDebugEnabled()) {
                logger.debug("Deleting document data: " + sessionId);
            }
            deleteBySessionId(sessionId);
            if (logger.isDebugEnabled()) {
                logger.debug("Deleted " + sessionId
                        + " documents. The execution time is "
                        + (System.currentTimeMillis() - execTime2) + "ms.");
            }
        }
        finishedSessionIdList.clear();

        if (logger.isInfoEnabled()) {
            logger.info("Deleted completed document data. "
                    + "The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
    }

    private void commitDocuments() {
        long execTime = System.currentTimeMillis();
        if (logger.isInfoEnabled()) {
            logger.info("Committing documents. ");
        }
        solrServerGroup.commit();
        if (logger.isInfoEnabled()) {
            logger.info("Committed documents. The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
    }

    private void sendDocuments(final List<SolrInputDocument> docList) {
        long execTime = System.currentTimeMillis();
        if (logger.isInfoEnabled()) {
            logger.info("Sending " + docList.size() + " document to a server.");
        }
        solrServerGroup.add(docList);
        if (logger.isInfoEnabled()) {
            logger.info("Sent " + docList.size()
                    + " documents. The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
        docList.clear();
    }

    public long getExecuteTime() {
        return executeTime;
    }

    public List<String> getSessionIdList() {
        return sessionIdList;
    }

    public void setSessionIdList(List<String> sessionIdList) {
        this.sessionIdList = sessionIdList;
    }

    public SolrServerGroup getSolrServerGroup() {
        return solrServerGroup;
    }

    public void setSolrServerGroup(SolrServerGroup solrServerGroup) {
        this.solrServerGroup = solrServerGroup;
    }

    public void setFinishCrawling(boolean finishCrawling) {
        this.finishCrawling = finishCrawling;
    }

    public long getDocumentSize() {
        return documentSize;
    }

    public void setCommitPerCount(long commitPerCount) {
        this.commitPerCount = commitPerCount;
    }

    @Binding(bindingType = BindingType.MAY)
    @Override
    public void setUncaughtExceptionHandler(UncaughtExceptionHandler eh) {
        super.setUncaughtExceptionHandler(eh);
    }

    @Binding(bindingType = BindingType.MAY)
    public static void setDefaultUncaughtExceptionHandler(
            UncaughtExceptionHandler eh) {
        Thread.setDefaultUncaughtExceptionHandler(eh);
    }

    public void setMaxSolrErrorCount(int maxSolrErrorCount) {
        this.maxSolrErrorCount = maxSolrErrorCount;
    }

    public void setUnprocessedDocumentSize(int unprocessedDocumentSize) {
        this.unprocessedDocumentSize = unprocessedDocumentSize;
    }
}
