/*
 * Copyright (C) 2008-2011 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/*
 * Copyright (C) 2011 OgakiSoft
 * 
 * Added method getAllStrokes(), save_gesture(), load_gesture().
 */

package ogakisoft.android.gesture.reform;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import ogakisoft.android.util.LOG;
import android.graphics.RectF;

/**
 * Utility Class for Gesture Sampling
 * 
 * @author The Android Open Source Project
 * @author noritoshi ogaki
 * @version 1.1
 */
public final class GestureUtilities {
    private static final float[] EMPTY_ARRAY = new float[0];

    private static final int TEMPORAL_SAMPLING_RATE = 8; // 16

    private static final String TAG = "GestureUtilities";

    private GestureUtilities() {
    }

    static float[] spatialSampling(Gesture gesture, int sampleMatrixDimension) {
        final float targetPatchSize = sampleMatrixDimension - 1;
        // edge inclusive
        final float[] sample = new float[sampleMatrixDimension * sampleMatrixDimension];
        final RectF rect = gesture.getBoundingBox();
        final float sx = targetPatchSize / rect.width();
        final float sy = targetPatchSize / rect.height();
        final float scale = (sx < sy) ? sx : sy;
        final float preDx = -rect.centerX();
        final float preDy = -rect.centerY();
        final float postDx = targetPatchSize / 2;
        final float postDy = targetPatchSize / 2;
        final List<GestureStroke> strokes = gesture.getStrokes();
        final int count = strokes.size();
        int size = 0;
        float xpos = 0;
        float ypos = 0;
        GestureStroke stroke = null;
        float[] strokepoints = null;
        float[] pts = null;
        float segmentEndX = 0f;
        float segmentEndY = 0f;
        float segmentStartX = 0f;
        float segmentStartY = 0f;
        float slope = 0f;
        float invertSlope = 0f;

        Arrays.fill(sample, 0);
        for (int index = 0; index < count; index++) {
            stroke = strokes.get(index);
            strokepoints = stroke.points;
            size = strokepoints.length;
            pts = new float[size];

            for (int i = 0; i < size; i += 2) {
                pts[i] = (strokepoints[i] + preDx) * scale + postDx;
                pts[i + 1] = (strokepoints[i + 1] + preDy) * scale + postDy;
            }
            segmentEndX = -1;
            segmentEndY = -1;

            for (int i = 0; i < size; i += 2) {
                segmentStartX = (pts[i] < 0) ? 0 : pts[i];
                segmentStartY = (pts[i + 1] < 0) ? 0 : pts[i + 1];
                if (segmentStartX > targetPatchSize) {
                    segmentStartX = targetPatchSize;
                }
                if (segmentStartY > targetPatchSize) {
                    segmentStartY = targetPatchSize;
                }
                plot(segmentStartX, segmentStartY, sample, sampleMatrixDimension);
                if (segmentEndX != -1) {
                    // evaluate horizontally
                    if (segmentEndX > segmentStartX) {
                        xpos = (float) Math.ceil(segmentStartX);
                        slope = (segmentEndY - segmentStartY) / (segmentEndX - segmentStartX);
                        for (; xpos < segmentEndX; xpos++) {
                            ypos = slope * (xpos - segmentStartX) + segmentStartY;
                            plot(xpos, ypos, sample, sampleMatrixDimension);
                        }
                    } else if (segmentEndX < segmentStartX) {
                        xpos = (float) Math.ceil(segmentEndX);
                        slope = (segmentEndY - segmentStartY) / (segmentEndX - segmentStartX);
                        for (; xpos < segmentStartX; xpos++) {
                            ypos = slope * (xpos - segmentStartX) + segmentStartY;
                            plot(xpos, ypos, sample, sampleMatrixDimension);
                        }
                    }

                    // evaluating vertically
                    if (segmentEndY > segmentStartY) {
                        ypos = (float) Math.ceil(segmentStartY);
                        invertSlope = (segmentEndX - segmentStartX) / (segmentEndY - segmentStartY);
                        for (; ypos < segmentEndY; ypos++) {
                            xpos = invertSlope * (ypos - segmentStartY) + segmentStartX;
                            plot(xpos, ypos, sample, sampleMatrixDimension);
                        }
                    } else if (segmentEndY < segmentStartY) {
                        ypos = (float) Math.ceil(segmentEndY);
                        invertSlope = (segmentEndX - segmentStartX) / (segmentEndY - segmentStartY);
                        for (; ypos < segmentStartY; ypos++) {
                            xpos = invertSlope * (ypos - segmentStartY) + segmentStartX;
                            plot(xpos, ypos, sample, sampleMatrixDimension);
                        }
                    }
                }

                segmentEndX = segmentStartX;
                segmentEndY = segmentStartY;
            }
        }
        return sample;
    }

    private static void plot(float x, float y, float[] sample, int sampleSize) {
        final float xx = (x < 0) ? 0 : x;
        final float yy = (y < 0) ? 0 : y;
        final int xFloor = (int) Math.floor(xx);
        final int xCeiling = (int) Math.ceil(xx);
        final int yFloor = (int) Math.floor(yy);
        final int yCeiling = (int) Math.ceil(yy);

        // if it's an integer
        if (xx == xFloor && yy == yFloor) {
            final int index = yCeiling * sampleSize + xCeiling;
            if (sample[index] < 1) {
                sample[index] = 1;
            }
        } else {
            final double topLeft = Math.sqrt(Math.pow(xFloor - xx, 2) + Math.pow(yFloor - yy, 2));
            final double topRight = Math
                    .sqrt(Math.pow(xCeiling - xx, 2) + Math.pow(yFloor - yy, 2));
            final double btmLeft = Math.sqrt(Math.pow(xFloor - xx, 2) + Math.pow(yCeiling - yy, 2));
            final double btmRight = Math.sqrt(Math.pow(xCeiling - xx, 2)
                    + Math.pow(yCeiling - yy, 2));
            final double sum = topLeft + topRight + btmLeft + btmRight;

            double value = topLeft / sum;
            int index = yFloor * sampleSize + xFloor;
            if (value > sample[index]) {
                sample[index] = (float) value;
            }
            value = topRight / sum;
            index = yFloor * sampleSize + xCeiling;
            if (value > sample[index]) {
                sample[index] = (float) value;
            }
            value = btmLeft / sum;
            index = yCeiling * sampleSize + xFloor;
            if (value > sample[index]) {
                sample[index] = (float) value;
            }
            value = btmRight / sum;
            index = yCeiling * sampleSize + xCeiling;
            if (value > sample[index]) {
                sample[index] = (float) value;
            }
        }
    }

    /**
     * Featurize a stroke into a vector of a given number of elements
     * 
     * @param stroke
     * @param sampleSize
     * @return a float array
     */
    public static float[] temporalSampling(GestureStroke stroke, int sampleSize) {
        final float increment = stroke.length / (sampleSize - 1);
        final int vectorLength = sampleSize * 2;
        final float[] vector = new float[vectorLength];
        float distanceSoFar = 0;
        final float[] pts = stroke.points;
        float lstPointX = pts[0];
        float lstPointY = pts[1];
        int index = 0;
        float currentPointX = Float.MIN_VALUE;
        float currentPointY = Float.MIN_VALUE;
        int i = 0;
        int count = 0;
        float deltaX = 0f;
        float deltaY = 0f;
        float distance = 0f;
        float ratio = 0f;
        float nx = 0f;
        float ny = 0f;

        // ogaki-->
        if (null == stroke || stroke.points.length < 2) {
            return EMPTY_ARRAY;
        }
        count = pts.length / 2;
        // <--ogaki
        vector[index] = lstPointX;
        index++;
        vector[index] = lstPointY;
        index++;
        while (i < count) {
            // if (currentPointX == Float.MIN_VALUE) {
            if (0 == (currentPointX - Float.MIN_VALUE)) {
                i++;
                if (i >= count) {
                    break;
                }
                currentPointX = pts[i * 2];
                currentPointY = pts[i * 2 + 1];
            }
            deltaX = currentPointX - lstPointX;
            deltaY = currentPointY - lstPointY;
            distance = (float) Math.sqrt(deltaX * deltaX + deltaY * deltaY);
            if (distanceSoFar + distance >= increment) {
                ratio = (increment - distanceSoFar) / distance;
                nx = lstPointX + ratio * deltaX;
                ny = lstPointY + ratio * deltaY;
                vector[index] = nx;
                index++;
                vector[index] = ny;
                index++;
                lstPointX = nx;
                lstPointY = ny;
                distanceSoFar = 0;
            } else {
                lstPointX = currentPointX;
                lstPointY = currentPointY;
                currentPointX = Float.MIN_VALUE;
                currentPointY = Float.MIN_VALUE;
                distanceSoFar += distance;
            }
        }

        for (i = index; i < vectorLength; i += 2) {
            vector[i] = lstPointX;
            vector[i + 1] = lstPointY;
        }
        return vector;
    }

    /**
     * Calculate the centroid
     * 
     * @param points
     * @return the centroid
     */
    static float[] computeCentroid(float[] points) {
        float centerX = 0;
        float centerY = 0;
        final float[] center = new float[2];
        final int count = points.length;
        for (int i = 0; i < count; i++) {
            centerX += points[i];
            i++;
            centerY += points[i];
        }
        center[0] = 2 * centerX / count;
        center[1] = 2 * centerY / count;

        return center;
    }

    /**
     * calculate the variance-covariance matrix, treat each point as a sample
     * 
     * @param points
     * @return the covariance matrix
     */
    private static double[][] computeCoVariance(float[] points) {
        final double[][] array = new double[2][2];
        final int count = points.length;
        float x = 0f;
        float y = 0f;
        array[0][0] = 0;
        array[0][1] = 0;
        array[1][0] = 0;
        array[1][1] = 0;
        for (int i = 0; i < count; i++) {
            x = points[i];
            i++;
            y = points[i];
            array[0][0] += x * x;
            array[0][1] += x * y;
            array[1][0] = array[0][1];
            array[1][1] += y * y;
        }
        array[0][0] /= (count / 2f);
        array[0][1] /= (count / 2f);
        array[1][0] /= (count / 2f);
        array[1][1] /= (count / 2f);

        return array;
    }

    static float computeTotalLength(float[] points) {
        float sum = 0;
        final int count = points.length - 4;
        float dx = 0f;
        float dy = 0f;
        for (int i = 0; i < count; i += 2) {
            dx = points[i + 2] - points[i];
            dy = points[i + 3] - points[i + 1];
            sum += Math.sqrt(dx * dx + dy * dy);
        }
        return sum;
    }

    static double computeStraightness(float[] points) {
        final float totalLen = computeTotalLength(points);
        final float dx = points[2] - points[0];
        final float dy = points[3] - points[1];
        return Math.sqrt(dx * dx + dy * dy) / totalLen;
    }

    static double computeStraightness(float[] points, float totalLen) {
        final float dx = points[2] - points[0];
        final float dy = points[3] - points[1];
        return Math.sqrt(dx * dx + dy * dy) / totalLen;
    }

    /**
     * Calculate the squared Euclidean distance between two vectors
     * 
     * @param vector1
     * @param vector2
     * @return the distance
     */
    static double squaredEuclideanDistance(float[] vector1, float[] vector2) {
        double squaredDistance = 0;
        final int size = vector1.length;
        float difference = 0f;
        for (int i = 0; i < size; i++) {
            difference = vector1[i] - vector2[i];
            squaredDistance += difference * difference;
        }
        return squaredDistance / size;
    }

    /**
     * Calculate the cosine distance between two instances
     * 
     * @param vector1
     * @param vector2
     * @return the distance between 0 and Math.PI
     */
    static double cosineDistance(float[] vector1, float[] vector2) {
        float sum = 0;
        final int len = vector1.length;
        for (int i = 0; i < len; i++) {
            sum += vector1[i] * vector2[i];
        }
        return Math.acos(sum);
    }

    static OrientedBoundingBox computeOrientedBoundingBox(List<GesturePoint> pts) {
        final GestureStroke stroke = new GestureStroke(pts);
        final float[] points = temporalSampling(stroke, TEMPORAL_SAMPLING_RATE);
        return computeOrientedBoundingBox(points);
    }

    static OrientedBoundingBox computeOrientedBoundingBox(float[] points) {
        final float[] meanVector = computeCentroid(points);
        return computeOrientedBoundingBox(points, meanVector);
    }

    static OrientedBoundingBox computeOrientedBoundingBox(float[] points, float[] centroid) {
        final double[][] array = computeCoVariance(points);
        final double[] targetVector = computeOrientation(array);
        float angle = 0f;
        float minx = Float.MAX_VALUE;
        float miny = Float.MAX_VALUE;
        float maxx = Float.MIN_VALUE;
        float maxy = Float.MIN_VALUE;
        final int count = points.length;

        translate(points, -centroid[0], -centroid[1]);
        if (0 == targetVector[0] && 0 == targetVector[1]) {
            angle = (float) -Math.PI / 2;
        } else { // -PI<alpha<PI
            angle = (float) Math.atan2(targetVector[1], targetVector[0]);
            rotate(points, -angle);
        }
        for (int i = 0; i < count; i++) {
            if (points[i] < minx) {
                minx = points[i];
            }
            if (points[i] > maxx) {
                maxx = points[i];
            }
            i++;
            if (points[i] < miny) {
                miny = points[i];
            }
            if (points[i] > maxy) {
                maxy = points[i];
            }
        }

        return new OrientedBoundingBox((float) (angle * 180 / Math.PI), centroid[0], centroid[1],
                maxx - minx, maxy - miny);
    }

    private static double[] computeOrientation(double[][] covarianceMatrix) {
        final double[] targetVector = new double[2];
        final double aa = -covarianceMatrix[0][0] - covarianceMatrix[1][1];
        final double bb = covarianceMatrix[0][0] * covarianceMatrix[1][1] - covarianceMatrix[0][1]
                * covarianceMatrix[1][0];
        double value = aa / 2;
        final double rightside = Math.sqrt(Math.pow(value, 2) - bb);
        final double lambda1 = -value + rightside;
        final double lambda2 = -value - rightside;

        if (0 == covarianceMatrix[0][1] || 0 == covarianceMatrix[1][0]) {
            targetVector[0] = 1;
            targetVector[1] = 0;
        }
        // if (lambda1 == lambda2) {
        if (0 == Math.abs(lambda1 - lambda2)) {
            targetVector[0] = 0;
            targetVector[1] = 0;
        } else {
            final double lambda = (lambda1 > lambda2) ? lambda1 : lambda2;
            targetVector[0] = 1;
            targetVector[1] = (lambda - covarianceMatrix[0][0]) / covarianceMatrix[0][1];
        }
        return targetVector;
    }

    static float[] rotate(float[] points, double angle) {
        final double cos = Math.cos(angle);
        final double sin = Math.sin(angle);
        final int size = points.length;
        float x = 0f;
        float y = 0f;
        for (int i = 0; i < size; i += 2) {
            x = (float) (points[i] * cos - points[i + 1] * sin);
            y = (float) (points[i] * sin + points[i + 1] * cos);
            points[i] = x;
            points[i + 1] = y;
        }
        return points;
    }

    static float[] translate(float[] points, float dx, float dy) {
        final int size = points.length;
        for (int i = 0; i < size; i += 2) {
            points[i] += dx;
            points[i + 1] += dy;
        }
        return points;
    }

    static float[] scale(float[] points, float sx, float sy) {
        final int size = points.length;
        for (int i = 0; i < size; i += 2) {
            points[i] *= sx;
            points[i + 1] *= sy;
        }
        return points;
    }

    /**
     * Get all strokes as one stroke
     * 
     * @param gesture
     * @return GestureStroke
     */
    public static GestureStroke getAllStroke(Gesture gesture) {
        final List<GesturePoint> points = new ArrayList<GesturePoint>();
        if (null != gesture && null != gesture.getStrokes()) {
            final List<GestureStroke> strokes = gesture.getStrokes();
            final int count = strokes.size();
            int size = 0;
            GestureStroke stroke = null;
            float[] strokepoints = null;
            for (int index = 0; index < count; index++) {
                stroke = strokes.get(index);
                strokepoints = stroke.points;
                size = strokepoints.length;
                for (int i = 0; i < size; i += 2) {
                    points.add(new GesturePoint(strokepoints[i], strokepoints[i + 1], 0));
                }
            }
        }
        return new GestureStroke(points);
    }

    public static float compareFeature(Feature f1, Feature f2) {
        float[] results = {
                f1.strokes_count, f2.strokes_count, f1.vertical_straight, f2.vertical_straight,
                f1.horizontal_straight, f2.horizontal_straight, f1.right_angle, f2.right_angle,
                f1.slant_right_up, f2.slant_right_up, f1.slant_right_down, f2.slant_right_down,
                f1.slant_left_up, f2.slant_left_up, f1.slant_left_down, f2.slant_left_down
        };
        int count = results.length;
        float r = 0f;
        for (int i = 0; i < count; i += 2) {
            if (results[i] != 0 && results[i + 1] != 0) {
                r *= results[i] / results[i + 1];
                // LOG.d(TAG, "compareFeature {0,number,#},{1,number,#}",
                // results[i], results[i + 1]);
            }
        }
        // LOG.d(TAG, "compareFeature {0,number,#.###}",r);
        return r;
    }

    public static Feature computeFeature(Gesture gesture) {
        final List<GestureStroke> strokes = gesture.getStrokes();
        final int count = strokes.size();
        final Feature f = new Feature();
        f.strokes_count = count;
        double[] temp;
        GestureStroke stroke;
        for (int i = 0; i < count; i++) {
            LOG.d(TAG, "--- stroke {0,number,#} start ---", i);
            stroke = strokes.get(i);
            // temp = computeStraightness(stroke);
            // if (temp[0] >= 0.6f && temp[1] == 0f) {
            // f.vertical_straight++;
            // } else if (temp[0] == 0f && temp[1] >= 0.6f) {
            // f.horizontal_straight++;
            // } else if (temp[0] >= 0.4f && temp[1] >= 0.4f) {
            // f.right_angle++;
            // }
            // temp = computeInclination(stroke);
            // if (temp[0] >= 0.5f && temp[1] < 0f && temp[1] > -80f) {
            // f.slant_right_up++;
            // } else if (temp[0] >= 0.5f && temp[1] > 0f && temp[1] < 80f) {
            // f.slant_right_down++;
            // } else if (temp[0] >= 0.5f && temp[1] < -100f & temp[1] > -170f)
            // {
            // f.slant_left_up++;
            // } else if (temp[0] >= 0.5f && temp[1] > 100f && temp[1] < 170f) {
            // f.slant_left_down++;
            // }
            temp = computeArguments(stroke);
            if (temp[0] > 250d) {
                LOG.d(TAG, "circle {0,number,#.###}", temp[0]);
            } else if (temp[1] >= 0d && temp[1] < 20d) {
                LOG.d(TAG, "straight {0,number,#.###}", temp[1]);
            }
            LOG.d(TAG, "--- stroke {0,number,#} end ---", i);
        }
        return f;
    }

    private static double[] computeArguments(GestureStroke stroke) {
        final int length = stroke.points.length;
        int last = -1, current = 0, next = -1;
        double vx1, vy1, vx2, vy2, len1, len2;
        double value, sum = 0, min = Double.MAX_VALUE, max = Double.MIN_VALUE;
        int count = 0;
        double[] array = new double[length / 2];
        for (int i = 0; i < length; i += 2) {
            current = i;
            last = i - 2;
            next = i + 2;
            if (last < 0 || next >= length)
                continue;
            vx1 = stroke.points[current] - stroke.points[last];
            vy1 = stroke.points[current + 1] - stroke.points[last + 1];
            vx2 = stroke.points[next] - stroke.points[current];
            vy2 = stroke.points[next + 1] - stroke.points[current + 1];
            len1 = Math.sqrt(vx1 * vx1 + vy1 * vy1);
            len2 = Math.sqrt(vx2 * vx2 + vy2 * vy2);
            value = (vx1 * vx2 + vy1 * vy2) / len1 / len2;
            if (len1 == 0 || len2 == 0)
                value = 1.0;
            else {
                if (value > 1.0)
                    value = 1.0;
                else if (value < -1.0)
                    value = -1.0;
            }
            double rad = Math.acos(value);
            double deg = (rad * 180) / Math.PI;
            LOG.d(TAG, "computeArg: {0,number,#.###}, {1,number,#.###}", rad, deg);
            array[count++] = rad;
            sum += deg;
            max = Math.max(max, deg);
            min = Math.min(min, deg);
        }
        LOG.d(TAG, "computeArg: sum={0,number,#.###}", sum);
        LOG.d(TAG, "computeArg: max={0,number,#.###}, min={1,number,#.###}", max, min);
        double[] results = {
                sum, max - min
        };
        return results;
    }

    // private static float[] computeInclination(GestureStroke stroke) {
    // final int length = stroke.points.length;
    // float lastX = Float.NaN, lastY = Float.NaN, dx = 0, dy = 0, theta = 0,
    // last = Float.NaN, sum = 0;
    // dx = stroke.points[length - 2] - stroke.points[0];
    // dy = stroke.points[length - 1] - stroke.points[1];
    // float total = (float) (Math.atan2(dy, dx) * 180.0f / Math.PI);
    // LOG.d(TAG, "computeInc: top-bottom={0,number,#.###}", total);
    // int count_slant = 0, count = 0;
    // float r;
    // for (int i = 0; i < length; i += 2) {
    // if (!Float.isNaN(lastX) && !Float.isNaN(lastY)) {
    // dx = stroke.points[i] - lastX;
    // dy = stroke.points[i + 1] - lastY;
    // LOG.d(TAG, "computeInc: x={0,number,#.###}, y={1,number,#.###}",
    // stroke.points[i], stroke.points[i+1]);
    // if (dx != 0 && dy != 0) {
    // count++;
    // theta = (float) (Math.atan2(dy, dx) * 180 / Math.PI);
    // r = theta / total;
    // if (0.8f <= r && r < 1.3f) {
    // count_slant++;
    // }
    // LOG.d(TAG, "computeInc: theta={0,number,#.###}, {1,number,#.###}", theta,
    // r);
    // if (!Float.isNaN(last)) sum += last - theta;
    // last = theta;
    // }
    // }
    // lastX = stroke.points[i];
    // lastY = stroke.points[i + 1];
    // }
    // LOG.d(TAG,
    // "computeInc: lines={0,number,#.###}, slanted={1,number,#.###}, sum={2,number,#.###}",
    // count,
    // count_slant, sum);
    // if (count == 0) count = 1;
    // final float[] results = {
    // count_slant / count, total
    // };
    // return results;
    // }
    //
    // private static float[] computeStraightness(GestureStroke stroke) {
    // final int length = stroke.points.length;
    // float lineV = 0f, lineH = 0f;
    // float lastX = 0, lastY = 0, delta = 0;
    // for (int i = 0; i < length; i++) {
    // if (lastX > 0) {
    // delta = lastX - stroke.points[i];
    // if (delta >= -5f && delta < 5f)
    // lineV++;
    // // LOG.d(TAG, "computeStraightness: deltaX={0,number,#.###}",
    // // temp);
    // }
    // lastX = stroke.points[i];
    // i++;
    // if (lastY > 0) {
    // delta = lastY - stroke.points[i];
    // if (delta >= -5f && delta < 5f)
    // lineH++;
    // // LOG.d(TAG, "computeStraightness: deltaY={0,number,#.###}",
    // // delta);
    // }
    // lastY = stroke.points[i];
    // }
    // float len = length / 2;
    // // LOG.d(TAG,
    // //
    // "computeStraightness: v={0,number,#}, h={1,number,#}, length={2,number,#.###}",
    // // lineV, lineH, len);
    // if (lineV != 0)
    // lineV = lineV / len;
    // if (lineH != 0)
    // lineH = lineH / len;
    // // LOG.d(TAG,
    // // "computeStraightness: v={0,number,#.###}, h={1,number,#.###}",
    // // lineV, lineH);
    // final float[] results = {
    // lineV, lineH
    // };
    // return results;
    // }

    public static List<GestureStroke> scaling(Gesture gesture, int width1, int height1, int width2,
            int height2) {
        LOG.d(TAG, "scaling: {0,number,#},{1,number,#}" + " -> {2,number,#},{3,number,#}", width1,
                height1, width2, height2);
        final List<GestureStroke> strokes = gesture.getStrokes();
        final int count_strokes = strokes.size();
        int count_points;
        GestureStroke stroke;
        float x, y;
        RectF box;
        /*
         * magnify
         */
        box = computeBoundingBox(strokes);
        float w = box.width();
        float h = box.height();
        LOG.d(TAG, "scaling: magnify w={0,number,#}, h={1,number,#}", w, h);
        float scaleX = width2 / w;
        float scaleY = height2 / h;
        if (h < w / 2) {
            scaleY = 1.0f;
        } else if (w < h / 2) {
            scaleX = 1.0f;
        }
        LOG.d(TAG, "scaling: magnify scaleX={0,number,#.###}, scaleY={1,number,#.###}", scaleX,
                scaleY);
        for (int i = 0; i < count_strokes; i++) {
            stroke = strokes.get(i);
            stroke.clearPath();
            count_points = stroke.points.length;
            for (int j = 0; j < count_points; j += 2) {
                x = (float) ((stroke.points[j] - box.left) * scaleX);
                y = (float) ((stroke.points[j + 1] - box.top) * scaleY);
                if (x < 0 || y < 0 || x > width2 || y > height2) {
                    LOG.e(TAG, "scaling: magnify ({0,number,#},{1,number,#})"
                            + " -> ({2,number,#},{3,number,#})", stroke.points[j],
                            stroke.points[j + 1], x, y);
                }
                stroke.points[j] = x;
                stroke.points[j + 1] = y;
            }
        }
        return strokes;
    }

    private static RectF computeBoundingBox(List<GestureStroke> strokes) {
        int count_strokes = strokes.size();
        RectF boundingBox = new RectF();
        RectF box;
        for (int i = 0; i < count_strokes; i++) {
            box = strokes.get(i).boundingBox;
            boundingBox.union(box);
        }
        return boundingBox;
    }

    // private static double[] computeFourierDescriptor(GestureStroke stroke) {
    // int length = stroke.points.length;
    // Complex[] z = new Complex[length / 2];
    // int count = 0;
    // for (int j = 0; j < length; j += 2) {
    // z[count++] = new Complex(stroke.points[j], stroke.points[j + 1]);
    // }
    // // compute arguments
    // length = z.length;
    // double[] arg = new double[length];
    // arg[0] = z[1].subtract(z[0]).getArgument();
    // Complex c1, c2;
    // for (int j = 1; j < length - 1; j++) {
    // c1 = z[j].subtract(z[j - 1]);
    // c2 = z[j + 1].subtract(z[j]);
    // arg[j] = FastMath.atan2(c2.getImaginary() - c1.getImaginary(),
    // c2.getReal() - c1.getReal());
    // LOG.d(TAG, "computeFourierDescriptor arg={0,number,#.###}",
    // arg[j]);
    // }
    // // compute curvature
    // double[] theta = new double[length];
    // theta[0] = arg[0];
    // for (int j = 1; j < length - 1; j++) {
    // theta[j] = theta[j - 1] + arg[j];
    // // LOG.d(TAG, "computeFourierDescriptor theta={0,number,#.###}",
    // // theta[j]);
    // }
    // // compute phase-expression
    // Complex[] omega = new Complex[length];
    // for (int j = 0; j < length; j++) {
    // omega[j] = new Complex(Math.cos(theta[j]), Math.sin(theta[j]));
    // // omega[j] = Math.exp(Complex.I.multiply(theta[j]).getReal());
    // // LOG.d(TAG,
    // // "computeFourierDescriptor omega={0,number,#.###},{1,number,#.###}",
    // // omega[j].getReal(), omega[j].getImaginary());
    // }
    // // computer p-type fourier descriptor
    // double[] conv = new double[length];
    // double pow;
    // for (int k = 0; k < length - 1; k++) {
    // conv[k] = 0f;
    // for (int j = 0; j < length - 1; j++) {
    // pow = (2 * Math.PI * j * k) / length;
    // c1 = new Complex(Math.cos(pow), -1 * Math.sin(pow));
    // c2 = omega[j].multiply(c1);
    // conv[k] += c2.getReal() + c2.getImaginary();
    // // LOG.d(TAG, "computeFourierDescriptor pow={0,number,#.###}",
    // // pow);
    // // LOG.d(TAG,
    // // "computeFourierDescriptor comp={0,number,#.###},{1,number,#.###}",
    // // c2.getReal(), c2.getImaginary());
    // }
    // // LOG.d(TAG, "computeFourierDescriptor conv={0,number,#.###}",
    // // conv[k]);
    // conv[k] /= length;
    // LOG.d(TAG,
    // "computeFourierDescriptor {0,number,#}:{1,number,#.###}",
    // k, conv[k]);
    // }
    // return conv;
    // }
}
