/*
 * Copyright (C) 2011 OgakiSoft
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package ogakisoft.android.gestureime; //com.android.gesture.builder;

import java.io.File;
import java.util.List;

import ogakisoft.android.gesture.reform.Gesture;
import ogakisoft.android.gesture.reform.GestureLibrary;
import ogakisoft.android.gesture.reform.GestureView;
import ogakisoft.android.gesture.reform.GestureView.OnGestureListener;
import ogakisoft.android.gesture.reform.GestureView.OnGesturePerformedListener;
import ogakisoft.android.util.LOG;
import android.app.Activity;
import android.app.AlertDialog;
import android.app.Dialog;
import android.app.ProgressDialog;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.SharedPreferences;
import android.content.res.Resources;
import android.os.AsyncTask;
import android.os.Bundle;
import android.os.Environment;
import android.preference.PreferenceManager;
import android.text.TextUtils;
import android.view.ContextMenu;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.widget.AdapterView;
import android.widget.Button;
import android.widget.EditText;
import android.widget.GridView;
import android.widget.TextView;
import android.widget.Toast;

/**
 * Check gesture and assigned string
 * 
 * @author noritoshi ogaki
 * @version 1.1
 */
public class CheckGestureActivity extends Activity {
	public final class GesturesLoadCallback implements
			GesturesLoadTask.Callback {
		public void after() {
			stopProgress();
			buttonEnable(true);
		}

		public void before() {
			buttonEnable(false);
			mAdapter.clear();
			startProgress();
		}

		public void cancel() {
		}

		public void fail(int status) {
			if (status == GesturesLoadTask.STATUS_NO_STORAGE) {
				LOG.d(TAG, "GesturesLoadCallback: no storage");
			}
		}

		public void post() {
			showData();
		}
	}

	private static final int DIALOG_RENAME_GESTURE = 1;
	private static final int MAX_NUM_OF_PREDICT = 5;
	private static final int MENU_ID_MODIFY = 3;
	private static final int MENU_ID_REMOVE = 2;
	private static final int MENU_ID_RENAME = 1;
	private static final int REQUEST_ADD_GESTURE = 1;
	private static final int REQUEST_DOWNLOAD_GESTURE = 2;
	private static final int REQUEST_MODIFY_GESTURE = 3;
	private static final String SaveInstanceStateId_Gesture = "gesture";
	private static final String TAG = "CheckGestureActivity";
	private GridViewAdapter mAdapter;
	private NamedGesture mCurrentNamedGesture;
	private EditText mInput;
	private int mPage;
	private ProgressDialog mProgressDialog;
	private Dialog mRenameDialog;
	private Resources mResources;
	private boolean mShowGestureArea;
	private GesturesLoadTask mTask;
	private final GesturesLoadCallback mTaskCallback = new GesturesLoadCallback();

	private void buttonEnable(boolean sw) {
		findViewById(R.id.recogButton).setEnabled(sw);
		findViewById(R.id.clearButton).setEnabled(sw);
		findViewById(R.id.showHideButton).setEnabled(sw);
		findViewById(R.id.addButton).setEnabled(sw);
		findViewById(R.id.reloadButton).setEnabled(sw);
	}

	private void cleanupRenameDialog() {
		if (null != mRenameDialog) {
			mRenameDialog.dismiss();
			mRenameDialog = null;
		}
		mCurrentNamedGesture = null;
	}

	private void deleteGesture(NamedGesture gesture) {
		if (null == gesture) {
			return;
		}
		final GridViewAdapter adapter = mAdapter;
		final GestureLibrary library = GestureLibrary.getInstance();
		// final NamedGesture gesture =
		// adapter.getItemById(deleteGesture.getId());
		library.removeGesture(gesture.getName(), gesture.getGesture());
		library.saveGestures();
		adapter.remove(gesture);
		adapter.sort();
		adapter.notifyDataSetChanged();
	}

	private void loadGestures() {
		final GestureLibrary library = GestureLibrary.getInstance();
		if (null == mTask) {
			mTask = new GesturesLoadTask(library, mTaskCallback);
		} else if (mTask.getStatus() == AsyncTask.Status.FINISHED) {
			mTask = new GesturesLoadTask(library, mTaskCallback);
		}
		if (mTask.getStatus() != AsyncTask.Status.RUNNING
				&& mTask.getStatus() != AsyncTask.Status.FINISHED) {
			mTask.execute();
		}
	}

	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		super.onActivityResult(requestCode, resultCode, data);
		switch (requestCode) {
		case REQUEST_DOWNLOAD_GESTURE:
			if (resultCode == RESULT_OK) {
				reload();
			} else if (resultCode == RESULT_CANCELED) {
				loadGestures();
			}
			break;
		case REQUEST_ADD_GESTURE:
			if (resultCode == RESULT_OK) {
				reload();
			}
			break;
		case REQUEST_MODIFY_GESTURE:
			if (resultCode == RESULT_OK) {
				// remove old entry
				// deleteGesture(mCurrentRenameGesture);
				mCurrentNamedGesture = null;
				reload();
			}
			break;
		}
	}

	public void onClickAddButton(View v) {
		final Intent intent = new Intent(this, CreateGestureActivity.class);
		startActivityForResult(intent, REQUEST_ADD_GESTURE);
	}

	public void onClickBackPageButton(View v) {
		if (mPage > 0) {
			--mPage;
			showData();
		}
	}

	public void onClickClearButton(View v) {
		final GestureView view = (GestureView) findViewById(R.id.gesture_view);
		view.clear(false);
	}

	public void onClickCloseButton(View v) {
		mAdapter.closePopup();
	}

	public void onClickDeleteButton(View v) {
		mAdapter.closePopup();
		NamedGesture gesture = mAdapter.getSelectedItem();
		onContextItemDeleteGesture(gesture);
	}

	public void onClickEditButton(View v) {
		mAdapter.closePopup();
		NamedGesture gesture = mAdapter.getSelectedItem();
		onContextItemModifyGesture(gesture);
	}

	public void onClickNextPageButton(View v) {
		if (mPage < GestureLibrary.MAX_NUM_OF_STORE - 1) {
			++mPage;
			showData();
		}
	}

	public void onClickRecogButton(View v) {
		final GestureView view = (GestureView) findViewById(R.id.gesture_view);
		final Gesture gesture = view.getGesture();
		if (null != gesture) {
			if (view.isFadeEnabled()) {
				view.cancelClearAnimation();
			}
			predict(gesture);
		}
	}

	public void onClickReloadButton(View v) {
		reload();
	}

	public void onClickShowHideButton(View v) {
		final GestureView view = (GestureView) findViewById(R.id.gesture_view);
		final Button b = (Button) findViewById(R.id.showHideButton);
		if (mShowGestureArea) {
			mShowGestureArea = false;
			view.setVisibility(android.view.View.GONE);
			b.setText(R.string.button_show_pad);
			findViewById(R.id.recogButton).setEnabled(false);
			findViewById(R.id.clearButton).setEnabled(false);
		} else {
			mShowGestureArea = true;
			view.setVisibility(android.view.View.VISIBLE);
			b.setText(R.string.button_hide_pad);
			findViewById(R.id.recogButton).setEnabled(true);
			findViewById(R.id.clearButton).setEnabled(true);
		}
	}

	private void onContextItemDeleteGesture(NamedGesture gesture) {
		deleteGesture(gesture);
		Toast.makeText(this, R.string.message_delete_success,
				Toast.LENGTH_SHORT).show();
	}

	private void onContextItemModifyGesture(NamedGesture gesture) {
		mCurrentNamedGesture = gesture;
		saveGesture(gesture.getGesture(), GestureLibrary.GESTURE_WIDTH,
				GestureLibrary.GESTURE_HEIGHT);
		final Intent intent = new Intent(this, CreateGestureActivity.class);
		startActivityForResult(intent, REQUEST_MODIFY_GESTURE);
	}

	private void onContextItemRenameGesture(NamedGesture gesture) {
		mCurrentNamedGesture = gesture;
		showDialog(DIALOG_RENAME_GESTURE);
	}

	@Override
	public boolean onContextItemSelected(MenuItem item) {
		final AdapterView.AdapterContextMenuInfo menuInfo = (AdapterView.AdapterContextMenuInfo) item
				.getMenuInfo();
		final NamedGesture gesture = (NamedGesture) menuInfo.targetView
				.getTag();
		switch (item.getItemId()) {
		case MENU_ID_RENAME:
			onContextItemRenameGesture(gesture);
			return true;
		case MENU_ID_REMOVE:
			onContextItemDeleteGesture(gesture);
			return true;
		case MENU_ID_MODIFY:
			onContextItemModifyGesture(gesture);
			return true;
		default:
			break;
		}
		return super.onContextItemSelected(item);
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		final GridView grid;
		final GestureView view;
		final SharedPreferences sp;

		super.onCreate(savedInstanceState);
		setContentView(R.layout.check_gesture_activity);

		grid = (GridView) findViewById(R.id.myGrid);
		view = (GestureView) findViewById(R.id.gesture_view);
		sp = PreferenceManager.getDefaultSharedPreferences(this);

		mShowGestureArea = true;
		mResources = getResources();
		mAdapter = new GridViewAdapter(this, mResources, getWindow());
		grid.setAdapter(mAdapter);
		registerForContextMenu(grid);

		Settings.setFromPreferences(mResources, sp, view);

		// check SDCARD
		final File f = Environment.getExternalStorageDirectory();
		if (!f.canRead() || !f.canWrite()) {
			LOG.e(TAG, "onCreate: can not access SDCARD");
			Toast.makeText(CheckGestureActivity.this,
					getResources().getString(R.string.error_sdcard),
					Toast.LENGTH_LONG).show();
		} else {
			// check file in SDCARD
			int count = GestureLibrary.DataFile.length;
			boolean nothing = true;
			// DataFile[0]: user defined data
			for (int i = 1; i < count; i++) {
				if (GestureLibrary.DataFile[i].exists()) {
					nothing = false;
					break;
				}
			}
			if (nothing) {
				final Intent intent = new Intent(this, DownloadActivity.class);
				startActivityForResult(intent, REQUEST_DOWNLOAD_GESTURE);
			} else {
				loadGestures();
			}
		}
		// need this because mHandleGestureActions=true in
		// GestureView.addOnGesturePerformedListener
		view.addOnGesturePerformedListener(new OnGesturePerformedListener() {
			public void onGesturePerformed(GestureView v, final Gesture gesture) {
			}
		});
		view.addOnGestureListener(new OnGestureListener() {
			@Override
			public void onGesture(GestureView v, MotionEvent e) {
			}

			@Override
			public void onGestureCancelled(GestureView v, MotionEvent e) {
			}

			@Override
			public void onGestureEnded(GestureView v, MotionEvent e) {
				if (v.isFadeEnabled()) {
					predict(v.getGesture());
				}
			}

			@Override
			public void onGestureStarted(GestureView v, MotionEvent e) {
			}
		});
	}

	@Override
	public void onCreateContextMenu(ContextMenu menu, View v,
			ContextMenu.ContextMenuInfo menuInfo) {
		final AdapterView.AdapterContextMenuInfo info;
		super.onCreateContextMenu(menu, v, menuInfo);
		info = (AdapterView.AdapterContextMenuInfo) menuInfo;
		final NamedGesture gesture = (NamedGesture) info.targetView.getTag();
		menu.setHeaderTitle(gesture.getName());
		menu.add(0, MENU_ID_RENAME, 0, R.string.gestures_rename);
		menu.add(0, MENU_ID_REMOVE, 0, R.string.gestures_delete);
		menu.add(0, MENU_ID_MODIFY, 0, R.string.gestures_modify);
	}

	@Override
	protected Dialog onCreateDialog(int id) {
		if (id == DIALOG_RENAME_GESTURE) {
			final View layout = View
					.inflate(this, R.layout.dialog_rename, null);
			final AlertDialog.Builder builder = new AlertDialog.Builder(this);
			mInput = (EditText) layout.findViewById(R.id.name);
			((TextView) layout.findViewById(R.id.label))
					.setText(R.string.gestures_rename_label);
			builder.setIcon(0);
			builder.setTitle(getString(R.string.gestures_rename_title));
			builder.setCancelable(true);
			builder.setOnCancelListener(new Dialog.OnCancelListener() {
				public void onCancel(DialogInterface dialog) {
					cleanupRenameDialog();
				}
			});
			builder.setNegativeButton(getString(R.string.cancel_action),
					new Dialog.OnClickListener() {
						public void onClick(DialogInterface dialog, int which) {
							cleanupRenameDialog();
						}
					});
			builder.setPositiveButton(getString(R.string.rename_action),
					new Dialog.OnClickListener() {
						public void onClick(DialogInterface dialog, int which) {
							renameGesture(mCurrentNamedGesture, mInput
									.getText().toString());
							mCurrentNamedGesture = null;
						}
					});
			builder.setView(layout);
			return builder.create();
		}
		return super.onCreateDialog(id);
	}

	@Override
	protected void onDestroy() {
		super.onDestroy();

		if (null != mTask
				&& mTask.getStatus() != GesturesLoadTask.Status.FINISHED) {
			mTask.cancel(true);
			mTask = null;
		}
		cleanupRenameDialog();

		mAdapter = null;
		mResources = null;
		mInput = null;
		mProgressDialog = null;
	}

	@Override
	protected void onPrepareDialog(int id, Dialog dialog) {
		super.onPrepareDialog(id, dialog);
		if (id == DIALOG_RENAME_GESTURE) {
			mInput.setText(mCurrentNamedGesture.getName());
		}
	}

	@Override
	protected void onRestart() {
		final GestureView view = (GestureView) findViewById(R.id.gesture_view);
		final SharedPreferences sp = PreferenceManager
				.getDefaultSharedPreferences(this);
		super.onRestart();
		Settings.setFromPreferences(getResources(), sp, view);
	}

	@Override
	protected void onRestoreInstanceState(Bundle state) {
		final Gesture gesture = state
				.getParcelable(SaveInstanceStateId_Gesture);
		if (null != gesture) {
			mCurrentNamedGesture = GestureLibrary.getInstance()
					.getNamedGesture(gesture.getId());
		}
		super.onRestoreInstanceState(state);
	}

	@Override
	protected void onSaveInstanceState(Bundle state) {
		super.onSaveInstanceState(state);
		if (null != mCurrentNamedGesture) {
			state.putParcelable(SaveInstanceStateId_Gesture,
					mCurrentNamedGesture.getGesture());
		}
	}

	private void predict(Gesture gesture) {
		if (null == gesture || 0 == gesture.getLength()) {
			return;
		}
		final GestureView view = (GestureView) findViewById(R.id.gesture_view);
		final int w = view.getWidth();
		final int h = view.getHeight();
		final GestureLibrary library = GestureLibrary.getInstance();
		final List<String> list = library.predict(gesture, w, h);
		final int size = list.size();
		String str = null;
		for (int i = 0; i < MAX_NUM_OF_PREDICT && i < size; i++) {
			str = list.get(i);
			Toast.makeText(CheckGestureActivity.this, str, Toast.LENGTH_SHORT)
					.show();
		}
		saveGesture(gesture, w, h);
	}

	private void reload() {
		final GestureLibrary library = GestureLibrary.getInstance();
		for (int i = 0; i < GestureLibrary.MAX_NUM_OF_STORE; i++) {
			if (library.hasChanged(i)) {
				GesturesLoadTask.LEARNER[i].delete();
			}
		}
		// reload
		loadGestures();
	}

	private void renameGesture(NamedGesture gesture, String label) {
		if (null == gesture || TextUtils.isEmpty(label)) {
			return;
		}
		final GridViewAdapter adapter = mAdapter;
		final GestureLibrary library = GestureLibrary.getInstance();
		// final NamedGesture gesture =
		// adapter.getItemById(renameGesture.getId());
		library.removeGesture(gesture.getName(), gesture.getGesture());
		gesture.setName(label);
		library.addGesture(label, gesture.getGesture());
		library.saveGestures();
		adapter.sort();
		adapter.notifyDataSetChanged();
	}

	private void saveGesture(Gesture gesture, int width, int height) {
		// Can CreateGestureActivity at any time,
		// Save gesture data in a file
		try {
			GestureLibrary.saveGesture(gesture, width, height);
		} catch (Exception e) {
			LOG.e(TAG, "saveGesture: {0}", e.getMessage());
			final String str = mResources
					.getString(R.string.error_save_gesture);
			Toast.makeText(CheckGestureActivity.this, str, Toast.LENGTH_LONG)
					.show();
		}
	}

	private void showData() {
		mAdapter.clear();
		final GestureLibrary library = GestureLibrary.getInstance();
		for (String label : library.getGestureEntries(mPage)) {
			for (Gesture gesture : library.getGestures(mPage, label)) {
				mAdapter.add(new NamedGesture(gesture, label));
			}
		}
		mAdapter.sort();
		mAdapter.notifyDataSetChanged();
		final TextView text = (TextView) findViewById(R.id.pageText);
		text.setText(String.valueOf(mPage + 1));
		// Toast.makeText(CheckGestureActivity.this,
		// R.string.message_gesture_click, Toast.LENGTH_LONG).show();
	}

	private void startProgress() {
		mProgressDialog = new ProgressDialog(this);
		mProgressDialog.setCancelable(false);
		mProgressDialog.setIndeterminate(false);
		mProgressDialog.setMessage(mResources
				.getString(R.string.message_loading));
		// mProgressDialog.setCancelMessage(mHandler
		// .obtainMessage(REQUEST_CANCEL));
		mProgressDialog.show();
	}

	private void stopProgress() {
		if (null != mProgressDialog) {
			if (mProgressDialog.isShowing()) {
				mProgressDialog.dismiss();
			}
			// mHandler.removeMessages(REQUEST_CANCEL);
		}
	}
}
