#include "stdafx.h"
#include <vector>

#include "GVONavish.h"
#include "GVOWorldMap.h"



namespace {
	const double k_scaleStep = 0.125;	// 12.5%
	const double k_minScale = 0.125;	// 12.5%
	const double k_maxScale = 4.00;		// 400%
}


bool GVOWorldMap::loadFromFile( const GVOConfig& config )
{
	std::wstring filePath;

	filePath = g_makeFullPath( config.m_mapFileName );
	if ( !m_mapImage.loadFromFile( filePath.c_str() ) ) {
		filePath = g_makeFullPath( config.m_defaultMapFileName );
		if ( !m_mapImage.loadFromFile( filePath.c_str() ) ) {
			return false;
		}
	}

	// 摜̍ƈܓx̔䗦̃XP[O̊ƂȂB
	const double mapHeight = m_mapImage.height();
	m_ratioForImageCoordFromWorldCoord = mapHeight / k_worldHeight;
	return true;
}


void GVOWorldMap::setViewSize( const SIZE& viewSize )
{
	m_viewSize = viewSize;
}


void GVOWorldMap::offsetFocusInViewCoord( const POINT& offset )
{
	const double dx = ((double)offset.x / m_viewScale) / m_mapImage.width();
	const double dy = ((double)offset.y / m_viewScale) / m_mapImage.height();

	LONG x = m_focusPointInWorldCoord.x + LONG( dx * k_worldWidth );
	LONG y = m_focusPointInWorldCoord.y + LONG( dy * k_worldHeight );
	y = max( 0, min( y, k_worldHeight ) );
	while ( x < 0 ) {
		x += k_worldWidth;
	}
	while ( k_worldWidth < x ) {
		x -= k_worldWidth;
	}

	m_focusPointInWorldCoord.x = x;
	m_focusPointInWorldCoord.y = y;
}


void GVOWorldMap::setConfig( const GVOConfig& config )
{
	m_positionUpdated = config.m_traceShipPositionEnabled;
	m_focusPointInWorldCoord = config.m_initialSurveyCoord;
	m_shipPointInWorld = config.m_initialSurveyCoord;
	m_previousDrawPointInWorld = m_shipPointInWorld;
	m_shipVectorLineEnabled = config.m_shipVectorLineEnabled;
}


void GVOWorldMap::setShipPosition( const POINT& worldCoord, bool isSyncCenter )
{
	if ( isSyncCenter ) {
		m_focusPointInWorldCoord = worldCoord;
	}
	if ( m_shipPointInWorld.x != worldCoord.x
		|| m_shipPointInWorld.y != worldCoord.y ) {

		m_positionUpdated = true;
		m_shipPointInWorld = worldCoord;
	}
}


bool GVOWorldMap::zoomIn()
{
	double scale = m_viewScale;
	double step = k_scaleStep;

	scale = m_viewScale + step;
	if ( k_maxScale < scale ) {
		scale = k_maxScale;
	}
	if ( m_viewScale != scale ) {
		m_viewScale = scale;
		return true;
	}
	return false;
}


bool GVOWorldMap::zoomOut()
{
	double scale = m_viewScale;
	double step = k_scaleStep;

	scale = m_viewScale - step;
	if ( scale < k_minScale ) {
		scale = k_minScale;
	}
	if ( m_viewScale != scale ) {
		m_viewScale = scale;
		return true;
	}
	return false;
}


void GVOWorldMap::drawMap( HDC hdc, const GVOShip& ship )
{
	const SIZE mapSize = scaledMapSize();
	GVOImage scaledMapImage;
	GVOImage *mapImage = &m_mapImage;

	::SaveDC( hdc );
	if ( m_viewScale < 1.0 ) {
		// k摜`RXgyׂɃLbV쐬B
		// pɂɏk䗦ύX鎞RXgtA
		// wǂ̏ꍇɂCPUׂyB
		if ( !m_reducedMapImageCache.isCompatible(mapSize) ) {
			m_reducedMapImageCache.stretchCopy( m_mapImage, mapSize );
		}
		mapImage = &m_reducedMapImageCache;
	}
	::SetStretchBltMode( hdc, COLORONCOLOR );

	HDC hdcMem = ::CreateCompatibleDC( hdc );
	::SaveDC( hdcMem );

	::SelectObject( hdcMem, mapImage->bitmapHandle() );

	const POINT mapTopLeft = mapOriginInView();

	int xDrawOrigin, yDrawOrigin;
	xDrawOrigin = mapTopLeft.x;
	yDrawOrigin = mapTopLeft.y;

	if ( 0 < xDrawOrigin ) {
		xDrawOrigin = (xDrawOrigin % mapSize.cx) - mapSize.cx;
	}
	const int xInitial = xDrawOrigin;	// [̕`JnxW
	int drawn = xInitial;				// `ςݍW

	// En}ɕׂĕ`
	// i`œK͏ȗj
	while ( drawn < m_viewSize.cx ) {
		::StretchBlt( hdc,
			xDrawOrigin, yDrawOrigin,
			mapSize.cx, mapSize.cy,
			hdcMem,
			0, 0,
			mapImage->width(), mapImage->height(),
			SRCCOPY );

		xDrawOrigin += mapSize.cx;
		drawn += mapSize.cx;
	}


	const POINT shipPointOffset = drawOffsetFromWorldCoord( m_shipPointInWorld );

	// jH\`
	if ( ship.isVectorEnabled() && m_shipVectorLineEnabled ) {
		const int penWidth = max( 1, int(1 * m_viewScale) );
		HPEN courseLinePen = ::CreatePen( PS_SOLID, penWidth, RGB( 255, 0, 255 ) );
		HGDIOBJ oldPen = ::SelectObject( hdc, courseLinePen );

		const LONG k_lineLength = k_worldHeight;
		const POINT reachPointOffset = drawOffsetFromWorldCoord(
			ship.pointFromOriginWithLength( m_shipPointInWorld, k_lineLength )
			);

		// Ăn}摜̕`悷
		drawn = xInitial;
		xDrawOrigin = xInitial;
		while ( drawn < m_viewSize.cx ) {
			const POINT shipPointInView = {
				xDrawOrigin + shipPointOffset.x,
				yDrawOrigin + shipPointOffset.y
			};
			const POINT reachPointInView = {
				xDrawOrigin + reachPointOffset.x,
				yDrawOrigin + reachPointOffset.y
			};
			::MoveToEx( hdc, shipPointInView.x, shipPointInView.y, NULL );
			::LineTo( hdc, reachPointInView.x, reachPointInView.y );
			xDrawOrigin += mapSize.cx;
			drawn += mapSize.cx;
		}

		::SelectObject( hdc, oldPen );
		::DeleteObject( courseLinePen );
	}

	// D̈ʒu`
	const SIZE shipMarkSize = { 6, 6 };
	HBRUSH shipBrush = ::CreateSolidBrush( RGB( 51, 238, 153 ) );
	HGDIOBJ prevBrush = ::SelectObject( hdc, shipBrush );

	// Ăn}摜̕`悷
	drawn = xInitial;
	xDrawOrigin = xInitial;
	while ( drawn < m_viewSize.cx ) {
		::Ellipse( hdc,
			xDrawOrigin + shipPointOffset.x - shipMarkSize.cx / 2,
			yDrawOrigin + shipPointOffset.y - shipMarkSize.cy / 2,
			xDrawOrigin + shipPointOffset.x + shipMarkSize.cx,
			yDrawOrigin + shipPointOffset.y + shipMarkSize.cy );

		xDrawOrigin += mapSize.cx;
		drawn += mapSize.cx;
	}
	::SelectObject( hdc, prevBrush );
	::DeleteObject( shipBrush );


	::RestoreDC( hdcMem, -1 );
	::DeleteDC( hdcMem );
	::RestoreDC( hdc, -1 );
}


void GVOWorldMap::updateShipRouteMap( HDC hdc )
{
	// ړĂȂΉȂ
	if ( !m_positionUpdated ) {
		return;
	}
	// ړĂĂqHqȂȂ`悵Ȃ
	if ( !m_linkRoute ) {
		m_previousDrawPointInWorld = m_shipPointInWorld;
		m_linkRoute = true;
		return;
	}

	const POINT& latestPoint = imageCoordFromWorldCoord( m_shipPointInWorld );
	const POINT& previousPoint = imageCoordFromWorldCoord( m_previousDrawPointInWorld );

	// `WꏏȂ牽Ȃ
	if ( latestPoint.x == previousPoint.x
		&& latestPoint.y == previousPoint.y ) {
		return;
	}


	// ŐVWO̍W֐
	// EׂꍇɍlKvB
	// ړʂ𒴂`揈Ȃ悤ɂ̂X}[g


	// EׂƂ݂Ȃ臒l
	const int k_distanceThreshold = m_mapImage.width() / 2;

	const LONG xMin = min( latestPoint.x, previousPoint.x );
	const LONG xMax = max( latestPoint.x, previousPoint.x );
	const int xDistance = xMax - xMin;


	HDC hdcMem = ::CreateCompatibleDC( hdc );
	::SaveDC( hdcMem );

	HPEN hpen = ::CreatePen( PS_SOLID, 1, RGB( 255, 255, 255 ) );

	::SelectObject( hdcMem, m_mapImage.bitmapHandle() );
	::SelectObject( hdcMem, hpen );

	if ( k_distanceThreshold < xDistance ) {
		if ( previousPoint.x < latestPoint.x ) {	// E𐼂ɌČׂ
			// ̉ʊO`
			LONG x3 = latestPoint.x - m_mapImage.width();

			::MoveToEx( hdcMem, x3, latestPoint.y, NULL );
			::LineTo( hdcMem, previousPoint.x, previousPoint.y );

			// ẺʊOɌ`
			LONG x4 = previousPoint.x + m_mapImage.width();
			::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
			::LineTo( hdcMem, x4, previousPoint.y );
		}
		else {				// E𓌂ɌČׂ
			// ̉ʊO`
			LONG x3 = previousPoint.x - m_mapImage.width();

			::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
			::LineTo( hdcMem, x3, previousPoint.y );

			// ẺʊOɌ`
			LONG x4 = latestPoint.x + m_mapImage.width();
			::MoveToEx( hdcMem, x4, latestPoint.y, NULL );
			::LineTo( hdcMem, previousPoint.x, previousPoint.y );
		}
	}
	else {
		// ׂȂ`
		::MoveToEx( hdcMem, latestPoint.x, latestPoint.y, NULL );
		::LineTo( hdcMem, previousPoint.x, previousPoint.y );
	}


	::RestoreDC( hdcMem, -1 );
	::DeleteDC( hdcMem );
	::DeleteObject( hpen );

	m_previousDrawPointInWorld = m_shipPointInWorld;
}


void GVOWorldMap::clearShipRoute()
{
	const std::wstring fileName = m_mapImage.fileName();
	m_mapImage.loadFromFile( fileName );
}


POINT GVOWorldMap::mapOriginInView()
{
	const POINT viewCenter = viewCenterPoint();
	const SIZE mapSize = scaledMapSize();
	const POINT worldPosInView = drawOffsetFromWorldCoord( m_focusPointInWorldCoord );

	POINT mapTopLeft = {
		viewCenter.x - worldPosInView.x,
		viewCenter.y - worldPosInView.y
	};
	if ( m_viewSize.cx < mapSize.cx ) {
		while ( 0 < mapTopLeft.x ) {
			mapTopLeft.x -= mapSize.cx;
		}
	}

	return mapTopLeft;
}


POINT GVOWorldMap::imageCoordFromWorldCoord( const POINT& worldCoord ) const
{
	const double xNormPos = worldCoord.x / (double)k_worldWidth;
	const double yNormPos = worldCoord.y / (double)k_worldHeight;
	const POINT worldPosInImage = {
		LONG( m_mapImage.width() * xNormPos ),
		LONG( m_mapImage.height() * yNormPos )
	};
	return worldPosInImage;
}

POINT GVOWorldMap::drawOffsetFromWorldCoord( const POINT&worldCoord ) const
{
	const POINT worldPosInImage = imageCoordFromWorldCoord( worldCoord );
	const POINT drawOffset = {
		LONG( worldPosInImage.x * m_viewScale ),
		LONG( worldPosInImage.y * m_viewScale )
	};
	return drawOffset;
}
