/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie <ashie@homa.ne.jp>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include "gobject-utils.h"
#include "intl.h"
#include "kz-metamenu.h"
#include "kz-icons.h"
#include "gtk-utils.h"


#define KZ_META_MENU_ITEM_KEY "KzMETAMenu::Item"


static void     kz_meta_menu_class_init            (KzMETAMenuClass *klass);
static void     kz_meta_menu_init                  (KzMETAMenu *metamenu);
static void     kz_meta_menu_dispose               (GObject   *object);

static void     kz_meta_menu_load_start            (KzMETA *meta,
						    KzMETAMenu *metamenu);
static void     kz_meta_menu_update                (KzMETA *meta,
						    KzMETAMenu *metamenu);
static gboolean kz_meta_menu_item_button_press_cb  (GtkWidget *widget,
						    GdkEventButton *event,
						    KzMETAMenu *metamenu);
static gboolean kz_meta_menu_all_button_press_cb   (GtkWidget *widget,
						    GdkEventButton *event,
						    KzMETAMenu *metamenu);
static void     kz_meta_menu_items_new             (KzWindow *kz,
						    KzMETAMenu *metamenu);

static gboolean kz_meta_menu_press_cb              (GtkWidget *widget,
						   GdkEvent *event,
						   KzMETAMenu *metamenu);
static void     kz_meta_menu_update_button_clicked (GtkWidget *widget,
						   KzMETAMenu *metamenu);


static GtkHBoxClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_meta_menu, "KzMETAMenu", KzMETAMenu,
		   kz_meta_menu_class_init, kz_meta_menu_init,
		   GTK_TYPE_HBOX)

static void
kz_meta_menu_class_init (KzMETAMenuClass *klass)
{
	GObjectClass *gobject_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *) klass;

	/* GObject signals */
	gobject_class->dispose = kz_meta_menu_dispose;
}

static void
kz_meta_menu_init (KzMETAMenu *metamenu)
{
	metamenu->kz            = NULL;
	metamenu->meta          = NULL;
	metamenu->sub_menu      = NULL;
	metamenu->menu_item     = NULL;
	metamenu->update_button = NULL;
	metamenu->image         = NULL;
}

static void
kz_meta_menu_dispose (GObject *object)
{
	KzMETAMenu *metamenu;

	g_return_if_fail (KZ_IS_META_MENU(object));

	metamenu = KZ_META_MENU(object);

	metamenu->kz = NULL;

	if (metamenu->meta)
	{
		g_signal_handlers_disconnect_by_func(G_OBJECT(metamenu->meta),
						     kz_meta_menu_load_start,
						     metamenu);
		g_signal_handlers_disconnect_by_func(G_OBJECT(metamenu->meta),
						     kz_meta_menu_update,
						     metamenu);
		g_object_unref(metamenu->meta);
		metamenu->meta = NULL;
	}

	if (metamenu->sub_menu)
	{
		gtk_widget_destroy(metamenu->sub_menu);
	}

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}

GtkWidget *
kz_meta_menu_new (KzWindow *kz, KzMETA *meta)
{
	KzMETAMenu *metamenu;
	GtkWidget *button, *image, *hbox, *label, *eventbox;

	g_return_val_if_fail(KZ_IS_WINDOW(kz), NULL);
	g_return_val_if_fail(KZ_IS_META(meta), NULL);

	metamenu = KZ_META_MENU(g_object_new(KZ_TYPE_META_MENU, NULL));
	metamenu->kz  = kz;
	metamenu->meta = meta;

	g_object_ref(metamenu->meta);

	hbox = gtk_hbox_new(FALSE, 0);

	/* label */
	label = gtk_label_new(meta->title);
	eventbox = gtk_event_box_new();
	gtk_container_add(GTK_CONTAINER(eventbox), label);
	g_signal_connect(G_OBJECT(eventbox), "button_press_event",
			 G_CALLBACK(kz_meta_menu_press_cb), metamenu);
	gtk_widget_show (label);

	/* button */
	metamenu->update_button = button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(button), GTK_RELIEF_NONE);
	gtk_container_set_border_width(GTK_CONTAINER(button), 0);
	g_signal_connect(G_OBJECT(button), "clicked", 
			 G_CALLBACK(kz_meta_menu_update_button_clicked), metamenu);

	metamenu->image = image
		= gtk_image_new_from_stock(KZ_STOCK_GREEN,
					   KZ_ICON_SIZE_BOOKMARK_MENU);

	gtk_container_add(GTK_CONTAINER(button), image);
	gtk_widget_set_size_request(button, 16, 16);
	gtk_widget_show(image);
	gtk_widget_show(button);

	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), eventbox, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(metamenu), hbox);
	gtk_container_set_border_width(GTK_CONTAINER(metamenu), 2);

	g_signal_connect(G_OBJECT(metamenu->meta), "update_start", 
			 G_CALLBACK(kz_meta_menu_load_start), metamenu);
	g_signal_connect(G_OBJECT(metamenu->meta), "update_completed", 
			 G_CALLBACK(kz_meta_menu_update), metamenu);

	kz_meta_menu_items_new(kz, metamenu);

	gtk_widget_show_all(GTK_WIDGET(metamenu));

	if (metamenu->meta->state == KZ_META_LOADING)
		kz_meta_menu_load_start(metamenu->meta, metamenu);
	
	return GTK_WIDGET(metamenu);
}

static void
kz_meta_menu_items_new (KzWindow *kz, KzMETAMenu *metamenu)
{
	GtkWidget *menu_item;
	GSList *items;
	guint item_num, n;
	KzMETA *meta = NULL;
	KzMETAItem *item = NULL;
	GtkTooltips *tooltip = NULL;

	g_return_if_fail(KZ_IS_META_MENU(metamenu));

	meta = metamenu->meta;

	if (metamenu->sub_menu)
	{
		gtk_widget_destroy(metamenu->sub_menu);
	}

	metamenu->sub_menu = gtk_menu_new();
	
	items = meta->items;
	item_num = g_slist_length(meta->items);

	/* Open all */
	menu_item = gtk_menu_item_new_with_label(_("Open all items"));
	gtk_menu_shell_append(GTK_MENU_SHELL(metamenu->sub_menu),
			      menu_item);
	g_signal_connect(G_OBJECT(menu_item), "button_press_event",
			 G_CALLBACK(kz_meta_menu_all_button_press_cb), metamenu);

	/* separator */
	menu_item = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(metamenu->sub_menu),
			      menu_item);

	/* items */
	for (n = 0; n < item_num; n++)
	{
		item = (KzMETAItem*) g_slist_nth_data(items, n);
		if (!item->title) continue;
		
		menu_item = gtk_menu_item_new_with_label(item->title);
		gtk_menu_shell_append(GTK_MENU_SHELL(metamenu->sub_menu),
				     menu_item);
		item->meta = meta;
		g_object_set_data(G_OBJECT(menu_item), KZ_META_MENU_ITEM_KEY, item);
		g_signal_connect(G_OBJECT(menu_item), "button_press_event",
				 G_CALLBACK(kz_meta_menu_item_button_press_cb), metamenu);
		tooltip = gtk_tooltips_new();
		gtk_tooltips_set_tip(tooltip, menu_item, item->description, NULL);
	}
	gtk_widget_show_all(metamenu->sub_menu);
}

static void 
kz_meta_menu_load_start (KzMETA *meta, KzMETAMenu *metamenu)
{
	g_return_if_fail(KZ_IS_META(meta));
	g_return_if_fail(KZ_IS_META_MENU(metamenu));

	gtk_image_set_from_stock(GTK_IMAGE(metamenu->image),
				 KZ_STOCK_RED,
				 KZ_ICON_SIZE_BOOKMARK_MENU);
}

static void 
kz_meta_menu_update (KzMETA *meta, KzMETAMenu *metamenu)
{
	KzWindow *kz;

	g_return_if_fail(KZ_IS_META(meta));
	g_return_if_fail(KZ_IS_META_MENU(metamenu));
	kz = metamenu->kz;

	kz_meta_menu_items_new(kz, metamenu);
	gtk_widget_show_all(GTK_WIDGET(metamenu));

	gtk_image_set_from_stock(GTK_IMAGE(metamenu->image),
				 KZ_STOCK_GREEN,
				 KZ_ICON_SIZE_BOOKMARK_MENU);
}

static gboolean
kz_meta_menu_all_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				 KzMETAMenu *metamenu)
{
	KzWindow *kz;
	KzMETA *meta;
	GSList *items, *node;

	g_return_val_if_fail(KZ_IS_META_MENU(metamenu), FALSE);

	kz = metamenu->kz;
	if (!KZ_IS_WINDOW(kz)) return FALSE;

	meta = metamenu->meta;
	if (!KZ_IS_META(meta)) return FALSE;

	items = meta->items;

	for (node = items; node; node = g_slist_next(node))
	{
		KzMETAItem *item = node->data;
		if (!item) continue;

		kz_window_open_new_tab(kz, item->link);
	}

	return TRUE;
}

static gboolean
kz_meta_menu_item_button_press_cb (GtkWidget *widget, GdkEventButton *event,
				  KzMETAMenu *metamenu)
{
	KzMETAItem *item;
	KzWindow *kz;

	g_return_val_if_fail(KZ_IS_META_MENU(metamenu), FALSE);
	kz = metamenu->kz;
	if (!KZ_IS_WINDOW(kz)) return FALSE;

	item = g_object_get_data(G_OBJECT(widget), KZ_META_MENU_ITEM_KEY);
	g_return_val_if_fail(item, FALSE);

	if (!item->link) return FALSE;

	switch (event->button) {
	case 1:
	case 3:
		kz_window_load_url(kz, item->link);
		return TRUE;
		break;
	case 2:
		kz_window_open_new_tab(kz, item->link);
		return TRUE;
		break;
	}

	return FALSE;
}

static gboolean
kz_meta_menu_press_cb (GtkWidget *widget, GdkEvent *event,
		      KzMETAMenu *metamenu)
{
	KzWindow *kz;
	GdkEventButton *event_button;
	GSList *items;
	guint item_num, n;
	KzMETAItem *item = NULL;

	kz = metamenu->kz;

	event_button = (GdkEventButton *)event;
	switch(event_button->button)
	{
	case 1:
		/* g_return_val_if_fail(GTK_IS_MENU(metamenu->sub_menu), FALSE); */
		if (!GTK_IS_MENU(metamenu->sub_menu)) return FALSE;

		gtk_menu_popup(GTK_MENU(metamenu->sub_menu), NULL, NULL, 
			       gtkutil_menu_position_under_widget,
			       GTK_WIDGET(metamenu),
			       event_button->button, event_button->time);
		break;
	case 2:
		items = metamenu->meta->items;
		item_num = g_slist_length(metamenu->meta->items);
		for (n = 0; n < item_num; n++)
		{
			item = (KzMETAItem*) g_slist_nth_data(items, n);
			if (item->link)
				kz_window_open_new_tab(kz, item->link);
		}
		break;
	case 3:
		/* kz_window_load_url(kz, metamenu->meta->meta_link); */
		break;
	}

	return FALSE;
}

static void
kz_meta_menu_update_button_clicked (GtkWidget *widget, KzMETAMenu *metamenu)
{
	g_return_if_fail(KZ_IS_META_MENU(metamenu));

	kz_meta_force_update(metamenu->meta);
}
