/* -*- mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-sidebar.h"

#include <string.h>
#include <gtk/gtk.h>

#include "gobject-utils.h"
#include "intl.h"
#include "kz-tabtree.h"
#include "kz-metatree.h"


#define ADD_ENTRIES(func) 						\
{									\
	KzSidebarEntry *entry;						\
	gint i;								\
	for (i = 0; (entry = func(i)); i++)				\
	{								\
		n_sidebar_entries++;					\
		sidebar_entries = g_realloc(sidebar_entries,		\
					    sizeof(KzSidebarEntry)	\
					    * n_sidebar_entries);	\
		sidebar_entries[n_sidebar_entries - 1] = entry;		\
	}								\
}


static void     kz_sidebar_class_init     (KzSidebarClass *klass);
static void     kz_sidebar_init           (KzSidebar *sidebar);

static gboolean kz_sidebar_change_content (KzSidebar *sidebar,
					   KzSidebarEntry *entry);

static void     cb_option_menu_changed    (GtkOptionMenu *option_menu,
					   KzSidebar     *sidebar);
static void     cb_close_button           (GtkWidget     *widget,
					   KzSidebar     *sidebar);

static GtkVBoxClass *parent_class = NULL;

static KzSidebarEntry **sidebar_entries   = NULL;
static gint             n_sidebar_entries = 0;


KZ_OBJECT_GET_TYPE(kz_sidebar, "KzSidebar", KzSidebar,
		   kz_sidebar_class_init, kz_sidebar_init,
		   GTK_TYPE_VBOX)

static void
kz_sidebar_class_init (KzSidebarClass *klass)
{
	parent_class = g_type_class_peek_parent(klass);

	ADD_ENTRIES(kz_tab_tree_get_entry);
	ADD_ENTRIES(kz_meta_tree_get_entry);
}

static void
kz_sidebar_init (KzSidebar *sidebar)
{
	GtkWidget *hbox, *option_menu, *menu, *close_button, *image;
	gint i;

	/* top hbox */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(sidebar), hbox, FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(hbox));

	/* option menu */
	option_menu = gtk_option_menu_new();
	gtk_button_set_relief(GTK_BUTTON(option_menu), GTK_RELIEF_NONE);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(GTK_WIDGET(option_menu));

	sidebar->content = NULL;
	sidebar->current_entry = NULL;

	menu = gtk_menu_new();
	for (i = 0; i < n_sidebar_entries; i++)
	{
		GtkWidget *item;
		item = gtk_menu_item_new_with_label(_(sidebar_entries[i]->label));
		gtk_menu_shell_append (GTK_MENU_SHELL(menu), item);
		gtk_widget_show (item);
	}
	gtk_option_menu_set_menu (GTK_OPTION_MENU(option_menu), menu);
	g_signal_connect (option_menu, "changed",
			  G_CALLBACK (cb_option_menu_changed), sidebar);

	/* close button */
	close_button = gtk_button_new();
	gtk_button_set_relief(GTK_BUTTON(close_button), GTK_RELIEF_NONE);
	gtk_box_pack_end(GTK_BOX(hbox), close_button, FALSE, FALSE, 0);
	gtk_widget_show(close_button);

	image = gtk_image_new_from_stock(GTK_STOCK_CLOSE,
					 GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(close_button), image);
	gtk_widget_show(image);

	g_signal_connect(G_OBJECT(close_button), "clicked",
			 G_CALLBACK(cb_close_button), sidebar);

	sidebar->kz            = NULL;
	sidebar->hbox          = hbox;
	sidebar->option_menu   = option_menu;
	sidebar->close_button  = close_button;
	sidebar->content       = NULL;
	sidebar->current_entry = NULL;
}

GtkWidget *
kz_sidebar_new (KzWindow *kz)
{
	KzSidebar *sidebar;

	sidebar = KZ_SIDEBAR(g_object_new(KZ_TYPE_SIDEBAR, NULL));
	sidebar->kz = kz; /* FIXME!! use property */

	kz_sidebar_set_current(sidebar, sidebar_entries[0]->label);

	return GTK_WIDGET(sidebar);
}

const gchar *
kz_sidebar_get_current (KzSidebar *sidebar)
{
	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), NULL);
	g_return_val_if_fail(sidebar->current_entry, NULL);

	return sidebar->current_entry->label;
}


static gboolean
kz_sidebar_change_content (KzSidebar *sidebar, KzSidebarEntry *entry)
{
	GtkWidget *widget;

	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), FALSE);
	g_return_val_if_fail(entry, FALSE);
	g_return_val_if_fail(entry->create, FALSE);

	widget = entry->create(sidebar);
	if (!GTK_IS_WIDGET(widget))
		return FALSE;

	if (sidebar->content)
		gtk_widget_destroy(sidebar->content);

	sidebar->content = widget;

	gtk_box_pack_start(GTK_BOX(sidebar), sidebar->content,
			   TRUE, TRUE, 0);
	gtk_widget_show(sidebar->content);

	sidebar->current_entry = entry;

	return TRUE;
}


gboolean
kz_sidebar_set_current (KzSidebar *sidebar, const gchar *label)
{
	gint i, num = n_sidebar_entries;

	g_return_val_if_fail(KZ_IS_SIDEBAR(sidebar), FALSE);
	g_return_val_if_fail(label && *label, FALSE);

	for (i = 0; i < num; i++)
	{
		gboolean ok;

		if (strcmp(sidebar_entries[i]->label, label)) continue;

		ok = kz_sidebar_change_content(sidebar, sidebar_entries[i]);
		if (!ok) continue;

		/* set option menu */
		g_signal_handlers_block_by_func(G_OBJECT(sidebar->option_menu),
						G_CALLBACK(cb_option_menu_changed),
						sidebar);
		gtk_option_menu_set_history(GTK_OPTION_MENU(sidebar->option_menu), i);
		g_signal_handlers_unblock_by_func(G_OBJECT(sidebar->option_menu),
						  G_CALLBACK(cb_option_menu_changed),
						  sidebar);

		return TRUE;
	}

	return FALSE;
}

static void
cb_option_menu_changed (GtkOptionMenu *option_menu, KzSidebar *sidebar)
{
	gint num;

	g_return_if_fail (GTK_IS_OPTION_MENU(option_menu));
	g_return_if_fail (KZ_IS_SIDEBAR(sidebar));

	num = gtk_option_menu_get_history(option_menu);
	g_return_if_fail(num >= 0 && num < n_sidebar_entries);

	kz_sidebar_change_content(sidebar, sidebar_entries[num]);
}

static void
cb_close_button (GtkWidget *widget, KzSidebar *sidebar)
{
	g_return_if_fail(KZ_IS_SIDEBAR(sidebar));
	gtk_widget_hide(GTK_WIDGET(sidebar));
}
