/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gnet.h"

#include "gobject-utils.h"
#include "intl.h"
#include "kz-http.h"
#include "kazehakase.h"
#include "kz-profile.h"

enum {
	START_SIGNAL,
	FETCH_COMPLETED_SIGNAL,
	LAST_SIGNAL
};

enum {
	PROP_0,
	PROP_URI,
	PROP_HOSTNAME,
	PROP_PATH,
};


static void kz_http_class_init      (KzHTTPClass *klass);
static void kz_http_init            (KzHTTP *http);
static void kz_http_set_property    (GObject *object,
				     guint prop_id,
				     const GValue *value,
				     GParamSpec *pspec);
static void kz_http_get_property    (GObject *object,
				     guint prop_id,
				     GValue *value,
				     GParamSpec *pspec);

static void kz_http_connect  (GTcpSocket *socket, 
			      GTcpSocketConnectAsyncStatus status,
			      gpointer data);
static gboolean kz_http_in   (GIOChannel *iochannel,
			      GIOCondition condition,
			      gpointer data);

static GObjectClass *parent_class = NULL;
static gint kz_http_signals[LAST_SIGNAL] = {0};

KZ_OBJECT_GET_TYPE(kz_http, "KzHTTP", KzHTTP,
		   kz_http_class_init, kz_http_init,
		   G_TYPE_OBJECT)

static void
kz_http_class_init (KzHTTPClass *klass)
{
	GObjectClass *object_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;

	object_class->dispose = kz_http_dispose;
	object_class->set_property = kz_http_set_property;
	object_class->get_property = kz_http_get_property;
	
	klass->fetch_completed     = NULL;
	
	g_object_class_install_property(
		object_class,
		 PROP_URI,
		 g_param_spec_string(
			 "uri",
			 _("URI"),
			 _("The URI of Fetch file"),
			 NULL,
			 G_PARAM_READWRITE));
	g_object_class_install_property(
		object_class,
		 PROP_HOSTNAME,
		 g_param_spec_string(
			 "hostname",
			 _("Hostname"),
			 _("The Hostname of the URI"),
			 NULL,
			 G_PARAM_READWRITE));
	g_object_class_install_property(
		object_class,
		 PROP_PATH,
		 g_param_spec_string(
			 "path",
			 _("Path"),
			 _("The Path of the URI"),
			 NULL,
			 G_PARAM_READWRITE));

	kz_http_signals[FETCH_COMPLETED_SIGNAL]
		= g_signal_new ("fetch_completed",
				G_TYPE_FROM_CLASS (klass),
				G_SIGNAL_RUN_LAST,
				G_STRUCT_OFFSET (KzHTTPClass, fetch_completed),
				NULL, NULL,
				g_cclosure_marshal_VOID__VOID,
				G_TYPE_NONE, 0);
}


static void
kz_http_init (KzHTTP *http)
{
	http->uri             = NULL;
	http->hostname        = NULL;
	http->path            = NULL;
	
	http->header          = TRUE;
	http->buf             = NULL;
}


void
kz_http_dispose (GObject *object)
{
	KzHTTP *http;

	g_return_if_fail(KZ_IS_HTTP(object));

	http = KZ_HTTP(object);
	if (http->uri)
		g_free(http->uri);
	if (http->hostname)
		g_free(http->hostname);
	if (http->path)
		g_free(http->path);
	if (http->buf)
		g_string_free(http->buf, TRUE);
	http->uri      = NULL;
	http->hostname = NULL;
	http->path     = NULL;
	http->buf      = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


static void
kz_http_set_property (GObject *object,
		      guint prop_id,
		      const GValue *value,
		      GParamSpec *pspec)
{
	KzHTTP *http = KZ_HTTP(object);

	switch (prop_id)
	{
	case PROP_URI:
		g_free(http->uri);
		http->uri = g_value_dup_string(value);
		break;
	case PROP_HOSTNAME:
		g_free(http->hostname);
		http->hostname = g_value_dup_string(value);
		break;
	case PROP_PATH:
		g_free(http->path);
		http->path = g_value_dup_string(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
kz_http_get_property (GObject *object,
		      guint prop_id,
		      GValue *value,
		      GParamSpec *pspec)
{
	KzHTTP *http = KZ_HTTP(object);

	switch (prop_id)
	{
	case PROP_URI:
		g_value_set_string(value, http->uri);
		break;
	case PROP_HOSTNAME:
		g_value_set_string(value, http->hostname);
		break;
	case PROP_PATH:
		g_value_set_string(value, http->path);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


KzHTTP *
kz_http_new (gchar *uri)
{
	GURI *guri;
	gchar *path;
	guri = gnet_uri_new(uri);
	
	if (guri->query)
		path = g_strdup_printf("%s?%s", guri->path, guri->query);
	else
		path = g_strdup(guri->path);
	
	KzHTTP *http = KZ_HTTP(g_object_new(KZ_TYPE_HTTP,
					 "uri", uri,
					 "hostname", guri->hostname,
					 "path", path,
					 NULL));
	
	http->buf = g_string_sized_new(1);
	gnet_uri_delete(guri);
	g_free(path);
	return http;
}


void
kz_http_file_fetch(KzHTTP *http)
{
	GTcpSocketConnectAsyncID sockID;
	GURI *uri = NULL;
	gchar proxy[1024];
	gboolean exist;
	guint port = 80;
	gchar *host;

	/* proxy check */
	exist = kz_profile_get_value(kz_global_profile,
				     "Global", "proxy",
				     proxy, sizeof(proxy),
				     KZ_PROFILE_VALUE_TYPE_STRING);
	if (exist)
	{
		uri = gnet_uri_new(proxy);
		http->use_proxy = TRUE;
		port = uri->port;
		host = uri->hostname;
	}
	else 
	{
		host = http->hostname;
	}

	sockID = gnet_tcp_socket_connect_async(host, port,
					       kz_http_connect, http);
	
	if(uri)
		gnet_uri_delete(uri);
}


static gboolean
kz_http_in(GIOChannel *iochannel, GIOCondition condition,
	   gpointer data)
{
	KzHTTP *http;
	GIOStatus iostatus;
	GError *e = NULL;
	GString *buffer = NULL;
	guint bytes_read;
	gchar buf[1024];

	http = KZ_HTTP(data);
	
	if ((condition & G_IO_ERR) || (condition & G_IO_HUP))
	{
		g_warning("IO ERROR");
		g_io_channel_unref(iochannel);
		return FALSE;
	}
			
	if (http->header) /* Header Section */
	{
		buffer = g_string_sized_new(0);
		/* Read the data into our buffer */
		iostatus = g_io_channel_read_line_string(iochannel,
						  buffer,
						  NULL,
						  &e);

		if ( iostatus == G_IO_STATUS_ERROR )
		{
			g_warning("%s", e->message);
			g_string_free(buffer, TRUE);
			return FALSE;
		}
		if (strncmp(buffer->str, "HTTP/1.1", 8) == 0)
		{
			if (strncmp(buffer->str, "HTTP/1.1 200 OK", 15) != 0)
			{
				g_warning("%s", buffer->str);
				g_string_free(buffer, TRUE);
				return FALSE;
			}
		}
		else if ( strncmp(buffer->str, "Content-Length:", 15) == 0)
		{
			http->content_length = strtol(buffer->str + 15, NULL, 10);
			http->buf = g_string_sized_new(http->content_length);
		}
		else if (strncmp(buffer->str, "\r\n", 2) == 0)
		{
			http->header = FALSE;
		}
		g_string_free(buffer, TRUE);
	}
	else /* Content Section */
	{
		/* Read the data into our buffer */
		iostatus = g_io_channel_read_chars(iochannel, buf, 
						   sizeof(buf),
						   &bytes_read,
						   &e);

		if ( iostatus == G_IO_STATUS_ERROR )
			g_warning("%s", e->message);

		g_string_append_len(http->buf, buf, bytes_read);
		/* Check for EOF */
		if (bytes_read == 0)
		{
			g_io_channel_unref(iochannel);
			g_signal_emit(G_OBJECT(http),
				      kz_http_signals[FETCH_COMPLETED_SIGNAL],
				      0);
			return FALSE;
		}
	}
	return TRUE;
}


static void 
kz_http_connect(GTcpSocket *socket,
		GTcpSocketConnectAsyncStatus status, gpointer data)
{
	KzHTTP *http;
	GIOChannel *iochannel;
	GIOStatus iostatus;
	gchar *command;
	GError *e = NULL;
	guint n;
	gchar *URL;

	if ( status != GTCP_SOCKET_CONNECT_ASYNC_STATUS_OK )
		return;
	http = KZ_HTTP(data);

	/* Get the IOChannel */
	iochannel = gnet_tcp_socket_get_io_channel(socket);
	g_assert (iochannel != NULL);

	g_io_channel_set_flags(iochannel, G_IO_FLAG_NONBLOCK, NULL);

	/* proxy */
	if (http->use_proxy)
	{
		URL = g_strdup_printf("http://%s%s",
				      http->hostname,
				      http->path);
	}
	else
	{
		URL = g_strdup(http->path);
	}
		
	/* Send GET command */
	command = g_strconcat("GET ", URL, " HTTP/1.1\r\n",
			      "Host: ", http->hostname, "\r\n",
			      "User-Agent: Kazehakase\r\n",
			      "Connection: close\r\n\r\n", NULL);

	iostatus = g_io_channel_write_chars(iochannel, command,
					    strlen(command), &n, &e);

	g_free(command);
	g_free(URL);

	if ( iostatus != G_IO_STATUS_NORMAL )
		g_warning("%s", e->message);

	g_io_channel_set_buffered(iochannel, TRUE);

	g_io_add_watch(iochannel, G_IO_IN | G_IO_PRI | G_IO_ERR | G_IO_HUP,
		       kz_http_in, http);
}
