/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <unistd.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "config.h"
#include "google.h"
#include "gtk-utils.h"
#include "utils.h"
#include "kz-icons.h"
#include "kz-bookmark-menu.h"
#include "kz-actions.h"
#include "intl.h"

#define GOOGLE_URI "http://www.google.com/search?lr=lang_ja&hl=ja&ie=UTF-8&oe=utf8&q=%s"

#define MAX_HISTORY 32

static gchar       *google_url_encode        (const gchar* utf8src);
static gchar       *google_entry_get_uri     (Google *google);
static const gchar *google_entry_get_keyword (Google *google);

/* callbacks */
static void     cb_google_entry_activate  (GtkEntry *widget, Google *google);
static gboolean cb_google_entry_key_press (GtkWidget *widget, GdkEventKey *event,
					   Google *google);

static gboolean cb_google_icon_press      (GtkWidget *widget, GdkEventButton *event,
					   Google *google);

static void
cb_google_destroy (GtkWidget *hbox, Google *google)
{
	/* kz_bookmark_save(google->history); */
	g_object_unref(G_OBJECT(google->history));
	
	g_free(google);
}


/* create new google search item */
GtkWidget *
google_new (KzWindow *kz)
{
	GtkWidget *hbox, *pixmap, *button;
	gchar *history_file;
	
	Google *google =NULL;
	google = g_new0(Google, 1);
	google->kz = kz;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	g_signal_connect(G_OBJECT(hbox), "destroy",
			 G_CALLBACK(cb_google_destroy), google);
	
	/* create google icon button */
	button = gtk_button_new();
	google->icon = button;
	gtk_button_set_relief((GtkButton *) button, GTK_RELIEF_NONE);
	pixmap = gtk_image_new_from_stock(KZ_STOCK_GOOGLE,
					  GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(button), pixmap);

	/* create the keyword entry */
	google->entry = gtk_entry_new();
	gtk_widget_set_size_request(google->entry, 100, -1);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), google->entry, TRUE, TRUE, 0);
	
	g_signal_connect(G_OBJECT(GTK_ENTRY(google->entry)), 
			 "activate",
			 G_CALLBACK(cb_google_entry_activate), google);
	g_signal_connect(G_OBJECT(google->entry),
			 "key-press-event",
			 G_CALLBACK(cb_google_entry_key_press), google);
	g_signal_connect(G_OBJECT(google->icon), 
			 "button_press_event",
			 G_CALLBACK(cb_google_icon_press), google);	
	gtk_widget_show_all(hbox);

	g_object_set_data(G_OBJECT(hbox), "Google::Google", google);
	
	/* history */
	/*
	history_file = g_strconcat(g_get_home_dir(),
				   "/.kazehakase/google_history.xml",
				   NULL);
	if (g_file_test(history_file, G_FILE_TEST_EXISTS))
	{
		google->history = kz_bookmark_file_new(history_file, _("Google Search History"),
						       "XBEL");
		kz_bookmark_load_in_memory_start(google->history);
	}
	else
		google->history = kz_bookmark_file_create_new(history_file, _("Google Search History"),
							      "XBEL");
	g_free(history_file);
	 */
	google->history = kz_bookmark_pure_folder_new();
	kz_bookmark_set_title(google->history, _("Google Search History"));
	google->history_menu = kz_bookmark_menu_create_submenu(google->history,
							       google->kz);
	return hbox;
}

Google *
google_get (GtkWidget *widget)
{
	g_return_val_if_fail(G_IS_OBJECT(widget), NULL);
	return g_object_get_data(G_OBJECT(widget), "Google::Google");
}

static gboolean
cb_google_icon_press (GtkWidget *widget, GdkEventButton *event,
		      Google *google)
{
	KzWindow *kz;
	GdkEventButton *event_button;

	kz = google->kz;

	event_button = (GdkEventButton *)event;

	switch(event_button->button)
	{
	case 1:
		if (!GTK_IS_MENU(google->history_menu)) return FALSE;

		gtk_menu_popup(GTK_MENU(google->history_menu), NULL, NULL, 
			       gtkutil_menu_position_under_widget,
			       GTK_WIDGET(google->icon),
			       event_button->button, event_button->time);
		break;
	case 2:
		break;
	case 3:
		kz_actions_popup_bookmark_menu_modal(kz, google->history,
						     event->button,
						     event->time);
		break;
	}

	return FALSE;
}


static void
google_add_history (Google *google, const gchar *uri)
{
	const gchar *title;
	KzBookmark *bookmark;
	GList *list, *node, *children;
	gint history_num;

	children = kz_bookmark_get_children(google->history);

	title = google_entry_get_keyword(google);
	bookmark = kz_bookmark_new_with_attrs(title, uri, NULL);
	kz_bookmark_prepend(google->history, bookmark);
	g_object_unref(bookmark);	
	
	history_num = g_list_length(children);
	
	if (history_num > MAX_HISTORY)
	{
		list = g_list_nth(children, MAX_HISTORY);
		for (node = list; node; node = g_list_next(node))
		{
			kz_bookmark_remove(google->history, node->data);
		}
	}	
	g_list_free(children);
}


static void
cb_google_entry_activate (GtkEntry *widget, Google *google)
{
	gchar *uri;
	gboolean new_tab;

	uri = google_entry_get_uri(google);

	KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);
	if (new_tab)
		kz_window_open_new_tab(google->kz, uri);
	else	
		kz_window_load_url(google->kz, uri);

	google_add_history(google, uri);
	gtk_entry_set_text(widget, "");

	g_free(uri);
}


static gboolean
cb_google_entry_key_press (GtkWidget *widget, GdkEventKey *event,
			   Google *google)
{
	gboolean new_tab;

	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter)
	    && (event->state & GDK_CONTROL_MASK))
	{
		gchar *uri;
		KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);

		uri = google_entry_get_uri(google);
		if (new_tab)
			kz_window_load_url(google->kz, uri);
		else 
			kz_window_open_new_tab(google->kz, uri);

		google_add_history(google, uri);
		gtk_entry_set_text(GTK_ENTRY(widget), "");

		g_free(uri);
		return TRUE;
	}

	return FALSE;
}


static const gchar *
google_entry_get_keyword (Google *google)
{
	GtkEntry *entry = GTK_ENTRY(google->entry);
	const gchar *text= gtk_entry_get_text(entry);
	
	return text;
}


static gchar *
google_entry_get_uri (Google *google)
{
	gchar *text_encoded;
	gchar *search_uri;

	text_encoded = google_url_encode(google_entry_get_keyword(google));
	search_uri = g_strdup_printf(GOOGLE_URI, text_encoded);
	g_free(text_encoded);
	return search_uri;
}


/* return hex-encoded UTF-8 data
 * please free returned gchar* if unnecessary
 */
static gchar *
google_url_encode(const gchar* utf8src)
{
	GString *dest;
	const gchar *ch = utf8src;
	unsigned char buf;

	dest = g_string_sized_new(strlen(utf8src));

	while(*ch != '\0')
	{
		if (((*ch >= 'A') && (*ch <= 'Z')) ||
		    ((*ch >= 'a') && (*ch <= 'z')) ||
		    ((*ch >= '0') && (*ch <= '9')))
		{
			g_string_append_c(dest, *ch);
		}
		else if (*ch == ' ')
		{
			g_string_append_c(dest, '+');

		}
		else
		{
			g_string_append_c(dest, '%');
			buf = (*ch >> 4) & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
			buf = *ch & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
		}
		ch++;
	}
	
	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}
