/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "gobject-utils.h"
#include "kz-bookmarkbar.h"
#include "prefs.h"
#include "rss-menu.h"
#include "utils.h"

static void kz_bookmark_bar_class_init    (KzBookmarkBarClass *class);
static void kz_bookmark_bar_init          (KzBookmarkBar *kzbkmkbar);

static GtkToolbarClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_bookmark_bar, "KzBookmarkBar", KzBookmarkBar,
		   kz_bookmark_bar_class_init, kz_bookmark_bar_init,
		   GTK_TYPE_TOOLBAR)

typedef enum {
	STATE_START,
	STATE_RSSBAR,
	STATE_RSS,
	STATE_END
} ParseState;

typedef struct _ParseContext ParseContext;
struct _ParseContext
{
	/* parser state information */
	ParseState state;

	GtkWidget *toolbar;
};

static void
start_element_handler (GMarkupParseContext *context,
		       const gchar         *element_name,
		       const gchar        **attribute_names,
		       const gchar        **attribute_values,
		       gpointer             user_data,
		       GError             **error)
{
	int i;
	ParseContext *ctx = user_data;

	switch (element_name[0])
	{
	 case 'r':
		if (ctx->state == STATE_START && !strcmp(element_name, "rssbar"))
		{
			ctx->state = STATE_RSSBAR;
		}
		else if (ctx->state == STATE_RSSBAR &&
			 !strcmp(element_name, "rss"))
		{
			gchar *uri = NULL;
			gchar *title = NULL;
			guint update_interval = 0;
			KzRSS *rss = NULL;
			GtkWidget *rss_menu;
			ctx->state = STATE_RSS;

			for (i = 0; attribute_names[i] != NULL; i++)
			{
				if (!strcmp(attribute_names[i], "uri"))
					uri = g_strdup(attribute_values[i]);
				else if (!strcmp(attribute_names[i], "title"))
					title = g_strdup(attribute_values[i]);
				else if (!strcmp(attribute_names[i], "interval"))
					update_interval = atoi(attribute_values[i]);
			}
			rss = kz_rss_new(uri, title, update_interval);
			rss_menu = rss_menu_new(KZ_BOOKMARK_BAR(ctx->toolbar)->parent_kazehakase,rss);
			gtk_toolbar_append_widget(GTK_TOOLBAR(ctx->toolbar),
						  rss_menu, NULL, NULL);
			g_free(uri);
			g_free(title);
		}
		break;
	}
}

static void
end_element_handler (GMarkupParseContext *context,
		     const gchar         *element_name,
		     gpointer             user_data,
		     GError             **error)
{
	ParseContext *ctx = user_data;
	
	switch (ctx->state)
	{
	 case STATE_START:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	 case STATE_RSSBAR:
		ctx->state = STATE_END;
		break;
	 case STATE_RSS:
		ctx->state = STATE_RSSBAR;
		break;
	 case STATE_END:
		g_warning("shouldn't get any end tags at this point");
		/* should do a GError here */
		break;
	}
}

static GMarkupParser ui_parser = {
	start_element_handler,
	end_element_handler,
	NULL,
	NULL,
	NULL
};

static gboolean
kz_bookmark_bar_parse_from_string (GtkWidget *toolbar, gpointer user_data,
				   const gchar *buffer, guint length,
				   GError **error)
{
	ParseContext ctx = { 0 };
	GMarkupParseContext *context;
	gboolean res = TRUE;

	g_return_val_if_fail(buffer != NULL, FALSE);

	ctx.state = STATE_START;
	ctx.toolbar = toolbar;

	context = g_markup_parse_context_new(&ui_parser, 0, &ctx, NULL);
	if (length < 0)
		length = strlen(buffer);

	if (g_markup_parse_context_parse(context, buffer, length, error))
	{
		if (!g_markup_parse_context_end_parse(context, error))
			res = FALSE;
	}
	else
		res = FALSE;

	g_markup_parse_context_free (context);

	return res;
}

static gboolean
kz_bookmark_bar_parse_from_file (GtkWidget *toolbar, gpointer user_data,
				 const gchar *filename, GError **error)
{
	gchar *buffer;
	gint length;
	gboolean res;

	if (!g_file_get_contents (filename, &buffer, &length, error))
		return FALSE;

	res = kz_bookmark_bar_parse_from_string(toolbar, user_data,
						buffer, length, error);
	g_free(buffer);

	return res;
}

static void
kz_bookmark_bar_class_init (KzBookmarkBarClass *class)
{
	GtkObjectClass *object_class;

	parent_class = g_type_class_peek_parent (class);

	object_class = (GtkObjectClass *) class;

	/* GtkObject signals */
	/* object_class->destroy = kz_bookmark_bar_destroy; */
}

static void
kz_bookmark_bar_init (KzBookmarkBar *kzbkmkbar)
{
	kzbkmkbar->parent_kazehakase = NULL;
	kzbkmkbar->items = NULL;
}

/* create new rssbar */
GtkWidget *
kz_bookmark_bar_new (KzWindow *kz)
{
	KzBookmarkBar *kzbkmkbar
		= KZ_BOOKMARK_BAR(g_object_new(KZ_TYPE_BOOKMARK_BAR, NULL));
        gchar *toolbarfile;

	/* build the filename */
        toolbarfile = g_strconcat(g_get_home_dir(),
				  "/.kazehakase/toolbar.xml",
				  NULL);
	
	kzbkmkbar->parent_kazehakase = kz;
	kzbkmkbar->items = NULL;

	/* load toolbar config file */
	kz_bookmark_bar_parse_from_file(GTK_WIDGET(kzbkmkbar), NULL,
					toolbarfile, NULL);

	g_free(toolbarfile);
	return GTK_WIDGET(kzbkmkbar);
}
