/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>
#include <unistd.h>
#include <string.h>
#include <gdk/gdkkeysyms.h>

#include "config.h"
#include "google.h"
#include "gtk-utils.h"
#include "utils.h"
#include "kz-icons.h"

#define GOOGLE_URI "http://www.google.com/search?lr=lang_ja&hl=ja&ie=UTF-8&oe=utf8&q=%s"

static gchar *google_url_encode    (const gchar* utf8src);
static gchar *google_entry_get_uri (Google *google);

/* callbacks */
static void     google_entry_activate_cb  (GtkEntry *widget, Google *google);
static gboolean google_entry_key_press_cb (GtkWidget *widget, GdkEventKey *event,
					   Google *google);

static gboolean google_icon_press_cb      (GtkWidget *widget, GdkEvent *event,
					   SearchResult *r);

static void
cb_google_destroy (GtkWidget *hbox, Google *google)
{
	g_free(google);
}


/* create new google search item */
GtkWidget *
google_new (KzWindow *kz)
{
	GtkWidget *hbox, *pixmap, *button;
	
	Google *google =NULL;
	google = g_new0(Google, 1);
	google->kz = kz;

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	g_signal_connect(G_OBJECT(hbox), "destroy",
			 G_CALLBACK(cb_google_destroy), google);
	
	/* create google icon button */
	button = gtk_button_new();
	google->Icon = button;
	gtk_button_set_relief((GtkButton *) button, GTK_RELIEF_NONE);
	pixmap = gtk_image_new_from_stock(KZ_STOCK_GOOGLE,
					  GTK_ICON_SIZE_SMALL_TOOLBAR);
	gtk_container_add(GTK_CONTAINER(button), pixmap);

	/* create the keyword entry */
	google->KeywordEntry = gtk_entry_new();
	gtk_widget_set_size_request(google->KeywordEntry, 100, -1);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), google->KeywordEntry, TRUE, TRUE, 0);
	
	g_signal_connect(G_OBJECT(GTK_ENTRY(google->KeywordEntry)), 
			 "activate",
			 G_CALLBACK(google_entry_activate_cb), google);
	g_signal_connect(G_OBJECT(google->KeywordEntry),
			 "key-press-event",
			 G_CALLBACK(google_entry_key_press_cb), google);
	g_signal_connect(G_OBJECT(google->Icon), 
			 "button_press_event",
			 G_CALLBACK(google_icon_press_cb), google);	
	gtk_widget_show_all(hbox);

	g_object_set_data(G_OBJECT(hbox), "Google::Google", google);
	
	return hbox;
}

Google *
google_get (GtkWidget *widget)
{
	g_return_val_if_fail(G_IS_OBJECT(widget), NULL);
	return g_object_get_data(G_OBJECT(widget), "Google::Google");
}

static gboolean
google_icon_press_cb(GtkWidget *widget, GdkEvent *event,
		     SearchResult *r)
{
#if 0	
	GdkEventButton *event_button;
	if (event->type == GDK_BUTTON_PRESS) {
		event_button = (GdkEventButton *)event;
		if (event_button->button == 1) {
			gtk_menu_popup(GTK_MENU(r->ResultMenu), NULL, NULL,
				       menu_position_under_widget,
				       widget,
				       event_button->button, event_button->time);
			return TRUE;
		}
	}
#endif

	return FALSE;
}

static void
google_entry_activate_cb (GtkEntry *widget, Google *google)
{
	gchar *uri;
	uri = google_entry_get_uri(google);

	kz_window_load_url(google->kz, uri);
	gtk_entry_set_text(widget, "");

	g_free(uri);
}


static gboolean
google_entry_key_press_cb (GtkWidget *widget, GdkEventKey *event,
			   Google *google)
{
	if ((event->keyval == GDK_Return || event->keyval == GDK_ISO_Enter) &&
	    (event->state & GDK_CONTROL_MASK))
	{
		gchar *uri;
		uri = google_entry_get_uri(google);
		kz_window_open_new_tab(google->kz, uri);
		gtk_entry_set_text(widget, "");
		g_free(uri);

		return TRUE;
	}

	return FALSE;
}


static gchar *
google_entry_get_uri (Google *google)
{
	GtkEntry *entry = GTK_ENTRY(google->KeywordEntry);
	const gchar *text= gtk_entry_get_text(entry);
	gchar *text_encoded;
	gchar *search_uri;

	text_encoded = google_url_encode((gchar*)text);
	search_uri = g_strdup_printf(GOOGLE_URI, text_encoded);
	g_free(text_encoded);
	return search_uri;
}


/* return hex-encoded UTF-8 data
 * please free returned gchar* if unnecessary
 */
static gchar *
google_url_encode(const gchar* utf8src)
{
	GString *dest;
	const gchar *ch = utf8src;
	unsigned char buf;

	dest = g_string_sized_new(strlen(utf8src));

	while(*ch != '\0')
	{
		if (((*ch >= 'A') && (*ch <= 'Z')) ||
		    ((*ch >= 'a') && (*ch <= 'z')) ||
		    ((*ch >= '0') && (*ch <= '9')))
		{
			g_string_append_c(dest, *ch);
		}
		else if (*ch == ' ')
		{
			g_string_append_c(dest, '+');

		}
		else
		{
			g_string_append_c(dest, '%');
			buf = (*ch >> 4) & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
			buf = *ch & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
		}
		ch++;
	}
	
	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}
