/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_privacy.h"
#include <glib/gi18n.h>

#include "mozilla-prefs.h"
#include "kazehakase.h"
#include "kz-window.h"

#define DATA_KEY "KzPrefsPrivacy::info"


static GtkWidget     *prefs_privacy_create          (void);
static void           prefs_privacy_response        (GtkWidget *widget,
						     gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/Privacy"),
	priority_hint:   0,
	ui_level:        KZ_UI_LEVEL_EXPERT,
	create:          prefs_privacy_create,
	response:        prefs_privacy_response,
};


typedef struct _KzPrefsPrivacy
{
	GtkWidget      *main_vbox;
	GtkWidget      *cookie_behaviour;
	GtkWidget      *cookie_lifetime;
	GtkWidget      *span_spin;
	GtkWidget      *except_check;
	gboolean       changed;
} KzPrefsPrivacy;


static void
cb_changed (GtkWidget *widget, KzPrefsPrivacy *prefsui)
{
	prefsui->changed = TRUE;
}


static void
cb_lifespan_span_changed (GtkToggleButton *button, KzPrefsPrivacy *prefsui)
{
	gtk_widget_set_sensitive(prefsui->span_spin,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}


static void
cb_lifespan_except_changed (GtkToggleButton *button, KzPrefsPrivacy *prefsui)
{
	gtk_widget_set_sensitive(prefsui->except_check,
				 gtk_toggle_button_get_active(button));
	prefsui->changed = TRUE;
}


static void
cb_password_button_clicked (GtkWidget *widget, KzPrefsPrivacy *prefsui)
{
	GtkWidget *top;
	KzWindow *kz;
	GtkAction *action;

	top = gtk_widget_get_toplevel(prefsui->main_vbox);
	if (!GTK_IS_WINDOW(top)) return;

	kz = KZ_WINDOW(gtk_window_get_transient_for(GTK_WINDOW(top)));
	if (!KZ_IS_WINDOW(kz)) return;

	action = gtk_action_group_get_action(kz->actions, "ShowPassword");
	gtk_action_activate(action);
}


static void
prefs_privacy_destroy (gpointer data)
{
	KzPrefsPrivacy *prefsui = data;

	g_free(prefsui);
}


static GtkWidget *
prefs_privacy_create (void)
{
	KzPrefsPrivacy *prefsui = g_new0(KzPrefsPrivacy, 1);
	GtkWidget *main_vbox, *vbox, *hbox, *frame;
	GtkWidget *radio1, *radio2, *radio3, *radio4;
	GtkWidget *label, *spin, *check;
	GtkWidget *button;
	GtkAdjustment *adj;
	gint behaviour, lifetime, span;
	gboolean except;

	main_vbox = gtk_vbox_new(FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full(G_OBJECT(main_vbox), DATA_KEY,
			       prefsui, prefs_privacy_destroy);

	label = kz_prefs_ui_utils_create_title(_("Privacy"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	/*
	 *  cookie acceptance policy
	 */
	frame = gtk_frame_new(_("Cookie Acceptance Policy"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	/*
	 *  cookie behaviour
	 */ 
	prefsui->cookie_behaviour = radio1
		= gtk_radio_button_new_with_mnemonic(NULL,
						  _("_Allow all cookies"));
	radio2 = gtk_radio_button_new_with_mnemonic_from_widget(
		 	GTK_RADIO_BUTTON(radio1),
			_("Allow cookies for the originating web site _only"));
	radio3 = gtk_radio_button_new_with_mnemonic_from_widget(
		 	GTK_RADIO_BUTTON(radio1),
			_("_Block cookies"));
	
	mozilla_prefs_get_int("network.cookie.cookieBehavior", &behaviour);
	switch (behaviour)
	{
	 case 0:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio1), TRUE);
		break;
	 case 1:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio2), TRUE);
		break;
	 case 2:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio3), TRUE);
		break;
	 default:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio1), TRUE);
		break;
	}
	
	gtk_box_pack_start(GTK_BOX(vbox), radio1, TRUE, TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), radio2, TRUE, TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), radio3, TRUE, TRUE, 2);
	g_signal_connect(G_OBJECT(radio1), "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	g_signal_connect(G_OBJECT(radio2), "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	g_signal_connect(G_OBJECT(radio3), "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	gtk_widget_show(radio1);
	gtk_widget_show(radio2);
	gtk_widget_show(radio3);

	/*
	 *  cookie lifetime policy
	 */
	frame = gtk_frame_new(_("Cookie Lifetime Policy"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	/*
	 *  cookie lifetime
	 */ 
	prefsui->cookie_lifetime = radio1
		= gtk_radio_button_new_with_mnemonic(NULL,
						  _("Accept cookie _normally"));
	gtk_box_pack_start(GTK_BOX(vbox), radio1, TRUE, TRUE, 2);

	radio2 = gtk_radio_button_new_with_mnemonic_from_widget(
		 	GTK_RADIO_BUTTON(radio1),
			_("Accept for current _session only"));
	gtk_box_pack_start(GTK_BOX(vbox), radio2, TRUE, TRUE, 2);

	radio3 = gtk_radio_button_new_with_mnemonic_from_widget(
		 	GTK_RADIO_BUTTON(radio1),
			_("Accept cookie _for"));
	hbox = gtk_hbox_new(FALSE, 0);	
	gtk_box_pack_start(GTK_BOX(hbox), radio3, FALSE, FALSE, 0);
	prefsui->span_spin = spin
		= gtk_spin_button_new_with_range(0, 1000, 1);
	adj = gtk_spin_button_get_adjustment(GTK_SPIN_BUTTON(spin));
	mozilla_prefs_get_int("network.cookie.lifetime.days", &span);
	gtk_adjustment_set_value(adj, span);
	gtk_box_pack_start(GTK_BOX(hbox), spin, FALSE, FALSE, 2);	
	gtk_widget_set_sensitive(spin, FALSE);
	g_signal_connect(G_OBJECT(spin), "value-changed",
			 G_CALLBACK(cb_changed), prefsui);
	label = gtk_label_new_with_mnemonic(_("_days"));
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), spin);
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 2);	
	gtk_box_pack_start(GTK_BOX(vbox), hbox,   TRUE, TRUE, 2);

	radio4 = gtk_radio_button_new_with_mnemonic_from_widget(
		 	GTK_RADIO_BUTTON(radio1),
			_("Ask for _each cookie"));
	hbox = gtk_hbox_new(FALSE, 0);	
	gtk_box_pack_start(GTK_BOX(hbox), radio4, FALSE, FALSE, 0);
	prefsui->except_check = check =
		gtk_check_button_new_with_mnemonic(_("e_xcept for session cookies"));
	mozilla_prefs_get_boolean("network.cookie.alwaysAcceptSessionCookies", &except);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), except);
	gtk_box_pack_start(GTK_BOX(hbox), check, FALSE, FALSE, 2);
	gtk_widget_set_sensitive(check, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, TRUE, TRUE, 2);
	
	mozilla_prefs_get_int("network.cookie.lifetimePolicy", &lifetime);
	switch (lifetime)
	{
	 case 0:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio1), TRUE);
		break;
	 case 1:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio2), TRUE);
		break;
	 case 2:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio3), TRUE);
		break;
	 case 3:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio4), TRUE);
		break;
	 default:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(radio1), TRUE);
		break;
	}
	
	g_signal_connect(G_OBJECT(radio1), "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	g_signal_connect(G_OBJECT(radio2), "toggled",
			 G_CALLBACK(cb_changed), prefsui);
	g_signal_connect(G_OBJECT(radio3), "toggled",
			 G_CALLBACK(cb_lifespan_span_changed), prefsui);
	g_signal_connect(G_OBJECT(radio4), "toggled",
			 G_CALLBACK(cb_lifespan_except_changed), prefsui);
	gtk_widget_show_all(vbox);

	/*
	 *  password manager 
	 */
	frame = gtk_frame_new(_("Password"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_container_set_border_width(GTK_CONTAINER(frame), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), frame, FALSE, FALSE, 2);
	gtk_widget_show(frame);

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 4);
	gtk_container_add(GTK_CONTAINER(frame), vbox);
	gtk_widget_show(vbox);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 0);
	gtk_widget_show(hbox);

	/* remove button */
	button = gtk_button_new_with_mnemonic(_("_Manage Stored Passwords..."));
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(G_OBJECT(button), "clicked",
			 G_CALLBACK(cb_password_button_clicked), prefsui);
	gtk_widget_show(button);

	prefsui->changed = FALSE;

	return main_vbox;
}


static void
prefs_privacy_response (GtkWidget *widget, gint response)
{
	KzPrefsPrivacy *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
	{
		GtkAdjustment *adj;
		GSList *group;
		GSList *radio;
		gint behaviour = 0;
		gint lifetime = 0;
		gint span, length;

		gboolean except;
		if (!prefsui->changed)
			break;
		
		/* cokkie behaviour */
		group  = gtk_radio_button_get_group(GTK_RADIO_BUTTON(prefsui->cookie_behaviour));
		length = g_slist_length(group);

		for (radio = group; radio; radio = g_slist_next(radio))
		{
			gboolean check;
			GtkRadioButton *button = radio->data;
			check = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
			if (check)
				break;
			behaviour++;
		}
		mozilla_prefs_set_int("network.cookie.cookieBehavior",
				      (length - 1) - behaviour);

		/* cokkie lifetime */
		group  = gtk_radio_button_get_group(GTK_RADIO_BUTTON(prefsui->cookie_lifetime));
		length = g_slist_length(group);

		for (radio = group; radio; radio = g_slist_next(radio))
		{
			gboolean check;
			GtkRadioButton *button = radio->data;
			check = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(button));
			if (check)
				break;
			lifetime++;
		}
		mozilla_prefs_set_int("network.cookie.lifetimePolicy",
				      (length - 1) - lifetime);
		
		/* cookie lifespan days */
		adj = gtk_spin_button_get_adjustment
			(GTK_SPIN_BUTTON(prefsui->span_spin));
		span = gtk_adjustment_get_value(adj);
		mozilla_prefs_set_int("network.cookie.lifetime.days",
				      span);
		
		/* except for session cookies */
		except = gtk_toggle_button_get_active
			      (GTK_TOGGLE_BUTTON(prefsui->except_check));
		mozilla_prefs_set_boolean("network.cookie.alwaysAcceptSessionCookies",
					  except);

		prefsui->changed = FALSE;

		break;
	}
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_privacy_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
