/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "gobject-utils.h"
#include <glib/gi18n.h>
#include "kz-file.h"

static void kz_file_class_init      (KzFileClass *klass);
static void kz_file_init            (KzFile *file);
static void kz_file_set_property    (GObject *object,
				     guint prop_id,
				     const GValue *value,
				     GParamSpec *pspec);
static void kz_file_get_property    (GObject *object,
				     guint prop_id,
				     GValue *value,
				     GParamSpec *pspec);
static void kz_file_dispose         (GObject *object);
static void kz_file_finalize        (GObject *object);

static void      file_start      (KzIO *io);
static GIOStatus read_from_io    (KzIO *io,
				  GIOChannel *iochannel);
static GIOStatus write_to_io     (KzIO *io,
				  GIOChannel *iochannel);

struct _KzFilePrivate 
{
	gchar *filename;
};

enum {
	PROP_0,
	PROP_FILENAME
};

#define BUFFER_SIZE 1024

static KzIOClass *parent_class = NULL;

KZ_OBJECT_GET_TYPE(kz_file, "KzFile", KzFile,
		   kz_file_class_init, kz_file_init,
		   KZ_TYPE_IO)

static void
kz_file_class_init (KzFileClass *klass)
{
	GObjectClass *object_class;
	KzIOClass *io_class;

	parent_class = g_type_class_peek_parent (klass);
	object_class = (GObjectClass *) klass;
	io_class     = (KzIOClass *) klass;

	object_class->dispose      = kz_file_dispose;
	object_class->finalize     = kz_file_finalize;
	object_class->set_property = kz_file_set_property;
	object_class->get_property = kz_file_get_property;
	
	io_class->read_from_io  = read_from_io;
	io_class->write_to_io   = write_to_io;
	io_class->io_start      = file_start;

	g_object_class_install_property(
		object_class,
		PROP_FILENAME,
		g_param_spec_string(
			"filename",
			_("Filename"),
			_("The Filename of the local file"),
			NULL,
			G_PARAM_READWRITE));
}


static void
kz_file_init (KzFile *file)
{
	file->priv = g_new0(KzFilePrivate, 1);
	
	file->priv->filename = NULL;
}


static void
kz_file_dispose (GObject *object)
{
	KzFile *file;

	g_return_if_fail(KZ_IS_FILE(object));

	file = KZ_FILE(object);

	if (file->priv->filename)
	{
		g_free(file->priv->filename);
	}
	file->priv->filename = NULL;

	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


KZ_OBJECT_FINALIZE(kz_file, KzFile)


static void
kz_file_set_property (GObject *object,
		      guint prop_id,
		      const GValue *value,
		      GParamSpec *pspec)
{
	KzFile *file = KZ_FILE(object);

	switch (prop_id)
	{
	case PROP_FILENAME:
		g_free(file->priv->filename);
		file->priv->filename = g_value_dup_string(value);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}


static void
kz_file_get_property (GObject *object,
		      guint prop_id,
		      GValue *value,
		      GParamSpec *pspec)
{
	KzFile *file = KZ_FILE(object);

	switch (prop_id)
	{
	case PROP_FILENAME:
		g_value_set_string(value, file->priv->filename);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
		break;
	}
}



KzFile *
kz_file_new (const gchar *uri)
{
	KzFile *file;

	file = g_object_new(KZ_TYPE_FILE,
			    "filename", uri,
			    NULL);
	
	return file;
}


static GIOStatus
kz_file_prepare_iochannel (KzIO *io)
{
	GIOStatus iostatus;
	KzFile *file;
	gchar *filename;

	file = KZ_FILE(io);
	
	filename = file->priv->filename;

	if (kz_io_get_mode(io) == KZ_IO_READ)
	{
		if (g_file_test(filename, G_FILE_TEST_EXISTS) == FALSE)
		{
			KZ_IO_CLASS (parent_class)->io_error(io);
			return G_IO_STATUS_ERROR;
		}
		io->iochannel = g_io_channel_new_file(filename, "r", NULL);
	}
	else if (kz_io_get_mode(io) == KZ_IO_WRITE)
		io->iochannel = g_io_channel_new_file(filename, "w", NULL);

	iostatus = g_io_channel_set_encoding(io->iochannel,
					     NULL,
					     NULL);
	return iostatus;
}


static GIOStatus
read_from_io(KzIO *io, GIOChannel *iochannel)
{
	GIOStatus iostatus;
	gsize bytes_read;
	gchar buffer[BUFFER_SIZE];

	/* Read the data into our buffer */
	iostatus = g_io_channel_read_chars(iochannel, buffer, 
					   sizeof(buffer),
					   &bytes_read,
					   NULL);

	if (iostatus == G_IO_STATUS_NORMAL)
	{	
		KZ_IO_CLASS (parent_class)->io_to_buffer(io, bytes_read, buffer);
		/* Check for EOF */
		if (bytes_read == 0)
			iostatus = G_IO_STATUS_EOF;
	}
	return iostatus;
}


static GIOStatus
write_to_io(KzIO *io, GIOChannel *iochannel)
{
	KZ_IO_CLASS (parent_class)->buffer_to_io(io, 0, NULL);
	return G_IO_STATUS_NORMAL;
}


static void
file_start (KzIO *io)
{
	GIOStatus iostatus;
	iostatus = kz_file_prepare_iochannel(io);
	
	if (iostatus != G_IO_STATUS_NORMAL)
	{
		KZ_IO_CLASS (parent_class)->io_error(io);
		return;
	}

	/* call io_set_iochannel, start to loading */
	KZ_IO_CLASS (parent_class)->io_set_channel(io);
}
