/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2005 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANDYNAMICILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-actions-dynamic.h"
#include "kz-smart-bookmark.h"

#include <string.h>
#include <glib/gi18n.h>
#include "gtk-utils.h"
#include "kazehakase.h"
#include "kz-icons.h"
#include "kz-navi.h"

#define KZ_ACTIONS_DYNAMIC_COPY_IN_USER_FORMAT_KEY "KzActionsDynamic::CopyInUserFormat"
#define KZ_ACTIONS_DYNAMIC_SMART_LIST_KEY "KzActionsDynamic::SmartList"


static void
cb_copy_in_user_format_preference_activate (GtkWidget *menuitem, KzWindow *kz)
{
	GtkAction *action;

	g_return_if_fail(KZ_IS_WINDOW(kz));
	
	action = gtk_action_group_get_action(kz->actions, "ShowCopyFormat");
	gtk_action_activate(action);
}

static void
cb_copy_in_user_format_menuitem_activate (GtkWidget *menuitem, KzTabLabel *kztab)
{
	KzEmbed *kzembed;
	gint idx = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(menuitem),
						     KZ_ACTIONS_DYNAMIC_COPY_IN_USER_FORMAT_KEY));
	gchar conf_key[] = "copy_document_format99";
	gchar *format;
	gchar *text;
	gchar *str;
	gchar *compress_str;
	gint i;
	gint counts = 0;

	g_return_if_fail(0 < idx && idx < 100);
	g_return_if_fail(KZ_IS_TAB_LABEL(kztab));

	kzembed = kztab->kzembed;
	if (!KZ_IS_EMBED(kzembed)) return;

	text = kz_embed_get_selection_string(kzembed);
	if (!text) text = g_strdup("");

	g_snprintf(conf_key, sizeof(conf_key), "copy_document_format%d", idx);
	format = KZ_CONF_GET_STR("Global" , conf_key);
	g_return_if_fail(format);

	for(i = 0; format[i] != '\0'; i++)
		if(format[i] == '%') counts++;
	counts++;

	if(counts != 1)
	{
		gchar **temp = g_alloca(sizeof(gchar*) * (counts + 1));
		gchar **splited = g_strsplit(format , "%" , counts);
		gint st = 0;

		for(i = 0; i < counts-1; i++)
		{
			if(st == 0)
			{
				/* first state */
				temp[i] = splited[i];
				st = 1;
			}
			else if (st == 2)
			{
				/* after %t% and so on */
				temp[i] = splited[i];
				st = 1;
			}
			else if (splited[i][0] == '\0')
			{
				/* "%%" */
				temp[i] = "%";
				st = 0;
			}
			else if (splited[i][1] != '\0')
			{
				/* %text% */
				temp[i] = splited[i];
				st = 1;
			}
			else if(splited[i][0] == 't')
			{
				/* %t% */
				const KzEmbedEventMouse *event;

				event = kz_window_get_mouse_event_info(kztab->kz);
				if (event && event->cinfo.linktext)
					temp[i] = event->cinfo.linktext;
				else
					temp[i] = (gchar *)kz_embed_get_title(kzembed);

				st = 2;
			}
			else if(splited[i][0] == 'u')
			{
				/* %u% */
				const KzEmbedEventMouse *event;

				event = kz_window_get_mouse_event_info(kztab->kz);
				if (event && event->cinfo.link)
					temp[i] = event->cinfo.link;
				else
					temp[i] = (gchar *)kz_embed_get_location(kzembed);

				st = 2;
			}
			else if(splited[i][0] == 's')
			{
				/* %s% */
				temp[i] = text;
				st = 2;
			}
			else
			{
				/* '%' + other char + '%' */
				temp[i] = splited[i];
				st = 1;
			}
		}
		temp[counts-1] = splited[counts-1];
		temp[counts] = NULL;

		str = g_strjoinv(NULL , temp);
		g_strfreev(splited);
	}
	else
	{
		str = g_strdup(format);
	}

	g_free(format);

	compress_str = g_strcompress(str);

	gtkutil_copy_text(compress_str);

	if(str) g_free(str);
	if(compress_str) g_free(compress_str);
	if(text) g_free(text);
}

void
kz_actions_dynamic_append_copy_in_user_format_menuitem (KzTabLabel *kztab,
						        GtkMenuItem *menu)
{
	GtkWidget *copy_in_user_format_submenu = NULL;
	GtkWidget *menuitem;
	gchar *title;
	gchar conf_key[] = "copy_document_format_title99";
	gint i;

	copy_in_user_format_submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu));
	if (copy_in_user_format_submenu)
	{
		gtk_menu_item_remove_submenu(GTK_MENU_ITEM(menu));
	}

	copy_in_user_format_submenu = gtk_menu_new();
	gtk_widget_show(copy_in_user_format_submenu);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu),
				  copy_in_user_format_submenu);

	/* append copy in user format menuitem */
	menuitem = gtk_menu_item_new_with_mnemonic(_("_Preference..."));
	g_signal_connect(menuitem, "activate",
			 G_CALLBACK(cb_copy_in_user_format_preference_activate),
			 kztab->kz);

	gtk_menu_shell_append(GTK_MENU_SHELL(copy_in_user_format_submenu),
			      menuitem);
	gtk_widget_show(menuitem);
	
	menuitem = gtk_separator_menu_item_new();
	gtk_menu_shell_append(GTK_MENU_SHELL(copy_in_user_format_submenu),
			      menuitem);
	gtk_widget_show(menuitem);

	for (i = 1; i < 100; i++)
	{
		g_snprintf(conf_key, sizeof(conf_key), "copy_document_format_title%d", i);
		title = KZ_CONF_GET_STR("Global" , conf_key);
		if(!title) break;

		menuitem = gtk_menu_item_new_with_label(title);

		g_object_set_data (G_OBJECT(menuitem),
				   KZ_ACTIONS_DYNAMIC_COPY_IN_USER_FORMAT_KEY,
				   GINT_TO_POINTER(i));

		g_signal_connect(menuitem, "activate",
				 G_CALLBACK(cb_copy_in_user_format_menuitem_activate),
				 kztab);

		gtk_menu_shell_append(GTK_MENU_SHELL(copy_in_user_format_submenu),
				      menuitem);
		gtk_widget_show(menuitem);

		g_free(title);
	}
}

static void
cb_add_feed_bookmark_menuitem_activate (GtkWidget *widget, KzNavi *navi)
{
	kz_navi_add_bookmark(navi);
}

void
kz_actions_dynamic_append_add_feed_bookmark (KzWindow *kz,
					     GtkMenuItem *menu)
{
	GtkWidget *add_feed_bookmark_submenu = NULL;
	GtkWidget *menuitem;
	KzEmbed *kzembed;
	GList *nav_link, *nav_links;
	KzNavi *navi;
	gchar *title;

	add_feed_bookmark_submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu));
	if (add_feed_bookmark_submenu)
	{
		gtk_menu_item_remove_submenu(GTK_MENU_ITEM(menu));
	}

	kzembed = KZ_WINDOW_CURRENT_PAGE(kz);
	if (!KZ_IS_EMBED(kzembed)) return;

	nav_links = kz_embed_get_nav_links(kzembed, KZ_EMBED_LINK_RSS);

	if (!nav_links) return;
	
	add_feed_bookmark_submenu = gtk_menu_new();
	gtk_widget_show(add_feed_bookmark_submenu);
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu),
				  add_feed_bookmark_submenu);

	/* append available feed menuitem */
	for (nav_link = nav_links;
	     nav_link;
	     nav_link = g_list_next(nav_link))
	{
		navi = nav_link->data;
		if (!navi) continue;
		
		title = navi->title ? navi->title : _("Feed");

		menuitem = gtk_menu_item_new_with_label(title);

		g_signal_connect(menuitem, "activate",
				 G_CALLBACK(cb_add_feed_bookmark_menuitem_activate),
				 navi);

		gtk_menu_shell_append(GTK_MENU_SHELL(add_feed_bookmark_submenu),
				      menuitem);
		gtk_widget_show(menuitem);
	}
}

static void
cb_open_smart_bookmark_menuitem_activate (GtkWidget *widget, KzWindow *kz)
{
	KzSmartBookmark *smart_bookmark;
	GtkWidget *embed;
	const gchar *text, *smart_uri;
	gboolean new_tab;

	smart_bookmark = g_object_get_data(G_OBJECT(widget),
					   KZ_ACTIONS_DYNAMIC_SMART_LIST_KEY);

	g_return_if_fail(KZ_IS_SMART_BOOKMARK(smart_bookmark));

	embed = KZ_WINDOW_CURRENT_PAGE(kz);
	text = kz_embed_get_selection_string(KZ_EMBED(embed));
	smart_uri = kz_smart_bookmark_get_smart_uri(smart_bookmark, text);

	KZ_CONF_GET("Global", "entry_open_in_new_tab", new_tab, BOOL);
	if (new_tab)
		kz_window_open_new_tab(kz, smart_uri);
	else
		kz_window_load_url(kz, smart_uri);

	g_free(smart_uri);
}

void
kz_actions_dynamic_append_open_smart_bookmark_menuitem (KzWindow *kz,
							GtkMenuItem *menu)
{
	GList *children, *node;
	GtkWidget *smart_bookmark_submenu = NULL;
	const gchar *title;
	KzBookmark *bookmark = kz_bookmarks->smarts;
	KzBookmark *child, *smart;

	smart_bookmark_submenu = gtk_menu_item_get_submenu(GTK_MENU_ITEM(menu));
	if (smart_bookmark_submenu)
	{
		gtk_menu_item_remove_submenu(GTK_MENU_ITEM(menu));
	}

	smart_bookmark_submenu = gtk_menu_new();
	gtk_menu_item_set_submenu(GTK_MENU_ITEM(menu), smart_bookmark_submenu);
	gtk_widget_show(smart_bookmark_submenu);

	children = kz_bookmark_get_children(bookmark);
	for (node = children; node; node = g_list_next(node))
	{
		GList *sub_children, *sub_node;
		GtkWidget *sub_folder, *sub_bookmark;
		GtkWidget *menuitem;

		child = node->data;
		if (!kz_bookmark_is_folder(child))
			continue;

		title = kz_bookmark_get_title(child);
		if (title && g_str_has_prefix(title, "LocationEntry"))
			continue;

		menuitem = gtk_menu_item_new_with_label(title);
		gtk_menu_shell_append(GTK_MENU_SHELL(smart_bookmark_submenu),
				      menuitem);

		sub_folder = gtk_menu_new();
		gtk_menu_item_set_submenu(GTK_MENU_ITEM(menuitem),
					  sub_folder);
		gtk_widget_show(sub_folder);
		gtk_widget_show(menuitem);

		sub_children = kz_bookmark_get_children(child);

		for (sub_node = sub_children;
		     sub_node;
		     sub_node = g_list_next(sub_node))
		{
			const gchar *bookmark_title;

			smart = sub_node->data;
			bookmark_title = kz_bookmark_get_title(smart);

			sub_bookmark =
				gtk_menu_item_new_with_label(bookmark_title);
			g_object_set_data (G_OBJECT(sub_bookmark),
					   KZ_ACTIONS_DYNAMIC_SMART_LIST_KEY,
					   smart);

			gtk_menu_shell_append(GTK_MENU_SHELL(sub_folder),
					      sub_bookmark);
			g_signal_connect(sub_bookmark, "activate",
					 G_CALLBACK(cb_open_smart_bookmark_menuitem_activate),
					 kz);
			gtk_widget_show(sub_bookmark);
		}
		g_list_free(sub_children);
	}
	g_list_free(children);
}
