/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *  Copyright (C) 2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *  $Id: kz-bookmark-edit.c 2113 2005-04-28 13:40:55Z ikezoe $
 */

#include <glib/gi18n.h>
#include "kz-bookmark-edit.h"
#include "kz-smart-bookmark.h"
#include "gobject-utils.h"
#include "gtk-utils.h"
#include "kz-thumbnail.h"
#include "kz-bookmark-file.h"

#define REGEX_ENTRY_WIDTH 100
#define URI_ENTRY_WIDTH 150
#define URLENCODE_WIDTH 60

enum
{
	COLUMN_TERMINATOR = -1,
	COLUMN_REGEX,
	COLUMN_URI,
	COLUMN_ENCODE,
	COLUMN_URLENCODE,
	COLUMN_EDITABLE,
	N_COLUMNS
};

struct _KzSmartBookmarkEdit
{
	GtkWidget    *box;
	GtkTreeView  *tree;
	GtkListStore *store;
};

static GtkVBoxClass *parent_class = NULL;


static void kz_bookmark_edit_class_init (KzBookmarkEditClass *klass);
static void kz_bookmark_edit_init       (KzBookmarkEdit *edit);
static void kz_bookmark_edit_dispose    (GObject        *object);

static void kz_bookmark_edit_set_sensitive   (KzBookmarkEdit *edit);
static void kz_bookmark_edit_sync_properties (KzBookmarkEdit *edit,
					      KzBookmark *bookmark);
static void kz_bookmark_edit_set_smart_list  (KzBookmarkEdit *edit,
					      const GList *smart_list);

static void cb_bookmark_notify           (GObject *object,
					  GParamSpec *pspec,
					  KzBookmarkEdit *edit);
static void cb_title_entry_changed       (GtkEditable    *editable,
					  KzBookmarkEdit *edit);
static void cb_uri_entry_changed         (GtkEditable    *editable,
					  KzBookmarkEdit *edit);
static void cb_location_entry_changed    (GtkEditable    *editable,
					  KzBookmarkEdit *edit);
static void cb_interval_value_changed    (GtkSpinButton  *button,
					  KzBookmarkEdit *edit);
static void cb_description_changed       (GtkTextBuffer  *textbuf,
					  KzBookmarkEdit *edit);
static void cb_xmlrpc_entry_changed      (GtkEditable    *editable,
					  KzBookmarkEdit *edit);
static void cb_xmlrpc_user_entry_changed (GtkEditable    *editable,
					  KzBookmarkEdit *edit);
static void cb_xmlrpc_pass_entry_changed (GtkEditable    *editable,
					  KzBookmarkEdit *edit);

static GtkWidget *create_widget_for_smart_property (KzBookmarkEdit *edit);

KZ_OBJECT_GET_TYPE(kz_bookmark_edit, "KzBookmarkEdit", KzBookmarkEdit,
		   kz_bookmark_edit_class_init, kz_bookmark_edit_init,
		   GTK_TYPE_VBOX)

static void
kz_bookmark_edit_class_init (KzBookmarkEditClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;
	GtkWidgetClass *widget_class;

	parent_class = g_type_class_peek_parent (klass);

	gobject_class = (GObjectClass *)   klass;
	object_class  = (GtkObjectClass *) klass;
	widget_class  = (GtkWidgetClass *) klass;

	gobject_class->dispose = kz_bookmark_edit_dispose;
}


static void
kz_bookmark_edit_init (KzBookmarkEdit *edit)
{
	GtkVBox *main_vbox;
	GtkWidget *hbox, *table, *scrwin;
	GtkWidget *title_label, *title_entry;
	GtkWidget *uri_label, *uri_entry;
	GtkWidget *location_label, *location_entry;
	GtkWidget *interval_label, *interval_spin;
	GtkWidget *memo_label, *textview;
	GtkWidget *thumbnail;
	GtkWidget *xmlrpc_label, *xmlrpc_user_label, *xmlrpc_pass_label;
	GtkWidget *xmlrpc_entry, *xmlrpc_user_entry, *xmlrpc_pass_entry;
	GtkTextBuffer *textbuf;

	edit->smart = g_new0(KzSmartBookmarkEdit, 1);
	
	main_vbox = GTK_VBOX(edit);

	table = gtk_table_new(6, 3, FALSE);
	gtk_container_set_border_width(GTK_CONTAINER(table), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), table, TRUE, TRUE, 0);
	gtk_widget_show(table);

	/* Title */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 0, 1,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	title_label = gtk_label_new(_("Title:"));
	gtk_box_pack_start(GTK_BOX(hbox), title_label, FALSE, FALSE, 0);
	gtk_widget_show(title_label);

	title_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), title_entry,
			 1, 2, 0, 1,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(title_entry, "changed",
			 G_CALLBACK(cb_title_entry_changed), edit);
	gtk_widget_show(title_entry);

	/* URI */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 1, 2,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	uri_label = gtk_label_new(_("URI:"));
	gtk_box_pack_start(GTK_BOX(hbox), uri_label, FALSE, FALSE, 0);
	gtk_widget_show(uri_label);

	uri_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), uri_entry,
			 1, 2, 1, 2,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(uri_entry, "changed",
			 G_CALLBACK(cb_uri_entry_changed), edit);
	gtk_widget_show(uri_entry);

	/* Location */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 2, 3,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	location_label = gtk_label_new(_("Location:"));
	gtk_box_pack_start(GTK_BOX(hbox), location_label, FALSE, FALSE, 0);

	location_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), location_entry,
			 1, 2, 2, 3,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(location_entry, "changed",
			 G_CALLBACK(cb_location_entry_changed), edit);

	/* Update interval */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 2, 3, 4,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	interval_label = gtk_label_new(_("Update interval (min):"));
	gtk_box_pack_start(GTK_BOX(hbox), interval_label, FALSE, FALSE, 0);

	interval_spin = gtk_spin_button_new_with_range(0, 60 * 24 * 7, 1);
	gtk_box_pack_start(GTK_BOX(hbox), interval_spin, FALSE, FALSE, 0);
	g_signal_connect(interval_spin, "value-changed",
			 G_CALLBACK(cb_interval_value_changed), edit);
#if 1
	/* XML-RPC interface for shared bookmark */
	/* Interface address */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 4, 5,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	xmlrpc_label = gtk_label_new(_("Interface:"));
	gtk_box_pack_start(GTK_BOX(hbox), xmlrpc_label, FALSE, FALSE, 0);

	xmlrpc_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), xmlrpc_entry,
			 1, 2, 4, 5,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(xmlrpc_entry, "changed",
			 G_CALLBACK(cb_xmlrpc_entry_changed), edit);
	
	/* User name */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 5, 6,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	xmlrpc_user_label = gtk_label_new(_("User name:"));
	gtk_box_pack_start(GTK_BOX(hbox), xmlrpc_user_label, FALSE, FALSE, 0);

	xmlrpc_user_entry = gtk_entry_new();
	gtk_table_attach(GTK_TABLE(table), xmlrpc_user_entry,
			 1, 2, 5, 6,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(xmlrpc_user_entry, "changed",
			 G_CALLBACK(cb_xmlrpc_user_entry_changed), edit);
	
	/* Password */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 1, 6, 7,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	xmlrpc_pass_label = gtk_label_new(_("Password:"));
	gtk_box_pack_start(GTK_BOX(hbox), xmlrpc_pass_label, FALSE, FALSE, 0);

	xmlrpc_pass_entry = gtk_entry_new();
	gtk_entry_set_visibility(GTK_ENTRY(xmlrpc_pass_entry), FALSE);
	gtk_table_attach(GTK_TABLE(table), xmlrpc_pass_entry,
			 1, 2, 6, 7,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, 0,
			 2, 2);       /* padding x, y  */
	g_signal_connect(xmlrpc_pass_entry, "changed",
			 G_CALLBACK(cb_xmlrpc_pass_entry_changed), edit);
#endif
	/* Memo */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_table_attach(GTK_TABLE(table), hbox,
			 0, 3, 4, 5,  /* xpos, ypos */
			 GTK_FILL, 0,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(hbox);

	memo_label = gtk_label_new(_("Memo:"));
	gtk_box_pack_start(GTK_BOX(hbox), memo_label, FALSE, FALSE, 0);
	gtk_widget_show(memo_label);

	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_table_attach(GTK_TABLE(table), scrwin,
			 1, 3, 4, 5,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND,
			 2, 2);       /* padding x, y  */
	gtk_widget_show(scrwin);

	textview = gtk_text_view_new();
	gtk_text_view_set_wrap_mode(GTK_TEXT_VIEW(textview), GTK_WRAP_WORD);
	gtk_container_add(GTK_CONTAINER(scrwin), textview);
	textbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(textview));
	g_signal_connect(textbuf, "changed",
			 G_CALLBACK(cb_description_changed), edit);
	gtk_widget_show(textview);

	/* properties for smart bookmark */
	edit->smart->box = create_widget_for_smart_property(edit);
	gtk_table_attach(GTK_TABLE(table), edit->smart->box,
			 0, 3, 5, 6,  /* xpos, ypos */
			 GTK_FILL | GTK_EXPAND, GTK_FILL | GTK_EXPAND,
			 2, 2);       /* padding x, y  */

	/* thumbnail */
	thumbnail = kz_thumbnail_new();
	gtk_table_attach(GTK_TABLE(table), thumbnail,
			 2, 3, 0, 4,
			 GTK_SHRINK, GTK_SHRINK,
			 2, 2);
	gtk_widget_show(thumbnail);

	/* initialize */
	edit->title_label       = title_label;
	edit->title_entry       = title_entry;
	edit->uri_label         = uri_label;
	edit->uri_entry         = uri_entry;
	edit->location_label    = location_label;
	edit->location_entry    = location_entry;
	edit->interval_label    = interval_label;
	edit->interval_spin     = interval_spin;
	edit->memo_label        = memo_label;
	edit->memo_textview     = textview;
	edit->memo_scrwin       = scrwin;
	edit->xmlrpc_label      = xmlrpc_label;
	edit->xmlrpc_user_label = xmlrpc_user_label;
	edit->xmlrpc_pass_label = xmlrpc_pass_label;
	edit->xmlrpc_entry      = xmlrpc_entry;
	edit->xmlrpc_user_entry = xmlrpc_user_entry;
	edit->xmlrpc_pass_entry = xmlrpc_pass_entry;

	edit->bookmark          = NULL;
	edit->thumbnail         = thumbnail;
	edit->changing          = FALSE;
}


static void
kz_bookmark_edit_dispose (GObject *object)
{
	KzBookmarkEdit *edit = KZ_BOOKMARK_EDIT(object);

	if (edit->bookmark)
	{
		g_object_unref(edit->bookmark);
		g_signal_handlers_disconnect_by_func
			(edit->bookmark,
			 G_CALLBACK(cb_bookmark_notify),
			 edit);
	}
	edit->bookmark = NULL;

	if (edit->smart && edit->smart->store)
	{
		g_object_unref(edit->smart->store);
		edit->smart->store = NULL;
	}

	if (edit->smart)
	{
		g_free(edit->smart);
		edit->smart = NULL;
	}
	if (G_OBJECT_CLASS (parent_class)->dispose)
		G_OBJECT_CLASS (parent_class)->dispose(object);
}


GtkWidget *
kz_bookmark_edit_new (void)
{
	KzBookmarkEdit *edit;

	edit = g_object_new(KZ_TYPE_BOOKMARK_EDIT, NULL);

	return GTK_WIDGET(edit);
}


void
kz_bookmark_edit_set (KzBookmarkEdit *edit, KzBookmark *bookmark)
{
	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	kz_bookmark_edit_clear(edit);

	edit->bookmark = bookmark;
	if (!edit->bookmark) return;
	g_object_ref(edit->bookmark);

	g_signal_connect(edit->bookmark, "notify",
			 G_CALLBACK(cb_bookmark_notify), edit);

	kz_bookmark_edit_sync_properties (edit, bookmark);
}


void
kz_bookmark_edit_clear (KzBookmarkEdit *edit)
{
	GtkTextBuffer *textbuf;
	GtkTextIter start_iter, end_iter;
	GtkAdjustment *adj;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->bookmark)
	{
		g_object_unref(edit->bookmark);
		g_signal_handlers_disconnect_by_func
			(edit->bookmark,
			 G_CALLBACK(cb_bookmark_notify),
			 edit);
	}
	edit->bookmark = NULL;

	gtk_entry_set_text(GTK_ENTRY(edit->title_entry),       "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->uri_entry),         "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->location_entry),    "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_entry),      "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_user_entry), "\0");
	gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_pass_entry), "\0");
	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	gtk_adjustment_set_value(adj, 0.0);

	textbuf = gtk_text_view_get_buffer(GTK_TEXT_VIEW(edit->memo_textview));
	gtk_text_buffer_get_start_iter(textbuf, &start_iter);
	gtk_text_buffer_get_end_iter(textbuf, &end_iter);
	gtk_text_buffer_delete(textbuf, &start_iter, &end_iter);

	kz_bookmark_edit_set_sensitive(edit);
}


static void
kz_bookmark_edit_set_sensitive (KzBookmarkEdit *edit)
{
	gboolean can_edit_title = TRUE;
	gboolean can_edit_uri = TRUE;
	gboolean can_edit_desc = TRUE;
	gboolean can_edit_location = FALSE;
	gboolean can_edit_interval = FALSE;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	/* set sensitive */
	if (!edit->bookmark ||
	    !kz_bookmark_is_editable(edit->bookmark) ||
	    kz_bookmark_is_separator(edit->bookmark))
	{
		can_edit_title    = FALSE;
		can_edit_uri      = FALSE;
		can_edit_desc     = FALSE;
		can_edit_location = FALSE;
		can_edit_interval = FALSE;
	}
	else if (edit->bookmark && kz_bookmark_is_pure_folder(edit->bookmark))
	{
		can_edit_uri = FALSE;
	}
	if (edit->bookmark && KZ_IS_BOOKMARK_FILE(edit->bookmark))
	{
		can_edit_location = TRUE;
		can_edit_interval = TRUE;
	}

	gtk_editable_set_editable(GTK_EDITABLE(edit->title_entry),
				  can_edit_title);
	gtk_editable_set_editable(GTK_EDITABLE(edit->uri_entry),
				  can_edit_uri);
	gtk_editable_set_editable(GTK_EDITABLE(edit->location_entry),
				  can_edit_location);
	gtk_widget_set_sensitive(edit->interval_spin, can_edit_interval);
	gtk_text_view_set_editable(GTK_TEXT_VIEW(edit->memo_textview),
				   can_edit_desc);

	/* set visible */
	if (edit->bookmark && KZ_IS_BOOKMARK_FILE(edit->bookmark))
	{
		gtk_widget_hide(edit->uri_label);
		gtk_widget_hide(edit->uri_entry);
		gtk_widget_hide(edit->memo_label);
		gtk_widget_hide(edit->memo_scrwin);

		gtk_widget_show(edit->location_label);
		gtk_widget_show(edit->location_entry);
		gtk_widget_show(edit->interval_label);
		gtk_widget_show(edit->interval_spin);

		gtk_widget_show(edit->xmlrpc_entry);
		gtk_widget_show(edit->xmlrpc_user_entry);
		gtk_widget_show(edit->xmlrpc_pass_entry);
		gtk_widget_show(edit->xmlrpc_label);
		gtk_widget_show(edit->xmlrpc_user_label);
		gtk_widget_show(edit->xmlrpc_pass_label);
	}
	else
	{
		if (edit->bookmark &&
		    kz_bookmark_is_pure_folder(edit->bookmark))
		{
			gtk_widget_hide(edit->uri_label);
			gtk_widget_hide(edit->uri_entry);
		}
		else
		{
			gtk_widget_show(edit->uri_label);
			gtk_widget_show(edit->uri_entry);
		}
		gtk_widget_show(edit->memo_label);
		gtk_widget_show(edit->memo_scrwin);

		if (edit->bookmark &&
		    KZ_IS_SMART_BOOKMARK(edit->bookmark))
		{
			gtk_widget_hide(edit->memo_label);
			gtk_widget_hide(edit->memo_scrwin);
			gtk_widget_show(edit->smart->box);
		}
		else
		{	
			gtk_widget_hide(edit->smart->box);
		}

		gtk_widget_hide(edit->location_label);
		gtk_widget_hide(edit->location_entry);
		gtk_widget_hide(edit->interval_label);
		gtk_widget_hide(edit->interval_spin);
		
		gtk_widget_hide(edit->xmlrpc_entry);
		gtk_widget_hide(edit->xmlrpc_user_entry);
		gtk_widget_hide(edit->xmlrpc_pass_entry);
		gtk_widget_hide(edit->xmlrpc_label);
		gtk_widget_hide(edit->xmlrpc_user_label);
		gtk_widget_hide(edit->xmlrpc_pass_label);
	}
}


static void
kz_bookmark_edit_sync_properties (KzBookmarkEdit *edit, KzBookmark *bookmark)
{
	GtkTextBuffer *buffer;
	GtkAdjustment *adj;
	const gchar *title, *uri, *location = NULL, *desc;
	guint interval = 0;

	edit->changing = TRUE;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));
	g_return_if_fail(KZ_IS_BOOKMARK(bookmark));

	title    = kz_bookmark_get_title(bookmark);
	uri      = kz_bookmark_get_link(bookmark);
	if (KZ_IS_BOOKMARK_FILE(bookmark))
	{
		location = kz_bookmark_file_get_location(KZ_BOOKMARK_FILE(bookmark));
		interval = kz_bookmark_file_get_interval(KZ_BOOKMARK_FILE(bookmark));
	}
	desc     = kz_bookmark_get_description(bookmark);

	buffer = gtk_text_view_get_buffer(GTK_TEXT_VIEW(edit->memo_textview));

	if (title)
		gtk_entry_set_text(GTK_ENTRY(edit->title_entry), title);
	if (uri)
	{
		gtk_entry_set_text(GTK_ENTRY(edit->uri_entry), uri);
		
		kz_thumbnail_set_thumbnail_from_uri(
				KZ_THUMBNAIL(edit->thumbnail),
				uri);
	}
	if (location)
		gtk_entry_set_text(GTK_ENTRY(edit->location_entry), location);
	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	gtk_adjustment_set_value(adj, interval);

	if (desc)
	{
		GtkTextView *textview = GTK_TEXT_VIEW(edit->memo_textview);
		GtkTextIter iter, end_iter;

		buffer = gtk_text_view_get_buffer(textview);
		gtk_text_buffer_get_start_iter(buffer, &iter);
		gtk_text_buffer_get_end_iter(buffer, &end_iter);
		gtk_text_buffer_delete(buffer, &iter, &end_iter);
		gtk_text_buffer_insert(buffer, &iter, desc, -1);
	}

	/* for smart bookmark */
	if (KZ_IS_SMART_BOOKMARK(bookmark))
	{
		const GList *smart_list;
		smart_list = kz_smart_bookmark_get_smart_list(KZ_SMART_BOOKMARK(bookmark));
		kz_bookmark_edit_set_smart_list(edit, smart_list);
	}

	/* for XML-RPC interface */
	if (KZ_IS_BOOKMARK_FILE(bookmark))
	{
		const gchar *xmlrpc, *xmlrpc_pass, *xmlrpc_user;

		xmlrpc      = kz_bookmark_file_get_xmlrpc(KZ_BOOKMARK_FILE(bookmark));
		xmlrpc_user = kz_bookmark_file_get_xmlrpc_user(KZ_BOOKMARK_FILE(bookmark));
		xmlrpc_pass = kz_bookmark_file_get_xmlrpc_pass(KZ_BOOKMARK_FILE(bookmark));
		if (xmlrpc)
			gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_entry), xmlrpc);
		if (xmlrpc_user)
			gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_user_entry), xmlrpc_user);
		if (xmlrpc_pass)
			gtk_entry_set_text(GTK_ENTRY(edit->xmlrpc_pass_entry), xmlrpc_pass);
	}

	kz_bookmark_edit_set_sensitive(edit);

	edit->changing = FALSE;
}


static void
kz_bookmark_edit_set_smart_list  (KzBookmarkEdit *edit, 
				  const GList *smart_list)
{
	const GList *node;
	GtkListStore *store;

	store = edit->smart->store;
	gtk_list_store_clear(store);
	
	for (node = smart_list; node; node = g_list_next(node))
	{
		GtkTreeIter iter;
		KzSmartBookmarkProperty *prop;
		
		prop = node->data;
		gtk_list_store_append(store, &iter);
		gtk_list_store_set(store, &iter,
				   COLUMN_REGEX,      prop->regex,
				   COLUMN_URI,        prop->uri,
				   COLUMN_ENCODE,     prop->encode,
				   COLUMN_URLENCODE,  prop->urlencode,
				   COLUMN_EDITABLE,   TRUE,
				   COLUMN_TERMINATOR);
	}
}

static void
kz_bookmark_edit_update_smart_list(KzBookmarkEdit *edit)
{
	GList *list = NULL;
	GtkTreeModel *model;
	GtkTreeIter iter;
	gboolean exist;

	model = gtk_tree_view_get_model(GTK_TREE_VIEW(edit->smart->tree));
	
	exist = gtk_tree_model_get_iter_first(model, &iter);
	for(; exist; exist = gtk_tree_model_iter_next(model, &iter))
	{
		gchar *regex = NULL, *uri = NULL, *encode = NULL;
		gboolean urlencode = FALSE;
		KzSmartBookmarkProperty *prop;

		gtk_tree_model_get(model, &iter,
				   COLUMN_REGEX,     &regex,
				   COLUMN_URI,       &uri,
				   COLUMN_ENCODE,    &encode,
				   COLUMN_URLENCODE, &urlencode,
				   COLUMN_TERMINATOR);

		prop = g_new0(KzSmartBookmarkProperty, 1);

		if (regex)
		{
			prop->regex = g_strdup(regex);
			g_free(regex);
		}
		if (uri)
		{
			prop->uri = g_strdup(uri);
			g_free(uri);
		}
		if (regex)
		{
			prop->encode = g_strdup(encode);
			g_free(encode);
		}
		if (urlencode)
			prop->urlencode = urlencode;
		list = g_list_append(list, prop);
	}

	kz_smart_bookmark_set_smart_list(KZ_SMART_BOOKMARK(edit->bookmark), list);
}

static void
cb_smart_up_button (GtkButton *button, KzBookmarkEdit *edit)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, prev;
	GtkTreePath *treepath;
	GtkListStore *store;

	store = edit->smart->store;
	selection = gtk_tree_view_get_selection (edit->smart->tree);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	treepath = gtk_tree_model_get_path(model, &iter);
	if (!gtk_tree_path_prev(treepath)) goto ERROR;
	gtk_tree_model_get_iter(model, &prev, treepath);

	gtk_list_store_swap(store, &iter, &prev);
	kz_bookmark_edit_update_smart_list(edit);
	
	gtk_tree_selection_select_path(selection, treepath);
ERROR:
	gtk_tree_path_free(treepath);
}


static void
cb_smart_down_button (GtkButton *button, KzBookmarkEdit *edit)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	GtkListStore *store;

	store = edit->smart->store;
	selection = gtk_tree_view_get_selection (edit->smart->tree);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

	next = iter;
	if (!gtk_tree_model_iter_next(model, &next)) return;

	gtk_list_store_swap(store, &iter, &next);
	kz_bookmark_edit_update_smart_list(edit);
	//gtk_tree_selection_select_iter(selection, &next);
}


static void
cb_smart_add_button (GtkButton *button, KzBookmarkEdit *edit)
{
	GtkTreeIter iter;
	GtkTreePath *treepath;
	GtkListStore *store;

	store = edit->smart->store;
	
	gtk_list_store_append(store, &iter);
	gtk_list_store_set(store, &iter,
			   COLUMN_REGEX,     "(.*)",
			   COLUMN_URI,       "%s",
			   COLUMN_ENCODE,    NULL,
			   COLUMN_URLENCODE, FALSE,
			   COLUMN_EDITABLE,  TRUE,
			   COLUMN_TERMINATOR);

	treepath = gtk_tree_model_get_path(GTK_TREE_MODEL(store),
					   &iter);
	gtk_tree_view_set_cursor(edit->smart->tree,
				 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);
	kz_bookmark_edit_update_smart_list(edit);
}


static void
cb_smart_remove_button (GtkButton *button, KzBookmarkEdit *edit)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model;
	GtkTreeIter iter, next;
	GtkTreePath *treepath;
	gboolean select;
	GtkListStore *store;

	store = edit->smart->store;
	selection = gtk_tree_view_get_selection (edit->smart->tree);
	if (!gtk_tree_selection_get_selected (selection, &model, &iter))
		return;

        /* get next row to select */
	next = iter;
	select = gtk_tree_model_iter_next(model, &next);
	if (select)
	{
		treepath = gtk_tree_model_get_path(model, &next);
	}
	else
	{
		treepath = gtk_tree_model_get_path(model, &iter);
		select = gtk_tree_path_prev(treepath);
	}
	if (select)
		gtk_tree_view_set_cursor(edit->smart->tree,
					 treepath, NULL, FALSE);
	gtk_tree_path_free(treepath);

	/* remove the row */
	gtk_list_store_remove(store, &iter);
	kz_bookmark_edit_update_smart_list(edit);
}


static void
cb_smart_regex_edited (GtkCellRendererText *cell,
		       const gchar *path_str,
		       const gchar *new_text,
		       KzBookmarkEdit *edit)
{
        GtkTreeIter  iter;
	GtkListStore *store;

	store = edit->smart->store;
        gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(store),
					    &iter, path_str);
	gtk_list_store_set(store, &iter,
			   COLUMN_REGEX, new_text,
			   COLUMN_TERMINATOR);
	kz_bookmark_edit_update_smart_list(edit);
}


static void
cb_smart_uri_edited (GtkCellRendererText *cell,
	             const gchar *path_str,
	             const gchar *new_text,
	             KzBookmarkEdit *edit)
{
        GtkTreeIter  iter;
	GtkListStore *store;

	store = edit->smart->store;

        gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(store),
					    &iter, path_str);
	gtk_list_store_set(store, &iter,
			   COLUMN_URI, new_text,
			   COLUMN_TERMINATOR);
	kz_bookmark_edit_update_smart_list(edit);
}

static void
cb_smart_encode_edited (GtkCellRendererText *cell,
	                const gchar *path_str,
	                const gchar *new_text,
	                KzBookmarkEdit *edit)
{
        GtkTreeIter  iter;
	GtkListStore *store;

	store = edit->smart->store;

        gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(store),
					    &iter, path_str);
	gtk_list_store_set(store, &iter,
			   COLUMN_ENCODE, new_text,
			   COLUMN_TERMINATOR);
	kz_bookmark_edit_update_smart_list(edit);
}

static void
cb_smart_urlencode_toggled (GtkCellRendererToggle *cell,
			    const gchar *path_str,
			    KzBookmarkEdit *edit)
{
        GtkTreeIter  iter;
	GValue val = { 0 };
	gboolean flag;
	GtkListStore *store;

	store = edit->smart->store;
	gtk_tree_model_get_iter_from_string(GTK_TREE_MODEL(store),
					    &iter, path_str);

	gtk_tree_model_get_value(GTK_TREE_MODEL(store), &iter,
				 COLUMN_URLENCODE, &val);
	flag = !(g_value_get_boolean(&val));

	gtk_list_store_set(store, &iter,
			   COLUMN_URLENCODE, flag,
			   COLUMN_TERMINATOR);
	kz_bookmark_edit_update_smart_list(edit);
}

static void
cb_smart_cursor_changed(GtkTreeView *tree_view, KzBookmarkEdit *edit)
{
}


static GtkWidget *
create_widget_for_smart_property(KzBookmarkEdit *edit)
{
	GtkListStore *store;
	GtkWidget *main_vbox, *hbox, *vbox1, *vbox2;
	GtkWidget *tree_view, *scrwin;
	GtkWidget *button, *arrow;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *column;

	main_vbox = gtk_vbox_new(FALSE, 0);

	/* hbox for list and up/down buttons */
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	/* scrolled window */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_container_set_border_width(GTK_CONTAINER(scrwin), 5);
	gtk_box_pack_start(GTK_BOX(hbox), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	/* tree view */
	store = gtk_list_store_new(N_COLUMNS,
				   G_TYPE_STRING,
				   G_TYPE_STRING,
				   G_TYPE_STRING,
				   G_TYPE_BOOLEAN,
				   G_TYPE_BOOLEAN);
	edit->smart->store = store;
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	edit->smart->tree = GTK_TREE_VIEW(tree_view);
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);

	/* Regex column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(cell, "edited",
			 G_CALLBACK(cb_smart_regex_edited), edit);
	column = gtk_tree_view_column_new_with_attributes
			(_("Regex"), cell,
			 "text",     COLUMN_REGEX,
			 "editable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

	/* URI column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(cell, "edited",
			 G_CALLBACK(cb_smart_uri_edited), edit);
	column = gtk_tree_view_column_new_with_attributes
			(_("URI"), cell,
			 "text",     COLUMN_URI,
			 "editable", COLUMN_EDITABLE,
			 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

        /* Encode column */
	cell = gtk_cell_renderer_text_new();
	g_signal_connect(cell, "edited",
			 G_CALLBACK(cb_smart_encode_edited), edit);
	column = gtk_tree_view_column_new_with_attributes
		(_("ENCODE"), cell,
		 "text", COLUMN_ENCODE,
		 "editable", COLUMN_EDITABLE,
		 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);
        
	/* URL encode flag  column */
	cell = gtk_cell_renderer_toggle_new();
	g_signal_connect(cell, "toggled",
			 G_CALLBACK(cb_smart_urlencode_toggled), edit);

	gtk_cell_renderer_toggle_set_radio
		(GTK_CELL_RENDERER_TOGGLE(cell), FALSE);

	column = gtk_tree_view_column_new_with_attributes
		(_("URL ENCODE"), cell,
		 "active", COLUMN_URLENCODE,
		 "activatable", COLUMN_EDITABLE,
		 NULL);
	gtk_tree_view_column_set_resizable(column, TRUE);
	gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (column, URLENCODE_WIDTH);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), column);

	gtk_container_add(GTK_CONTAINER(scrwin), tree_view);
	gtk_widget_show(tree_view);

	/* button box */
	vbox1 = gtk_vbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(hbox), vbox1, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox1), 0);
	gtk_widget_show(vbox1);

	vbox2 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), vbox2, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox2), 0);
	gtk_widget_show(vbox2);

	/* up button */
	button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_UP, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 5);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_smart_up_button), edit);
	gtk_widget_show(button);

	/* down button */
	button = gtk_button_new();
	arrow = gtk_arrow_new(GTK_ARROW_DOWN, GTK_SHADOW_NONE);
	gtk_container_add(GTK_CONTAINER(button), arrow);
	gtk_widget_show(arrow);
	gtk_box_pack_start(GTK_BOX(vbox2), button, FALSE, FALSE, 5);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_smart_down_button), edit);
	gtk_widget_show(button);

	/* button box */
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_widget_show(hbox);

	/* add button */
	button = gtk_button_new_from_stock(GTK_STOCK_ADD);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_smart_add_button), edit);
	gtk_widget_show(button);

	/* remove button */
	button = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);
	g_signal_connect(button, "clicked",
			 G_CALLBACK(cb_smart_remove_button), edit);
	gtk_widget_show(button);

	/* signal handler for GtkTreeView */
	g_signal_connect(tree_view, "cursor-changed",
			 G_CALLBACK(cb_smart_cursor_changed), edit);
	return main_vbox;
}

static void
cb_bookmark_notify (GObject *object, GParamSpec *pspec,
		    KzBookmarkEdit *edit)
{
	KzBookmark *bookmark;

	g_return_if_fail(KZ_IS_BOOKMARK(object));
	bookmark = KZ_BOOKMARK(object);

	kz_bookmark_edit_sync_properties(edit, bookmark);
}


static void
cb_title_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *title;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	title = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_set_title(edit->bookmark, title);
}


static void
cb_uri_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *uri;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	uri = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_set_link(edit->bookmark, uri);
}


static void
cb_location_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *uri;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;
	if (!KZ_IS_BOOKMARK_FILE(edit->bookmark)) return;

        uri = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_file_set_location(KZ_BOOKMARK_FILE(edit->bookmark), uri);
}


static void
cb_interval_value_changed (GtkSpinButton *button, KzBookmarkEdit *edit)
{
	GtkAdjustment *adj;
	gint interval;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (!KZ_IS_BOOKMARK_FILE(edit->bookmark)) return;

	adj = gtk_spin_button_get_adjustment
		(GTK_SPIN_BUTTON(edit->interval_spin));
	interval = gtk_adjustment_get_value(adj);
	kz_bookmark_file_set_interval(KZ_BOOKMARK_FILE(edit->bookmark), interval);
}


static void
cb_description_changed (GtkTextBuffer *textbuf, KzBookmarkEdit *edit)
{
	GtkTextIter start_iter, end_iter;
	gchar *desc;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;

	gtk_text_buffer_get_start_iter(textbuf, &start_iter);
	gtk_text_buffer_get_end_iter(textbuf, &end_iter);

	desc = gtk_text_buffer_get_text(GTK_TEXT_BUFFER(textbuf),
					&start_iter, &end_iter,
					FALSE);

	kz_bookmark_set_description(edit->bookmark, desc);

	g_free(desc);
}

static void
cb_xmlrpc_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *xmlrpc;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;
	if (!KZ_IS_BOOKMARK_FILE(edit->bookmark)) return;

        xmlrpc = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_file_set_xmlrpc(KZ_BOOKMARK_FILE(edit->bookmark), xmlrpc);
}

static void
cb_xmlrpc_user_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *user;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;
	if (!KZ_IS_BOOKMARK_FILE(edit->bookmark)) return;

        user = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_file_set_xmlrpc_user(KZ_BOOKMARK_FILE(edit->bookmark), user);
}

static void
cb_xmlrpc_pass_entry_changed(GtkEditable *editable, KzBookmarkEdit *edit)
{
	const gchar *pass;

	g_return_if_fail(KZ_IS_BOOKMARK_EDIT(edit));

	if (edit->changing) return;
	if (!edit->bookmark) return;
	if (!KZ_IS_BOOKMARK_FILE(edit->bookmark)) return;

        pass = gtk_entry_get_text(GTK_ENTRY(editable));
	kz_bookmark_file_set_xmlrpc_pass(KZ_BOOKMARK_FILE(edit->bookmark), pass);
}

