/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *  Copyright (C) 2004 Hiroyuki Ikzoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-password-manager-dialog.h"

#include <stdio.h>
#include <gdk/gdkkeysyms.h>
#include <glib/gi18n.h>

#include "kz-embed.h"
#include "kz-embed-prefs.h"
#include "kz-icons.h"

enum {
	PROP_0,
	PROP_PARENT_WINDOW
};

enum {
	TERMINATOR = -1,
	COLUMN_HOST,
	COLUMN_USERNAME,
	N_COLUMNS
};

static GObject *constructor        (GType           type,
				    guint           n_props,
				    GObjectConstructParam *props);
static void     dispose            (GObject        *object);
static void     set_property       (GObject       *object,
				    guint          prop_id,
				    const GValue  *value,
				    GParamSpec    *pspec);
static void     get_property       (GObject       *object,
				    guint          prop_id,
				    GValue        *value,
				    GParamSpec    *pspec);
static void     response           (GtkDialog      *dialog,
				    gint            arg);

static void     cb_remove_button_clicked           (GtkWidget      *widget,
						    KzPasswordManagerDialog  *password_manager);

G_DEFINE_TYPE(KzPasswordManagerDialog, kz_password_manager_dialog, GTK_TYPE_DIALOG)

static void
kz_password_manager_dialog_class_init (KzPasswordManagerDialogClass *klass)
{
	GObjectClass *gobject_class;
	GtkDialogClass *dialog_class;

	gobject_class = G_OBJECT_CLASS(klass);
	dialog_class  = GTK_DIALOG_CLASS(klass);

	gobject_class->constructor  = constructor;
	gobject_class->dispose      = dispose;
	gobject_class->set_property = set_property;
	gobject_class->get_property = get_property;

	/* GtkDialog signals */
	dialog_class->response = response;

	g_object_class_install_property
		(gobject_class,
		 PROP_PARENT_WINDOW,
		 g_param_spec_object("parent-window",
				      _("ParentWindow"),
				     _("The parent kazehakase window"),
				     GTK_TYPE_WINDOW,
				     G_PARAM_READWRITE |
				     G_PARAM_CONSTRUCT_ONLY));
}


static void
kz_password_manager_dialog_init (KzPasswordManagerDialog *password_manager)
{
	GtkWidget *main_vbox, *scrwin, *tree_view, *hbox, *remove_button;
	GtkListStore *store;
	GtkCellRenderer *cell;
	GtkTreeViewColumn *col;
	GtkTreeSelection *selection;

	password_manager->parent_window = NULL;

	main_vbox = GTK_DIALOG(password_manager)->vbox;
	gtk_box_set_spacing(GTK_BOX(main_vbox), 4);
	gtk_window_set_default_size(GTK_WINDOW(password_manager), 600, 450);
	gtk_window_set_icon(GTK_WINDOW(password_manager), kz_icon);
	gtk_window_set_destroy_with_parent(GTK_WINDOW(password_manager), TRUE);

	gtk_dialog_add_buttons(GTK_DIALOG(password_manager),
			       GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
			       NULL);

	/* scrolled window */
	scrwin = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrwin),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
        gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(scrwin),
					    GTK_SHADOW_IN);
	gtk_box_pack_start(GTK_BOX(main_vbox), scrwin, TRUE, TRUE, 0);
	gtk_widget_show(scrwin);

	/* tree view */
	store = gtk_list_store_new(N_COLUMNS,
				   G_TYPE_STRING,
				   G_TYPE_STRING);
	tree_view = gtk_tree_view_new_with_model(GTK_TREE_MODEL(store));
	gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (tree_view), TRUE);
	selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tree_view));
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	/* host column */
	cell = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes
			(_("Host"), cell, "text", COLUMN_HOST, NULL);
	gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (col, 200);
	gtk_tree_view_column_set_resizable(col, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), col);

	/* username column */
	cell = gtk_cell_renderer_text_new();
	col = gtk_tree_view_column_new_with_attributes
			(_("User Name"), cell, "text", COLUMN_USERNAME, NULL);
	gtk_tree_view_column_set_sizing(col, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_fixed_width (col, 360);
	gtk_tree_view_column_set_resizable(col, TRUE);
	gtk_tree_view_append_column(GTK_TREE_VIEW(tree_view), col);

	gtk_container_add(GTK_CONTAINER(scrwin), tree_view);
	gtk_widget_show(tree_view);

	/* button box */
	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_widget_show(hbox);

	/* remove button */
	remove_button = gtk_button_new_from_stock(GTK_STOCK_REMOVE);
	gtk_box_pack_start(GTK_BOX(hbox), remove_button, FALSE, FALSE, 0);
	g_signal_connect(remove_button, "clicked",
			 G_CALLBACK(cb_remove_button_clicked), password_manager);
	gtk_widget_show(remove_button);

	password_manager->scrolled_window = scrwin;
	password_manager->tree_view       = GTK_TREE_VIEW(tree_view);
	password_manager->list_store      = store;
}


static GObject*
constructor (GType                  type,
	     guint                  n_props,
	     GObjectConstructParam *props)
{
	GtkWidget *widget;
	KzPasswordManagerDialog *password_manager;
	GObject *object;
	GObjectClass *klass = G_OBJECT_CLASS(kz_password_manager_dialog_parent_class);
	GList *list = NULL, *node;
	gboolean result;
	KzEmbedPrefs *embed_prefs;

	object = klass->constructor(type, n_props, props);

	password_manager = KZ_PASSWORD_MANAGER_DIALOG(object);
	widget = GTK_WIDGET(password_manager);

	gtk_window_set_title(GTK_WINDOW(password_manager), _("Password Manager"));
	gtk_window_set_transient_for(GTK_WINDOW(password_manager),
	    			     GTK_WINDOW(password_manager->parent_window));

#warning FIXME! KzProfile should handle password.
	embed_prefs = KZ_EMBED_PREFS(kz_embed_new("gecko"));
        if (!embed_prefs)
                return object;

	result = kz_embed_prefs_get_passwords(embed_prefs, &list);
	gtk_widget_destroy(GTK_WIDGET(embed_prefs));

	if (!result || !list) return object;

	for (node = list; node; node = g_list_next(node))
	{
		PasswordInfo *link = node->data;
		GtkTreeIter iter;
		const gchar *host = link->host;
		const gchar *user = link->username;

		if (!host || !*host) continue;

		gtk_list_store_append(password_manager->list_store, &iter);
		gtk_list_store_set(password_manager->list_store, &iter,
				   COLUMN_HOST, host,
				   COLUMN_USERNAME, user,
				   TERMINATOR);
	}

	g_list_free(list);

	return object;
}


static void
dispose (GObject *object)
{
	KzPasswordManagerDialog *password_manager = KZ_PASSWORD_MANAGER_DIALOG(object);

	if (password_manager->parent_window)
	{
		g_object_unref(password_manager->parent_window);
		password_manager->parent_window = NULL;
	}

	if (password_manager->list_store)
	{
		g_object_unref(password_manager->list_store);
		password_manager->list_store = NULL;
	}

	if (G_OBJECT_CLASS (kz_password_manager_dialog_parent_class)->dispose)
		G_OBJECT_CLASS (kz_password_manager_dialog_parent_class)->dispose(object);
}


static void
set_property (GObject         *object,
	      guint            prop_id,
	      const GValue    *value,
	      GParamSpec      *pspec)
{
	KzPasswordManagerDialog *password_manager = KZ_PASSWORD_MANAGER_DIALOG(object);
  
	switch (prop_id)
	{
	case PROP_PARENT_WINDOW:
		password_manager->parent_window = g_object_ref(g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


static void
get_property (GObject         *object,
	      guint            prop_id,
	      GValue          *value,
	      GParamSpec      *pspec)
{
	KzPasswordManagerDialog *password_manager = KZ_PASSWORD_MANAGER_DIALOG(object);

	switch (prop_id)
	{
	case PROP_PARENT_WINDOW:
		g_value_set_object(value, password_manager->parent_window);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
		break;
	}
}


GtkWidget *
kz_password_manager_dialog_new (GtkWindow *parent_window)
{
	return GTK_WIDGET(g_object_new(KZ_TYPE_PASSWORD_MANAGER_DIALOG,
				       "parent-window", parent_window,
				       NULL));

}


static void
response (GtkDialog *dialog, gint arg)
{
	KzPasswordManagerDialog *password_manager;

	g_return_if_fail (KZ_IS_PASSWORD_MANAGER_DIALOG(dialog));

	password_manager = KZ_PASSWORD_MANAGER_DIALOG(dialog);
	if (!password_manager->parent_window) return;

	switch (arg) {
	case GTK_RESPONSE_CLOSE:
		gtk_widget_destroy(GTK_WIDGET(dialog));
		break;
	default:
		break;
	}
}


static void
cb_remove_button_clicked (GtkWidget *widget,
			  KzPasswordManagerDialog *password_manager)
{
	GtkTreeSelection *selection;
	GtkTreeModel *model = NULL;
	GList *list, *node, *pass = NULL;
	KzEmbedPrefs *embed_prefs;

	g_return_if_fail(KZ_IS_PASSWORD_MANAGER_DIALOG(password_manager));

	embed_prefs = KZ_EMBED_PREFS(kz_embed_new("gecko"));
        if (!embed_prefs)
                return;

	selection = gtk_tree_view_get_selection(password_manager->tree_view);
	
	list = gtk_tree_selection_get_selected_rows(selection,
						    &model);
	for (node = list; node; node = g_list_next(node))
	{
		gchar *host, *user;
		PasswordInfo *p;
		GtkTreeIter iter;
		GtkTreePath *path = node->data;
		
		gtk_tree_model_get_iter(model, &iter, path);
	
		gtk_tree_model_get(model,
				   &iter, 
				   0, &host,
				   1, &user,
				   -1);
		p = g_new0(PasswordInfo, 1);
		p->host = g_strdup(host);
		p->username = g_strdup(user);
		g_free(host);
		g_free(user);
		pass = g_list_append(pass, p);
		gtk_list_store_remove(GTK_LIST_STORE(model), &iter);
		gtk_tree_path_free(path);
	}
#warning FIXME! KzProfile should handle password.
	kz_embed_prefs_remove_passwords(embed_prefs, pass);
	gtk_widget_destroy(GTK_WIDGET(embed_prefs));

	g_list_free(list);
	g_list_free(pass);
}

