/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-zoom-action.h"

#include <glib/gi18n.h>
#include "kazehakase.h"

typedef struct _KzZoomActionPrivate KzZoomActionPrivate;
struct _KzZoomActionPrivate 
{
	KzWindow *kz;
	GtkWidget *spin;
};

#define KZ_ZOOM_ACTION_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), KZ_TYPE_ZOOM_ACTION, KzZoomActionPrivate))

enum {
	PROP_0,
	PROP_KZ_WINDOW
};


static void dispose          (GObject *obj);
static void set_property     (GObject           *object,
                              guint              prop_id,
                              const GValue      *value,
                              GParamSpec        *pspec);
static void get_property     (GObject           *object,
                              guint              prop_id,
                              GValue            *value,
                              GParamSpec        *pspec);

static GtkWidget *create_tool_item (GtkAction  *action);

static void cb_spin_value_changed (GtkSpinButton *spin, KzZoomAction *action);

G_DEFINE_TYPE(KzZoomAction, kz_zoom_action, GTK_TYPE_ACTION)

static void
kz_zoom_action_class_init (KzZoomActionClass *klass)
{
	GObjectClass *object_class;
	GtkActionClass *action_class;

	object_class = G_OBJECT_CLASS(klass);
	action_class = GTK_ACTION_CLASS(klass);

	object_class->set_property     = set_property;
	object_class->get_property     = get_property;
	object_class->dispose          = dispose;

	action_class->create_tool_item = create_tool_item;

	g_object_class_install_property
		(object_class,
		 PROP_KZ_WINDOW,
		 g_param_spec_object ("kz-window",
			 _("KzWindow"),
			 _("The KzWindow to add a zoom button"),
			 KZ_TYPE_WINDOW,
			 G_PARAM_READWRITE |
			 G_PARAM_CONSTRUCT_ONLY));
	g_type_class_add_private (object_class, sizeof(KzZoomActionPrivate));
}


static void
kz_zoom_action_init (KzZoomAction *action)
{
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);

	priv->kz   = NULL;
	priv->spin = NULL;
}


static void
dispose (GObject *obj)
{
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (obj);

	if (priv->kz)
		g_object_unref(priv->kz);
	priv->kz = NULL;
	priv->spin = NULL;

	if (G_OBJECT_CLASS(kz_zoom_action_parent_class)->dispose)
		G_OBJECT_CLASS(kz_zoom_action_parent_class)->dispose(obj);
}


static void
set_property (GObject         *object,
              guint            prop_id,
              const GValue    *value,
              GParamSpec      *pspec)
{
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (object);

	switch (prop_id)
	{
		case PROP_KZ_WINDOW:
			priv->kz = g_value_dup_object(value);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}


static void
get_property (GObject         *object,
              guint            prop_id,
              GValue          *value,
              GParamSpec      *pspec)
{
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (object);

	switch (prop_id)
	{
		case PROP_KZ_WINDOW:
			g_value_set_object(value, priv->kz);
			break;
		default:
			G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
			break;
	}
}


KzZoomAction *
kz_zoom_action_new(KzWindow *kz)
{
	KzZoomAction *action;

	action = g_object_new(KZ_TYPE_ZOOM_ACTION,
			      "name",     "Zoom",
			      "label",    _("_Zoom"),
			      "tooltip",  _("Change the text size of the current page"),
			      "stock_id",  GTK_STOCK_ZOOM_100,
			      "kz-window", kz,
			      NULL);

	return action;
}

static void
cb_spin_button_destroy (GtkWidget *widget, GtkAction *action)
{
	KzZoomActionPrivate *priv;
	g_return_if_fail(KZ_IS_ZOOM_ACTION(action));
	
	priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);
	priv->spin = NULL;
}


static GtkWidget * 
create_tool_item (GtkAction  *action)
{
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);
	GtkWidget *widget, *spin;

	widget = GTK_WIDGET(g_object_new(GTK_TYPE_TOOL_ITEM, NULL));

	spin = gtk_spin_button_new_with_range(10, 990, 10);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), 100);
	gtk_container_add(GTK_CONTAINER(widget), spin);
	gtk_widget_show(spin);

	g_signal_connect(spin, "value-changed",
			 G_CALLBACK(cb_spin_value_changed), action);
	g_signal_connect(spin, "destroy",
			 G_CALLBACK(cb_spin_button_destroy), action);
	priv->spin = spin;

	return widget;
}


static void
cb_spin_value_changed (GtkSpinButton *spin, KzZoomAction *action)
{
	gint ratio;
	KzWeb *web;
	KzZoomActionPrivate *priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);

	if (!priv->spin || !gtk_widget_get_visible(priv->spin))
		return;

	web = KZ_WINDOW_CURRENT_WEB(priv->kz);
	if (!web)
		return;
	
	ratio = gtk_spin_button_get_value_as_int (spin);
	
	kz_web_zoom(web, (gdouble)ratio / 100);
}


void
kz_zoom_action_set_ratio (KzZoomAction *action,
			  gdouble ratio) 
{
	GtkWidget *spin;
	KzZoomActionPrivate *priv;
	g_return_if_fail(KZ_IS_ZOOM_ACTION(action));
	
	priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);

	if (!priv->spin || !gtk_widget_get_visible(priv->spin))
		return;
	
	spin = priv->spin;
	g_signal_handlers_block_by_func(spin,
			G_CALLBACK(cb_spin_value_changed), action);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(spin), ratio * 100);
	g_signal_handlers_unblock_by_func(spin,
			G_CALLBACK(cb_spin_value_changed), action);
}

gdouble
kz_zoom_action_get_ratio (KzZoomAction *action)
{
	gint ratio;
	KzZoomActionPrivate *priv;
	
	g_return_val_if_fail(KZ_IS_ZOOM_ACTION(action), 1.0);
	
	priv = KZ_ZOOM_ACTION_GET_PRIVATE (action);
	
	if (!priv->spin || !gtk_widget_get_visible(priv->spin))
		return 1.0;
	ratio = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(priv->spin)) / 100;
    
	return ratio;
}

