/* -*- Mode: C; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2009 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "kz-font-utils.h"

#include <gtk/gtk.h>

typedef struct _kz_font_list_set {
	GList **fonts;
	PangoLanguage *pango_language;
} kz_font_list_set;

static gboolean
each_font (PangoFontset *fontset, PangoFont *font, kz_font_list_set *list_set)
{
    PangoCoverage *coverage;
    PangoFontDescription *font_description;
    const gchar *sample_string, *p;
    glong i, len;

    coverage = pango_font_get_coverage(font, list_set->pango_language);
    sample_string = pango_language_get_sample_string(list_set->pango_language);
    len = g_utf8_strlen(sample_string, -1);
    p = sample_string;
    for (i = 0; i < len; i++) {
        gunichar index = g_utf8_get_char(p);
        if (pango_coverage_get(coverage, index) < PANGO_COVERAGE_EXACT) {
            pango_coverage_unref(coverage);
            return FALSE;
        }
        p = g_utf8_find_next_char(p, NULL);
    }

    font_description = pango_font_describe(font);
    *(list_set->fonts) = g_list_prepend(*(list_set->fonts), 
                                        g_strdup(pango_font_description_get_family(font_description)));
    pango_font_description_free(font_description);
    pango_coverage_unref(coverage);

    return FALSE;
}

void
kz_font_utils_get_font_list (const gchar *language, GList **fonts)
{
    PangoContext *pango_context;
    PangoLanguage *pango_language;
    PangoFontMap *font_map;
    PangoFontset *fontset;
    PangoFontDescription *font_description;
    kz_font_list_set list_set;

    *fonts = NULL;

    pango_context = gdk_pango_context_get();
    pango_language = pango_language_from_string(language);
    font_map = pango_cairo_font_map_get_default();
    font_description = pango_font_description_new();
    fontset = pango_font_map_load_fontset(font_map, 
                                          pango_context,
                                          font_description,
                                          pango_language);
    list_set.fonts = fonts;
    list_set.pango_language = pango_language;
    pango_fontset_foreach(fontset, (PangoFontsetForeachFunc)each_font, &list_set);

    *fonts = g_list_sort(*fonts, (GCompareFunc)g_strcmp0);

    g_object_unref(fontset);
    g_object_unref(pango_context);
    pango_font_description_free(font_description);
}

void
kz_font_utils_get_all_font_list (GList **fonts)
{
    PangoContext *pango_context;
    PangoFontFamily **families = NULL;
    gint n_families, i;

    *fonts = NULL;

    pango_context = gdk_pango_context_get();
    pango_context_list_families(pango_context, &families, &n_families);

    for (i = 0; i < n_families; i++) {
        gchar *font_name;
        font_name = g_strdup(pango_font_family_get_name(families[i]));
        *fonts = g_list_prepend(*fonts, font_name);
    }

    *fonts = g_list_sort(*fonts, (GCompareFunc)g_strcmp0);

    g_object_unref(pango_context);
    g_free(families);
}

void
kz_font_utils_free_font_list (GList *fonts)
{
    if (!fonts)
        return;

    g_list_foreach(fonts, (GFunc)g_free, NULL);
    g_list_free(fonts);
}

/* 
vi:ts=4:nowrap:ai:expandtab:sw=4
*/
