/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003-2004 Hiroyuki Ikezoe
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>
#include <ctype.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <glib/gstdio.h>
#include <stdlib.h>
#ifdef HAVE_UNISTD_H
#  include <unistd.h>
#endif
#ifdef HAVE_SYS_UTSNAME_H
#  include <sys/utsname.h>
#endif

#include "utils.h"
#include "glib-utils.h"
#include "gtk-utils.h"
#include "config.h"
#include "kazehakase.h"
#include "kz-search.h"
#include "kz-site.h"

#ifdef G_OS_WIN32
#  include <windows.h>
#endif

#define BUFFER_SIZE 1024

#define TIME_STAMP_FORMAT "%ld,%s\n"

gboolean
kz_utils_cp (const gchar *from, const gchar *to)
{
	FILE *fp;
	gboolean success = FALSE;
	gchar *contents;
	gsize length;

	if (!g_file_get_contents(from, &contents, &length, NULL))
		return FALSE;

	fp = g_fopen(to, "wb");
	if (fp)
	{
		success = TRUE;
		if (fwrite(contents, length, 1, fp) != 1)
			success = FALSE;
		fclose(fp);

		if (!success)
			g_unlink(to);
	}

	g_free(contents);
	return success;
}

gboolean
key_seems_sequential (const gchar *key, const gchar *prefix)
{
	gint len;
	const gchar *tail;
	gint i;
	gboolean valid;

	g_return_val_if_fail(key && *key, FALSE);
	g_return_val_if_fail(prefix && *prefix, FALSE);

	len = strlen(prefix);
	tail = key + len;

	if (strncmp(key, prefix, len)) return FALSE;

	len = strlen(tail);
	if (len <= 0) return FALSE;

	valid = TRUE;
	for (i = 0; i < len; i++)
	{
		if (!isdigit(tail[i]))
		{
			valid = FALSE;
			break;
		}
	}

	return valid;
}


gchar *
remove_tag (const gchar *string, gsize len)
{
	GString *work_string;
	guint i = 0;
	
	work_string = g_string_new(NULL);
	while (string[i] != '\0' && i < len)
	{
		if (string[i] == '<')
		{
			while (string[i] != '>' && string[i] != '\0' && i < len)
			{
				i++;
			}
		}
		else
			g_string_append_c(work_string, string[i]);
		i++;
	}
	
	return g_string_free(work_string, FALSE);
}


gchar *
html_to_text (const gchar *html)
{
	GString *work_string;
	guint i=0;
	
	work_string = g_string_new(NULL);
	while (html[i] != '\0')
	{
		gboolean ignore = FALSE;
		switch (html[i])
		{
		case '<':
		{
			gchar *name = NULL;
			if (!g_ascii_strncasecmp(html+i+1, "script", 6))
			{
				name = "/script";
				ignore = TRUE;
			}
			else if (!g_ascii_strncasecmp(html+i+1, "style", 5)) 
			{
				name = "/style"; 
				ignore = TRUE;
			}
			else if (!g_ascii_strncasecmp(html+i+1, "noscript", 8)) 
			{
				name = "/noscript"; 
				ignore = TRUE;
			}
			while (html[i] != '>' && html[i] != '\0')
				i++;
			if (ignore)
			{
				/* ignore until close tag */
				while (html[i] != '<' && html[i] != '\0' &&
				       g_ascii_strncasecmp(html+i+1, name, strlen(name)))
					i++;
				while (html[i] != '>' && html[i] != '\0')
					i++;
			}
			break;
		}
		case '&':
			if (!strncmp(html+i, "&amp;", 5))
			{
				g_string_append_c(work_string, '&');
				i+=4;
			}
			else if (!strncmp(html+i, "&quot;", 6))
			{
				g_string_append_c(work_string, '"');
				i+=5;
			}
			else if (!strncmp(html+i, "&lt;", 4))
			{
				g_string_append_c(work_string, '<');
				i+=3;
			}
			else if (!strncmp(html+i, "&gt;", 4))
			{
				g_string_append_c(work_string, '>');
				i+=3;
			}
			else
			{
				g_string_append_c(work_string, html[i]);
			}
			break;
		default:
			g_string_append_c(work_string, html[i]);
			break;
		}
		i++;
	}
	
	return g_string_free(work_string, FALSE);
}


gchar *
create_filename_from_uri (const gchar *uri)
{
	gchar *filename;
	gint i = 0;
	
	filename = g_strdup(uri);

	while (filename[i] != '\0')
	{
		if (filename[i] == '/')
			filename[i] = '_';
		i++;
	}

	return filename;
}


gchar *
create_filename_with_path_from_uri (const gchar *uri)
{
	gchar *filename;
	gchar *pos;
	gchar *scheme;
	
	pos = strstr(uri, "://");
	
	if (!pos)
	{
		pos = (gchar*)uri;
		scheme = g_strdup("");
	}
	else
	{
		scheme = g_strndup(uri, pos - uri);
		pos += 3;
	}

	if (g_str_has_suffix(uri, "/"))
	{
		filename = g_strconcat(scheme, G_DIR_SEPARATOR_S, pos, "_", NULL);
	}
	else
	{
		gchar *query_pos = strchr(uri, '?');
		if (query_pos)
		{
			gchar *string = g_strndup(pos, query_pos - pos);
			filename = g_strconcat(scheme,
					       G_DIR_SEPARATOR_S,
					       string,
					       "_"G_DIR_SEPARATOR_S,
					       query_pos + 1,
					       NULL);
			g_free(string);
		}
		else
		{
			filename = g_strconcat(scheme,
					       G_DIR_SEPARATOR_S,
					       pos,
					       NULL);
		}
	}

	g_free(scheme);

	return filename;
}


/**
 * create uri string from local cache file (ex. /users/http/path.to/data to http://path.to/data)
 * @param filename filename to create uri for local file path
 * @return new allocated uri string, that should be g_free
 */
gchar *
create_uri_from_filename (const gchar *filename)
{
	gchar *uri, *pos, *scheme, *path;
	
	pos = strstr(filename, G_DIR_SEPARATOR_S);
	if (!pos)
		return g_strdup(filename);

	scheme = g_strndup(filename, pos - filename);
	if (g_str_has_suffix(pos + 1, G_DIR_SEPARATOR_S"_"))
	{
		path = g_strndup(pos + 1, strlen(pos) - 2);
	}
	else
	{
		gchar *query_pos = g_strrstr(pos + 1, "_"G_DIR_SEPARATOR_S);
		if (query_pos)
		{
			gchar *string = g_strndup(pos + 1, query_pos - pos - 1);
			path = g_strconcat(string, "?", query_pos + 2, NULL);
			g_free(string);
		}
		else
		{
			path = g_strdup(pos + 1);
		}
	}

	if (!strcmp(scheme, "file"))
	{
		uri = g_strconcat(scheme, ":///", path, NULL);
	}
	else
	{
		uri = g_strconcat(scheme, "://", path, NULL);
	}

	g_free(scheme);
	g_free(path);

	return uri;
}


gchar *
url_decode(const gchar *src)
{
	GString *dest;
	gint len, i=0;

	if (!src) return NULL;
	len = strlen(src);
	dest = g_string_sized_new(len);

	while (src[i] != '\0' && i < len)
	{
		if (src[i] == '%')
		{
			if (i + 2 <= len &&
			    g_ascii_isxdigit(src[i+1]) && 
			    g_ascii_isxdigit(src[i+2]))
			{				
				g_string_append_c(dest,
						  g_ascii_xdigit_value(src[i+1]) * 16 + 
						  g_ascii_xdigit_value(src[i+2]));
				i+=2;
			}
		}
		else
		{
			g_string_append_c(dest, src[i]);
		}
		i++;
	}

	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
create_profile_key_from_uri (const gchar *string)
{
	gchar *key, *pos;
	gint len, i = 0;
	
	if (!string)
		return NULL;
	pos = strchr(string, '?');
	len = strlen(string);
	if (pos)
		len = pos - string;
	key = g_strndup(string, len);

	while (key[i] != '\0' && i < len)
	{
		if (key[i] == '=')
			key[i] = '_';
		i++;
	}

	return key;
}

/* return hex-encoded UTF-8 data
 * please free returned gchar* if unnecessary
 */
gchar *
url_encode(const gchar* utf8src)
{
	GString *dest;
	const gchar *ch = utf8src;
	unsigned char buf;

	if (!utf8src) return "";

	dest = g_string_sized_new(strlen(utf8src));

	while(*ch != '\0')
	{
		if (((*ch >= 'A') && (*ch <= 'Z')) ||
		    ((*ch >= 'a') && (*ch <= 'z')) ||
		    ((*ch >= '0') && (*ch <= '9')))
		{
			g_string_append_c(dest, *ch);
		}
		else if (*ch == ' ')
		{
			g_string_append_c(dest, '+');

		}
		else
		{
			g_string_append_c(dest, '%');
			buf = (*ch >> 4) & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
			buf = *ch & 0x0f;
			g_string_append_c(dest,
					( (buf < 10) ? buf + '0' : buf + ('A' - 10)) );
		}
		ch++;
	}
	
	/* Free gstring and reserve its data.*/
	return g_string_free(dest, FALSE);
}


gchar *
kz_utils_complement_scheme (const gchar* url)
{
	gchar *file;

	if (g_file_test(url, G_FILE_TEST_EXISTS))
	{
		if (!g_path_is_absolute(url))
		{
			gchar *current= g_get_current_dir();
			file = g_strdup_printf("file://%s/%s",
					       current,
					       url);
			g_free(current);
		}
		else
		{
			file = g_strdup_printf("file://%s",
					       url);
		}
		return file;
	}
	else
		return g_strdup(url);
}

gchar *
xml_get_content (const gchar *buffer)
{
	gchar *pos1, *pos2, *pos3;
	gchar *name;
	gchar *content = NULL;
	
	pos1 = strchr(buffer, '>');
	pos3 = strchr(buffer, ' ');
	if (pos1)
	{
		gchar *string;
		guint len;
		if (pos3 && pos1 > pos3)
			len = pos3 - buffer - 1;
		else
			len = pos1 - buffer - 1;
		name = g_strndup(buffer + 1, len); 
		string = g_strconcat("</", name, NULL);
		pos2 = strstr(pos1, string);
		if (pos2)
			content = g_strndup(pos1 +1 , pos2 - pos1 - 1);
		g_free(string);
		g_free(name);
	}
	return content;
}

gchar *
xml_get_attr (const gchar *buffer, const gchar *attr_name)
{
	gchar *pos1, *pos2;
	gchar *string;
	gchar *attr = NULL;
	guint len;

	pos1 = strchr(buffer, '>');
	if (!pos1)
		return NULL;

	string = g_strdup_printf("%s=\"", attr_name);
	pos1 = g_strstr_len(buffer, pos1 - buffer, string); 
	if (pos1)
	{
		len = strlen(string);
		pos1 += len;
		pos2 = strchr(pos1, '"'); 
		if (pos2)
			attr = g_strndup(pos1, pos2 - pos1); 
	}
	g_free(string);
	
	return attr;
}

static void
_append_time_stamp(const gchar *path, FILE *time_stamp)
{
	struct stat st;

	if (g_stat(path, &st) == 0)
		g_fprintf(time_stamp, TIME_STAMP_FORMAT, st.st_mtime, path);
}

void
kz_utils_append_time_stamp(const gchar *target_file,
			   const gchar *time_stamp_path)
{
	FILE *fp;

	fp = g_fopen(time_stamp_path, "a");
	if (!fp)
		return;

	_append_time_stamp(target_file, fp);
	fclose(fp);
}

static void
_make_time_stamp (const gchar *dir, FILE *time_stamp)
{
	GDir *gdir;
	const gchar *entry;

	gdir = g_dir_open(dir, 0, NULL);
	if (!gdir) return;

	while ((entry = g_dir_read_name(gdir)))
	{
		gchar *file_name;
		file_name = g_build_filename(dir, entry, NULL);

		if (g_file_test(file_name, G_FILE_TEST_IS_DIR))
			_make_time_stamp(file_name, time_stamp);
		else
			_append_time_stamp(file_name, time_stamp);
		g_free(file_name);
	}
	g_dir_close(gdir);
}

void
kz_utils_make_time_stamp(const gchar *target_dir, const gchar *time_stamp_path)
{
	FILE *fp;

	fp = g_fopen(time_stamp_path, "w");
	if (!fp)
		return;

	_make_time_stamp(target_dir, fp);
	fclose(fp);
}

void
kz_utils_purge_files(const gchar *target_dir, time_t limit_seconds)
{
	GDir *dir;
	const gchar *entry;

	dir = g_dir_open(target_dir, 0, NULL);

	if (!dir) return;

	while ((entry = g_dir_read_name(dir)))
	{
		gchar *file_name;
		file_name = g_build_filename(target_dir, entry, NULL);

		if (g_file_test(file_name, G_FILE_TEST_IS_DIR))
		{
			kz_utils_purge_files(file_name, limit_seconds);
		}
		else
		{
			struct stat st;
			time_t t;
			GTimeVal now;

			g_get_current_time(&now);
			t = (time_t)now.tv_sec;

			if (g_stat(file_name, &st) == 0 &&
			    st.st_mtime < t - limit_seconds)
				g_unlink(file_name);
		}
		g_free(file_name);
	}
	g_dir_close(dir);
}

void
kz_utils_purge_files_by_time_stamp(const gchar *target_dir,
				   const gchar *time_stamp_path,
				   time_t limit_seconds)
{
	gchar buf[BUFFER_SIZE];
	gchar *tmp_file;
	gint fd, target_dir_len;
	FILE *fp;
	GTimeVal now;
	time_t t;
	KzSearch *search;

	fp = g_fopen(time_stamp_path, "r");
	if (!fp)
	{
		g_unlink(time_stamp_path);
		return;
	}

	fd = g_file_open_tmp("kzXXXXXX", &tmp_file, NULL);
	if (fd == -1)
		return;

	search = KZ_GET_SEARCH;
	g_get_current_time(&now);
	t = now.tv_sec;
	target_dir_len = strlen(target_dir);
	while (fgets(buf, sizeof(buf), fp))
	{
		gchar *pos, *time, *path, *path_end, *dir_name;
		gint time_int;
		struct stat st;
		gboolean path_exists;

		pos = strstr(buf, ",");
		time = g_strndup(buf, pos - buf);
		path = g_strdup(pos+1);

		path_end = strstr(path, "\n");
		if (path_end)
			path[path_end - path] = '\0';

		time_int = atoi(time);
		if (t - time_int < limit_seconds)
		{
			ssize_t size;
			size = write(fd, buf, strlen(buf));
			goto CLEANUP;
		}

		path_exists = g_stat(path, &st) == 0;
		if (!path_exists)
			goto CLEANUP;

		if (t - st.st_mtime < limit_seconds)
		{
			gchar *entry;
			ssize_t size;

			entry = g_strdup_printf(TIME_STAMP_FORMAT,
						st.st_mtime, path);
			size = write(fd, entry, strlen(entry));
			goto CLEANUP;
		}


		if (search && strlen(path) > target_dir_len)
		{
			gchar *uri;
			gchar *relative_path;

			relative_path = path + target_dir_len;
			uri = create_uri_from_filename(relative_path);
			kz_search_unregister_document(search, uri);
			g_free(uri);
		}
		/* remove file over storage period */
		g_unlink(path);

		dir_name = g_path_get_dirname(path);
		g_rmdir(dir_name); /* ignore error */
		g_free(dir_name);

	  CLEANUP:
		g_free(time);
		g_free(path);
	}
	close(fd);
	fclose(fp);

	g_unlink(time_stamp_path);
	kz_utils_cp(tmp_file, time_stamp_path);
	g_unlink(tmp_file);
}

gchar *
ensure_encode_string(const gchar *text,
		     const gchar *encode,
		     gboolean urlencode)
{
	GError *e = NULL;
	gchar *encode_string = NULL, *url_string = NULL;

	if (!text) 
	{
		encode_string = g_strdup("");
		return encode_string;
	}

	if (encode)
	{
		encode_string = 
			g_convert(text,
				  strlen(text),
				  encode, "UTF-8",
				  NULL, NULL,
				  &e);
	}

	if (e || !encode)
	{
		encode_string =
		    g_strdup(text);
		if (e)
			g_error_free(e);
	}

	if (urlencode)	
	{
		url_string = url_encode(encode_string);
	}
	else
	{
		url_string = g_strdup(encode_string);
	}

	g_free(encode_string);
	
	return url_string;
}

GTime 
thumbnail_get_last_modified (const gchar *uri, EggPixbufThumbnailSize size)
{
	gchar *thumb_filename;
	struct stat st;
	int ret;

	thumb_filename = egg_pixbuf_get_thumbnail_filename(uri, size);

	ret = g_stat(thumb_filename, &st);
	g_free(thumb_filename);

	if (ret == 0)
		return st.st_mtime;
	else
		return 0;
}

GTime 
history_get_last_modified (const gchar *uri)
{
	gchar *history_filename, *filename;
	struct stat st;
	int ret;
	
	filename = create_filename_with_path_from_uri(uri);
	history_filename = g_build_filename(g_get_home_dir(),
					    HISTORY_DIR,
					    filename,
					    NULL);
	
	ret = g_stat(history_filename, &st);
	g_free(history_filename);
	g_free(filename);

	if (ret == 0)
		return st.st_mtime;
	else
		return 0;
}

gboolean
str_isdigit (const gchar *str)
{
	int i;
	size_t len;
	gboolean is_digit = TRUE;

	if (!str) return FALSE;

	len = strlen(str);
	for (i = 0; i < len; i++)
	{
		if (!isdigit(str[i]))
		{
			is_digit = FALSE;
			break;
		}
	}
	return is_digit;
}

/* the following tow functions are picked from utils/gul-general.c in galeon-1.3.18 */
static void
find_file_recursive (const gchar *path,
		     const gchar *fname,
		     GSList **l,
		     gint depth,
		     gint maxdepth)
{
	GDir *d = g_dir_open (path, 0, NULL);
	const gchar *f;
	if (d)
	{
		while ((f = g_dir_read_name (d)))
		{
			gchar *new_path = g_build_filename (path, f, NULL);
			if (depth < maxdepth)
			{
				find_file_recursive (new_path, fname, l,
							 depth + 1, maxdepth);
			}
			if (!strcmp (f, fname))
			{
				*l = g_slist_prepend (*l, new_path);
			}
			else
			{
				g_free (new_path);
			}
		}
		g_dir_close (d);
	}
}

/*
 * Looks for a file in a directory, recursively.
 */
GSList *
find_file (const gchar *path, 
	   const gchar *fname,
	   gint maxdepth)
{
	GSList *ret = NULL;
	find_file_recursive (path, fname, &ret, 0, maxdepth);
	return ret;
}

void
kz_str_replace_char (gchar *str, gchar replace_char, gchar new_char)
{
	gint i = 0;

	while (str[i] != '\0')
	{
		if (str[i] == replace_char)
			str[i] = new_char;
		i++;
	}
}

gchar *
kz_uri_encode_last_component (const gchar *uri)
{
	gchar *last_component_encoded_uri, *last_sep_pos;

	last_sep_pos = g_strrstr(uri, "/");
	if (last_sep_pos)
	{
		gchar *pre_component, *last_component;

		pre_component = g_strndup(uri, last_sep_pos - uri + 1);
		last_component = url_encode(last_sep_pos + 1);
		last_component_encoded_uri =
			g_strconcat(pre_component, last_component, NULL);
		g_free(pre_component);
	}
	else
	{
		last_component_encoded_uri = g_strdup(uri);
	}

	return last_component_encoded_uri;
}

#ifdef G_OS_WIN32
static gchar *
kz_win32_get_system_name (void)
{
	gchar *system = NULL;
	OSVERSIONINFO version;

	version.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);
	GetVersionEx(&version);

	switch (version.dwPlatformId)
	{
	case VER_PLATFORM_WIN32s:
		break;
	case VER_PLATFORM_WIN32_WINDOWS:
		if (version.dwMinorVersion == 0)
		{
			system = g_strdup("Windows 95");
			break;
		}

		if (version.dwMinorVersion == 10)
		{
			system = g_strdup("Windows 98");
			break;
		}

		if (version.dwMinorVersion == 90)
		{
			system = g_strdup("Windows Me");
			break;
		}

		break;
	case VER_PLATFORM_WIN32_NT:
		if (version.dwMajorVersion == 5 &&
		    version.dwMinorVersion == 0)
		{
			system = g_strdup_printf("Windows 2000 with %s",
						 version.szCSDVersion);
			break;
		}

		if (version.dwMajorVersion == 5 &&
		    version.dwMinorVersion == 1)
		{
			system = g_strdup_printf("Windows XP %s",
						 version.szCSDVersion);
			break;
		}

		if (version.dwMajorVersion <= 4)
		{
			system = g_strdup_printf("Windows NT %ld.%ld with %s",
						 version.dwMajorVersion,
						 version.dwMinorVersion,
						 version.szCSDVersion);
			break;
		}

		break;
	}

	if (!system)
		system = g_strdup_printf("Windows %ld.%ld",
					 version.dwMajorVersion,
					 version.dwMinorVersion);

	return system;
}
#endif

gchar *
kz_utils_get_system_name (void)
{
	gchar *system = NULL;

#ifdef HAVE_SYS_UTSNAME_H
	{
		struct utsname name;
		if (uname(&name) >= 0)
		{
			system = g_strdup_printf("%s %s",
						 name.sysname,
						 name.machine);
		}
	}
#endif

#ifdef G_OS_WIN32
	if (!system)
		system = kz_win32_get_system_name();
#endif

	if (!system)
		system = g_strdup("Unknown");

	return system;
}

static gchar *win32_base_path = NULL;

const gchar *
kz_win32_base_path (void)
{
#ifndef G_OS_WIN32
        return win32_base_path;
#else
	if (win32_base_path)
		return win32_base_path;
#if GLIB_CHECK_VERSION(2, 16, 0)
	win32_base_path = g_win32_get_package_installation_directory_of_module(NULL);
#else
	win32_base_path = g_win32_get_package_installation_directory(PACKAGE, NULL);
#endif

	return win32_base_path;
#endif
}

static void
append_site (KzBookmark *bookmark, GList **list)
{
	KzSite *site;

	site = kz_site_new(kz_bookmark_get_title(bookmark),
			   kz_bookmark_get_link(bookmark));
	*list = g_list_append(*list, site);
}

void
kz_utils_bookmark_folder_to_site_list (KzBookmarkFolder *folder, GList **list, guint *current_position)
{
	kz_bookmark_folder_foreach_child(folder, (GFunc)append_site, list);
	*current_position = kz_bookmark_folder_get_current_position(folder);
}

void
kz_utils_site_list_to_bookmark_folder (const GList *list, guint current_position, KzBookmarkFolder *folder)
{
	GList *node, *bookmark_children, *bookmark_node;

	bookmark_children = kz_bookmark_folder_get_children(folder);
	bookmark_node = g_list_first(bookmark_children);
	for (node = g_list_first((GList*)list); node; node = g_list_next(node))
	{
		KzSite *site = (KzSite*)node->data;
		KzBookmark *child;

		if (!bookmark_node) {
			child = kz_bookmark_new_with_attrs(site->title, site->uri, NULL);
			kz_bookmark_folder_append(folder, child);
		} else {
			child = KZ_BOOKMARK(bookmark_node->data);
			kz_bookmark_set_title(child, site->title);
			kz_bookmark_set_link(child, site->uri);
			bookmark_node = g_list_next(bookmark_node);
		}	

	}
	kz_bookmark_folder_set_current_position(folder, current_position);
	g_list_free(bookmark_children);
}

