package jp.kirikiri.tvp2.env;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.DataLine;
import javax.sound.sampled.FloatControl;
import javax.sound.sampled.Line;
import javax.sound.sampled.LineEvent;
import javax.sound.sampled.LineListener;
import javax.sound.sampled.LineUnavailableException;
import javax.sound.sampled.Mixer;
import javax.sound.sampled.SourceDataLine;

public class SoundMixer implements LineListener {
	private Mixer mCurrentMixer;

	private FloatControl mPanControl;
	private FloatControl mGainControl;
	private FloatControl mVolumeControl;
	private float mMinPanValue;
	private float mMaxPanValue;
	private float mMinGainValue;
	private float mMaxGainValue;
	private float mMinVolumeValue;
	private float mMaxVolumeValue;

	public void initialize() {
		// 一般的に利用されるオーディオフォーマットをサポートしているかテスト
		AudioFormat[] requestFormat = new AudioFormat[12];
		requestFormat[0]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 44100, 16, 2, 4, 44100, false ); // 44.1kHz 16bit stereo
		requestFormat[1]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 44100,  8, 2, 2, 44100, false ); // 44.1kHz 8bit stereo
		requestFormat[2]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 44100, 16, 1, 2, 44100, false ); // 44.1kHz 16bit mono
		requestFormat[3]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 44100,  8, 1, 1, 44100, false ); // 44.1kHz 8bit mono
		requestFormat[4]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 22050, 16, 2, 4, 22050, false ); // 22.05kHz 16bit stereo
		requestFormat[5]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 22050,  8, 2, 2, 22050, false ); // 22.05kHz 8bit stereo
		requestFormat[6]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 22050, 16, 1, 2, 22050, false ); // 22.05kHz 16bit mono
		requestFormat[7]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 22050,  8, 1, 1, 22050, false ); // 22.05kHz 8bit mono
		requestFormat[8]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 11025, 16, 2, 4, 11025, false ); // 11.025kHz 16bit stereo
		requestFormat[9]  = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 11025,  8, 2, 2, 11025, false ); // 11.025kHz 8bit stereo
		requestFormat[10] = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 11025, 16, 1, 2, 11025, false ); // 11.025kHz 16bit mono
		requestFormat[11] = new AudioFormat( AudioFormat.Encoding.PCM_SIGNED, 11025,  8, 1, 1, 11025, false ); // 11.025kHz 8bit mono

		DataLine.Info[] info = new DataLine.Info[12];
		for( int i = 0; i < requestFormat.length; i++ ) {
			info[i] = new DataLine.Info(SourceDataLine.class, requestFormat[i]);
		}

		mCurrentMixer = AudioSystem.getMixer(null);
		if( mCurrentMixer != null) {
			// デフォルトミキサーがサポートしているフォーマットを確認
			Mixer mixer = mCurrentMixer;
			Line.Info[] infos = mixer.getSourceLineInfo(); // ソースラインをサポートしているか確認
			if( infos.length > 0 ) {
				boolean supportAll = true;
				for( int j = 0; j < info.length; j++ ) {
					try {
						if( mixer.isLineSupported(info[j]) == false ) {
							supportAll = false;
							break;
						}
					} catch( IllegalArgumentException e ) {
						supportAll = false;
						break;
					}
				}
				if( supportAll ) { // 要求するフォーマットがサポートされているのでデフォルトミキサーを使う
					return;
				}
			}
		}

		// 要求するフォーマットをサポートしていないので、全てのミキサーの中からサポートしているものを探す
		Mixer.Info[] mixers = AudioSystem.getMixerInfo();
		final int count = mixers.length;
		for( int i = 0; i < count; i++ ) {
			Mixer mixer = AudioSystem.getMixer(mixers[i]);
			Line.Info[] infos = mixer.getSourceLineInfo();
			if( infos.length > 0 ) {
				boolean supportAll = true;
				for( int j = 0; j < info.length; j++ ) {
					try {
						if( mixer.isLineSupported(info[j]) == false ) {
							supportAll = false;
							break;
						}
					} catch( IllegalArgumentException e ) {
						supportAll = false;
						break;
					}
				}
				if( supportAll ) {
					mCurrentMixer = mixer;
					return; // 見付かったので、これを使う
				}
			}
		}
		// 要求するものをサポートするミキサーが見付からないので、デフォルトミキサーにしておく
		mCurrentMixer = AudioSystem.getMixer(null);
		if( mCurrentMixer == null ) return;

		try {
			mCurrentMixer.open();
			mCurrentMixer.addLineListener( this );
			mPanControl = (FloatControl) mCurrentMixer.getControl(FloatControl.Type.PAN);
			mGainControl = (FloatControl) mCurrentMixer.getControl(FloatControl.Type.MASTER_GAIN);
			mVolumeControl = (FloatControl) mCurrentMixer.getControl(FloatControl.Type.VOLUME);
			mMinPanValue = mPanControl.getMinimum();
			mMaxPanValue = mPanControl.getMaximum();
			mMinGainValue = mGainControl.getMinimum();
			mMaxGainValue = mGainControl.getMaximum();
			mMinVolumeValue = mVolumeControl.getMinimum();
			mMaxVolumeValue = mVolumeControl.getMaximum();
		} catch (LineUnavailableException e) {
		}
	}
	public void close() {
		if( mCurrentMixer != null ) {
			mCurrentMixer.close();
		}
	}
	@Override
	public void update(LineEvent event) {
		LineEvent.Type type = event.getType();
		if( LineEvent.Type.CLOSE.equals(type) ) {
		} else if( LineEvent.Type.OPEN.equals(type) ) {
		} else if( LineEvent.Type.START.equals(type) ) {
		} else if( LineEvent.Type.STOP.equals(type) ) {
		}
	}

	public void setPan( float pan ) {
		if( mPanControl == null ) return;
		if( pan < mMinPanValue ) pan = mMinPanValue;
		else if( pan > mMaxPanValue ) pan = mMaxPanValue;
		mPanControl.setValue( pan );
	}
	public float getPan() {
		if( mPanControl != null )
			return mPanControl.getValue();
		else
			return 0;
	}
	public void setGain( float gain ) {
		if( mGainControl == null ) return;
		if( gain < mMinGainValue ) gain = mMinGainValue;
		else if( gain > mMaxGainValue ) gain = mMaxGainValue;
		mGainControl.setValue(gain);
	}
	public float getGain() {
		if( mGainControl != null )
			return mGainControl.getValue();
		else
			return 0;
	}
	public void setVolume( float volume ) {
		if( mVolumeControl == null ) return;
		if( volume < mMinVolumeValue ) volume = mMinVolumeValue;
		else if( volume > mMaxVolumeValue ) volume = mMaxVolumeValue;
		mVolumeControl.setValue(volume);
	}
	public float getVolume() {
		if( mVolumeControl != null )
			return mVolumeControl.getValue();
		else
			return 0;
	}
	public SourceDataLine getSourceDataLine( DataLine.Info info ) throws LineUnavailableException {
		return (SourceDataLine) mCurrentMixer.getLine(info);
	}
	public void setGlobalVolume( int vol ) {
		if( vol < 0 ) vol = 0;
		else if( vol > 100000  ) vol = 100000;

		vol = vol - 100000;
		float v = vol / Math.abs(mMinGainValue);
		setGain( v );
	}
	public int getGlobalVolume() {
		float v = getGain();
		if( v > 0 ) {
			return 100000;
		} else {
			float vol = v * 100000 / Math.abs(mMinGainValue);
			vol = 100000 - vol;
			return (int)vol;
		}
	}

}
