﻿module y4d_draw.fontloader;

private import ytl.y4d_result;
private import y4d_aux.cacheloader;
private import y4d_draw.font;

///	文字フォントのcacheシステム。
/**
	Font を CacheLoader で cache する仕組みです。<BR>

	文字フォントはリソースが大きいので、ゲーム中に読み込みや解放を
	繰り返すべきではないと考えます。<BR>

	そこで、一度読み込んだフォントを(読み込みフォントの総容量が一定に
	達するまでは)解放しない仕組みが必要となります。<BR>

	このクラスは、その機能を実現します。

<PRE>
	y4d_result	loadDefFile(char[] filename);
	で読み込む定義ファイルは、
			msgothic.ttc , 0 , 16 , 0
			msmincho.ttc , 1 , 24 , 1
			msmincho.ttc , 2 , 40 , 0
	のように書き、ひとつ目は、ファイル名、二つ目は、読み込む番号、3,4つ目は
	option1,2である。option1はフォントサイズ,option2はフォントのindex
	(Font.openの時に指定するindex)を指定する。これらは指定しない場合0を
	意味するのでoption1は必ず指定する必要がある。
</PRE>

	番号は指定してもしなくても良い。指定しない場合、自動的に0から始まる
	連番となる。指定した場合、そこから連番となる。(CやDのenumと似たものだと
	考えれば良い。)<BR>

	また、フォントは文字サイズごとに生成する必要がある。そこで頻繁に
	文字サイズを変更するとリソースの再読み込みで結構時間がかかることになる。
	これを回避するためには、 TextureFontRepository を用いると良い。<BR>

	このクラスでキャッシュするフォントのサイズは、ディフォルトでは32MB。
	MS明朝,MSゴシックは10M程度あるので、32MBではひょっとすると少ないかも
	知れないが。

*/
class FontLoader : CacheLoader {
	///	noのフォントを生成して返す
	Font get(int no) {
		return cast(Font)getHelpper(no);
	}

	///	指定された番号のオブジェクトを構築して返す
	/**
		暗黙のうちにオブジェクトは構築されるが、openはされない。
		(この点、 get とは異なる)
	*/
	Font getObj(int no) {
		return cast(Font) getHelpperObj(no);
	}

	///	Fontのcloneメソッド。
	Font createInstance() { return new Font; }

	int opApply(int delegate(inout Font) dg)
	{
		int result = 0;
		foreach(int key;filelist.getKeys())
		{
			Font t = getObj(key);
			result = dg(t);
			if (result)	break;
		}
		return result;
	}

	///	foreachに対応させるためのもの
	/**
		事前にすべてのを読み込んでおきたいならば、
		foreachがあったほうが便利である。<BR>

		keyを返すバージョン。
	*/
	int opApply(int delegate(inout int) dg)
	{ return super.opApply(dg); }

	this() {
		cacheSize = 32*1024*1024; // 32MBのfont cache
	}

protected:
	///	フォントを読み込むための関数。LoadCacheからのオーバーロード
	y4d_result loadFile(char[] filename,inout Object obj,int option1,int option2) {
		if (obj) {
			Font s = cast(Font)obj;
			return s.open(filename,option1,option2);
		} else {
			Font s = new Font;
			y4d_result result = s.open(filename,option1,option2);
			obj = s;
			return result;
		}
	}

	///	フォントを解放するための関数。LoadCacheからのオーバーロード
	y4d_result releaseFile(inout Object obj) {
		if (obj) {
			Font s = cast(Font)obj;
			s.close();
		} else {
			//	読み込んでないから解放しないでいいんよね？
		}
		return y4d_result.no_error;
	}

}
