using System;
using System.Text;
using System.Diagnostics;
using System.Collections.Generic;

namespace imagefile{
	class Fds : GameImage{
		//public static readonly long MAX_SIZE = (long) (HEADER.Length + Disksystem.Filesystem.SIDE_SIZE * 8);
		public const long MAX_SIZE = 0x100000;
		public const string SAVEFILE_GUESS = "<guess>";
		public const string PATCHFILE_AUTO = "<auto>";
		static readonly byte [] HEADER = {
			(byte)'F', (byte)'D', (byte)'S', 0x1a, 0, 0, 0, 0,
			0, 0, 0, 0, 0, 0, 0, 0
		};
		//---- member and property ----
		Disksystem.Filesystem [] m_side;
		int Sidenum{
			get {return m_side.Length;}
		}
		//---- method ----
		public Fds() : base(mdc5.Script.imagetype.disk)
		{
		}
		bool compare(byte [] a0, byte[] a1, int offset, int length)
		{
			for(int i = 0; i < length; i++){
				int j = offset + i;
				if(a0[j] != a1[j]){
					return false;
				}
			}
			return true;
		}
		int header_check(byte [] data, out int offset)
		{
			//header check
			offset = 0;
			if(compare(HEADER, data, offset, 4) == false){
				return 0;
			}
			offset += 4;
			int side_num = (int) data[offset];
			if((side_num == 0) || (side_num >= 5)){
				return 0;
			}
			offset += 1;
			if(compare(HEADER, data, offset, HEADER.Length - offset) == false){
				return 0;
			}
			//image file size check
			int SIDE_SIZE = Disksystem.Filesystem.SIDE_SIZE;
			if(data.Length != (SIDE_SIZE * side_num + HEADER.Length)){
				return 0;
			}
			offset = HEADER.Length;
			return side_num;
		}
		//対象の面の data の配列を切り出してから渡す
		bool filesystem_set(
			byte [] data, ref int offset,
			Interface.ImageHash hash, 
			ref Disksystem.Filesystem side, string savefile_name
		)
		{
			int SIDE_SIZE = Disksystem.Filesystem.SIDE_SIZE;
			byte [] sidedata = new byte[SIDE_SIZE];
			Array.Copy(data, offset, sidedata, 0, SIDE_SIZE);
			offset += SIDE_SIZE;
			side = new Disksystem.Filesystem();
			if(side.SetAndCheck(sidedata, savefile_name) == false){
				return false;
			}
			hash.Transform(side.HashByte);
			return true;
		}
		
		override public bool Load(string diskimage_name)
		{
			m_name = diskimage_name;
			byte [] image;
			if(Static.Utility.binary_load(diskimage_name, out image) == false){
				return false;
			}
			int offset;
			int side_num = header_check(image, out offset);
			if(side_num == 0){
				return false;
			}
			m_hash = new Interface.ImageHash();
			m_side = new Disksystem.Filesystem[side_num];
			for(int i = 0; i < side_num; i++){
				if(filesystem_set(image, ref offset, m_hash, ref m_side[i], SAVEFILE_GUESS) == false){
					return false;
				}
			}
			m_hash.Final();
			return true;
		}

		override public bool Load(mdc5.Script k)
		{
			mdc5.DiskScript s = (mdc5.DiskScript) k;
			m_name = s.BiosName;
			m_code = s.GameCode;
			byte [] image;
			if(Static.Utility.binary_load(s.ImageFilename, out image) == false){
				return false;
			}
			int offset;
			int side_num = header_check(image, out offset);
			if(side_num == 0){
				return false;
			}
			m_hash = new Interface.ImageHash();
			m_side = new Disksystem.Filesystem[side_num];
			for(int i = 0; i < side_num; i++){
				if(filesystem_set(image, ref offset, m_hash, ref m_side[i], s.SavefileNameGet(i)) == false){
					return false;
				}
			}
			m_hash.Final();
			return true;
		}
		string sidename_get(int side)
		{
			string ret = "disk" + ((side / 2) + 1).ToString() + ".";
			ret += "side" + ((side & 1) == 0 ? "A" : "B") + ".";
			return ret;
		}
		override public string [] HashListGet()
		{
			List<string> list = new List<string>();
			int i = 0;
			foreach(Disksystem.Filesystem t in m_side){
				string prefix = sidename_get(i) + "hash = ";
				list.Add(prefix + t.HashStr);
				if(t.GuessedSavefileName != ""){
					string ayay = sidename_get(i) + "savefilename guessed" + t.GuessedSavefileName;
					list.Add(ayay);
				}
				i += 1;
			}
			return list.ToArray();
		}
		override public patchresult PatchManual(RomRecoard.MotorolaSRecoard r, out string log)
		{
			Debug.Assert(r.Valid == true);
			int found_count = 0;
			int side_index = 0, file_index = 0;
			for(int i = 0; i < m_side.Length; i++){
				int t = m_side[i].PatchSearch(r.Address, r.TargetFilename, ref file_index);
				found_count += t;
				if(t != 0){
					side_index = i;
				}
			}
			switch(found_count){
			case 0:
				log = "error: patchable area not found!";
				return patchresult.AREA_NOTFOUND;
			case 1: //正常
				break;
			default:
				log = "error: patchable area too much!";
				return patchresult.AREA_TOOMUCH;
			}
			log = sidename_get(side_index);
			m_side[side_index].PatchManual(file_index, r, ref log);
			return patchresult.OK;
		}
		
		override public patchresult PatchAuto(out string [] log)
		{
			int side_index = 0;
			List<string> list = new List<string>();
			foreach(Disksystem.Filesystem d in m_side){
				d.PatchAuto(sidename_get(side_index++), list);
			}
			log = list.ToArray();
			return patchresult.OK;
		}
		
		override public byte [] ToRomImage(int pagesize, ref int pageoffset, out int [] sideoffset, out mdc5.SaveArgument [] saveinfo)
		{
			sideoffset = new int [this.Sidenum];
			int totalpage = 0;
			{
				int i = 0;
				foreach(Disksystem.Filesystem t in m_side){
					sideoffset[i++] = pageoffset;
					int usepage = (t.EmptyOffset - 1) / pagesize;
					usepage += 1;
					pageoffset += usepage;
					totalpage += usepage;
				}
			}
			byte [] rom = new byte[totalpage * pagesize];
			int romoffset = 0;
			List<mdc5.SaveArgument> save = new List<mdc5.SaveArgument>();
			for(int i = 0; i < this.Sidenum; i++){
				mdc5.SaveArgument s;// = new mdc5.SaveArgument();
				if(m_side[i].ToRom(pagesize, ref rom, ref romoffset, out s) == true){
					s.Side = i;
					save.Add(s);
				}
				Debug.Assert((romoffset % pagesize) == 0);
			}
			saveinfo = save.ToArray();
			return rom;
		}
		
		public string [] FilesystemShow()
		{
			List<string> list = new List<string>();
			int i = 0;
			string prefix = String.Format(
				"{0,-8} {1,-11} {2,-13} {3}",
				"name", "region", "address-range", "length"
			);
			list.Add(prefix);
			foreach(Disksystem.Filesystem side in m_side){
				list.Add("-- " + sidename_get(i) + " -------------------------");
				foreach(string au in side.FilesystemShow()){
					list.Add(au);
				}
				i++;
			}
			return list.ToArray();
		}
		//pagesize = mdc5.GameImage.PAGESIZE
		override public int UsingPage(int pagesize)
		{
			int page = 0;
			foreach(Disksystem.Filesystem side in m_side){
				int side_capacity = side.EmptyOffset / pagesize;
				if((side.EmptyOffset % pagesize) != 0){
					side_capacity += 1;
				}
				page += side_capacity;
			}
			return page;
		}
	}
	namespace Disksystem{
		class Filesystem{
			public static readonly int SIDE_SIZE = 0xffdc;
			public const int OFFSET_UNKNOWN = 0;
			File[] m_file;
			int m_empty_offset = OFFSET_UNKNOWN;
			Interface.ImageHash m_hash;
			string m_guessed_savefilename = "";
			
			public int EmptyOffset{
				get{return m_empty_offset;}
			}
			public byte [] HashByte{
				get {return m_hash.Hash;}
			}
			public string HashStr{
				get {return m_hash.ToString();}
			}
			public string GuessedSavefileName{
				get {return m_guessed_savefilename;}
			}
			void savefile_guess(File[] file, ref string log)
			{
				FileHeader header = null;
				foreach(File t in file){
					switch(t.Id){
					case File.blockid.FILE_HEADER:
						header = (FileHeader) t;
						break;
					case File.blockid.EMPTY:
						header.SaveFileGuess();
						break;
					}
				}
				if(header.Savedata == true){
					log += String.Format(" {0} #0x{1:x4} byte", header.Name, header.DataSize);
				}
			}
			Interface.ImageHash hash_calc(File[] file)
			{
				Interface.ImageHash s = new Interface.ImageHash();
				FileHeader header = null; //header をなめた後に data を算出対象か判断するためにつける
				foreach(File t in file){
					switch(t.Id){
					case File.blockid.DISK_HEADER: //個別情報なので算出範囲外
					case File.blockid.FILE_AMOUNT: //あてにならないので算出範囲外
						break;
					case File.blockid.FILE_HEADER:{
						header = (FileHeader) t;
						s.Transform(header.DataWithoutSavedata);
						break;
						}
					case File.blockid.EMPTY:
						s.Final(t.Data);
						break;
					}
				}
				return s;
			}
			public bool SetAndCheck(byte [] data, string savefile)
			{
				int offset = 0;
				List<File> fileQueue = new List<File>();
				while(offset < data.Length){
					File.blockid blockid = (File.blockid) data[offset];
					offset += 1;
					Debug.Assert(blockid != File.blockid.FILE_DATA);
					switch(blockid){
					case File.blockid.DISK_HEADER:
						{
							DiskHeader h = new DiskHeader(blockid, data, ref offset);
							fileQueue.Add(h);
						}
						break;
					case File.blockid.FILE_AMOUNT:
						{
							FileAmount a = new FileAmount(blockid, data, ref offset);
							fileQueue.Add(a);
						}
						break;
					case File.blockid.FILE_HEADER:
						{
							//filedata.header and data
							FileHeader h = new FileHeader(blockid, data, ref offset);
							h.SaveFileSet(savefile);
							fileQueue.Add(h);
						}
						break;
					case File.blockid.EMPTY:
						{
							m_empty_offset = offset;
							Empty e = new Empty(blockid, data, ref offset, SIDE_SIZE - offset);
							if(e.Error == true){
								return false;
							}
							fileQueue.Add(e);
						}
						break;
					default: //image error
						return false;
					}
				}
				m_file = fileQueue.ToArray();
				//ファイル2以上は空のディスクイメージでセーブファイルを特定すると落ちるので対策
				if((savefile == Fds.SAVEFILE_GUESS) && (m_file.Length >= 2)){
					savefile_guess(m_file, ref m_guessed_savefilename);
				}
				m_hash = hash_calc(m_file);
				return true;
			}
			
			public int PatchSearch(uint address, string filename, ref int data_index)
			{
				int found = 0;
				//loop の途中で index を得るので for を使用する
				for(int i = 0; i < m_file.Length; i++){
					if(m_file[i].Id == File.blockid.FILE_HEADER){
						Disksystem.FileHeader h;
						h = (Disksystem.FileHeader) m_file[i];
						if(h.IsArea(address, filename) == true){
							found += 1;
							data_index = i;
						}
					}
				}
				return found;
			}
			public void PatchManual(int file_index, RomRecoard.MotorolaSRecoard r, ref string log)
			{
				FileHeader h = (FileHeader) m_file[file_index];
				h.PatchManual(r, ref log);
			}
			public void PatchAuto(string prefix, List<string> log)
			{
				foreach(File f in m_file){
					if(f.Id == File.blockid.FILE_HEADER){
						//この dynamic cast をなくさないと...
						Disksystem.FileHeader h;
						h = (Disksystem.FileHeader) f;
						h.PatchAuto(prefix, log);
					}
				}
			}
			
			public bool ToRom(int pagesize, ref byte [] rom, ref int rom_offset, out mdc5.SaveArgument save)
			{
				int side_offset = 0;//面の先頭からのoffset
				bool save_found = false;
				save = new mdc5.SaveArgument();
				foreach(File f in m_file){
					rom[rom_offset] = (byte) f.Id;
					if(f.Id == File.blockid.EMPTY){
						break;
					}
					rom_offset += 1;
					side_offset += 1;
					if(f.Id == File.blockid.FILE_HEADER){
						FileHeader h = (FileHeader) f;
						if(h.Savedata == true){
							save_found = true;
							save.Offset = side_offset; //header data offset なので
							save.Offset += 0x0f + 1; //header分と block ID を飛ばす
							save.Name = h.Name;
							save.Length = h.DataSize;
						}
					}
					foreach(byte t in f.Data){
						rom[rom_offset++] = t;
						side_offset += 1;
					}
				}
				int fillsize = rom_offset % pagesize;
				if(fillsize != 0){
					fillsize  = pagesize - fillsize;
					for(int i = 0; i < fillsize; i++){
						rom[rom_offset + i] = 0;
					}
					rom_offset += fillsize;
				}
				return save_found;
			}
			
			public string [] FilesystemShow()
			{
				List<string> list = new List<string>();
				foreach(File f in m_file){
					if(f.Id == File.blockid.FILE_HEADER){
						FileHeader h = (FileHeader) f;
						string t = String.Format(
							"{0,-8} {1,-11} 0x{2:x4}-0x{3:x4} 0x{4:x4}", 
							h.Name, h.Region, 
							h.Address, h.Address + h.DataSize - 1,
							h.DataSize
						);
						list.Add(t);
					}
				}
				return list.ToArray();
			}
		}

		class File{
			public enum blockid{
				EMPTY = 0, DISK_HEADER,
				FILE_AMOUNT, FILE_HEADER, FILE_DATA
			}
			protected byte [] m_data; //先頭の blockid を抜いた data
			protected blockid m_id;
			virtual public byte[] Data{
				get{return m_data;}
			}
			public blockid Id{
				get{return m_id;}
			}
			protected byte [] cut(byte [] data, ref int offset, int length)
			{
				byte[] filedata = new byte[length];
				Array.Copy(data, offset, filedata, 0, length);
				offset += length;
				return filedata;
			}
			uint shift_left(byte data, int count)
			{
				uint t = (uint) data;
				t &= 0xff;
				t <<= count;
				return t;
			}
			protected uint pack(byte [] data, ref int offset, int length)
			{
				Debug.Assert((length >= 1) && (length < 4));
				uint ret = 0;
				int shift = 0;
				switch(length){
				case 4:
					ret |= shift_left(data[offset++], shift);
					shift += 8;
					goto case 3;
				case 3:
					ret |= shift_left(data[offset++], shift);
					shift += 8;
					goto case 2;
				case 2:
					ret |= shift_left(data[offset++], shift);
					shift += 8;
					goto case 1;
				case 1:
					ret |= shift_left(data[offset++], shift);
					shift += 8;
					break;
				}
				return ret;
			}
			public File(blockid id, byte [] data, ref int offset, int length)
			{
				m_data = cut(data, ref offset, length);
				m_id = id;
			}
		}
/*
fds loader からの dumped image は 末尾から0x02-0x80byte程度に不規則な
ゴミデータが入っている場合がある。ゴミを許容するので末尾 0xc0 byte を
check せずに 0 で fill する
*/
		class Empty : File{
			bool m_error;
			public bool Error{
				get{return m_error;}
			}
			public Empty(blockid id, byte [] data, ref int offset, int length) 
			  :base(id, data, ref offset, length)
			{
/*				foreach(byte t in m_data){
					if(t != 0){
						m_error = true;
						return;
					}
				}*/
				int i;
				for(i = 0; i < m_data.Length - 0x100; i++){
					if(m_data[i] != 0){
						m_error = true;
						return;
					}
				}
				for(; i < m_data.Length; i++){
					m_data[i] = 0;
				}
				m_error = false;
			}
		}
/*
SIZE CONTENTS
14   FC Disk String  "**NINTENDO-HVC**"
 1   Manufacture Code  
 4   Game Name Code
 1   Game Version Number
 1   Side Number  0: Side-A  1: Side-B
 1   Disk Number
 1   Err.9  (Ext Disk Indicate No)
 1   Err.10 (Ext Disk Indicate No)
 1   Boot Read File Code  
       Specify ??? code read on boot
 5   Unknown
 3   Manufacture Permit Date(???)  
       Recorded in BCD, in the year of "showa"(+1925)
10   Unknown
 3   Created Date  
       Recorded in BCD, in the year of "showa"(+1925)
 9   Unknown*/
		/*
		disk個別で内容が異なるデータが含まれているのでcheckは行わない。
		imagefile 生成ツールにおいてはここの情報を欠落させる。
		*/
		class DiskHeader : File{
			public DiskHeader(blockid id, byte [] data, ref int offset) 
				:base(id, data, ref offset, 0x38 - 1)
			{
			}
		}
/* 
SIZE CONTENTS
 1   File Amount*/
		/*
		コピーツール対策にファイル数が一致しないものがあるので無効な情報とみなす。よって check はおこわない。
		*/
		class FileAmount :File{
			public FileAmount(blockid id, byte [] data, ref int offset) 
				:base(id, data, ref offset, 2 - 1)
			{
			}
		}
/*
SIZE CONTENTS
 1   File Number
 1   File Indicate Code (file identification code)
     ID specified at disk-read function call
 8   File Name
 2   File Address
     the destination address when loading
 2   File Size
 1   Kind of File  
       0:Program (CPU $0000-$07ff or $6000-$dfff)
       1:Character(PPU $0000-$1fff)
       2:Name table(PPU $2000-$2fff)
     The destination is shown.*/
		class FileHeader :File{
			public enum region{
				CPU_DATA, PPU_PATTERN, PPU_NAME
			};
			byte [] m_name_byte = new byte[8];
			string m_name_string;
			uint m_address;
			int m_size;
			region m_region;
			bool m_savedata = false;
			FileData m_content;
			
			public uint Address{
				get{return m_address;}
			}
			public int DataSize{
				get{return m_size;}
			}
			public region Region{
				get{return m_region;}
			}
			public string Name{
				get{return m_name_string;}
			}
			public bool Savedata{
				get{return m_savedata;}
				//set{m_savedata = value;}
			}
			override public byte [] Data{
				get{
					return header_with_data_get(m_data, m_content);
				}
			}
			//savefile 以外は filedata をまとめた data を渡す
			public byte [] DataWithoutSavedata{
				get{
					if(this.Savedata == true){
						return m_data;
					}
					return header_with_data_get(m_data, m_content);
				}
			}
			byte [] header_with_data_get(byte [] headerdata, FileData content)
			{
				int length = headerdata.Length + 1 + content.Data.Length;
				byte [] ret = new byte[length];
				int offset = 0;
				for(int i = 0; i < headerdata.Length; i++){
					ret[offset] = headerdata[i];
					offset += 1;
				}
				ret[offset++] = (byte) content.Id;
				for(int i = 0; i < content.Data.Length; i++){
					ret[offset] = content.Data[i];
					offset += 1;
				}
				return ret;
			}
			string name_set(byte [] bytename)
			{
				string s = "";
				bool nullfound = false;
				foreach(byte t in bytename){
					if(t == 0){
						nullfound = true;
					}else if(
						(t >= 0x20 && t < 0x7f) &&
						(nullfound == false)
					){
						//1文字だけ変換する method がわからない...
						byte [] au = new byte[2];
						au[0] = t;
						au[1] = 0;
						s += BitConverter.ToChar(au, 0);
					}else {
						s += String.Format("${0:x2}", t);
					}
				}
				if(s == ""){
					s = "$00";
				}
				return s.TrimEnd();
			}
			public FileHeader(blockid id, byte [] data, ref int dataoffset) 
			  :base(id, data, ref dataoffset, 0x10 - 1)
			{
				int offset = 1+1;
				m_name_byte = cut(m_data, ref offset, 8);
				m_name_string = name_set(m_name_byte);
				m_address = pack(m_data, ref offset, 2);
				m_size = (int) pack(m_data, ref offset, 2);
				m_region = (region) m_data[offset];
				
				blockid dataid = (blockid) data[dataoffset++];
				m_content = new FileData(dataid, data, ref dataoffset, m_size);
			}
			public bool IsArea(uint address, string filename)
			{
				if(m_region != region.CPU_DATA){
					return false;
				}
				if((filename == Fds.PATCHFILE_AUTO) || (m_name_string == filename)){
					uint start = m_address;
					uint end = m_address + (uint)(m_size) - 1;
					if(address >= start && address <= end){
						return true;
					}
				}
				return false;
			}

			public void SaveFileGuess()
			{
				if((m_region == region.CPU_DATA) && (m_size < 0x3000)){
					m_savedata = true;
					//Console.Write(m_name_string);
				}
			}
			public void SaveFileSet(string name)
			{
				if(m_name_string == name){
					m_savedata = true;
				}
			}
			public void PatchManual(RomRecoard.MotorolaSRecoard r, ref string log)
			{
				log += m_name_string + " ";
				r.TargetOffset = m_address;
				m_content.PatchManual(r, ref log);
			}
			public void PatchAuto(string prefix, List<string> log)
			{
				if(m_region != region.CPU_DATA){
					return;
				}
				if(m_address < 0x6000){
					return;
				}
				m_content.PatchAuto(m_address, prefix + m_name_string + " ", log);
			}
		}
/* 
SIZE CONTENTS
 1   File Amount*/
		class FileData :File{
			public FileData(blockid id, byte [] data, ref int offset, int length) 
			  :base(id, data, ref offset, length)
			{
			}
			public void PatchManual(RomRecoard.MotorolaSRecoard r, ref string log)
			{
				GameImage.PatchMemory(r, ref m_data, "0x{0:x4}", ref log);
			}
			const byte STORE_A_ABS = 0x8d, STORE_X_ABS = 0x8e, STORE_Y_ABS = 0x8c;
			const uint RP2C33_CONTROL = 0x4025;
			const byte JSR_ABS = 0x20;
			const uint MDC5_MIRROR_SET_A = 0xf400;
			const uint MDC5_MIRROR_SET_X = 0xf409;
			const uint MDC5_MIRROR_SET_Y = 0xf40C;
			void patching(uint jsr_address, string jsr_name, uint mapped_address, ref int offset, string prefix, List<string> log)
			{
				int t = offset + 1;
				if(pack(m_data, ref t, 2) == RP2C33_CONTROL){
					string au = prefix + String.Format("0x{0:x4}/${1:x4}: jsr {2} ", offset, mapped_address + offset, jsr_name);
					log.Add(au);
					m_data[offset++] = JSR_ABS;
					m_data[offset++] = (byte) (jsr_address & 0xff);
					m_data[offset++] = (byte) ((jsr_address >> 8) & 0xff);
				}else{
					offset += 1;
				}
			}
			public void PatchAuto(uint mapped_address, string prefix, List<string> log)
			{
				int offset = 0;
				// - 2 は 3byte 連続参照で m_data の範囲を超えないようにするため
				while(offset < m_data.Length - 2){
					switch(m_data[offset]){
					case STORE_A_ABS:
						//引数が多すぎ...
						patching(MDC5_MIRROR_SET_A, "mdc5_mirror_set_a", mapped_address, ref offset, prefix, log);
						break;
					case STORE_X_ABS:
						patching(MDC5_MIRROR_SET_X, "mdc5_mirror_set_x", mapped_address, ref offset, prefix, log);
						break;
					case STORE_Y_ABS:
						patching(MDC5_MIRROR_SET_Y, "mdc5_mirror_set_y", mapped_address, ref offset, prefix, log);
						break;
					default:
						offset += 1;
						break;
					}
				}
			}
		}
	}
}

