package jp.bne.deno.ordermaid.print;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.text.SimpleDateFormat;
import java.util.List;

import jp.bne.deno.ordermaid.dao.MenuItemTable;
import jp.bne.deno.ordermaid.dao.MenuTypeTable;
import jp.bne.deno.ordermaid.dao.SystemTable;
import jp.bne.deno.ordermaid.model.MenuType;
import jp.bne.deno.ordermaid.model.OrderData;
import jp.bne.deno.ordermaid.model.OrderDetail;
import jp.bne.deno.ordermaid.model.OrderItem;

import com.google.inject.Inject;
import com.lowagie.text.Cell;
import com.lowagie.text.Chunk;
import com.lowagie.text.Document;
import com.lowagie.text.DocumentException;
import com.lowagie.text.PageSize;
import com.lowagie.text.Paragraph;
import com.lowagie.text.Table;
import com.lowagie.text.pdf.PdfWriter;

/**
 * 注文票発行用ドキュメントビルダ.
 * 
 * @author hasegawt
 * 
 */
public class OrderSheetBuilder extends DocumentBuilder {

	@Inject
	SystemTable systemTable;

	@Inject
	MenuItemTable itemTable;

	@Inject
	MenuTypeTable typeTable;

	/**
	 * コンストラクタ.
	 */
	public OrderSheetBuilder() {
		super();
	}

	public File build(OrderData data, File file) throws IOException {
		String pageSize = systemTable.getSystemItem("pageSize", "A6")
				.getValue();
		Document document = new Document(PageSize.getRectangle(pageSize), 1, 1,
				1, 1);
		try {
			PdfWriter writer = PdfWriter.getInstance(document,
					new FileOutputStream(file));
			writer.setPageSize(PageSize.getRectangle(pageSize));
			writer.setMargins(0, 0, 0, 0);
			SimpleDateFormat format = new SimpleDateFormat("yy/MM/dd HH:mm:SS");
			document.open();

			List<MenuType> menuTypes = typeTable.findAll();

			int total = 0;
			for (OrderDetail detail : data.getDetails()) {
				for (OrderItem item : detail.getItems()) {
					total += item.getMenuItem().getPrice();
				}
			}

			document.add(new Paragraph("注文伝票  "
					+ format.format(data.getOrderDate()) + "   Ver.["
					+ data.getVersion() + "]    " + "[合計\\"
					+ Integer.toString(total) + "]", gothic16));
			document.add(new Paragraph(systemTable.getSystemItem("printHeader",
					"").getValue(), gothic10));

			for (OrderDetail detail : data.getDetails()) {
				String sex = "";
				if (detail.getSex().equals("1")) {
					sex = "(男)";
				} else if (detail.getSex().equals("2")) {
					sex = "(女)";
				} else {
					sex = "(他)";
				}
				String header2 = "";
				if (detail.isTakeOut()) {
					header2 = "テイクアウト [\\ " + detail.getSumPrice() + "]" + sex;
				} else {
					header2 = "席番:" + detail.getSheetId() + "  [\\ "
							+ detail.getSumPrice() + "]" + sex;
				}
				document.add(new Paragraph(header2, gothic12));

				int tableSize = menuTypes.size();
				Table orderTable = new Table(tableSize);
				orderTable.setPadding(1);
				orderTable.setCellsFitPage(true);
				orderTable.setTableFitsPage(true);
				orderTable.setWidth(100);
				// 商品見出し.
				for (MenuType type : menuTypes) {
					Cell cell = new Cell();
					cell.setHeader(true);
					cell.addElement(new Paragraph(type.getMenuType(), gothic8));
					cell.setHorizontalAlignment(Cell.ALIGN_LEFT);
					cell.setVerticalAlignment(Cell.ALIGN_TOP);
					orderTable.addCell(cell);
				}
				orderTable.endHeaders();
				// 商品.
				for (MenuType type : menuTypes) {
					Cell cell = new Cell("");
					cell.setHorizontalAlignment(Cell.ALIGN_MIDDLE);
					cell.setVerticalAlignment(Cell.ALIGN_TOP);
					for (OrderItem orderItem : detail.getItems()) {
						if (!orderItem.isCancel()
								&& type.equals(orderItem.getMenuItem()
										.getMenuType())) {
							Chunk chunk = getChunk(orderItem.getShortName());
							cell.addElement(chunk);
						}
					}
					orderTable.addCell(cell);
				}
				orderTable.endHeaders();
				document.add(orderTable);
			}
			document.add(new Paragraph(systemTable.getSystemItem("printFooter",
					"").getValue(), gothic10));
		} catch (DocumentException e) {
			e.printStackTrace();
			throw new IOException(e);
		} finally {
			// 終了処理.
			document.close();
		}
		return file;
	}

	private Chunk getChunk(String shortName) {
		Chunk chunk;
		switch (shortName.length()) {
		case 1:
			chunk = new Chunk(shortName + "\r\n", gothic14);
			break;
		case 2:
			chunk = new Chunk(shortName + "\r\n", gothic14);
			break;
		case 3:
			chunk = new Chunk(shortName + "\r\n", gothic14);
			break;
		default:
			chunk = new Chunk(shortName + "\r\n", gothic14);
			break;

		}
		// chunk.setBackground(Color.black);

		return chunk;

	}

}
