#pragma once

#include <cassert>
#include <vector>

#include "OpenCVTextureSetter.h"


namespace lib_gl
{


//! OpenCVˑ̃rfIeNX`NX
class CvVideoTexture
{
public:
	CvVideoTexture(void);
	virtual ~CvVideoTexture(void);

	bool OpenFile( const char* filename );
	bool OpenWebcamera(void);

	void Release(void);

	bool LoadNextFrame( void );
	bool SetGLLastFrame( TEXTURE_MODE tex_mode = TEXTURE_NORMAL );
	bool SetGLNextFrame( TEXTURE_MODE tex_mode = TEXTURE_NORMAL );


	bool IsFromVideoFile(void) const;
	bool IsFromWebcamera(void) const;

	bool IsValid(void) const;
	bool IsLastFrame(void) const;

	double GetNextFrameTime(void) const;
	void   SetNextFrameTime(double i_time);

	double GetNextFrameIndex(void) const;
	void   SetNextFrameIndex(double i_index);

	double GetNextNormalizedFramePos(void) const;
	void   SetNextNormalizedFramePos(double i_pos);

	double GetWidth  (void) const;
	double GetHeight (void) const;

	double GetFPS(void) const;
	double GetInterval(void) const;

	double GetFrameCount(void) const;


	const CvCapture* GetCapture   (void) const { return m_Capture;   }
	const IplImage*  GetLastFrame (void) const { return m_LastFrame; }


private:
	double GetProperty( int i_propertyId ) const;
	void   SetProperty( int i_propertyId , double i_value );


private:
	CvCapture*   m_Capture;
	IplImage*    m_LastFrame;

	bool         m_VideoFileOpened;
	bool         m_WebcameraOpened;
};



inline CvVideoTexture::CvVideoTexture(void)
{
	m_Capture = NULL;
	m_LastFrame   = NULL;

	m_VideoFileOpened = false;
	m_WebcameraOpened = false;
}

inline CvVideoTexture::~CvVideoTexture(void)
{
	Release();
}


//! t@CJ
inline bool CvVideoTexture::OpenFile( const char* filename )
{
	// UȂƑ̃t@C͊JȂ.
	_ASSERTE( m_Capture == NULL );
	if( m_Capture != NULL )
		return false;

	m_Capture = cvCaptureFromFile( filename );
	if( m_Capture == NULL )
		return false;

	m_VideoFileOpened = true;
	return true;
}

//! WebJJ
inline bool CvVideoTexture::OpenWebcamera(void)
{
	_ASSERTE( m_Capture == NULL );
	if( m_Capture != NULL )
		return false;

	m_Capture = cvCaptureFromCAM( CV_CAP_ANY );
	if( m_Capture == NULL )
		return false;

	m_WebcameraOpened = true;
	return true;
}


//! .
inline void CvVideoTexture::Release(void)
{
	if( m_Capture != NULL )
	{
		cvReleaseCapture( &m_Capture );
		m_Capture    = NULL;
		m_LastFrame  = NULL;
	}

	m_VideoFileOpened = false;
	m_WebcameraOpened = false;
}


//! ̃t[擾.
inline bool CvVideoTexture::LoadNextFrame( void )
{
	_ASSERTE( m_Capture != NULL );
	if( m_Capture == NULL )
		return false;

	m_LastFrame = cvQueryFrame( m_Capture );
	if( m_LastFrame == NULL )
		return false;

	return true;
}

//! Ōɓǂݍ񂾃t[OpenGL̃eNX`ɃZbg.
inline bool CvVideoTexture::SetGLLastFrame( TEXTURE_MODE tex_mode )
{
	_ASSERTE( m_Capture != NULL );
	if( m_Capture == NULL || m_LastFrame == NULL )
		return false;

	return SetCVTexture( m_LastFrame , tex_mode );
}

//! ̃t[擾OpenGL̃eNX`ɃZbg.
inline bool CvVideoTexture::SetGLNextFrame( TEXTURE_MODE tex_mode )
{
	_ASSERTE( m_Capture != NULL );
	if( m_Capture == NULL )
		return false;

	if( !LoadNextFrame() ) return false;
	if( !SetGLLastFrame( tex_mode ) ) return false;

	return true;
}


//! ݊JĂ铮eNX`t@CȂtrueԂ.
inline bool CvVideoTexture::IsFromVideoFile(void) const
{
	if( !IsValid() )
		return false;

	return m_VideoFileOpened;
}

//! ݊JĂ铮eNX`WebJȂtrueԂ.
inline bool CvVideoTexture::IsFromWebcamera(void) const
{
	if( !IsValid() )
		return false;

	return m_WebcameraOpened;
}


//! rfIJĂ邩mF.
inline bool CvVideoTexture::IsValid(void) const
{
	return ( m_Capture != NULL );
}

//! Ō̃t[ɓBmF.
inline bool CvVideoTexture::IsLastFrame(void) const
{
	// TODO : V[N𐳂擾łȂꍇ.

	// ȂőCfbNX = t[-2 ƂȂĂ̂ŉLɂȂ.
	return ( GetFrameCount() - GetNextFrameIndex() <= 2 );
}

	
//! ̃t[̎Ԃ߂.
inline double CvVideoTexture::GetNextFrameTime(void) const
{
	return GetProperty( CV_CAP_PROP_POS_MSEC );
}

inline void CvVideoTexture::SetNextFrameTime(double i_time)
{
	SetProperty( CV_CAP_PROP_POS_MSEC , i_time );
}

//! ̃t[̃CfbNX߂.
inline double CvVideoTexture::GetNextFrameIndex(void) const
{
	return GetProperty( CV_CAP_PROP_POS_FRAMES );
}

inline void CvVideoTexture::SetNextFrameIndex(double i_index)
{
	SetProperty( CV_CAP_PROP_POS_FRAMES , i_index );
}

//! ̃t[̃rfIł̐Kʒu(0`1)߂.
inline double CvVideoTexture::GetNextNormalizedFramePos(void) const
{
	return GetProperty( CV_CAP_PROP_POS_AVI_RATIO );
}

inline void CvVideoTexture::SetNextNormalizedFramePos(double i_pos)
{
	SetProperty( CV_CAP_PROP_POS_AVI_RATIO , i_pos );
}

//! rfỈ擾.
inline double CvVideoTexture::GetWidth(void) const
{
	return GetProperty( CV_CAP_PROP_FRAME_WIDTH );
}

//! rfȈc擾.
inline double CvVideoTexture::GetHeight(void) const
{
	return GetProperty( CV_CAP_PROP_FRAME_HEIGHT );
}

//! FPS擾.
inline double CvVideoTexture::GetFPS(void) const
{
	return GetProperty( CV_CAP_PROP_FPS );
}

//! t[Ԃ̃C^[o(==FPS̋t) 擾.
inline double CvVideoTexture::GetInterval(void) const
{
	double fps = GetFPS();
	if( fps <= 0.0 )
		return 0.0;
	
	return 1000.0 / fps;
}

//! t[擾.
inline double CvVideoTexture::GetFrameCount(void) const
{
	return GetProperty( CV_CAP_PROP_FRAME_COUNT );
}


inline double CvVideoTexture::GetProperty( int i_propertyId ) const
{
	_ASSERTE( m_Capture != NULL );
	if( m_Capture == NULL )
		return 0;

	return cvGetCaptureProperty( m_Capture , i_propertyId );
}

inline void CvVideoTexture::SetProperty( int i_propertyId , double i_value )
{
	_ASSERTE( m_Capture != NULL );
	if( m_Capture == NULL )
		return;

	cvSetCaptureProperty( m_Capture , i_propertyId , i_value );
}


}
