#include <QtGui>
#include "EditViewBase.h"
#include "Settings.h"
#include "charEncoding.h"
#include "CompletionWidget.h"

bool isLetterOrNumberOrUnderbar(const QChar &ch);
bool isLetterOrUnderbar(const QChar &ch);
void incCmdStatistics(const QString &keyText);

extern QHash<QString, uint> g_cmdStatistics;

EditViewBase::EditViewBase(Settings *settings, QWidget *parent)
	: QTextEdit(parent)
	, m_charEncoding(CharEncoding::UTF8)
{
	m_settings = settings;
	setAcceptDrops(false);			//	D&D ANZvg֎~
	setAcceptRichText(false);		//	b`eLXg[h OFF
	setLineWrapMode(QTextEdit::NoWrap);		//	܂Ԃ OFF
	setViewportMargins(leftMarginWidth(), 0, 0, 0);
	m_lmWidget = new QWidget(this);
	m_lmWidget->installEventFilter(this);
}

EditViewBase::~EditViewBase()
{

}
int EditViewBase::leftMarginWidth()
{
	return fontMetrics().width('8') * (2 + (settings()->m_lineNumber ? 5 : 0));
}
bool EditViewBase::modified() const
{
	return document()->isModified();
}
void EditViewBase::setModified(bool b, bool forced)
{
	if( b != modified() || forced ) {
		document()->setModified(b);
	}
}
void EditViewBase::setFullPath(const QString &fullPath)
{
	m_fullPath = fullPath;
	QFileInfo fi(fullPath);
	m_title = fi.fileName();
}
void EditViewBase::clearSelection()
{
	QTextCursor cursor = textCursor();
	cursor.clearSelection();
	setTextCursor(cursor);
}
QString EditViewBase::currentDir() const
{
	QString curPath = fullPath();
	if( curPath.isEmpty() ) return QString();
	QDir dir(curPath);
	dir.cdUp();
	return dir.absolutePath();
}
void EditViewBase::cdCurrentDir() const
{
	QString curDir = currentDir();
	if( !curDir.isEmpty() )
		QDir::setCurrent(curDir);
}
QString EditViewBase::toolTipText() const
{
	QString text(fullPath());
	if( text.isEmpty() ) text = title();
	cchar *encName = codecName(charEncoding());
	if( encName != 0 )
		text += QString(" (%1)").arg(encName);
	return text;
}
void EditViewBase::setCharEncoding(uchar enc)
{
	m_charEncoding = enc;
	emit encodingChanged();
}
#if 0
void EditViewBase::paste()
{
	QClipboard *cb = QApplication::clipboard();
	const QMimeData *mime = cb->mimeData();
	if( !mime ) return;
	if( mime->hasText() ) {
		QString text = mime->text();
		QTextCursor cur = textCursor();
		cur.insertText(text);
	}
}
#endif
QString EditViewBase::autoIndentText(QTextCursor cur)
{
	QString tab(settings()->m_softTab ? QString(settings()->m_tabWidth, QChar(' ')) : "\t");
	QString text;
	//QTextCursor cur = textCursor();
	QTextBlock block = cur.block();
	QString blockText = block.text();
	int i = 0;
    while( (i < blockText.length() && blockText[i] == ' ') || blockText[i] == '\t' ) {
		text += blockText[i++];
	}
	//	begin class def if unless else for while until when Ŏn܂s
	//	Aend ܂ } ŏIĂꍇ̓CfgȂ
	QRegExp exp("^\\s*(begin|class|module|def|if|unless|else|elsif|for|while|until|when)\\b");
	QRegExp expClose("(end|\\})(\\s*#.*)?$");
	if( exp.indexIn(blockText) >= 0
		&& cur.positionInBlock() >= exp.matchedLength() &&
		expClose.indexIn(blockText) < 0 )
	{
		text += tab;		//	undone ݒˑ
		return text;
	}
	//	J[\ { ȍ~ɂA{ (|.+|)?(Rg)? ŏIs
	QRegExp exp2("\\{(\\s*\\|.+\\|)?(\\s*#.*)?$");
	if( (i = exp2.indexIn(blockText)) >= 0
		&& cur.positionInBlock() >= i + 1 )
	{
		text += tab;		//	undone ݒˑ
		return text;
	}
	//	J[\ do ɂAdo (|.+|)?(Rg)? ŏIs
	QRegExp expDo("\\bdo(\\s*\\|.+\\|)?(\\s*#.*)?$");
	if( (i = expDo.indexIn(blockText)) >= 0
		&& cur.positionInBlock() >= i + 2 )
	{
		text += tab;		//	undone ݒˑ
		return text;
	}
	return text;
}
bool EditViewBase::eventFilter(QObject *obj, QEvent *event)
{
#if 1
	if( obj == m_lmWidget && event->type() == QEvent::Paint ) {
        drawLeftMargin();
        return true;
    }
#endif
    return false;
}
void EditViewBase::drawLeftMargin()
{
	QPainter painter(m_lmWidget);
	QRect wr = m_lmWidget->rect();
	painter.fillRect(wr, settings()->color(Settings::LEFT_MARGIN));
	if( !settings()->m_lineNumber )
		return;
	const int ht = fontMetrics().ascent();
	QTextCursor cur = textCursor();
	cur.setPosition(0);
	int lineNumber = 1;
	QRect r;
	while( 1 ) {
		r = cursorRect(cur);
		if( r.bottom() >= 0 ) break;
		if( !cur.movePosition(QTextCursor::NextBlock) )
			return;
		++lineNumber;
	}
	while( 1 ) {
		if( settings()->m_lineNumber ) {
			painter.setPen(settings()->color(Settings::LINE_NUMBER));
			const QString lnText = QString("%1:").arg(lineNumber);
			int x = wr.width() - 8 - fontMetrics().width(lnText);
			painter.drawText(x, r.y() + ht, lnText);
		}
		if( !cur.movePosition(QTextCursor::NextBlock) )
			return;
		r = cursorRect(cur);
		++lineNumber;
	}
}
void EditViewBase::paintEvent(QPaintEvent * event)
{
	//QPainter painter(this);
	//painter.fillRect(rect(), m_settings->color(Settings::BACKGROUND));
	QTextEdit::paintEvent(event);
	if( !hasFocus() ) {		//	tH[JXꍇ́AOŃJ[\`
		QPainter painter(viewport());
		painter.setPen(QPen(QBrush(m_settings->color(Settings::TEXT)), 0.0, Qt::DotLine));
		//painter.setPen(m_settings->color(Settings::TEXT));
		painter.drawRect(cursorRect(textCursor()));
	}
	m_lmWidget->update();		//	XN[Ȃǂ̂߂ɕKv
}
//	IsEVtgiCfgj
void EditViewBase::shiftRight()
{
	QTextCursor cur = textCursor();
	int pos = cur.position();
	int last = cur.anchor();
	if( pos > last ) qSwap(pos, last);	//	make sure pos <= last
	QString tab = settings()->m_softTab ? QString(settings()->m_tabWidth, QChar(' ')) : "\t";
	cur.setPosition(pos);
	QTextBlock block = cur.block();
	cur.beginEditBlock();
	while( block.isValid() && block.position() < last ) {
		cur.setPosition(block.position());		//	s擪ʒu
		cur.insertText(tab);
		block = block.next();
		last += tab.length();
	}
	cur.endEditBlock();
}
//	IsVtgitCfgj
void EditViewBase::shiftLeft()
{
	QTextCursor cur = textCursor();
	int pos = cur.position();
	int last = cur.anchor();
	if( pos > last ) qSwap(pos, last);	//	make sure pos <= last
	QString tab = settings()->m_softTab ? QString(settings()->m_tabWidth, QChar(' ')) : "\t";
	cur.setPosition(pos);
	QTextBlock block = cur.block();
	cur.beginEditBlock();
	while( block.isValid() && block.position() < last ) {
		QString blockText = block.text();
		int n = 0;
		if( !blockText.isEmpty() ) {
			if( blockText[0] == QChar('\t') )
				n = 1;
			else {
				while( n <= block.length() && blockText[n] == QChar(' ') ) {
					if( ++n == settings()->m_tabWidth )
						break;
				}
			}
			if( n != 0 ) {
				cur.setPosition(block.position());		//	s擪ʒu
				cur.movePosition(QTextCursor::Right, QTextCursor::KeepAnchor, n);
				cur.deleteChar();
				last -= n;
			}
		}
		block = block.next();
	}
	cur.endEditBlock();
}


void EditViewBase::focusOutEvent ( QFocusEvent * e )
{
	//doOutput(QString("focusOutEvent %1\n").arg((qlonglong)this));
	QTextEdit::focusOutEvent(e);
	//doOutput(QString("focusOutEvent %1\n").arg((qlonglong)this));
	emit focusOut();
}
void EditViewBase::focusInEvent ( QFocusEvent * e )
{
	//doOutput(QString("focusInEvent %1\n").arg((qlonglong)this));
	QTextEdit::focusInEvent(e);
	//doOutput(QString("focusInEvent %1\n").arg((qlonglong)this));
	emit focusIn();
}
void EditViewBase::mousePressEvent ( QMouseEvent * event )
{
	incCmdStatistics("Edit.mousePress");
	QTextEdit::mousePressEvent ( event );
}
void EditViewBase::mouseDoubleClickEvent ( QMouseEvent * event )
{
	incCmdStatistics("Edit.mouseDoubleClick");
	QTextEdit::mouseDoubleClickEvent ( event );
}
void EditViewBase::keyPressEvent ( QKeyEvent * e )
{
	const bool ctrl = (e->modifiers() & Qt::ControlModifier) != 0;
	const bool shift = (e->modifiers() & Qt::ShiftModifier) != 0;
	const bool alt = (e->modifiers() & Qt::AltModifier) != 0;
	QString keyText;	//	R}hvL[
	if( !alt ) {
		if( ctrl ) {
			if( shift ) {
			} else {
				//	Ctrl + ̂
				switch( e->key() ) {
				case Qt::Key_Home:	keyText = "Edit.StartOfDoc";	break;
				case Qt::Key_End:	keyText = "Edit.EndOfDoc";	break;
				case Qt::Key_Left:	keyText = "Edit.PrevWord";	break;
				case Qt::Key_Right:	keyText = "Edit.NextWord";	break;
				case Qt::Key_Enter:	keyText = "Edit.OpenNextLine";	break;
				case Qt::Key_Delete:	keyText = "Edit.WordDelete";	break;
				case Qt::Key_Backspace:	keyText = "Edit.WordBackspace";	break;
				case Qt::Key_A:	keyText = "Edit.SelectAll";	break;
				}
			}
		} else {
			if( shift ) {
				//	Shift + ̂
				switch( e->key() ) {
				case Qt::Key_Home:	keyText = "Edit.SelToStartOfBlock";	break;
				case Qt::Key_End:	keyText = "Edit.SelToEndOfBlock";	break;
				case Qt::Key_Left:	keyText = "Edit.SelLeft";	break;
				case Qt::Key_Right:	keyText = "Edit.SelRight";	break;
				case Qt::Key_Up:	keyText = "Edit.SelUp";	break;
				case Qt::Key_Down:	keyText = "Edit.SelDown";	break;
				case Qt::Key_Enter:	keyText = "Edit.OpenPrevLine";	break;
				}
			} else {
				//	Cq
				switch( e->key() ) {
				case Qt::Key_Home:	keyText = "Edit.StartOfBlock";	break;
				case Qt::Key_End:	keyText = "Edit.EndOfBlock";	break;
				case Qt::Key_Left:	keyText = "Edit.Left";	break;
				case Qt::Key_Right:	keyText = "Edit.Right";	break;
				case Qt::Key_Up:	keyText = "Edit.Up";	break;
				case Qt::Key_Down:	keyText = "Edit.Down";	break;
				case Qt::Key_Delete:	keyText = "Edit.Delete";	break;
				case Qt::Key_Backspace:	keyText = "Edit.Backspace";	break;
				case Qt::Key_PageUp:	keyText = "Edit.PageUp";	break;
				case Qt::Key_PageDown:	keyText = "Edit.PageDown";	break;
				case Qt::Key_Tab:	break;		//	ŏ
				default: keyText = "Edit.InsNormalChar"; break;
				}
			}
		}
	}
	incCmdStatistics(keyText);
	QTextCursor cur = textCursor();
	if( e->key() == Qt::Key_Return ) {
		if( e->modifiers() == Qt::NoModifier ) {
			incCmdStatistics("Edit.InsEnter");
			QString text =  "\n" + autoIndentText(textCursor());
			cur.beginEditBlock();
			cur.insertText(text);
			//	J[\̋󔒗ނ폜
			//cur = textCursor();
			QTextBlock block = cur.block();
			QString blockText = block.text();
			int pos = cur.positionInBlock();		//	sJ[\ʒu
			int ix = pos;
			while( ix < blockText.length() && blockText[ix].isSpace() )
				++ix;
			if( ix > pos ) {
				cur.movePosition(QTextCursor::Right, QTextCursor::KeepAnchor, ix - pos);
				cur.deleteChar();
			}
			cur.endEditBlock();
			setTextCursor(cur);
			return;
		}
		if( e->modifiers() == Qt::ShiftModifier ) {
			incCmdStatistics("Edit.openLineAbove");
			openLineAbove();
			return;
		}
	}
	if( e->key() == Qt::Key_Tab ) {
		if( e->modifiers() == Qt::NoModifier ) {
			if( cur.hasSelection() ) {
				shiftRight();
				return;
			}
			if( settings()->m_softTab ) {
				if( cur.hasSelection() )
					cur.deleteChar();
				const int offset = cur.positionInBlock();	//	s̃ItZbg
				const int n = settings()->m_tabWidth - offset % settings()->m_tabWidth;
				cur.insertText(QString(n, QChar(' ')));
				return;
			}
		} /*else if( e->modifiers() == Qt::ShiftModifier ) {		//	Shift + Tab
			shiftLeft();
			return;
		}*/
	}
	if( (e->modifiers() & (Qt::ControlModifier|Qt::AltModifier|Qt::MetaModifier)) != 0 ) {
		QTextEdit::keyPressEvent(e);
		return;
	}
	QTextEdit::keyPressEvent(e);		//	}
}
void EditViewBase::wheelEvent(QWheelEvent * event)
{
	const bool ctrl = (event->modifiers() & Qt::ControlModifier) != 0;
	const bool shift = (event->modifiers() & Qt::ShiftModifier) != 0;
	const bool alt = (event->modifiers() & Qt::AltModifier) != 0;
	if( !alt && ctrl && !shift ) {
		incCmdStatistics("Edit.WheelZoom");
		QFont font = this->font();
		int fontSize = font.pointSize();
		if( event->delta() > 0 )
			fontSize = qMin(MAX_FONT_SIZE, fontSize + 1);
		else
			fontSize = qMax(MIN_FONT_SIZE, fontSize - 1);
		font.setPointSize(fontSize);
		setFont(font);
		emit fontSizeChanged(fontSize);
		//onSettingsChanged();
		//update();
		return;
	}
	QTextEdit::wheelEvent(event);
}
void EditViewBase::resizeEvent(QResizeEvent *event)
{
	QTextEdit::resizeEvent(event);
	updateLeftMarginSize();
}
void EditViewBase::updateLeftMarginSize()
{
	QRect r = rect();
	const int leftMgn = leftMarginWidth();
	m_lmWidget->setGeometry(QRect(r.left() + 1, r.top() + 1, leftMgn , r.height() - 2));
	setViewportMargins(leftMgn, 0, 0, 0);
}
void EditViewBase::onSettingsChanged()
{
	setTabStopWidth(fontMetrics().width(QString(m_settings->m_tabWidth, QChar(' '))));
	QPalette plt = palette();
	plt.setColor(QPalette::Base, settings()->color(Settings::BACKGROUND));
	plt.setColor(QPalette::Text, settings()->color(Settings::TEXT));
	plt.setColor(QPalette::Highlight, settings()->color(Settings::SEL_BACKGROUND));
	plt.setColor(QPalette::HighlightedText, settings()->color(Settings::SEL_TEXT));
	setPalette(plt);
	updateLeftMarginSize();
	update();
}
void EditViewBase::jumpToLine(int lineNum)
{
	if( lineNum < 1 || lineNum > document()->lineCount() ) return;
	QScrollBar *vScrollBar = verticalScrollBar();
	const int vpos = vScrollBar->value();
	QTextCursor cur = textCursor();
	cur.movePosition(QTextCursor::Start);
	if( lineNum != 0 ) {
		cur.movePosition(QTextCursor::NextBlock, QTextCursor::MoveAnchor, lineNum - 1);
	}
	setTextCursor(cur);
	if( lineNum != 0 && vpos != vScrollBar->value() ) {
		scrollCurToTop();
#if 0
		QRect rect = cursorRect(cur);
		vScrollBar->setValue(vScrollBar->value() + rect.y());
#endif
	}
	update();
}
void EditViewBase::scrollCurToTop()
{
	QTextCursor cur = textCursor();
	QRect rect = cursorRect(cur);
	QScrollBar *vScrollBar = verticalScrollBar();
	vScrollBar->setValue(vScrollBar->value() + rect.y());
}
bool gotoMatchedParen(QTextCursor &cur)
{
    QTextBlock block = cur.block();
    int blockPos = block.position();
    QString blockText = block.text();
    int ix = cur.position() - blockPos;
    bool forward = true;
    QChar paren, dst;
    while( ix < blockText.length() ) {
        if( blockText[ix] == '{' ) { paren = '{'; dst = '}'; break; }
        if( blockText[ix] == '(' ) { paren = '('; dst = ')'; break; }
        if( blockText[ix] == '[' ) { paren = '['; dst = ']'; break; }
        if( blockText[ix] == '}' ) { paren = '}'; dst = '{'; forward = false; break; }
        if( blockText[ix] == ')' ) { paren = ')'; dst = '('; forward = false; break; }
        if( blockText[ix] == ']' ) { paren = ']'; dst = '['; forward = false; break; }
        ++ix;
    }
    if( paren == QChar() ) return false;
    int count = 1;
    if( forward ) {
        for(;;) {
            while( ++ix < blockText.length() ) {
                if( blockText[ix] == paren )
                    ++count;
                else if( blockText[ix] == dst && !--count ) {
                    cur.setPosition(blockPos + ix);
                    return true;
                }
            }
            block = block.next();
            if( !block.isValid() )
                break;
            blockPos = block.position();
            blockText = block.text();
            ix = -1;
        }
    } else {
        for(;;) {
            while( --ix >= 0 ) {
                if( blockText[ix] == paren )
                    ++count;
                else if( blockText[ix] == dst && !--count ) {
                    cur.setPosition(blockPos + ix);
                    return true;
                }
            }
            block = block.previous();
            if( !block.isValid() )
                break;
            blockPos = block.position();
            blockText = block.text();
            ix = blockText.length();
        }
    }
    return false;
}
void EditViewBase::gotoMatchedParen()
{
	QTextCursor cur = textCursor();
	if( ::gotoMatchedParen(cur) ) {
		setTextCursor(cur);
		update();
	}
}
//	J[\sOɍs}
void EditViewBase::openLineAbove()
{
	QTextCursor cur = textCursor();
	cur.clearSelection();
	if( cur.movePosition(QTextCursor::PreviousBlock) ) {
		setTextCursor(cur);
		openLineBelow();
	} else {	//	1sڂ̏ꍇ
		cur.movePosition(QTextCursor::StartOfBlock);
		cur.insertText("\n");
		cur.movePosition(QTextCursor::PreviousBlock);
		setTextCursor(cur);
	}
#if 0
	cur.movePosition(QTextCursor::StartOfBlock);
	QTextCursor cur2(cur);
	cur2.movePosition(QTextCursor::PreviousBlock);
	QString text = autoIndentText(cur2);
	//setTextCursor(cur);
	cur.insertText(text + "\n");
	cur.movePosition(QTextCursor::PreviousBlock);
	setTextCursor(cur);
#endif
}
//	J[\sɍs}
void EditViewBase::openLineBelow()
{
	QTextCursor cur = textCursor();
	cur.clearSelection();
	cur.movePosition(QTextCursor::EndOfBlock);
	QString text = autoIndentText(cur);
	//setTextCursor(cur);
	cur.insertText("\n" + text);
	//cur.movePosition(QTextCursor::PreviousBlock);
	setTextCursor(cur);
}
//	I͈͍sRg
//	IԂ̏ꍇ́As # }
void EditViewBase::enComment()
{
	QTextCursor cur = textCursor();
	if( !cur.hasSelection() ) {
		cur.movePosition(QTextCursor::StartOfBlock);
		cur.insertText("# ");
		setTextCursor(cur);
	} else {
		cur.beginEditBlock();
		int first = cur.position();
		int last = cur.anchor();
		if( first > last ) qSwap(first, last);	//	make sure first <= last
		cur.setPosition(last);
		//QTextBlock block = cur.block();
		if( cur.positionInBlock() != 0 )		//	sʒuŖꍇ
			cur.movePosition(QTextCursor::NextBlock);
		cur.insertText("=end\n");
		last = cur.position();
		cur.setPosition(first);
		cur.movePosition(QTextCursor::StartOfBlock);
		first = cur.position();
		cur.insertText("=begin\n");
		last += strlen("=begin\n") ;
		cur.endEditBlock();
		cur.setPosition(first);
		cur.setPosition(last, QTextCursor::KeepAnchor);
		setTextCursor(cur);
	}
}
#if 0
//	position pg[N
//	position ̓g[NʒuԂ
QString EditViewBase::getToken(int &position)
{
}
#endif
void EditViewBase::setupCompletionCandidates(QStringList &lst, const QString &text)
{
	QStringList lst2;		//	J[\ȍ~̕⊮
	lst.clear();
	if( text.isEmpty() ) return;
	const int curPosition = textCursor().position();
	QTextBlock block = document()->begin();
	while( block.isValid() ) {
		const QString blockText = block.text();
		int ix = 0;
		while( ix < blockText.length() ) {
			ix = blockText.indexOf(text, ix);
			if( ix < 0 ) break;
			if( !ix || !isLetterOrNumberOrUnderbar(blockText[ix-1]) ) {	//	PꋫȄꍇ
				QString t(text);
				ix += text.length();
				while( ix < blockText.length() && isLetterOrNumberOrUnderbar(blockText[ix]) )
					t += blockText[ix++];
				if( block.position() + ix <= curPosition ) {
					//	J[\ʒuȌꍇ
					int k = lst.indexOf(t);
					if( k >= 0 )
						lst.removeAt(k);	//	d̓J[\ɋ߂D
					lst.push_back(t);
				} else {
					int k = lst2.indexOf(t);
					if( k < 0 )		//	d͓o^Ȃ
						lst2.push_back(t);
				}
			} else
				ix += text.length();
		}
		block = block.next();
	}
	lst += lst2;
}
void EditViewBase::completion(bool keyword)
{
	QTextCursor cur = textCursor() ;
	if( cur.hasSelection() ) {
		cur.clearSelection();
		setTextCursor(cur);
	}
	int first = cur.position();
	QRect rect = cursorRect(cur);
	QPoint p(rect.x() + leftMarginWidth(), rect.bottom());		//	EditView Wn
	QString text;
	QString blockText = cur.block().text();
	int ix = cur.positionInBlock();
#if 0
	if( atJustBehindRegexp(textCursor()) ) {
	} else
#endif
	if( ix > 0 && isLetterOrUnderbar(blockText[ix - 1]) ) {
		cur.movePosition(QTextCursor::StartOfWord, QTextCursor::KeepAnchor);
		if( !cur.hasSelection() )
			cur.movePosition(QTextCursor::PreviousWord, QTextCursor::KeepAnchor);
		first = cur.position();
		text = cur.selectedText();
	}
	QStringList lst;
	if( !keyword ) {
		setupCompletionCandidates(lst, text);
		if( lst.isEmpty() ) return;		//	⊮△
	}
	CompletionWidget cmpl(this, lst, text);
	if( !cmpl.count() ) {
		showMessage(tr("No matched keyword."));
		return;
	}
	cmpl.move(mapToGlobal(p));		//	EditView Wn  O[oWnϊ
	if( cmpl.count() == 1 ) {
		cur.insertText(cmpl.text());
		return;
	}
	const int rc = cmpl.exec();
	switch( rc ) {
	case QDialog::Accepted:
		cur = textCursor();
		cur.setPosition(first, QTextCursor::KeepAnchor);
		cur.insertText(cmpl.text());
#if 0
		if( cmpl.text() == "end" || cmpl.text() == "else" ||
			cmpl.text() == "when" || cmpl.text() == "rescue" )
		{
			revAutoIndent(cmpl.text());
		}
#endif
		break;
	case '\b':
		textCursor().deletePreviousChar();
		break;
	}
}
void EditViewBase::completion()
{
	completion(false);
}
void EditViewBase::keywordCompletion()
{
	completion(true);
}
bool EditViewBase::isMatch(const QString &text)
{
	QTextCursor cur = textCursor();
	QString blockText;
	if( cur.hasSelection() ) {
		blockText = cur.selectedText();
		//	undone K\Ή
		//return text == cur.selectedText();
	} else
		blockText = cur.block().text().mid(cur.positionInBlock());
	if( m_settings->m_regexp ) {
		QString t(text);
		if( t[0] != QChar('^') ) t = "^" + text;
		Qt::CaseSensitivity cs = m_settings->m_ignoreCase ? Qt::CaseInsensitive : Qt::CaseSensitive;
		QRegExp exp(t, cs);
		return exp.indexIn(blockText) == 0;
	} else {
		return blockText.startsWith(text);
	}
}
bool EditViewBase::doFindSub(const QString &text, bool backward)
{
	if( m_settings->m_regexp )
		return findRegexp(text, backward);
	else {
        QTextDocument::FindFlags options = m_settings->m_ignoreCase ?
                    (QTextDocument::FindFlag)0 : QTextDocument::FindCaseSensitively;
		if( m_settings->m_wordSearch )
			options |= QTextDocument::FindWholeWords;
		if( backward )
			options |= QTextDocument::FindBackward;
		return find(text, options);
	}
}
bool EditViewBase::find(const QString &text, QTextDocument::FindFlags options)
{
	const int vpos = verticalScrollBar()->value();
	const bool rc = QTextEdit::find(text, options);
	if( verticalScrollBar()->value() != vpos ) {
		//	XN[ꍇ́AJ[\ʍŏ㕔ɎXN[
		scrollCurToTop();
	}
	return rc;
}
bool EditViewBase::doFind(const QString &text, bool backward, bool noLoop)
{
	QTextCursor cur = textCursor();
	bool rc = doFindSub(text, backward);
	if( rc || noLoop )
		return rc;
	if( m_settings->m_findLoop == Settings::FIND_NO_LOOP ||
        (!backward && cur.atStart()) ||		//	擪珇Ɍꍇ
        (backward && cur.atEnd()) )	//	tɌꍇ
	{
		QApplication::beep();
		emit showMessage(tr("'%1' not found.").arg(text));
		return false;
	}
	if( m_settings->m_findLoop == Settings::FIND_CONFIRM_LOOP &&
		QMessageBox::Yes != QMessageBox::question(0,
							tr("Find Loop"),
							tr("reached begin/end of document.\n"
								"search again from end/begin ?"),
							QMessageBox::Yes | QMessageBox::No) )
	{
		return false;
	}
	QTextCursor c(cur);
	if( !backward )
		c.movePosition(QTextCursor::Start);
	else
		c.movePosition(QTextCursor::End);
	setTextCursor(c);		//	ł̓XN[Ă܂
	rc = doFindSub(text, backward);
	if( !rc ) {
		setTextCursor(cur);		//	̃J[\ʒuɖ߂
		QApplication::beep();
		emit showMessage(tr("'%1' not found.").arg(text));
	}
	return rc;
}
//	K\AIvV Settings Q
bool EditViewBase::findRegexp(const QString &text, bool backward)
{
	Qt::CaseSensitivity cs = m_settings->m_ignoreCase ? Qt::CaseInsensitive : Qt::CaseSensitive;
	QRegExp exp(text, cs);
    QTextDocument::FindFlags options =  backward ?
                        QTextDocument::FindBackward : (QTextDocument::FindFlag)0;

	int vpos = verticalScrollBar()->value();
	QTextCursor cur = document()->find(exp, textCursor(), options);
	if( cur.isNull() )
		return false;
	setTextCursor(cur);
	if( verticalScrollBar()->value() != vpos ) {
		//	XN[ꍇ́AJ[\ʍŏ㕔ɎXN[
		scrollCurToTop();
	}
	return true;
}

