﻿using System;
using System.Collections.Generic;
using System.IO;

using HgCo.WindowsLive.SkyDrive;

namespace SkyDriveFolder
{
    #region イベント情報

    #region サインイン開始イベント情報
    public class SigninBeginEventArgs : EventArgs
    {
        public string ID
        {
            get;
            private set;
        }

        public SigninBeginEventArgs(string id)
        {
            this.ID = id;
        }
    }
    #endregion

    #region サインイン終了イベント情報
    public class SigninEndEventArgs : EventArgs
    {
        public string ID
        {
            get;
            private set;
        }

        public Exception Exception
        {
            get;
            private set;
        }

        public SigninEndEventArgs(string id, Exception exception)
        {
            this.ID = id;
            this.Exception = exception;
        }
    }
    #endregion

    #region リクエスト・フォルダ開始イベント情報
    public class RequestFolderBeginEventArgs : EventArgs
    {
        public SkyDriveFolder ParentFolder
        {
            get;
            private set;
        }

        public RequestFolderBeginEventArgs(SkyDriveFolder parentFolder)
        {
            this.ParentFolder = parentFolder;
        }
    }
    #endregion

    #region リクエスト・フォルダ終了イベント情報
    public class RequestFolderEndEventArgs : EventArgs
    {
        public SkyDriveFolder ParentFolder
        {
            get;
            private set;
        }

        public Exception Exception
        {
            get;
            private set;
        }

        public RequestFolderEndEventArgs(SkyDriveFolder parentFolder, Exception exception)
        {
            this.ParentFolder = parentFolder;
            this.Exception = exception;
        }
    }
    #endregion

    #region ルート・フォルダ作成開始イベント情報
    public class CreateRootFolderBeginEventArgs : EventArgs
    {
        public string FolderName
        {
            get;
            private set;
        }

        public CreateRootFolderBeginEventArgs(string folderName)
        {
            this.FolderName = folderName;
        }
    }
    #endregion

    #region ルート・フォルダ作成終了イベント情報
    public class CreateRootFolderEndEventArgs : EventArgs
    {
        public string FolderName
        {
            get;
            private set;
        }

        public Exception Exception
        {
            get;
            private set;
        }

        public CreateRootFolderEndEventArgs(string folderName, Exception exception)
        {
            this.FolderName = folderName;
            this.Exception = exception;
        }
    }
    #endregion

    #region フォルダ削除開始イベント情報
    public class DeleteFolderBeginEventArgs : EventArgs
    {
        public SkyDriveFolder Folder
        {
            get;
            private set;
        }

        public DeleteFolderBeginEventArgs(SkyDriveFolder folder)
        {
            this.Folder = folder;
        }
    }
    #endregion

    #region フォルダ削除終了イベント情報
    public class DeleteFolderEndEventArgs : EventArgs
    {
        public SkyDriveFolder Folder
        {
            get;
            private set;
        }

        public Exception Exception
        {
            get;
            private set;
        }

        public DeleteFolderEndEventArgs(SkyDriveFolder folder, Exception exception)
        {
            this.Folder = folder;
            this.Exception = exception;
        }
    }
    #endregion

    #region フォルダ選択イベント情報
    public class OpenedFolderEventArgs : EventArgs
    {
        public SkyDriveFolder OpenedFolder
        {
            get;
            private set;
        }

        public OpenedFolderEventArgs(SkyDriveFolder openedFolder)
        {
            this.OpenedFolder = openedFolder;
        }
    }
    #endregion

    #region アップロード・ファイル開始イベント情報
    public class UploadFileBeginEventArgs : EventArgs
    {
        public SkyDriveFolder ParentFolder
        {
            get;
            private set;
        }

        public FileInfo LocalFile
        {
            get;
            private set;
        }

        public UploadFileBeginEventArgs(SkyDriveFolder parentFolder, FileInfo localFile)
        {
            this.ParentFolder = parentFolder;
            this.LocalFile = localFile;
        }
    }
    #endregion

    #region アップロード・ファイル終了イベント情報
    public class UploadFileEndEventArgs : EventArgs
    {
        public SkyDriveFolder ParentFolder
        {
            get;
            private set;
        }

        public FileInfo LocalFile
        {
            get;
            private set;
        }

        public Exception Exception
        {
            get;
            private set;
        }

        public UploadFileEndEventArgs(SkyDriveFolder parentFolder, FileInfo localFile, Exception exception)
        {
            this.ParentFolder = parentFolder;
            this.LocalFile = localFile;
            this.Exception = exception;
        }
    }
    #endregion

    #region アップロード・ファイル進捗イベント情報
    public class UploadFileProgressedEventArgs : EventArgs
    {
        public SkyDriveFolder ParentFolder
        {
            get;
            private set;
        }

        public FileInfo LocalFile
        {
            get;
            private set;
        }

        public long BytesSent
        {
            get;
            private set;
        }

        public long TotalBytesToSent
        {
            get;
            private set;
        }

        public int ProgressPercentage
        {
            get;
            private set;
        }

        public UploadFileProgressedEventArgs(SkyDriveFolder parentFolder, FileInfo localFile, long bytesSent, long totalBytesToSent, int progressPercentage)
        {
            this.ParentFolder = parentFolder;
            this.LocalFile = localFile;
            this.BytesSent = bytesSent;
            this.TotalBytesToSent = TotalBytesToSent;
            this.ProgressPercentage = progressPercentage;
        }
    }
    #endregion

    #endregion

    public class SkyDriveModel
    {
        public event EventHandler<SigninBeginEventArgs> SigninBeginEvent;

        public event EventHandler<SigninEndEventArgs> SigninEndEvent;

        public event EventHandler<RequestFolderBeginEventArgs> RequestFolderBeginEvent;

        public event EventHandler<RequestFolderEndEventArgs> RequestFolderEndEvent;

        public event EventHandler<OpenedFolderEventArgs> OpenedFolderEvent;

        public event EventHandler<CreateRootFolderBeginEventArgs> CreateRootFolderBeginEvent;

        public event EventHandler<CreateRootFolderEndEventArgs> CreateRootFolderEndEvent;

        public event EventHandler<DeleteFolderBeginEventArgs> DeleteFolderBeginEvent;

        public event EventHandler<DeleteFolderEndEventArgs> DeleteFolderEndEvent;

        public event EventHandler<UploadFileBeginEventArgs> UploadFileBeginEvent;

        public event EventHandler<UploadFileEndEventArgs> UploadFileEndEvent;

        public event EventHandler<UploadFileProgressedEventArgs> UploadFileProgressedEvent;

        private string _id;

        private SkyDriveWebClient _client;

        private SkyDriveFolder _rootFolder;

        private SkyDriveFolder _openedFolder;

        private List<ISkyDriveItem> _selectedItemList;

        public SkyDriveModel()
        {
            _client = new SkyDriveWebClient();

            _selectedItemList = new List<ISkyDriveItem>();
        }

        public string ID
        {
            get
            {
                return _id;
            }
        }

        public SkyDriveFolder RootFolder
        {
            get
            {
                return _rootFolder;
            }
        }

        public SkyDriveFolder OpenedFolder
        {
            get
            {
                return _openedFolder;
            }
            set
            {
                _openedFolder = value;

                if (OpenedFolderEvent != null)
                {
                    OpenedFolderEventArgs e = new OpenedFolderEventArgs(_openedFolder);
                    OpenedFolderEvent(this, e);
                }
            }
        }

        public void Signin(string id, string password)
        {
            if (SigninBeginEvent != null)
            {
                SigninBeginEventArgs beginE = new SigninBeginEventArgs(id);
                SigninBeginEvent(this, beginE);
            }

            SigninEndEventArgs endE;

            try
            {
                _client.LogOn(id, password);

                _id = id;

                _rootFolder = new SkyDriveFolder(null);

                endE = new SigninEndEventArgs(id, null);
            }
            catch (Exception ex)
            {
                endE = new SigninEndEventArgs(id, ex);
            }

            if (SigninEndEvent != null)
            {
                SigninEndEvent(this, endE);
            }

            if (endE.Exception != null)
            {
                throw endE.Exception;
            }
        }

        public void RequestFolder(SkyDriveFolder parentFolder)
        {
            if (RequestFolderBeginEvent != null)
            {
                RequestFolderBeginEventArgs beginE = new RequestFolderBeginEventArgs(parentFolder);
                RequestFolderBeginEvent(this, beginE);
            }

            RequestFolderEndEventArgs endE;

            try
            {
                WebFolderItemInfo[] webItems;

                if (parentFolder.Info == null)
                {
                    webItems = _client.ListRootWebFolderItems();
                }
                else
                {
                    webItems = _client.ListSubWebFolderItems(parentFolder.Info);
                }

                parentFolder.LastRequestDateTime = DateTime.Now;
                parentFolder.FolderList.Clear();
                parentFolder.FileList.Clear();

                foreach (WebFolderItemInfo webItem in webItems)
                {
                    if (webItem is WebFolderInfo)
                    {
                        WebFolderInfo webFolder = (WebFolderInfo)webItem;
                        SkyDriveFolder folder = new SkyDriveFolder(webFolder);
                        parentFolder.FolderList.Add(folder);
                    }
                    else if (webItem is WebFileInfo)
                    {
                        WebFileInfo webFile = (WebFileInfo)webItem;
                        SkyDriveFile file = new SkyDriveFile(webFile);
                        parentFolder.FileList.Add(file);
                    }
                    else
                    {
                        throw new ApplicationException(string.Format("予期しないアイテムです。\n{0}", webItem.ToString()));
                    }
                }

                endE = new RequestFolderEndEventArgs(parentFolder, null);
            }
            catch (Exception ex)
            {
                endE = new RequestFolderEndEventArgs(parentFolder, ex);
            }

            if (RequestFolderEndEvent != null)
            {
                RequestFolderEndEvent(this, endE);
            }

            if (endE.Exception != null)
            {
                throw endE.Exception;
            }
        }

        public void UploadFile(SkyDriveFolder parentFolder, List<FileSystemInfo> fsInfoList)
        {
            UploadFileExecutor exe = new UploadFileExecutor(this, parentFolder, fsInfoList);
            exe.Execute();
        }

        private class UploadFileExecutor
        {
            private SkyDriveModel _model;

            private SkyDriveFolder _parentFolder;

            private List<FileSystemInfo> _fsInfoList;

            private FileInfo _currentLocalFile;

            public UploadFileExecutor(SkyDriveModel model, SkyDriveFolder parentFolder, List<FileSystemInfo> fsInfoList)
            {
                _model = model;
                _parentFolder = parentFolder;
                _fsInfoList = fsInfoList;

                _model._client.UploadWebFileProgressChanged += new EventHandler<UploadWebFileProgressChangedEventArgs>(_client_UploadWebFileProgressChanged);
            }

            void _client_UploadWebFileProgressChanged(object sender, UploadWebFileProgressChangedEventArgs e)
            {
                UploadFileProgressedEventArgs eArgs = new UploadFileProgressedEventArgs(_parentFolder, _currentLocalFile, e.BytesSent, e.TotalBytesToSent, e.ProgressPercentage);
                _model.UploadFileProgressedEvent(_model, eArgs);
            }

            public void Execute()
            {
                foreach (FileSystemInfo fsInfo in _fsInfoList)
                {
                    if (fsInfo is FileInfo)
                    {
                        _currentLocalFile = (FileInfo)fsInfo;

                        if (_model.UploadFileBeginEvent != null)
                        {
                            UploadFileBeginEventArgs beginE = new UploadFileBeginEventArgs(_parentFolder, _currentLocalFile);
                            _model.UploadFileBeginEvent(_model, beginE);
                        }

                        UploadFileEndEventArgs endE;

                        try
                        {
                            _model._client.UploadWebFile(_currentLocalFile.FullName, _parentFolder.Info);

                            endE = new UploadFileEndEventArgs(_parentFolder, _currentLocalFile, null);
                        }
                        catch (Exception ex)
                        {
                            endE = new UploadFileEndEventArgs(_parentFolder, _currentLocalFile, ex);
                        }

                        if (_model.UploadFileEndEvent != null)
                        {
                            _model.UploadFileEndEvent(_model, endE);
                        }
                    }
                }
            }
        }

        public void CreateRootFolder(string folderName)
        {
            if (this.CreateRootFolderBeginEvent != null)
            {
                CreateRootFolderBeginEventArgs beginE = new CreateRootFolderBeginEventArgs(folderName);
                this.CreateRootFolderBeginEvent(this, beginE);
            }

            CreateRootFolderEndEventArgs endE;

            try
            {

                this._client.CreateRootWebFolder(folderName, WebFolderItemShareType.Private);

                endE = new CreateRootFolderEndEventArgs(folderName, null);
            }
            catch (Exception ex)
            {
                endE = new CreateRootFolderEndEventArgs(folderName, ex);
            }

            if (this.CreateRootFolderEndEvent != null)
            {
                this.CreateRootFolderEndEvent(this, endE);
            }

            if (endE.Exception != null)
            {
                throw endE.Exception;
            }
        }

        public void DeleteFolder(SkyDriveFolder folder)
        {
            if (this.DeleteFolderBeginEvent != null)
            {
                DeleteFolderBeginEventArgs beginE = new DeleteFolderBeginEventArgs(folder);
                this.DeleteFolderBeginEvent(this, beginE);
            }

            DeleteFolderEndEventArgs endE;

            try
            {
                this._client.DeleteWebFolder(folder.Info);

                endE = new DeleteFolderEndEventArgs(folder, null);
            }
            catch (Exception ex)
            {
                endE = new DeleteFolderEndEventArgs(folder, ex);
            }

            if (this.DeleteFolderEndEvent != null)
            {
                this.DeleteFolderEndEvent(this, endE);
            }

            if (endE.Exception != null)
            {
                throw endE.Exception;
            }
        }
    }

    #region ファイル・フォルダ情報
    public interface ISkyDriveItem
    {
    }

    public class SkyDriveFile : ISkyDriveItem
    {
        public WebFileInfo Info
        {
            get;
            set;
        }

        public DateTime? LastRequestDateTime
        {
            get;
            set;
        }

        public SkyDriveFile(WebFileInfo info)
        {
            this.Info = info;
        }
    }

    public class SkyDriveFolder : ISkyDriveItem
    {
        public WebFolderInfo Info
        {
            get;
            set;
        }

        public DateTime? LastRequestDateTime
        {
            get;
            set;
        }

        public List<SkyDriveFolder> FolderList
        {
            get;
            private set;
        }

        public List<SkyDriveFile> FileList
        {
            get;
            private set;
        }

        public SkyDriveFolder(WebFolderInfo info)
        {
            this.Info = info;
            this.FolderList = new List<SkyDriveFolder>();
            this.FileList = new List<SkyDriveFile>();
        }
    }
    #endregion
}
