/*!
******************************************************************************

	@file	cpu.cpp

	Copyright (C) 2008-2009 Vsun86 Development Project. All rights reserved.

******************************************************************************
*/

#include "vsun86.h"
#include "printf.h"
#include "atomic.h"
#include "cpu.h"

static volatile int  cpu_intflag_access;
static volatile bool cpu_interrupt_flag;

static CPU_CONTEXT cpu_context[CPU_CONTEXT_MAX];
static u32 cpu_context_used;

bool cpu_init( void )
{
#ifndef _VSUN86_PCSIM
	if ( !x86_init() )
		return false;
#endif	//!_VSUN86_PCSIM

	cpu_intflag_access	= 0;
	cpu_interrupt_flag	= false;
	cpu_context_used	= 0;

	return true;
}

void cpu_enable_interrupt( void )
{
	lock( &cpu_intflag_access );
	{
		if ( cpu_interrupt_flag ) {
			vmm_printf( VMM_DEBUG, "cpu: cpu_enable_interrupt() ignored. (interrupt_flag == 1)\n" );
		}
		else {
#ifndef _VSUN86_PCSIM
			X86_STI();
#endif	//_VSUN86_PCSIM
			cpu_interrupt_flag = true;
		}
	}
	unlock( &cpu_intflag_access );
}

void cpu_disable_interrupt( void )
{
	lock( &cpu_intflag_access );
	{
		if ( !cpu_interrupt_flag ) {
			vmm_printf( VMM_DEBUG, "cpu: cpu_disable_interrupt() ignored. (interrupt_flag == 0)\n" );
		}
		else {
#ifndef _VSUN86_PCSIM
			X86_CLI();
#endif	//_VSUN86_PCSIM
			cpu_interrupt_flag = false;
		}
	}
	unlock( &cpu_intflag_access );
}

bool cpu_get_interrupt_flag( void )
{
	bool flag;

	lock( &cpu_intflag_access );
	{
		flag = cpu_interrupt_flag;
	}
	unlock( &cpu_intflag_access );

	return flag;
}

void halt( void )
{
#ifndef _VSUN86_PCSIM
	X86_HLT();
#else	//_VSUN86_PCSIM
	vmm_printf( VMM_ERROR, "halt()\n" );
#endif	//_VSUN86_PCSIM
}

CPU_CONTEXT * cpu_alloc_context( const void *code, const void *stack, size_t stack_size )
{
	if ( cpu_context_used >= CPU_CONTEXT_MAX )
		return NULL;

	CPU_CONTEXT *context = &cpu_context[cpu_context_used];

#ifndef _VSUN86_PCSIM
	context->tss = x86_alloc_tss( code, stack, stack_size );
	if ( context->tss == NULL )
		return NULL;
#else	//_VSUN86_PCSIM
	memset( context, 0, sizeof(CPU_CONTEXT) );
	context->code_ptr	= code;
	context->stack_ptr	= ((u8 *)stack) + stack_size - 4;
#endif

	cpu_context_used++;
	return context;
}

void cpu_switch_context( CPU_CONTEXT *old_context, CPU_CONTEXT *new_context )
{
#ifndef _VSUN86_PCSIM
	x86_switch_task( old_context->tss, new_context->tss );
#else	//_VSUN86_PCSIM
#ifdef	_MSC_VER
	__asm {
		mov		eax, old_context;
		mov		CPU_CONTEXT[eax]._r1, ecx;
		mov		CPU_CONTEXT[eax]._r2, edx;
		mov		CPU_CONTEXT[eax]._r3, ebx;
		mov		CPU_CONTEXT[eax]._r5, ebp;
		mov		CPU_CONTEXT[eax]._r6, esi;
		mov		CPU_CONTEXT[eax]._r7, edi;
		mov		CPU_CONTEXT[eax].code_ptr,  offset task_switch_return;
		mov		CPU_CONTEXT[eax].stack_ptr, esp;
		pushfd	;
		pop		CPU_CONTEXT[eax].flags;
		mov		eax, new_context;
		mov		ecx, CPU_CONTEXT[eax]._r1;
		mov		edx, CPU_CONTEXT[eax]._r2;
		mov		ebx, CPU_CONTEXT[eax]._r3;
		mov		esp, CPU_CONTEXT[eax].stack_ptr;
		mov		ebp, CPU_CONTEXT[eax]._r5;
		mov		esi, CPU_CONTEXT[eax]._r6;
		mov		edi, CPU_CONTEXT[eax]._r7;
		push	CPU_CONTEXT[eax].flags;
		popfd	;
		jmp		CPU_CONTEXT[eax].code_ptr;
task_switch_return:
	}
#else	//!_MSC_VER
	register CPU_CONTEXT *_old = old_context;
	register CPU_CONTEXT *_new = new_context;
	__ASM__(
		"movl	$task_switch_return,  %[old_eip]	\n\t"
		"pushfl										\n\t"
		"popl	%[old_eflags]						\n\t"
		"movl	%%ecx, %[old_ecx]					\n\t"
		"movl	%%edx, %[old_edx]					\n\t"
		"movl	%%ebx, %[old_ebx]					\n\t"
		"movl	%%esp, %[old_esp]					\n\t"
		"movl	%%ebp, %[old_ebp]					\n\t"
		"movl	%%esi, %[old_esi]					\n\t"
		"movl	%%edi, %[old_edi]					\n\t"
		"pushl	%[new_eflags]						\n\t"
		"popfl										\n\t"
		"movl	%[new_esp], %%esp					\n\t"
		"pushl	%[new_eip]							\n\t"
		"pushl	%[new_ecx]							\n\t"
		"pushl	%[new_edx]							\n\t"
		"pushl	%[new_ebx]							\n\t"
		"pushl	%[new_esp]							\n\t"
		"pushl	%[new_ebp]							\n\t"
		"pushl	%[new_esi]							\n\t"
		"pushl	%[new_edi]							\n\t"
		"popal										\n\t"
		"jmp	*%%eax								\n\t"
"task_switch_return:								\n\t"
		:	  [old_eip]		"=o"(_old->code_ptr)
			, [old_eflags]	"=o"(_old->flags)
			, [old_ecx]		"=o"(_old->_r1)
			, [old_edx]		"=o"(_old->_r2)
			, [old_ebx]		"=o"(_old->_r3)
			, [old_esp]		"=o"(_old->stack_ptr)
			, [old_ebp]		"=o"(_old->_r5)
			, [old_esi]		"=o"(_old->_r6)
			, [old_edi]		"=o"(_old->_r7)
		:	  [new_eip]		"a" (_new->code_ptr)
			, [new_eflags]	"o" (_new->flags)
			, [new_ecx]		"o" (_new->_r1)
			, [new_edx]		"o" (_new->_r2)
			, [new_ebx]		"o" (_new->_r3)
			, [new_esp]		"o" (_new->stack_ptr)
			, [new_ebp]		"o" (_new->_r5)
			, [new_esi]		"o" (_new->_r6)
			, [new_edi]		"o" (_new->_r7)
		:	"memory", "cc"
	);
#endif	//!_MSC_VER
#endif	//_VSUN86_PCSIM
}
